#!/usr/bin/env python
############################################################
# STAC.py
# Maintained by S.Brockhauser
# 7th Oct 2005
# 
# A wrapper for the strategy program STAC (Brockhauser)
# 
# 
# $Id: STAC.py,v 1.4 2006/06/22 06:57:20 sudol Exp $
# 
############################################################

# generic includes

import os, sys

import time

# check the DNA set up

if os.environ.has_key('DNAHOME'):
    dna = os.environ['DNAHOME']
else:
    raise RuntimeError, 'DNAHOME not configured'

sys.path.append(dna + '/scheduler/Scheduler/Driver')

import Driver, Output

sys.path.append(dna + '/xsd/python')
import XSD

# access a few singleton classes
from Somewhere import Somewhere
from Messenger import Messenger

# how to run STAC
# stac.module {params}

def GetDetectorName():
    if False and os.environ['DNANAME'] == 'id29':
        return 'd-id29'
    # add more here for other beamlines
    else:
        if not os.environ.has_key('DNA_DETECTOR_NAME'):
            return None

    detector = os.environ['DNA_DETECTOR_NAME']

    return detector

class STAC(Driver.Driver):

    def __init__(self):
        Driver.Driver.__init__(self)

        #test to run an arbitrary executable that accepts input
        self.setExecutable('java')

        self.resolution = None
        self.detector = 'adsc'
        self.exposure_time = 1.0
        self.dat = ''
        self.par = ''
        self.hkl = []
        self.segments = None
        self.interactive = False
        self.strategy = None
        self.diameter = None

    def setStrategy(self, start, end, width, time):
        self.strategy = (start, end, width, time)
        self.interactive = True

    def setSegments(self, segments):
        self.segments = segments

    def addHkl(self, hkl):
        self.hkl.append(hkl)

    def setPar(self, par):
        self.par = par

    def setDat(self, dat):
        self.dat = dat

    def setExposure_time(self, time):
        self.exposure_time = time

    def setDetector(self, detector, diameter = None):
        self.detector = detector
        self.diameter = diameter

        if diameter and detector == 'mar':
            intdiameter = int(10 * diameter)
            self.detector = self.detector + str(intdiameter)

        if GetDetectorName():
            self.detector = GetDetectorName()
            Messenger.log_write('Using DNANAME %s detector is %s' \
                                % (os.environ['DNANAME'], self.detector))
        else:
            Messenger.log_write('Detector type for best is %s' % self.detector)

    def setResolution(self, resolution):
        self.resolution = resolution

    def printMatrix(selfself,matrix):
        return ''+ \
            matrix.getE11()+'\n'+matrix.getE12()+'\n'+matrix.getE13()+'\n'+ \
            matrix.getE21()+'\n'+matrix.getE22()+'\n'+matrix.getE23()+'\n'+ \
            matrix.getE31()+'\n'+matrix.getE32()+'\n'+matrix.getE33()
            
    def printCell(selfself,cell):
        return ''+ \
            cell.getA()+'\n'+cell.getB()+'\n'+cell.getC()+'\n'+ \
            cell.getAlpha()+'\n'+cell.getBeta()+'\n'+cell.getGamma()
            

    def kappa_alignment(self,alignment_request,autoindexResult,kappaSettings):
        if not self.getWorkingDirectory():
            actWorkingDir = ""
        else:
            actWorkingDir = self.getWorkingDirectory() + '/'
        #
        # bestfile.par
        #
        try:
            import shutil
            import os
            bestfile=os.path.realpath(os.path.join(actWorkingDir,os.pardir,'integrate_1','bestfile.par'))
            bestf2=os.path.realpath(os.path.join(actWorkingDir,'bestfile.par'))
            shutil.copy(bestfile,bestf2)
        except:
            Messenger.log_write('Error: Could not copy bestfile.par')
        #
        # GUI request
        #
        #Messenger.log_write(actWorkingDir+'DNA_kappa_alignment_request')
        #passing the desired orientation        
        f=open(actWorkingDir+'DNA_kappa_alignment_request','w')
        f.write(alignment_request.marshal())
        f.close()
        #desiredOrientations=alignment_request.getDesired_orientation()
        ##Messenger.log_write('Desired Orientations received (%d): ' % (desiredOrientations.__len__()))
        #self.input(desiredOrientations.__len__())
        #for desOrientElem in desiredOrientations:
        #    #Messenger.log_write('v1: %s; v2: %s' % (desOrientElem.getV1(), desOrientElem.getV2()))
        #    self.input(desOrientElem.getV1())
        #    self.input(desOrientElem.getV2())
        #
        # DNA data
        #
        #Messenger.log_write(actWorkingDir+'DNA_STAC_Index_response')
        #TODO: comment for faster debugging
        f=open(actWorkingDir+'DNA_STAC_Index_response','w')
        f.write(autoindexResult.marshal())
        f.close()        
        #END of commented section
        #passing the OM
        #passing the cellparams
        #passing the symmetry
        #passing the wavelength
        #passing the detector distance
        #passing the image name template
        #passing ???
        f=open(actWorkingDir+'DNA_STAC_Kappa_Settings','w')
        if not kappaSettings is None:
            f.write(kappaSettings.marshal())
        f.close()
        

        command = os.getenv('STAC_PARAMETERS')+' stacgui.STAC_DNA_listener kappa_alignment -'+actWorkingDir
        #command = os.environ.get('STAC_PARAMETERS')+' stacgui.STAC_DNA_listener kappa_alignment -'+actWorkingDir
        #command = '-DSTACDIR=$STACDIR -DSPECDEF=$SPECDEF -DBCMDEF=$BCMDEF'+' stacgui.STAC_DNA_listener kappa_alignment -'+actWorkingDir
        self.start(command)
        #simulating?! the calculation
        #time.sleep(5)
        # RUN control
        self.close()
        while 1:
            line = self.output()
            if not line:
                break
        self.kill()

        #
        # parse and gather up the output...
        #
        #Convert an output file into DNA
        f=open(actWorkingDir+'STAC_DNA_kappa_alignment_response','r')
        solStr=f.read()
        f.close
        strategy_response = XSD.Kappa_alignment_response()
        strategy_response.unmarshal(solStr)
        # add extra things
        #status = XSD.Status()
        #status.setCode('ok')
        #strategy_response.setStatus(status)
        Messenger.log_write(strategy_response.getComment())
        #send the result back
        return strategy_response
        
    def kappa_strategy(self,strategy_request,autoindexResult):
        if not self.getWorkingDirectory():
            actWorkingDir = ""
        else:
            actWorkingDir = self.getWorkingDirectory() + '/'
        
        #
        # GUI request
        #
        #Messenger.log_write(actWorkingDir+'DNA_kappa_alignment_request')
        #passing the desired orientation        
        f=open(actWorkingDir+'DNA_kappa_strategy_request','w')
        f.write(strategy_request.marshal())
        f.close()
        #desiredOrientations=alignment_request.getDesired_orientation()
        ##Messenger.log_write('Desired Orientations received (%d): ' % (desiredOrientations.__len__()))
        #self.input(desiredOrientations.__len__())
        #for desOrientElem in desiredOrientations:
        #    #Messenger.log_write('v1: %s; v2: %s' % (desOrientElem.getV1(), desOrientElem.getV2()))
        #    self.input(desOrientElem.getV1())
        #    self.input(desOrientElem.getV2())
        #
        # DNA data
        #
        #Messenger.log_write(actWorkingDir+'DNA_STAC_Index_response')
        #TODO: comment for faster debugging
        f=open(actWorkingDir+'DNA_STAC_Index_response','w')
        f.write(autoindexResult.marshal())
        f.close()
        #END of commented section
        #passing the OM
        #passing the cellparams
        #passing the symmetry
        #passing the wavelength
        #passing the detector distance
        #passing the image name template
        #passing ???        

        command = os.getenv('STAC_PARAMETERS')+' stacgui.STAC_DNA_listener kappa_strategy -'+actWorkingDir
        #command = os.environ.get('STAC_PARAMETERS')+' stacgui.STAC_DNA_listener kappa_alignment -'+actWorkingDir
        #command = '-DSTACDIR=$STACDIR -DSPECDEF=$SPECDEF -DBCMDEF=$BCMDEF'+' stacgui.STAC_DNA_listener kappa_alignment -'+actWorkingDir
        self.start(command)
        #simulating?! the calculation
        #time.sleep(5)
        # RUN control
        self.close()
        while 1:
            line = self.output()
            if not line:
                break
        self.kill()

        #
        # parse and gather up the output...
        #
        #Convert an output file into DNA
        f=open(actWorkingDir+'STAC_DNA_kappa_strategy_response','r')
        solStr=f.read()
        f.close
        strategy_response = XSD.Kappa_strategy_response()
        strategy_response.unmarshal(solStr)
        # add extra things
        #status = XSD.Status()
        #status.setCode('ok')
        #strategy_response.setStatus(status)
        Messenger.log_write(strategy_response.getComment())
        #send the result back
        return strategy_response
        
    def kappa_collect_settings(self,kappaSettings):
        if not self.getWorkingDirectory():
            actWorkingDir = ""
        else:
            actWorkingDir = self.getWorkingDirectory() + '/'
        
        #
        # GUI request
        #
        #Messenger.log_write(actWorkingDir+'DNA_kappa_alignment_request')
        #passing the desired orientation        
        f=open(actWorkingDir+'DNA_kappa_collect_settings_request','w')
        f.write(kappaSettings.marshal())
        f.close()

        command = os.getenv('STAC_PARAMETERS')+' stacgui.STAC_DNA_listener kappa_collect_settings -'+actWorkingDir
        self.start(command)
        # RUN control
        self.close()
        while 1:
            line = self.output()
            if not line:
                break
        self.kill()

        #
        # parse and gather up the output...
        #
        #Convert an output file into DNA
        f=open(actWorkingDir+'STAC_DNA_kappa_collect_settings_status','r')
        solStr=f.read()
        f.close
        kappa_collect_settings_status = XSD.Status()
        kappa_collect_settings_status.unmarshal(solStr)
        # add extra things
        Messenger.log_write(kappa_collect_settings_status.getMessage())
        #send the result back
        return kappa_collect_settings_status

    def kappa_settings_request(self):
        if not self.getWorkingDirectory():
            actWorkingDir = ""
        else:
            actWorkingDir = self.getWorkingDirectory() + '/'
        
        if not (os.getenv('STAC_PARAMETERS') is None):
            command = os.getenv( 'STAC_PARAMETERS' )+' stacgui.STAC_DNA_listener kappa_settings_request -'+actWorkingDir
            self.start( command )
            # RUN control
            self.close()
            while 1:
                line = self.output()
                if not line:
                    break
            self.kill()
    
            #
            # parse and gather up the output...
            #
            #Convert an output file into DNA
            f=open( actWorkingDir+'STAC_DNA_kappa_settings_response', 'r' )
            solStr=f.read()
            f.close
            kappa_settings_response = XSD.Kappa_collect_settings()
            kappa_settings_response.unmarshal( solStr )
            # add extra things
            Messenger.log_write( kappa_settings_response.getComment() )
            #send the result back
            return kappa_settings_response

    def run(self):

        # check here

        # delete any BEST strategy stuff in Somewhere
        Somewhere.store('STAC_output', None)


        # look at the environment for a DNA_DETECTOR_NAME

        if os.environ.has_key('DNA_DETECTOR_NAME'):
            old_detector = self.detector
            
            self.detector = os.environ['DNA_DETECTOR_NAME']

            Messenger.log_write('Obtained detector name %s' % \
                                self.detector, 2)
            Messenger.log_write('Overridden from %s' % old_detector, 2)
            
        elif GetDetectorName():
            self.detector = GetDetectorName()
        command = '-f %s -t %s ' % (self.detector, self.exposure_time)
        if (self.resolution):
            command += '-r %f ' % self.resolution
        if self.segments:
            command += '-e %s ' % (str(self.segments))
        if self.interactive:
            command += '-i '
        command += '-mos -dna best.stf %s %s ' % (self.dat, self.par)
        for hkl in self.hkl:
            command += '%s ' % hkl

        self.start(command)

        if (self.strategy):
            # self.resolution = 1.5
            self.input('phi %s %s' % (str(self.strategy[0]), \
                                      str(self.strategy[1])))
            self.input('width %s' % (str(self.strategy[2])))
            if self.diameter:
                self.input('constant %s %s %s %f' % \
                           (str(self.resolution), str(self.strategy[2]),
                            str(self.strategy[3]), self.diameter))
            else:
                self.input('constant %s %s %s' % \
                           (str(self.resolution), str(self.strategy[2]),
                            str(self.strategy[3])))
            self.input('stats')

        self.close()


        while 1:
            line = self.output()

            if not line:
                break


        self.kill()

        # parse and gather up the output...

        AlignmentFilename='STAC_filtered_align.vec'
        StrategyFilename='strategies.txt'
        if not self.getWorkingDirectory():
            output = Output.Output('best.stf')
            outputAlignment = Output.Output(AlignmentFilename)
            outputStrategy = Output.Output(StrategyFilename)
        else:
            output = Output.Output(self.getWorkingDirectory() + '/best.stf')
            outputAlignment = Output.Output(self.getWorkingDirectory() + '/'+ AlignmentFilename)
            outputStrategy = Output.Output(self.getWorkingDirectory() + '/'+ StrategyFilename)

        if not self.strategy:
            for j in output.searchforlist('general_inform', 'short_range_'):
                i = int(j)

        # do not want to print this since it is misleading!
        if 1 == 2:
            if 1 == 2:
                Messenger.log_write('Data collection %d start %s end %s' % \
                                    (i, output.get('general_inform',
                                                   'short_range_%d' % i,
                                                   'phi_start')[0],
                                     output.get('general_inform',
                                                'short_range_%d' % i,
                                                'phi_end')[0]))

        Messenger.log_write('Summary of statistics')

        if not self.strategy:
            Messenger.log_write(
                'Fraction of achievable reflections measured: %s%%' % \
                (output.get('general_inform', 'rel_achievable_refl_1', \
                            'fraction_achievable_%')[0]))
            
        else:
            Messenger.log_write(
                'Fraction of achievable reflections measured: %s%%' % \
                (output.get('statistical_prediction',
                            'user_data_collection_strategy_1',
                            'completeness')[0]))
            
        # bug 821 reported by gleb
        # this only works if using BEST strategy else need above change!
        # output.get('general_inform', 'rel_achievable_refl_1', \
        # 'fraction_achievable_%')[0])


        if not self.strategy:
            Messenger.log_write('Redundancy: %s' %
                                (output.get('data_collection_strategy',
                                            'summary_1', 'redundancy')[0]))
        else:
            Messenger.log_write('Redundancy: %s' %
                                (output.get('statistical_prediction',
                                            'user_data_collection_strategy_1',
                                            'redundancy')[0]))

        if not self.strategy:
            Messenger.log_write('Data collection run info')
            line = '----------------------------------------'
            Messenger.log_write(line)
            Messenger.log_write('Enter this information to use the BEST ' + \
                                'strategy')
            Messenger.log_write('Start   Width   Time    Number  1st Image')

        runs = []
        if not self.strategy:
            for j in output.searchforlist('data_collection_strategy',
                                          'collection_run_'):
                i = int(j)
                runs.append(i)
            
            runs.sort()

            # this seems to be broken in the BEST output... in some cases
            # fixme this needs to be investigated!
            if 1 == 1:
                Somewhere.store('BEST exposure time', \
                                output.get('data_collection_strategy', \
                                           'collection_run_%d' % i, \
                                           'exposure_time')[0])
            else:
                # will need to be
                keys = output.searchforlist('dc_optimal_time',
                                            'compl_time_vs_resolution_')
                
                indexes = []
                for k in keys:
                    indexes.append(int(k))
                indexes.sort()

                # now work through to find the right record of resolution
                index = 0
                for i in indexes:
                    index = i
                    if float(output.get('dc_optimal_time',
                                        'compl_time_vs_resolution_%d' % \
                                        i,
                                        'resolution_max')[0]) < \
                                        self.resolution:
                        break
                high = index
                exposure_time = float(
                    output.get('dc_optimal_time',
                               'compl_time_vs_resolution_%d' % \
                               high,
                               'ex_time_i_sigma_3')[0])
                Somewhere.store('BEST exposure time', exposure_time)
                
            for i in runs:
                Messenger.log_write(
                    '%8s%8s%8s%8s    1' % \
                    (output.get('data_collection_strategy',
                                'collection_run_%d' % i, 'phi_start')[0],
                     output.get('data_collection_strategy',
                                'collection_run_%d' % i, 'phi_width')[0],
                     output.get('data_collection_strategy',
                                'collection_run_%d' % i, 'exposure_time')[0],
                     output.get('data_collection_strategy',
                                'collection_run_%d' % i, \
                                'number_of_images')[0]))
                    
            Messenger.log_write(line)
            
        bins = []
        for j in output.searchforlist('statistical_prediction', \
                                      'resolution_bin_'):
            i = int(j)
            bins.append(i)
            
        bins.sort()

        # take off the last bin
        bin = bins[-1]
        bins = bins[0:-1]

        # these information need to be stored in such a way that the
        # results can be recovered for writing on a web page - and
        # making graphs of.
        
        Messenger.log_write('%s\t%s\t%s\t%s\t%s' %
                            ('Min', 'Max', 'I/sig', 'R', 'Overload'))
        for i in bins:
            Messenger.log_write(
                '%s\t%s\t%s\t%s\t%s' % \
                (output.get('statistical_prediction', \
                            'resolution_bin_%d' % i, 'min_resolution')[0],
                 output.get('statistical_prediction', \
                            'resolution_bin_%d' % i, 'max_resolution')[0],
                 output.get('statistical_prediction', \
                            'resolution_bin_%d' % i, \
                            'average_i_over_sigma')[0],
                 output.get('statistical_prediction', \
                            'resolution_bin_%d' % i, 'R_factor')[0],
                 output.get('statistical_prediction', \
                            'resolution_bin_%d' % i, 'fract_overload')[0]))
                
        Messenger.log_write('Overall')
        Messenger.log_write('%s\t%s\t%s\t%s\t%s' %
                            ('Min', 'Max', 'I/sig', 'R', 'Overload'))
        i = bin
        Messenger.log_write(
            '%s\t%s\t%s\t%s\t%s' % \
            (output.get('statistical_prediction', \
                        'resolution_bin_%d' % i, 'min_resolution')[0],
             output.get('statistical_prediction', \
                        'resolution_bin_%d' % i, 'max_resolution')[0],
             output.get('statistical_prediction', \
                        'resolution_bin_%d' % i, \
                        'average_i_over_sigma')[0],
             output.get('statistical_prediction', \
                        'resolution_bin_%d' % i, 'R_factor')[0],
             output.get('statistical_prediction', \
                        'resolution_bin_%d' % i, 'fract_overload')[0]))

        Somewhere.store('BEST_output', output)

        return Strategy_response(output)

def Kappa_strategy_response(): #output):
    '''Convert an output file into DNA'''

    strategy_response = XSD.Kappa_strategy_response()
    #checking the expert status report by simulating a calculation
    time.sleep(2)
    status = XSD.Status()
    status.setCode('ok')
    strategy_response.setStatus(status)
    strategy_response.setComment('Strategy sent by STAC.py')

    return strategy_response
    
def Strategy_response(output):
    '''Convert an output best.stf file into DNA'''

    # check for overloaded reflections in the strategy - if there are 
    # overloaded reflectiosn then we need to reduce the exposure time
    # - maybe by a factor of two...

    bins = []
    for j in output.searchforlist('statistical_prediction', \
                                  'resolution_bin_'):
        i = int(j)
        bins.append(i)
        
    bins.sort()
        
    # take off the last bin
    bin = bins[-1]
    overload = float(output.get('statistical_prediction', \
                                'resolution_bin_%d' % bins[0], \
                                'fract_overload')[0])

    # this is in percent!
    if overload > 1.00:
        strategy_response = XSD.Strategy_response()
        status = XSD.Status()
        status.setCode('error')
        status.setMessage('Detector overloaded %f' % overload)
        strategy_response.setStatus(status)
        return strategy_response


    strategy_response = XSD.Strategy_response()
    status = XSD.Status()
    status.setCode('ok')
    strategy_response.setStatus(status)
    completeness = XSD.Completeness()
    completeness.setStandard(output.get('general_inform', \
                                        'rel_achievable_refl_1', \
                                        'fraction_achievable_%')[0])
    strategy_response.setCompleteness(completeness)

    runs = []

    for j in output.searchforlist('data_collection_strategy',
                                  'collection_run_'):
        i = int(j)
        runs.append(i)
            
    runs.sort()

    # we should have exactly one "run" at the moment!
    if len(runs) > 0:
        i = runs[0]
        start = float(output.get('data_collection_strategy',
                                 'collection_run_%d' % i, 'phi_start')[0])
        # this is really an int but...
        number = float(output.get('data_collection_strategy',
                                  'collection_run_%d' % i, 'number_of_images')[0])
        width = float(output.get('data_collection_strategy',
                                 'collection_run_%d' % i, 'phi_width')[0])
    else:
        # we put in a strategy
        start = float(output.get('statistical_prediction',
                                 'user_data_collection_strategy_1', \
                                 'phi_start')[0])
        number = float(output.get('statistical_prediction',
                                  'user_data_collection_strategy_1', \
                                  'number_of_images')[0])
        end = float(output.get('statistical_prediction',
                               'user_data_collection_strategy_1', \
                               'phi_end')[0])
        width = (end - start) / number

    end = start + (number * width)

    strategy_summary = XSD.Strategy_summary()
    strategy_summary.setNumber_of_segments(1)
    segment = XSD.Segment()
    oscillation = XSD.Oscillation_sequence()
    prediction = XSD.Predicted_spots()

    oscillation.setStart(start)
    oscillation.setRange(end - start)
    oscillation.setNumber_of_images(number)

    # this is in the statistics... maybe I should get it
    prediction.setFull(0.0)
    prediction.setOverlap(0.0)
    
    segment.setPredicted_spots(prediction)
    segment.setOscillation_sequence(oscillation)
    
    strategy_response.addSegment(segment)
    strategy_summary.addSegment(segment)
    strategy_response.addStrategy_summary(strategy_summary)

    return strategy_response

if __name__ == '__main__':

    Check()

    argv = sys.argv

    if len(argv) == 1:

        print 'Testing with default settings'
        
        b = STAC()
        
        hkl1 = '/tmp/beer/integrate_1/bestfile.hkl'
        hkl91 = '/tmp/beer/integrate_91/bestfile.hkl'
        par = '/tmp/beer/integrate_1/bestfile.par'
        dat = '/tmp/beer/autoindex_1/bestfile.dat'
        
        b.setDat(dat)
        b.setPar(par)
        b.addHkl(hkl1)
        b.addHkl(hkl91)
        
        b.run()

        b.setResolution(1.5)
        b.setStrategy(0, 360, 0.1)
        
        b.run()

    else:
        print 'Now testing interactive'
        
        b = STAC()

        hkl1 = '/tmp/beer/integrate_1/bestfile.hkl'
        hkl91 = '/tmp/beer/integrate_91/bestfile.hkl'
        par = '/tmp/beer/integrate_1/bestfile.par'
        dat = '/tmp/beer/autoindex_1/bestfile.dat'
        
        b.setDat(dat)
        b.setPar(par)
        b.addHkl(hkl1)
        b.addHkl(hkl91)
        
        b.setSegments(int(argv[1]))
        
        b.run()
