#!/usr/bin/env python
############################################################
# Processor.py
# Maintained by G.Winter
# A script to automatically process an existing data set.
# This will just look at the files which are available and
# attempt to process them using the DNA automation core.
# 5th January 2005
# 
# This should be run as a "main" routine, viz:
# 
# ./Processor.py [-directory /data/graeme/images] \
#                [-template foo_1_###.bar] \
#                [-image /data/graeme/images/foo_1_001.bar] \
#                [-start 1 -end 15] \
#                [-wavelength 0.979] \
#                [-beam 81.0 81.0] \
#                [-maxcell 150.0] \
#                [-mad] \
#                [-residues 123] \
#                [-label peak] \
#                [-do (strategy|process)]
#                [-spacegroup p2] \
#                [-nonullpix] \
#                [-norlimit]
# 
# $Id: Processor.py,v 1.3 2005/11/22 13:38:15 svensson Exp $
# 
############################################################

import os, sys
import re, string, time, exceptions, traceback

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

dna = os.environ['DNAHOME']
sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')
sys.path.append(dna + '/scheduler/Scheduler/CCP4')
sys.path.append(dna + '/scheduler/DiffractionImage/lib')

sys.path.append(dna + '/expertise/python/graeme/Expert')

import XSD
import MosflmInterface
import Mosflm
import MosflmLog
import CCP4Interface
import CCP4Log
import Screen
import Expert
import Bits
import Reindex
import ProcessorBits

from Messenger import Messenger
from Somewhere import Somewhere

if __name__ != '__main__':
    raise RuntimeError, 'Module "Processor.py" not designed for library use'

do_index = True
do_single_integrate = True
do_strategy = True
do_refine = True
do_integrate = True
do_scale = True
do_truncate = True

do_anomalous = False
do_nullpix = True
do_rlimit = True

do_sdadd = True
do_radiation = True

max_cell = None

template = None
directory = None

residues = None

args = sys.argv

start_at = None
end_at = None

beam = None
wavelength = None

spacegroup = ''

for i in range(len(args)):
    if args[i] == '-directory':
        directory = args[i + 1]
    if args[i] == '-template':
        template == args[i + 1]
    if args[i] == '-image':
        directory, template = ProcessorBits.untangle_image(args[i + 1])
    if args[i] == '-residues':
        residues = int(args[i + 1])
    if args[i] == '-mosaic':
        default = float(args[i + 1])
        Somewhere.store('default mosaic', default)
    if args[i] == '-nonullpix':
        do_nullpix = False
    if args[i] == '-noradiation':
        do_radiation = False
    if args[i] == '-nosdadd':
        do_sdadd = False
    if args[i] == '-nullpix':
        do_nullpix = False
        nullpix = int(args[i + 1])
        Somewhere.store('NULLPIX', nullpix)
    if args[i] == '-spacegroup':
        spacegroup = args[i + 1]
    if args[i] == '-norlimit':
        do_rlimit = False
    if args[i] == '-maxcell':
        max_cell = float(args[i + 1])
        Somewhere.store('max_cell', max_cell)
    if args[i] == '-wavelength':
        wavelength = float(args[i + 1])

    if args[i] == '-label':
        Somewhere.store('HKLOUT_label', args[i + 1])

    if args[i] == '-start':
        start_at = int(args[i + 1])
    if args[i] == '-end':
        end_at = int(args[i + 1])
        
    if args[i] == '-beam':
        beam = (float(args[i + 1]), float(args[i + 2]))

    if args[i] == '-mad':
        do_anomalous = True

    if args[i] == '-do':
        howfar = args[i + 1]
        if howfar == 'strategy':
            do_index = True
            do_single_integrate = True
            do_strategy = True
            do_refine = False
            do_integrate = False
            do_scale = False
            do_truncate = False
        elif howfar == 'process':
            do_index = True
            do_single_integrate = False
            do_strategy = False
            do_refine = True
            do_integrate = True
            do_scale = False
            do_truncate = False
        else:
            do_index = True
            do_single_integrate = True
            do_strategy = True
            do_refine = True
            do_integrate = True
            do_scale = True
            do_truncate = True

if directory is None:
    directory = os.getcwd()

fileinfo = XSD.Fileinfo()
fileinfo.setDirectory(directory)
fileinfo.setTemplate(template)

Somewhere.project(template)
Somewhere.job('setup')

MosflmInterface.SetMosflmInterfaceFile('mosflm.log')

Messenger.log_write('################################################')
Messenger.log_write('Running Processor.py for template %s' % template)

images = ProcessorBits.get_images(template, directory)

if len(images) == 0:
    raise RuntimeError, 'No images match the template %s in directory %s' % \
          (template, directory)

images.sort()

# next check to see if the list of images is consecutive - if not set a
# flat that integration will fail!

for i in range(len(images)):
    if images[i] != images[0] + i:
        if do_integrate:
            Messenger.log_write('The integration will fail horribly!')
            Messenger.log_write('Image %d is missing' % (images[0] + i))
            sys.exit(1)
        break

one = images[0]

# get hold of the image header for this image to allow the expertise to
# do some digesting

filename = Bits.Image(fileinfo, one)
header = Screen.GetHeader(filename)
two = Expert.SelectImagesForAutoindex(header, images)

# try to pick up the central beam position - if not already defined
if not beam:
    try:
        line = open(directory + '/.beam', 'r').read().split()
        beam = (float(line[0]), float(line[1]))
        Messenger.log_write('Direct beam file picked up: %f %f' % \
                            (beam[0], beam[1]))
    except:
        beam = None
else:
    Messenger.log_write('Direct beam given as: %f %f' % \
                        (beam[0], beam[1]))

extra_commands = XSD.Extra_commands()
if wavelength:
    Messenger.log_write('Using wavelength %f' % wavelength)
    mosflm_commands = XSD.Mosflm_commands()
    mosflm_commands.addCommand('wavelength %f' % wavelength)
    extra_commands.setMosflm_commands(mosflm_commands)

# compute a NULLPIX value and store

if do_nullpix:
    image1 = Screen.Filename(directory, template, one)
    image2 = Screen.Filename(directory, template, two)

    nullpix = Screen.Nullpix(image1, image2)
    
    Somewhere.store('NULLPIX', nullpix)

Somewhere.job('indexing')

index_request = XSD.Index_request()
index_request.setFileinfo(fileinfo)
detector = XSD.Detector()
# this needs to be implemented!
# detector.setType(getFromSomewhere('detector_type', fileinfo)
detector.setType('adsc')
index_request.setDetector(detector)
index_request.addImage(one)
index_request.addImage(two)
if beam:
    xsd_beam = XSD.Beam()
    xsd_beam.setX(beam[0])
    xsd_beam.setY(beam[1])
    index_request.setBeam(xsd_beam)
if spacegroup != '':
    target = XSD.Target()
    target.setSymmetry(spacegroup)
    index_request.setTarget(target)
index_request.setExtra_commands(extra_commands)
    
try:
    index_response = MosflmInterface.do_index_request(index_request)
except Mosflm.MosflmException, e:
    Messenger.log_write('Failed processing index for %s' % template)
    commands = MosflmInterface.get_commands()
    error = 'Initial exception was:\n"%s"' % str(e)
    for c in commands:
        error += c + '\n'
    Messenger.log_write('Error processing:\n%s' % error)
    sys.exit(1)
except exceptions.Exception, e:
    Messenger.log_write('Trapped Exception %s' % str(e))
    sys.exit(1)

if index_response.getStatus().getCode() == 'error':
    Messenger.log_write('Success - an error was recorded')
    Messenger.log_write('Error in indexing %s' % \
                        index_response.getStatus().getMessage())
    sys.exit(1)

mosaic = index_response.getMosaicity_value()

# trap if the mosaic spread was < 0.0 - if this is the case then the image
# integration will fail - all reflections will be reported as "full"
# why does the mosaic spread estimation fail?

if mosaic < 0.0:
    Messenger.log_write('Mosaic spread estimation failed')
    Messenger.log_write('Cannot proceed to integration')
    sys.exit(1)

a = index_response.getSolution().getOrientation().getCell().getA()
b = index_response.getSolution().getOrientation().getCell().getB()
c = index_response.getSolution().getOrientation().getCell().getC()
alpha = index_response.getSolution().getOrientation().getCell().getAlpha()
beta = index_response.getSolution().getOrientation().getCell().getBeta()
gamma = index_response.getSolution().getOrientation().getCell().getGamma()
    
Messenger.log_write('Unit cell is %f %f %f %f %f %f' %
                    (a, b, c, alpha, beta, gamma))
Messenger.log_write('Mosaic spread estimated to be %f' % mosaic)
Messenger.log_write('Spacegroup %s' % \
                    index_response.getSolution().getSymmetry())

selected_symmetry = index_response.getSolution().getSymmetry()

calculated_resolution = 1.5

Somewhere.job('single integrate')

if do_single_integrate:
    single_integrate_request = XSD.Single_integrate_request()

    single_integrate_request.setFileinfo(fileinfo)
    single_integrate_request.addImage(one)
    single_integrate_request.addImage(two)
    single_integrate_request.setExtra_commands(extra_commands)

    integrate_response = MosflmInterface.do_single_integrate_request(
        single_integrate_request)

    if integrate_response.getStatus().getCode() == 'error':
        Messenger.log_write('Stopping due to an error in integration!')
        sys.exit(1)

    # get the resolution here...
    calculated_resolution = integrate_response.getCalculated_resolution().getUpper()
    Messenger.log_write('Calculated resolution from integration %f' %
                        calculated_resolution)

Somewhere.job('strategy')

if do_strategy:
    strategy_request = XSD.Strategy_request()
    strategy_settings = XSD.Strategy_settings()
    resolution = XSD.Resolution()
    resolution.setUpper(calculated_resolution)

    strategy_settings.setResolution(resolution)

    # strategy_settings.setOverlap_limit(2.0)

    strategy_request.setStrategy_settings(strategy_settings)

    strategy_response = MosflmInterface.do_strategy_request(strategy_request)

    output = Somewhere.get('BEST_output')
    if output:
        if not os.environ.has_key('DNALOGDIR'):
            log_directory = '.'
        else:
            log_directory = '%s' % os.environ['DNALOGDIR']

        Messenger.log_write('Strategy results going to %s' % log_directory)
        MosflmLog.Strategy(output, log_directory)

if not do_refine:
    Messenger.log_write('Success with template %s!!' % template)
    time.sleep(1)
    sys.exit()

Somewhere.job('cell refinement')

start, end = Expert.SelectImagesForCellRefinement(header, mosaic, images)

start_images = XSD.Start_images()
for s in start:
    start_images.addImage(s)

end_images = XSD.End_images()
for e in end:
    end_images.addImage(e)

cell_refinement_request = XSD.Cell_refinement_request()
cell_refinement_request.setFileinfo(fileinfo)
cell_refinement_request.setStart_images(start_images)
cell_refinement_request.setEnd_images(end_images)
cell_refinement_request.setExtra_commands(extra_commands)

if do_refine:

    try:
        cell_refinement_response = MosflmInterface.do_refine_cell_request(
            cell_refinement_request)
    except exceptions.Exception, e:
        commands = MosflmInterface.get_commands()
        error = 'Error "%s"\n' % str(e)
        for c in commands:
            error += c + '\n'
        raise RuntimeError, 'Error processing:\n %s' % error
    except:
        commands = MosflmInterface.get_commands()
        error = ''
        for c in commands:
            error += c + '\n'
        raise RuntimeError, 'Error processing:\n %s' % error


    a = cell_refinement_response.getRefined_cell().getA()
    b = cell_refinement_response.getRefined_cell().getB()
    c = cell_refinement_response.getRefined_cell().getC()
    alpha = cell_refinement_response.getRefined_cell().getAlpha()
    beta = cell_refinement_response.getRefined_cell().getBeta()
    gamma = cell_refinement_response.getRefined_cell().getGamma()
    
    Messenger.log_write('Refined unit cell is %f %f %f %f %f %f' % \
                        (a, b, c, alpha, beta, gamma))

Somewhere.job('integration')

start = images[0]
end = images[-1]

if start_at:
    start = start_at

if end_at:
    end = end_at

# used in Scala.py to decide on the batch ranges
Somewhere.store('batch_range', (start, end))

dna_processors = 1
if os.environ.has_key('DNA_PROCESSORS'):
    dna_processors = int(os.environ['DNA_PROCESSORS'])

integrate_request = XSD.Integrate_request()
integrate_request.setFileinfo(fileinfo)
integrate_request.setNumber_of_batches(dna_processors)
integrate_request.setStart(start)
integrate_request.setEnd(end)
integrate_request.setExtra_commands(extra_commands)


if do_integrate:
    try:
        start_time = time.time()
        integrate_response = MosflmInterface.do_integrate_request(integrate_request)

        # try and get the overall "output" information from integration...
        output = Somewhere.get('integration_stuff')

        # next write a "log" of this information - in
        # $DNALOGDIR/integration

        if not os.environ.has_key('DNALOGDIR'):
            log_directory = '.'
        else:
            log_directory = '%s' % os.environ['DNALOGDIR']

        try:
            Messenger.log_write('Integration log going to %s' % log_directory)
            MosflmLog.Integrate(output, log_directory)
        except:
            pass
        end_time = time.time()
        Messenger.log_write('Time spent integrating %f' % \
                            (end_time - start_time))
        
    except exceptions.Exception, e:
        commands = MosflmInterface.get_commands()
        error = 'Error in integration "%s"\n' % str(e)
        for c in commands:
            error += c + '\n'

        for s in traceback.extract_tb(sys.exc_traceback):
            error += str(s) + '\n'

        raise RuntimeError, 'Error processing:\n %s' % error
    
    except:
        commands = MosflmInterface.get_commands()
        error = ''
        for c in commands:
            error += c + '\n'
        raise RuntimeError, 'Error processing:\n %s' % error

    Messenger.log_write('Integration status %s' % \
                        integrate_response.getStatus().getCode())

    if integrate_response.getStatus().getCode() == 'error':
        Messenger.log_write('Error integrating!')
        Messenger.log_write('%s' % integrate_response.getStatus().getMessage())
        sys.exit(1)

Somewhere.job('scaling')

mtzfiles = Somewhere.get('integrate_mtz')

if not mtzfiles:
    mtzfile = '../integrate_%d_%d/integrate.mtz' % (start, end)

sortfile = '../sort/sorted.mtz'
scalefile = '../scale/scaled.mtz'
truncatefile = '../truncate/truncated.mtz'
polishfile = '../scale/polish.sca'

sort_reflections_request = XSD.Sort_reflections_request()
sort_reflections_request.setSort_key('H K L M/ISYM BATCH')
input_reflections = XSD.Input_reflections()
output_reflections = XSD.Output_reflections()
output_reflections.setHklout(sortfile)
if mtzfiles:
    for m in mtzfiles:
        input_reflections.addHklin('../%s' % m)
else:
    input_reflections.addHklin(mtzfile)

sort_reflections_request.setInput_reflections(input_reflections)
sort_reflections_request.setOutput_reflections(output_reflections)

if do_scale:

    sort_response = CCP4Interface.sortmtz(sort_reflections_request)

    Messenger.log_write('%s' % sort_response.marshal())
    if sort_response.getStatus().getCode() == 'error':
        raise RuntimeError, 'Error sorting reflections: %s' % \
              sort_response.getStatus().getMessage()

    # limit the resolution of this file before going in to the point group
    # determination - purely for the sake of speed!

    smallfile = '../reindex/low_resolution.mtz'

    CCP4Interface.limit_resolution(sortfile, smallfile, 3.0)

    # Have a go now at determining the point group

    if start_at and end_at:
        Reindex.DeterminePointGroup(smallfile, '../reindex/point',
                                    selected_symmetry, one + start_at,
                                    two + start_at)
    else:
        Reindex.DeterminePointGroup(smallfile, '../reindex/point',
                                    selected_symmetry, one, two)
    
    Messenger.log_write('Sorting result: %s' % sort_response.getStatus().getCode())

if do_scale:
    if do_anomalous:
        anomalous = 'on'
    else:
        anomalous = 'off'

    scale_reflections_request = XSD.Scale_reflections_request()
    input_reflections = XSD.Input_reflections()
    output_reflections = XSD.Output_reflections()
    input_reflections.addHklin(sortfile)
    output_reflections.setHklout(scalefile)
    scale_reflections_request.setInput_reflections(input_reflections)
    scale_reflections_request.setOutput_reflections(output_reflections)
    scaling_options = XSD.Scaling_options()
    scaling_options.setAnomalous_scattering(anomalous)
    scaling_options.setBfactor_refinement('on')
    scaling_options.setSpacing(5)
    scaling_options.setSecondary(6)
    scaling_options.setCycle_limit(20)
    
    scale_reflections_request.setScaling_options(scaling_options)

    scale_reflections_response = CCP4Interface.scala(scale_reflections_request)
    
    # next rerun with a resolution limit and sdcorrection in there

    sdcorrection = scale_reflections_response.getStandard_deviation_parameters()

    # fetch the sdadd parameters from Somewhere

    sdadd_full = Somewhere.get('sdadd_full')
    sdadd_partial = Somewhere.get('sdadd_partial')

    if sdadd_full and sdadd_partial and do_sdadd:
        sdcorrection.setSdadd_full(sdadd_full)
        sdcorrection.setSdadd_partial(sdadd_partial)

    resolution = Somewhere.get('Complete resolution')
    radiation_damage = Somewhere.get('Radiation_Damage_Limit')

    if radiation_damage and do_radiation:
        Messenger.log_write('Radiation damage found at image %d' % \
                            radiation_damage)

    good_resolution = Somewhere.get('data_good_to')
    if good_resolution:
        # then we only have data to a given resolution - truncate it!
        Messenger.log_write('I/sigma limit found at %f' % good_resolution)
        Messenger.log_write('Limiting resolution range for scaling')
        do_rlimit = True
        resolution = good_resolution

    if radiation_damage:
        batches_xml = '<start>%d</start><end>%d</end>' % \
                      (one, radiation_damage)
    else:
        batches_xml = ''

    # let's hope that this does not include a copy.deepcopy() when
    # setting takes place, so that the pointers are maintained!

    if radiation_damage:
        scaling_options.setStart(one)
        scaling_options.setEnd(radiation_damage)
        
    if resolution > 0.0 and do_rlimit:
        xsd_resolution = XSD.Resolution()
        xsd_resolution.setUpper(resolution)
        scaling_options.setResolution(xsd_resolution)
    scaling_options.setUnmerged_polish_output(polishfile)
    scaling_options.setStandard_deviation_parameters(sdcorrection)
        
    output = Somewhere.get('scaling_results')
    if not os.environ.has_key('DNALOGDIR'):
        log_directory = '.'
    else:
        log_directory = '%s' % os.environ['DNALOGDIR']

    CCP4Log.Scaling(output, log_directory)

    if resolution > 0.0 and do_rlimit:
        Messenger.log_write('Scaling again with new resolution limit %f' % \
                            resolution)
    else:
        Messenger.log_write('Scaling again with same resolution limit')
        
    Messenger.log_write('and sdcorrection parameters of %f %f %f %f' % \
                        (sdcorrection.getSdfac_full(), \
                         sdcorrection.getSdadd_full(), \
                         sdcorrection.getSdfac_partial(), \
                         sdcorrection.getSdadd_partial()))
    
    scale_reflections_response = CCP4Interface.scala(scale_reflections_request)
    
    Messenger.log_write('Scaling status: %s' % \
                        scale_reflections_response.getStatus().getCode())
    sdcorrection = scale_reflections_response.getStandard_deviation_parameters()
    Messenger.log_write('Refined sdcorrection parameters are %f %f %f %f' % \
                        (sdcorrection.getSdfac_full(), \
                         sdcorrection.getSdadd_full(), \
                         sdcorrection.getSdfac_partial(), \
                         sdcorrection.getSdadd_partial()))
    
if not residues:
    Messenger.log_write('Cannot proceed to TRUNCATE without residues - see -residues')
    Messenger.log_write('Will guess from asu volume')

Somewhere.job('truncating')

if do_truncate:
    truncate_reflections_request = XSD.Truncate_reflections_request()
    input_reflections = XSD.Input_reflections()
    output_reflections = XSD.Output_reflections()
    input_reflections.addHklin(scalefile)
    output_reflections.setHklout(truncatefile)
    truncate_reflections_request.setInput_reflections(input_reflections)
    truncate_reflections_request.setOutput_reflections(output_reflections)
    if residues:
        truncate_options = XSD.Truncate_options()
        truncate_options.setResidue_count(residues)
        truncate_reflections_request.setTruncate_options(truncate_options)

    truncate_reflections_response = CCP4Interface.truncate(
        truncate_reflections_request)

    Messenger.log_write('Twinning results:')
    
    Messenger.log_write('BFactor is %f' % \
                        truncate_reflections_response.getBfactor_value())

    Messenger.log_write('Twinning came back as "%s"' % \
                        truncate_reflections_response.getTwinning())

    Messenger.log_write('Solvent content is about %f' % \
                        truncate_reflections_response.getSolvent_content())
