#!/usr/bin/env python
############################################################
# MosflmLog.py
# A html log making mechanism for DNA
# 
# Maintained by G.Winter
# 
# 23rd August 2004
#
# $Id: MosflmLog.py,v 1.7 2005/11/22 13:38:15 svensson Exp $
# 
############################################################

import os, sys, math

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined!'

dna = os.environ['DNAHOME']
sys.path.append('%s/xsd/python' % dna)
sys.path.append('%s/scheduler/Scheduler/Mosflm' % dna)

from Messenger import Messenger
from Somewhere import Somewhere

sys.path.append('%s/scheduler/Scheduler/Driver' % dna)

import Plotter

def MeanSd(list):
    if len(list) == 0:
        return 0.0
    a = 0.0
    for l in list:
        a += float(l)
    a /= len(list)
    mean = a

    a = 0.0
    for l in list:
        a += (float(l) - mean) * (float(l) - mean)
    a /= len(list)
    sd = math.sqrt(a)

    return mean, sd

def Integrate(output, directory):
    '''Write a nice HTML log of the integration results'''

    try:
        os.mkdir('%s/integration' % directory)
    except:
        # assume that the directory already existed
        Messenger.log_write('Failed to make directory integration in %s' \
                            % directory, 2)

    # first check that everything was ok

    if output.key('error') == 1:
        warnings = output.get('error')
        message = ''
        for w in warnings.keys():
            message = message + str(warnings[w]) 

        html_out = open('%s/integration/index.html' % directory, 'w')
        html_out.write('<html><head><title>Integration Results</title>' +
                       '</head><body>')
        html_out.write(message)
        html_out.write('</body></html>')
        html_out.close()
        return

    # then get a list of all of the images integrated

    images = output.get('processing_image', 'image', 'number').keys()
    images.sort()

    # ok then - next get a few statistics from the refinement of
    # parameters for these images

    distance_list = []
    isigall_list = []
    isigout_list = []
    yscale_list = []
    full_list = []
    partial_list = []
    overlap_list = []
    negative_list = []
    bad_list = []

    for image in images:
        distance = output.getFloat('summary_information', 'summary', \
                                   'distance', image)
        distance_list.append(distance)
        isigall = output.getFloat('summary_information', 'summary', \
                                  'isigall', image)
        isigall_list.append(isigall)
        isigout = output.getFloat('summary_information', 'summary', \
                                  'isigout', image)
        isigout_list.append(isigout)
        yscale = output.getFloat('summary_information', 'summary', \
                                 'yscale', image)
        yscale_list.append(yscale)
        
        full = output.getInt('summary_information', 'spots', \
                             'full', image)
        full_list.append(full)
        partial = output.getInt('summary_information', 'spots', \
                                'partial', image)
        partial_list.append(partial)
        overlap = output.getInt('summary_information', 'spots', \
                                'overlap', image)
        overlap_list.append(overlap)
        negative = output.getInt('summary_information', 'spots', \
                                 'negative', image)
        negative_list.append(negative)
        bad = output.getInt('summary_information', 'spots', \
                            'bad_spots', image)
        bad_list.append(bad)

    # ok then that's all the lists made - now let's make some graphs
    # and so on - need the Plotter for this

    html_out = open('%s/integration/index.html' % directory, 'w')
    html_out.write('<html><head><title>Integration Results</title>' +
                   '</head><body>')
    html_out.write('<h1>Integration results</h1>')
    html_out.write('<h3>Integration graphs</h3>')
    
    plotter = Plotter.Plotter()

    # distance
    plotter.setLegend('Distance vs. Batch', 'Batch', 'Distance')
    plotter.setData('/tmp/plotter-logger-%s' % os.environ['USER'])
    plotter.setX(images)
    plotter.setY(distance_list)

    # set a suitable range for the distance - 1% should do it!
    mean, sd = MeanSd(distance_list)
    if sd / mean < 0.03:
        top = 1.01 * mean
        bottom = 0.99 * mean
    else:
        top = mean + 3.0 * sd
        bottom = mean - 3.0 * sd
    plotter.setYrange(bottom, top)
    plotter.setGraph('%s/integration/distance.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="distance.gif"/></p>')
    # add some comments about this
    html_out.write('<p>The RMS variation in distance was %f in %f ' % \
                   (sd, mean))
    html_out.write('which should be less than 0.1%. It is ')
    if sd / mean > 0.001:
        html_out.write(
            '%f%% which is greater than this - check the processing.' %
            (100 * (sd / mean)))
    else:
        html_out.write('%f%% which is fine.' % (100 * (sd / mean)))

    html_out.write('</p>')

    # reset the y range
    plotter.setYrange()

    # i/sig
    plotter.setLegend('I/sigma vs. Batch (overall)', 'Batch', 'I/sigma')
    plotter.setY(isigall_list)
    plotter.setGraph('%s/integration/isigma.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="isigma.gif"/></p>')

    # i/sigout
    plotter.setLegend('I/sigma vs. Batch (outer)', 'Batch', 'I/sigma')
    plotter.setY(isigout_list)
    plotter.setGraph('%s/integration/isigmaout.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="isigmaout.gif"/></p>')

    # numbers of spots - should have a number of lines on one graph for this
    plotter.setLegend('Full spots vs. Batch', 'Batch', 'Count')
    plotter.setY(full_list)
    plotter.setGraph('%s/integration/full.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="full.gif"/></p>')

    plotter.setLegend('Partial spots vs. Batch', 'Batch', 'Count')
    plotter.setY(partial_list)
    plotter.setGraph('%s/integration/partial.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="partial.gif"/></p>')

    # next fulls and partials together - should tell something about
    # the variation in the mosaic spread and so on
    plotter.setLegend('(Full + Partial) vs. Batch', 'Batch', 'Count')
    fp_list = []
    for i in range(len(full_list)):
        fp_list.append(int(full_list[i]) +
                       int(partial_list[i]))
    plotter.setY(fp_list)
    plotter.setGraph('%s/integration/fullandpartial.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="fullandpartial.gif"/></p>')

    plotter.setLegend('Overlap spots vs. Batch', 'Batch', 'Count')
    plotter.setY(overlap_list)
    plotter.setGraph('%s/integration/overlap.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="overlap.gif"/></p>')

    # now have a look at this overlapped spots output!
    happy = True
    most = 0
    for o in overlap_list:
        if int(o) > 0:
            happy = False
            if int(o) > most:
                most = int(o)

    if happy:
        html_out.write('<p>No overlapped reflections!</p>')
    else:
        html_out.write('<p>There were %d overlapping reflections in one batch'
                       % most)
        html_out.write('</p>')


    plotter.setLegend('Negative spots vs. Batch', 'Batch', 'Count')
    plotter.setY(negative_list)
    plotter.setGraph('%s/integration/negative.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="negative.gif"/></p>')

    plotter.setLegend('Bad spots vs. Batch', 'Batch', 'Count')
    plotter.setY(bad_list)
    plotter.setGraph('%s/integration/bad.gif' % directory)
    plotter.plot()

    html_out.write('<p><img src="bad.gif"/></p>')

    # next write out the MTZ file outputs...
    integrate_mtz = Somewhere.get('integrate_mtz')

    html_out.write('<p>The following reflection files were made: ')

    k = 0
    for i in integrate_mtz:
        k += 1
        html_out.write('<a href="%s">Batch %d</a> ' % \
                       (i, k))
    
    html_out.write('</p>')
    

    html_out.write('</body></html>')
    
    html_out.close()
    
def Strategy(output, directory):
    '''Write out the statistics predicted by a BEST strategy as a graph'''

    # this will have to make graphs from the printed output which is
    # displayed at the moment by the Messenger. I guess that this could
    # incorporate the data from the single image integration too...

    # first make the "best_strategy" directory

    try:
        os.mkdir('%s/best_strategy' % directory)
    except:
        Messenger.log_write('Failed to make directory %s/best_strategy' % \
                            directory, 2)

    html_out = open('%s/best_strategy/index.html' % directory, 'w')
    html_out.write('<html><head><title>Strategy Results</title>' +
                   '</head><body>')
    html_out.write('<h1>Strategy results</h1>')
    html_out.write('<h3>Strategy graphs</h3>')

    # ok now need to decide what information we wish to display
    # I think that the data with max, min, i/sig, r and overloads
    # will do nicely!

    # probably good to add a comment about the suggested data collection
    # strategy too...

    bins = []
    for j in output.searchforlist('statistical_prediction', \
                                  'resolution_bin_'):
        i = int(j)
        bins.append(i)
            
    bins.sort()

    # take off the last bin
    bin = bins[-1]
    bins = bins[0:-1]
    min_list = []
    max_list = []
    isigma_list = []
    r_list = []
    overload_list = []

    for i in bins:
        mn = float(output.get('statistical_prediction', \
                              'resolution_bin_%d' % i, 'min_resolution')[0])
        mx = float(output.get('statistical_prediction', \
                              'resolution_bin_%d' % i, 'max_resolution')[0])
        isigma = float(output.get('statistical_prediction', \
                                   'resolution_bin_%d' % i, \
                                   'average_i_over_sigma')[0])
        r = float(output.get('statistical_prediction', \
                             'resolution_bin_%d' % i, 'R_factor')[0])
        overload = float(output.get('statistical_prediction', \
                                    'resolution_bin_%d' % i,
                                    'fract_overload')[0])

        min_list.append(mn)
        max_list.append(mx)
        isigma_list.append(isigma)
        r_list.append(r)
        overload_list.append(overload)

    # now look at some of these results

    # for the graphs use the mid point for resolutions BUT use the upper
    # and lower limits for the range - hence need a setXrange() method
    # an alternative would be to try and describe this as a step function..

    # therefore do some result mashing
    length = len(max_list)

    resolutions = []
    isigma_list2 = []
    r_list2 = []
    overload_list2 = []

    for i in range(length):
        resolutions.append(min_list[i])
        resolutions.append(max_list[i])
        isigma_list2.append(isigma_list[i])
        isigma_list2.append(isigma_list[i])
        r_list2.append(r_list[i])
        r_list2.append(r_list[i])
        overload_list2.append(overload_list[i])
        overload_list2.append(overload_list[i])

    plotter = Plotter.Plotter()
    plotter.setLegend('I/sigma vs. Resolution', 'Resolution', 'I/Sigma')
    plotter.setData('/tmp/plotter-logger-%s' % os.environ['USER'])
    plotter.setX(resolutions)
    plotter.setY(isigma_list2)
    plotter.setGraph('%s/best_strategy/isigma.gif' % directory)
    
    plotter.plot()

    html_out.write('<p><img src="%s/best_strategy/isigma.gif"></p>' % \
                   directory)

    plotter = Plotter.Plotter()
    plotter.setLegend('R factor vs. Resolution', 'Resolution', 'R factor')
    plotter.setData('/tmp/plotter-logger-%s' % os.environ['USER'])
    plotter.setX(resolutions)
    plotter.setY(r_list2)
    plotter.setGraph('%s/best_strategy/r.gif' % directory)
    
    plotter.plot()

    html_out.write('<p><img src="%s/best_strategy/r.gif"></p>' % \
                   directory)

    if max(overload_list) == 0.0:
        html_out.write('<p>No overloads predicted using this strategy.</p>')
    else:
        plotter = Plotter.Plotter()
        plotter.setLegend('Overloads vs. Resolution',
                          'Resolution', 'Overloads')
        plotter.setData('/tmp/plotter-logger-%s' % os.environ['USER'])
        plotter.setX(resolutions)
        plotter.setY(overload_list2)
        plotter.setGraph('%s/best_strategy/overload.gif' % directory)
        
        plotter.plot()
        
        html_out.write('<p><img src="%s/best_strategy/overload.gif"></p>' % \
                       directory)
        
    # finish off...

    html_out.write('</body></html>')
    
    html_out.close()
