#!/usr/bin/env python
##################################################
# MosflmInterface.py Version 2 for the new Scheduler
# Maintained by G.Winter
# 5th March 2004
# 
##################################################
# 
# import all of the things which are used from the
# Mosflm class side of things - this is in this directory

import exceptions
import traceback
import Mosflm, BEST, STAC
import os, math
from Messenger import *
from Somewhere import Somewhere
from SchedulerDNAConfig import *
import sys, re
import shutil

# get hold of the expertise
if os.environ.has_key('DNAHOME'):
    dna = os.environ['DNAHOME']
else:
    raise RuntimeError, 'DNAHOME not defined'

sys.path.append(dna + '/expertise/python/graeme/Expert')
sys.path.append(dna + '/expertise/python/olof')

# get access to the Driver class to implement the Abort() methods
sys.path.append(dna + '/scheduler/Scheduler/Driver')
import Driver
sys.path.append(dna + '/scheduler/Scheduler/New')
import AutoPointgroup
try:
    sys.path.append(dna + '/scheduler/Scheduler/LABELIT')
except:
    pass

import Expert

use_labelit = False

sys.path.append(dna + '/scheduler/Scheduler/New')

from QuickScaler import QuickScale

from BESTWrapper import do_strategy_rq, guess_detector_type

if os.environ.has_key('DNA_LABELIT'):
    if os.environ['DNA_LABELIT'] == 'on':
        Messenger.log_write('LABELIT switched on by DNA_LABELIT=on env', 2)
        use_labelit = True

try:
    import LABELITInterface
except:
    pass

# "database" things

from MosflmDatabase import MosflmDatabase

# web page writer
import ResultWebPages

mosflmInterfaceFile = None
mosflmInterfaceFileName = None

# create a mosflm - this is "global"
mosflm = Mosflm.Mosflm()

def InitMessenger(ES_host, ES_port):
    Messenger.set_host_port(ES_host, ES_port)
    SchedulerDNAConfig.set_host_port(ES_host, ES_port)

def SetMosflmInterfaceFile(filename):
    """Set a log file to echo the mosflm output to"""
    global mosflmInterfaceFile, mosflmInterfaceFileName
    mosflmInterfaceFileName = filename
    mosflmInterfaceFile = open(mosflmInterfaceFileName, "a")
    mosflm.drvOutputCopy = MosflmInterfaceFileWrite

def MoveMosflmInterfaceFile(new_filename):
    """Moves the log file to the given directory and starts a new log file"""
    global mosflmInterfaceFile
    mosflmInterfaceFile.close()
    shutil.move(mosflmInterfaceFileName, new_filename)
    mosflmInterfaceFile = open(mosflmInterfaceFileName, "w")
    mosflm.drvOutputCopy = MosflmInterfaceFileWrite
    
def GetMosflmInterfaceLog():
    """Returns the current MOSFLM log as a string"""
    global mosflmInterfaceFile
    mosflmInterfaceFile.close()
    logFile = open(mosflmInterfaceFileName, "r")
    logText = logFile.read()
    logFile.close()
    mosflmInterfaceFile = open(mosflmInterfaceFileName, "a")
    return logText

def MosflmInterfaceFileWrite(line):
    """Actually copy the output to this file"""
    global mosflmInterfaceFile
    if mosflmInterfaceFile:
        mosflmInterfaceFile.write(line)
        # flush the file to stop the OS buffering
        mosflmInterfaceFile.flush()

def SetMosflmExecutable(newExec):
    """Set the executable which is run for mosflm"""
    mosflm.setExecutable(newExec + ' DNA mosflm.stf')

    # record for later use
    Mosflm.MosflmFactory.set(newExec)

def RestartMosflm(debug = 0):
    """Restart the mosflm object"""
    global mosflm, mosflmInterfaceFile

    # Messenger.log_write('Resetting the scheduler')

    # kill the running mosflm
    try:
        mosflm.kill()
    except:
        # do this just in case it is running already!
        pass

    # delete the object - just to be sure!
    del mosflm
    
    # the configuration of this Mosflm object is persistant, at least
    # in terms of the executable. - the working directory *is not persistant*
    mosflm = Mosflm.Mosflm()
    if mosflmInterfaceFile:
        # there is already a file open, so continue writing to it
        # note well, that this will override the debug statement above
        mosflm.drvOutputCopy = MosflmInterfaceFileWrite
    

def SetWorkingDirectory(newDir):
    mosflm.setWorkingDirectory(newDir)
    mosflm.path = newDir

    Mosflm.MosflmFactory.setWd(newDir)

    if use_labelit:
        LABELITInterface.SetWorkingDirectory(newDir)

def make_image_name(template, number):
    r = re.compile(r'(.*)_(\#*)\.(.*)')
    m = r.match(template)
    prefix = m.group(1)
    hashes = m.group(2)
    extension = m.group(3)
    format = '%0' + str(len(hashes)) + 'd'
    filename = prefix + '_' + format % number + '.' + extension
    return filename

def do_index_request(index_request, jpeg = 0):

    # restart to reset all of the previous odds and ends stored in the Mosflm
    # object - now with a more helpful message
    Messenger.log_write('Initialising data processing system ... ')
    RestartMosflm()

    # flush comments from previous experiments
    Messenger.clear()
    Messenger.log_write("Starting autoindex...", 2)

    mosflm.setTimeout(120.0)

    #
    # after indexing, the actual kappa settings must be also updated
    #
    kappa_settings=XSD.Kappa_collect_settings()
    try:
        current_fileinfo=SchedulerDNAConfig.getDna_context().getCurrent_fileinfo()
        istream = open( os.path.join(current_fileinfo.getDirectory(),current_fileinfo.getTemplate()+"_kappa_settings.xml"), "r" )
        kapSet=istream.read()
        istream.close()    
        kappa_settings.unmarshal(kapSet)
    except:
        kappa_settings=None
    mosflm.kappaSettings = kappa_settings

    if use_labelit:
        index_response = LABELITInterface.do_index_request(index_request)
        mosflm.dnaResult = index_response
        mosflm.autoindexResult = index_response
        mosflm.autoindexRequest = index_request

    else:

        index_response = mosflm.autoindex(index_request, jpeg)
        
        # open('index_request.xml', 'w').write(index_request.marshal())
        # open('index_response.xml', 'w').write(index_response.marshal())

        if index_response.getPowder_ice_rings():
            Messenger.log_write('Powder rings flag : %s' % \
                                index_response.getPowder_ice_rings())
        
        if index_response == None:
            raise RuntimeError, 'Index response is None!'

        try:
            directory = SchedulerDNAConfig.getDna_context(
                ).getCurrent_log_dir()
        except:
            if os.environ.has_key('DNATMPDIR'):
                directory = os.environ['DNATMPDIR']
            else:
                directory = '.'
    
        # check for an error state
        if index_response.getStatus().getCode() == 'error':
            Messenger.log_write('Error detected', 2)
            Messenger.log_write(index_response.getStatus().getMessage(), 2)

            # make jpeg images to show the failure

            fileinfo = index_request.getFileinfo()
            images = index_request.getImage()

            for image in images:
                mosflm.jpeg(fileinfo.getDirectory(),
                            fileinfo.getTemplate(),
                            int(image))

            ResultWebPages.write_failed_index_result(directory,
                                                     index_request)
            
            return index_response

        # zero is overall, one is the first image and two is the second

        # write the results to a log page
        keys = mosflm.results.keys()
        keys.sort()
    
        zero = mosflm.results[keys[0]]
        one = mosflm.results[keys[1]]
        two = mosflm.results[keys[2]]

        try:
        
            ResultWebPages.write_index_result(directory,
                                              index_request,
                                              one, two, zero)

        except:
            Messenger.log_write('Failed to write results pages', 2)


        # store the result in the database

        shift = index_response.getSolution().getRefinement().getBeam_shift().getShift()
        shift_amount = math.sqrt(shift.getX() * shift.getX() + shift.getY() * shift.getY())
        
        reflections = index_response.getSolution().getRefinement().getReflections()
        spot_frac_rejected = 1.0 - 1.0*reflections.getUsed() / \
                             reflections.getUsed_in_indexing()
        MosflmDatabase.add(index_request.getFileinfo().getTemplate(),
                           index_response.getMosaicity_value(),
                           index_response.getSolution().getSpot_deviation(),
                           shift_amount,
                           spot_frac_rejected)

        score_list = MosflmDatabase.scores()

        Messenger.log_write('Lattice: %s   Assumed pointgroup: %s' % \
                            (index_response.getSolution().getLattice(),
                             index_response.getSolution().getPointgroup()))

        # never do this
        if len(score_list) < 0:
            Messenger.log_write( \
                "The scores for recent templates - small is good", 2)
            for score in score_list:
                Messenger.log_write("Score for template %s: %.2f" % \
                                    (score[0], score[1]), 2)

    mosflm.setTimeout(0.0)

    # check that the status is ok - this could just be a reflection of
    # a higher failure than just mosaic spreads not working!
    
    if index_response.getStatus().getCode() != 'error':
        if index_response.getMosaicity_value() < 0.0:
            Messenger.log_write('The mosaic spread estimation failed')
            Messenger.log_write(
                'This could suggest that the wrong lattice was chosen')
            Messenger.log_write('However the indexing may have succeeded')

            if 1 == 0:
            
                # cannot proceed without an estimate of mosaic spread
                index_response.getStatus().setCode('error')
                index_response.getStatus().setMessage(
                    'The mosaic spread estimation failed!')

                # make jpeg images to show the failure
                
                fileinfo = index_request.getFileinfo()
                images = index_request.getImage()

                for image in images:
                    mosflm.jpeg(fileinfo.getDirectory(),
                                fileinfo.getTemplate(),
                                int(image))

                ResultWebPages.write_failed_index_result(directory,
                                                         index_request)
            else:

                # plough ahead and just mosaic = 0.7 and warn!!
                # FIXME is 0.7 a good default???

                Messenger.log_write('============ WARNING!!! =========')

                Messenger.log_write(
                    'Assuming that mosaic spread = 0.7 degrees')
                index_response.setMosaicity_value(0.7)
            
                Messenger.log_write('============ WARNING!!! =========')

    return index_response

# a jiffy function to decide where the results files should be going

def get_results_directory():
    try:
        directory = SchedulerDNAConfig.getDna_context(
            ).getCurrent_log_dir()
    except:
        if os.environ.has_key('DNATMPDIR'):
            directory = os.environ['DNATMPDIR']
        else:
            directory = '.'

    return directory

def do_new_funky_best_strategy_request(strategy_request):
    '''A new implementation of the strategy request - using BEST
    all the way.'''
    
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    workingDir = os.path.join(mosflm.getWorkingDirectory(),
                              'best3')
    Messenger.log_write('BEST working directory: %s' % workingDir, 2)

    # get the exposure time
    try:
        exposure = mosflm.headers[1]['ExposureTime']
    except:
        # return an exception at this stage
        Messenger.log_write('Error getting exposure time - strategy failed.')
        raise RuntimeError, 'error getting exposure time'

    detector = guess_detector_type(mosflm.headers[1])

    Messenger.log_write('Guessed detector class of "%s"' % detector)

    Messenger.log_write('In strategy: Anomalous = %s' % \
                        strategy_request.getStrategy_settings().getAnomalous(),
                        2)

    result = do_strategy_rq(workingDir, exposure, strategy_request)

    try:
        Messenger.log_write(result.marshal(), 2)
    except:
        pass
    
    return result

def do_strategy_request(strategy_request):

    Messenger.log_write(strategy_request.marshal(), 2)
    
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    Messenger.log_write("Starting strategy calculation...", 2)

    strategy_response = do_new_funky_best_strategy_request(strategy_request)

    if True:
        return strategy_response

    strategy_response = mosflm.strategy(strategy_request)

    # decide where results web pages should go

    directory = get_results_directory()
    
    if not os.environ.has_key('besthome'):
        # we ain't got no BEST!
        ResultWebPages.write_basic_strategy(directory,
                                            strategy_request,
                                            strategy_response)

        return strategy_response

    try:        
        strategy_response_best = do_best_strategy_request(strategy_request)
        strategy_response = Expert.ConsiderBestStrategy(strategy_response,
                                                        strategy_response_best)
        # now recalculate the statistics for this strategy
        Messenger.log_write('Now recalculating statistics for the strategy', 2)
        strategy_response_again = do_best_strategy_calculation(
            strategy_request, strategy_response)
        if strategy_response_again.getStatus().getCode() == 'error':
            # then this new strategy will result in overloads!
            overload = float(strategy_response_again.getStatus(
                ).getMessage().split()[2])
            Messenger.log_write('Warning - this data collection strategy ' + \
                                'will result in %f%% overloads' % overload)
            Messenger.log_write('in the lowest resolution bin')

        # now write out a web page with these results in
        try:
            ResultWebPages.write_basic_strategy(directory,
                                                strategy_request,
                                                strategy_response)
        except exceptions.Exception, e:
            Messenger.log_write('Writing results failed!!!', 2)
            Messenger.log_write(str(e))

    except exceptions.Exception, e:
        # BEST strategy did not work for some reason!
        Messenger.log_write('BEST strategy did not work!')
        Messenger.log_write('BEST error was %s' % str(e))
        # Messenger.log_write('%s' % str(e))

        # so write out a web page with limited results in 
        ResultWebPages.write_basic_strategy(directory,
                                            strategy_request,
                                            strategy_response)

        if 1 == 0:

            stack = traceback.extract_tb(sys.exc_traceback)

            # at this stage I also want to write an error report to the
            # standard error including all of the relevent information
            
            Messenger.log_write('-------------------------------------------\n')
            for s in stack:
                Messenger.log_write('MEXS> %s\n' % str(s))
            Messenger.log_write('-------------------------------------------\n')
        
    # open('strategy_request.xml', 'w').write(strategy_request.marshal())
    # open('strategy_response.xml', 'w').write(strategy_response.marshal())

    # Messenger.log_write(strategy_response.marshal())

    return strategy_response

def prepareWorkingDir(module,workDir):
    # configure a working directory
    if mosflm.getWorkingDirectory() == '':
        here = Driver.getcwd()
    else:
        here = mosflm.getWorkingDirectory()
        
    workingDirectory = here + '/' + workDir
    module.setWorkingDirectory(workingDirectory)

    try:
        os.mkdir(workingDirectory)
    except:
        pass

    
def do_kappa_alignment_request(strategy_request):
    #temp commented out for better testability TODO: 
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    Messenger.log_write("Starting kappa alignment calculation...")

    stac_modul = STAC.STAC()
    prepareWorkingDir(stac_modul,'STAC')
    strategy_response = stac_modul.kappa_alignment(strategy_request,mosflm.autoindexResult,mosflm.kappaSettings)

    return strategy_response

def do_kappa_strategy_request(strategy_request):
#    temp commented out for better testability TODO: 
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    Messenger.log_write("Starting kappa strategy calculation...")

    stac_modul = STAC.STAC()
    
    prepareWorkingDir(stac_modul,'STAC')    
    strategy_response = stac_modul.kappa_strategy(strategy_request,mosflm.autoindexResult)

    return strategy_response

def do_kappa_bcm_request(kappa_collect_settings):
    Messenger.log_write("Apply special kappa settings...")
    stac_modul = STAC.STAC()
    prepareWorkingDir(stac_modul,'STAC')    
    kappa_collect_status = stac_modul.kappa_collect_settings(kappa_collect_settings)
    Messenger.log_write("Kappa settings has been applied")
    return kappa_collect_status

def do_kappa_bcm_settings_request():
    Messenger.log_write("Query kappa settings...")
    stac_modul = STAC.STAC()
    prepareWorkingDir(stac_modul,'STAC')    
    kappa_settings_response = stac_modul.kappa_settings_request()
    Messenger.log_write("Kappa settings has been read")
    return kappa_settings_response

def do_best_strategy_request(strategy_request):
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    if mosflm.singleIntegrateResult is None:
        Messenger.error_write("Cannot calculate strategy without integrating")
        raise RuntimeError, 'Cannot calculate strategy before integrating'

    Messenger.log_write("Starting BEST strategy calculation...", 2)

    # this is now false - it gets the integrated image numbers from
    # Somewhere
    Messenger.log_write('Using hkl in ../integrate_1 and ../integrate_91', 2)
    Messenger.log_write('Using orientation and parameters from ' + \
                        '../autoindex_all and ../integrate_1', 2)

    # configure a working directory
    if mosflm.getWorkingDirectory() == '':
        here = Driver.getcwd()
    else:
        here = mosflm.getWorkingDirectory()
        
    b = BEST.BEST()

    workingDirectory = here + '/best'
    b.setWorkingDirectory(workingDirectory)

    try:
        os.mkdir(workingDirectory)
    except:
        pass

    # these shouldn't be hard coded!
    # so FIX IT!
    
    # hkl1 = '../integrate_1/bestfile.hkl'
    # hkl91 = '../integrate_91/bestfile.hkl'
    hkl = Somewhere.get('best_hkl')
    if not hkl:
        raise MosflmException, 'No BEST hkl found!'
    
    par = '../integrate_1/bestfile.par'
    dat = '../autoindex_all/bestfile.dat'

    # get the exposure time
    try:
        exposure = mosflm.headers[1]['ExposureTime']
        b.setExposure_time(float(exposure))
        Messenger.log_write('Obtained an exposure time for image %d of %5.2f' \
                            % (1, float(exposure)), 2)
    except:
        Messenger.log_write('Cannot access image exposure time - assuming ' + \
                            '1.0 seconds', 2)
        b.setExposure_time(1.0)

    b.setDat(dat)
    b.setPar(par)
    # b.addHkl(hkl1)
    # b.addHkl(hkl91)
    for h in hkl:
        b.addHkl(h)

    detector = mosflm.autoindexRequest.getDetector().getType()
    if detector:
        diameter = None
        if detector == 'mar':
            # assume this means MAR image plate - determine the diameter
            diameter = float(mosflm.headers[1]['PixelX']) * \
                       float(mosflm.headers[1]['Width']) * 0.001
            Messenger.log_write('Detector diameter is %3.2f' % diameter, 2)
        Messenger.log_write('Setting detector type in BEST to %s' % detector,
                            2)
        b.setDetector(detector, diameter)

    try:
        resolution = float(strategy_request.getStrategy_settings().getResolution().getUpper())
        b.setResolution(resolution)
    except:
        pass
    
    b.setSegments(1)
    strategy_response = b.run()

    return strategy_response
    
def do_best_strategy_calculation(strategy_request, strategy_response):
    if mosflm.autoindexResult is None:
        Messenger.error_write("Cannot calculate strategy without indexing")
        raise RuntimeError, 'Cannot calculate strategy before indexing'

    if mosflm.singleIntegrateResult is None:
        Messenger.error_write("Cannot calculate strategy without integrating")
        raise RuntimeError, 'Cannot calculate strategy before integrating'

    # get the strategy out of here
    oscillations = strategy_response.getStrategy_interpretation().getOscillation_sequence()[0]
    exp_time = float(oscillations.getExposure_time())
    start = float(oscillations.getStart())
    range = float(oscillations.getRange())
    number = float(oscillations.getNumber_of_images())
    end = start + range * number
    width = range

    # check that start and end are in [0, 360] - if not shift by +180 or
    # -180. This is OK for non anomalous data I hope! i.e. measure -h -k -l
    # rather than h k l

    # note well - this assumes that you are collecting less than 180 degrees
    # of data

    if end > 360.0:
        start -= 180.0
        end -= 180.0
    else:
        if start < 0.0:
            start += 180.0
            end += 180.0
        

    Messenger.log_write('Starting BEST calculation for strategy:', 2)
    Messenger.log_write('start %3.2f end %3.2f width %3.2f time %5.2f' % \
                        (start, end, width, exp_time), 2)

    Messenger.log_write('Using hkl in ../integrate_1 and ../integrate_91', 2)
    Messenger.log_write('Using orientation and parameters from ' + \
                        '../autoindex_all and ../integrate_1', 2)

    # configure a working directory
    if mosflm.getWorkingDirectory() == '':
        here = Driver.getcwd()
    else:
        here = mosflm.getWorkingDirectory()
        
    b = BEST.BEST()

    workingDirectory = here + '/best'
    b.setWorkingDirectory(workingDirectory)

    try:
        os.mkdir(workingDirectory)
    except:
        pass

    # these shouldn't be hard coded!
    # so FIX IT!
    
    # hkl1 = '../integrate_1/bestfile.hkl'
    # hkl91 = '../integrate_91/bestfile.hkl'
    par = '../integrate_1/bestfile.par'
    dat = '../autoindex_all/bestfile.dat'

    hkl = Somewhere.get('best_hkl')
    if not hkl:
        raise MosflmException, 'No BEST hkl found!'

    # b.setExposure_time(exp_time)
    try:
        exposure = mosflm.headers[1]['ExposureTime']
        b.setExposure_time(float(exposure))
        Messenger.log_write('Obtained an exposure time for image %d of %5.2f'
                            % \
                            (1, float(exposure)), 2)
    except:
        Messenger.log_write('Cannot access image exposure time - assuming ' + \
                            '1.0 seconds', 2)
        b.setExposure_time(1.0)


    b.setStrategy(start, end, width, exp_time)

    b.setDat(dat)
    b.setPar(par)
    # b.addHkl(hkl1)
    # b.addHkl(hkl91)

    for h in hkl:
        b.addHkl(h)

    detector = mosflm.autoindexRequest.getDetector().getType()
    if detector:
        diameter = None
        if detector == 'mar':
            # assume this means MAR image plate - determine the diameter
            diameter = float(mosflm.headers[1]['PixelX']) * \
                       float(mosflm.headers[1]['Width']) * 0.001
        Messenger.log_write('Setting detector type in BEST to %s' % detector,
                            2)
        b.setDetector(detector, diameter)


    try:
        resolution = float(strategy_request.getStrategy_settings().getResolution().getUpper())
        b.setResolution(resolution)
    except:
        pass
    
    strategy_response = b.run()
    return strategy_response

def do_integrate_request(integrate_request):
    """Perform a basic integration job and simply return a status
    so that we can get things rolling with this part of the DNA
    project"""
    Messenger.log_write("Starting integration...", 2)
    integrate_result = mosflm.integrate(integrate_request)

    # open('integrate_request.xml', 'w').write(integrate_request.marshal())
    # open('integrate_response.xml', 'w').write(integrate_response.marshal())

    return integrate_result

def do_single_integrate_request(single_integrate_request):

    mosflm.setTimeout(120.0)

    # clear the somewhere store of best_hkl values

    Somewhere.store('best_hkl', None)
    
    Messenger.log_write("Starting single image integration...", 2)
    single_integrate_response = \
                              mosflm.single_integrate(single_integrate_request)

    try:
        directory = SchedulerDNAConfig.getDna_context().getCurrent_log_dir()
    except:
        if os.environ.has_key('DNATMPDIR'):
            directory = os.environ['DNATMPDIR']
        else:
            directory = '.'
    
    ResultWebPages.write_single_integrate_result(directory, \
                                                 single_integrate_request, \
                                                 single_integrate_response)

    Expert.AnalyseIntegrateResponse(single_integrate_response)

    # print some helpful information to the screen so that the user may
    # feel happy about the data
    integrated_images = single_integrate_response.getIntegrated_image()
    imagestring = ""

    resolutionTotal = 0.0
    resolutionCount = 0

    if not integrated_images:
        Messenger.log_write('Error integrating reference images!')

    if integrated_images:
        for i in integrated_images:
            e = Expert.EstimateResolution(i)
            if e == 0.0:
                Messenger.log_write("0.0 came back from resolution est. for " +
                                    str(i.getImage()), 2)
            else:
                Messenger.log_write('Resolution for image %d is %2.2f' % \
                                    (i.getImage(), e), 2)
                resolutionTotal += e
                resolutionCount += 1.0
                imagestring += " " + str(i.getImage())

        resolution = 0.0

        if resolutionCount > 0.0:
            resolution = resolutionTotal / resolutionCount
        else:
            resolution = 0.0

        # this should be stored in Somewhere
        mosflm.resolutionForStrategy = resolution

        Messenger.log_write("Integrated images:" + imagestring)
        Messenger.log_write("Average resolution is %2.2f" % resolution)

    mosflm.setTimeout(0.0)

    return single_integrate_response

def do_refine_cell_request(cell_refinement_request):

    Messenger.log_write("Starting cell refinement...", 2)
    cell_refinement_response = mosflm.refine(cell_refinement_request)

    cell = cell_refinement_response.getRefined_cell()
    Messenger.log_write("Refined cell edges are %4.2f, %4.2f, %4.2f" % \
                        (cell.getA(), cell.getB(), cell.getC()))
    Messenger.log_write("Refined cell angles are %4.2f, %4.2f, %4.2f" % \
                        (cell.getAlpha(), cell.getBeta(), cell.getGamma()))
    
    # open('cell_refinement_request.xml', 'w').write(cell_refinement_request.marshal())
    # open('cell_refinement_response.xml', 'w').write(cell_refinement_response.marshal())

    do_pointless = True

    if do_pointless:
        # Then run the auto-pointless on these frames as well - to
        # see what it says basically (this will not be acted on!!)
        fileinfo = cell_refinement_request.getFileinfo()
        results = AutoPointgroup.PointlessDNAThing(fileinfo.getTemplate(),
                                                   fileinfo.getDirectory())

        # now write about the results of this to the messenger

        spacegroup, reindex_op, conf = results

        if not spacegroup is None:

            Messenger.log_write('.                               .')
            Messenger.log_write('      IMPORTANT INFORMATION!!')
            Messenger.log_write('.                               .')
            Messenger.log_write(' With confidence %d/3' % conf)
            Messenger.log_write(
                ' Pointless thinks the spacegroup should be %s' \
                % spacegroup)
            Messenger.log_write('.                               .')

        else:
            Messenger.log_write('Sorry, will need more data.')

    return cell_refinement_response

def get_commands():
    return mosflm.debug()


# abort and reset commands to implement the system wide "sbort" button in DNA

def AbortAllNow():
    '''KILL all currently running processes - this also sets a flag to
    prevent starting of new processes'''

    Messenger.log_write('Aborting all running processes NOW!')
    Driver.Abort()

    return

def ResetAbortAllNow():
    '''Reset the abort flag for running processes'''

    Messenger.log_write('Resetting the abort flag in the Scheduler now')
    Driver.Reset()

    return

# a jiffy to write out an XDS.INP and XSCALE.INP file

def WriteXDSInputFiles(index_response,
                       header_info):
    '''Write an XDS input file using the results from autoindexing and
    the information from an image header measured at the correct
    distance'''

    # need to extract the header information to set up XDS
    # correctly and also configure the resolution ranges for
    # XSCALE

    # then need to obtain the useful information from the
    # autoindex response including the refined direct beam
    # position - note that this may require determining the
    # appropriate transformation

    raise RuntimeError, 'This is not yet implemented'

def do_quick_scale_request(quick_scale_request):
    '''Implement a connection to the QuickScale function in QuickScale.py.'''

    # note well - the fileinfo should be set up by the ES to point to the
    # MTZ files - wherever they have been copied to

    fileinfo = quick_scale_request.getFileinfo()

    directory = fileinfo.getDirectory()
    template = fileinfo.getTemplate()

    resolution = quick_scale_request.getHigh_resolution_limit()

    if template is None:
        status = XSD.Status()
        status.setCode('error')
        status.setMessage('template is None')
        return Status

    # first look for the reflection files - build up a list

    files = os.listdir(directory)
    files.sort()

    # 19/DEC/05 this regular expression should be just '*'
    # instead of however many [0-9] because of the complex
    # naming scheme DNA employs... e.g.
    # prefix_date_process_start_end.mtz

    startend = template.split('#')
    
    regexp = re.compile(startend[0] + '.*' + startend[-1])

    debugging = False

    if debugging:

        Messenger.log_write('Regular expression for pattern matching is %s' % \
                            startend[0] + '.*' + startend[-1], 2)

    mtz_list = []
    
    for f in files:
        if debugging:
            Messenger.log_write('Found file: %s' % f, 2)
        match = regexp.match(f)
        if match:
            Messenger.log_write('Found matching reflection file: %s' % f, 2)
            mtz_list.append(os.path.join(directory, f))

    # fix towards bug # 1475 - some proper error trapping if no reflection
    # files are found

    if len(mtz_list) == 0:
        status = XSD.Status()
        status.setCode('error')
        Messenger.log_write('No reflection files found in %s matching %s' % \
                            (directory, template), 2)
        status.setMessage('No reflection files found in %s matching %s' % \
                            (directory, template), 2)
        return status

    hklout = '%s_scaled.mtz' % template.split('#')[0]
    # scale the files
    
    results = QuickScale(mtz_list, hklout,
                         working_dir = mosflm.getWorkingDirectory(),
                         resolution = resolution)

    # report some stuff

    r_merge = results['r_merge']
    i_over_sig = results['i_over_sig']
    dmin, dmax = results['resolution_range']
    compl = results['completeness']
    multi = results['multiplicity']
    nref = results['n_ref_tot']
    nuniq = results['n_ref_unique']

    hr_merge = results['r_merge_hr']
    hi_over_sig = results['i_over_sig_hr']
    hdmin, hdmax = results['resolution_range_hr']
    hcompl = results['completeness_hr']
    hmulti = results['multiplicity_hr']
    hnref = results['n_ref_tot_hr']
    hnuniq = results['n_ref_unique_hr']

    Messenger.log_write('.                                             .')
    Messenger.log_write('.                                             .')
    Messenger.log_write('         IMPORTANT INFORMATION!!')
    Messenger.log_write('.                                             .')
    Messenger.log_write('Resolution   %6.2f - %5.2f (%5.2f - %5.2f)' % \
                        (dmax, dmin, hdmax, hdmin))
    Messenger.log_write('R merge         %5.3f     (%5.3f)' % \
                        (r_merge, hr_merge))
    Messenger.log_write('I/sig           %5.1f     (%5.1f)' % \
                        (i_over_sig, hi_over_sig))
    Messenger.log_write('Completeness    %6.2f     (%6.2f)' % \
                        (compl, hcompl))
    Messenger.log_write('Multiplicity    %5.2f      (%5.2f)' % \
                        (multi, hmulti))
    Messenger.log_write('Reflections     %d         (%d)' % \
                        (nref, hnref))
    Messenger.log_write('Unique          %d         (%d)' % \
                        (nuniq, hnuniq))
    Messenger.log_write('.                                             .')
    Messenger.log_write('.                                             .')
    Messenger.log_write(
        'Note: The completeness in the high resolution shell (%f to %f)' \
        % (hdmax, hdmin))
    Messenger.log_write(
        'may be low if the reflections fall off the detector.')
    Messenger.log_write('.                                             .')
    

    # return OK

    quick_scale_response = XSD.Quick_scale_response()

    status = XSD.Status()
    status.setCode('ok')

    quick_scale_response.setStatus(status)

    return quick_scale_response

if __name__ == '__main__':
    if len(sys.argv) <= 1:
        do = '__moon__'
    else:
        do = '__main__'

else:
    do = '__nothing__'

if do == "__moon__":
    # then test this

    import sys, os
    sys.path.append('/home/graeme/CVS/dna/xsd/python')
    import XSD

    if not os.environ.has_key('DNATMPDIR'):
        os.environ['DNATMPDIR'] = '/tmp/beer'

    SetMosflmExecutable('/home/graeme/bin/ipmosflm')
    try:
        os.mkdir(os.environ['DNATMPDIR'])
    except:
        pass
    
    SetWorkingDirectory(os.environ['DNATMPDIR'])

    SetMosflmInterfaceFile(os.environ['DNATMPDIR'] + '/fluffy-bunny.out')

    xml = "<?xml version='1.0'?><!DOCTYPE index_request><index_request><fileinfo><template>trypsin_1_###.img</template><directory>/data/graeme/images</directory><matrix_file></matrix_file></fileinfo><detector><type>adsc</type></detector><beam><x>94.8</x><y>92.9</y></beam><image>001</image><image>091</image></index_request>"

    index_request = XSD.Index_request()

    # this shouldn't be a method...
    dict = index_request.XMLStringToDict(xml)

    # print dict["index_request"]

    index_request.fromDict(dict["index_request"])

    index_response = do_index_request(index_request)

    print index_response.marshal()

    do_refine = 1
    if do_refine == 1:

        # refine the unit cell next....
        document = "<?xml version='1.0'?><!DOCTYPE cell_refinement_request><cell_refinement_request><fileinfo><template>trypsin_1_###.img</template><directory>/data/graeme/images</directory><matrix_file></matrix_file></fileinfo><start_images><image>1</image><image>46</image></start_images><end_images><image>3</image><image>48</image></end_images></cell_refinement_request>"
        
        cell_refinement_request = XSD.Cell_refinement_request()
        cell_refinement_request.unmarshal(document)
        
        response = do_refine_cell_request(cell_refinement_request)

        print response.marshal()

    do_p1 = 0
    if do_p1 == 1:
        document = "<?xml version='1.0'?><!DOCTYPE cell_refinement_request><cell_refinement_request><symmetry>p1</symmetry><fileinfo><template>trypsin_1_###.img</template><directory>/data/graeme/images</directory><matrix_file></matrix_file></fileinfo><start_images><image>1</image><image>46</image></start_images><end_images><image>3</image><image>48</image></end_images></cell_refinement_request>"
        
        cell_refinement_request = XSD.Cell_refinement_request()
        cell_refinement_request.unmarshal(document)
        
        response = do_refine_cell_request(cell_refinement_request)



    single_integrate_request = XSD.Single_integrate_request()

    xml = "<?xml version='1.0'?><!DOCTYPE single_integrate_request><single_integrate_request><image>1</image><image>91</image><fileinfo><template>trypsin_1_###.img</template><directory>/data/graeme/images</directory><matrix_file></matrix_file></fileinfo></single_integrate_request>"

    single_integrate_request.unmarshal(xml)

    integrate_response = do_single_integrate_request(single_integrate_request)

    print integrate_response.marshal()

    strategy = 1
    if strategy == 1:
    
        strategy_request = XSD.Strategy_request();

        try:
            resolution = float(mosflm.resolutionForStrategy)
        except:
            resolution = 1.0

        xml = "<?xml version='1.0'?><!DOCTYPE strategy_request><strategy_request><strategy_settings><resolution_limits><upper><isotropic>%f</isotropic></upper></resolution_limits></strategy_settings><symmetry>p222</symmetry></strategy_request>" % resolution
        
        
        print 'BEST'
        strategy_request.unmarshal(xml)
        strategy_response = do_best_strategy_request(strategy_request)
        
        print strategy_response.marshal()
        
        strategy_request = XSD.Strategy_request();
        
        xml = "<?xml version='1.0'?><!DOCTYPE strategy_request><strategy_request><strategy_settings><resolution_limits><upper><isotropic>%f</isotropic></upper></resolution_limits></strategy_settings><symmetry>p222</symmetry></strategy_request>" % resolution
        
                
        print 'Mosflm'
        strategy_request.unmarshal(xml)
        strategy_response = do_strategy_request(strategy_request)
        
        print strategy_response.marshal()

    big_integrate = 0
    if big_integrate == 1:

        xml = "<?xml version='1.0'?><!DOCTYPE integrate_request><integrate_request><start>1</start><end>91</end><fileinfo><template>trypsin_1_###.img</template><directory>/data/graeme/images</directory><matrix_file></matrix_file></fileinfo><number_of_batches>9</number_of_batches></integrate_request>"
        
        integrate_request = XSD.Integrate_request()
        integrate_request.unmarshal(xml)
        
        integrate_result = do_integrate_request(integrate_request)
        
        print integrate_result.marshal()

if do == "__main__":
    # then test this

    import sys, os
    sys.path.append('/home/graeme/CVS/dna/xsd/python')
    import XSD

    if not os.environ.has_key('DNATMPDIR'):
        os.environ['DNATMPDIR'] = '/tmp/raimond'

    SetMosflmExecutable('/home/graeme/bin/ipmosflm')
    try:
        os.mkdir(os.environ['DNATMPDIR'])
    except:
        pass
    
    SetWorkingDirectory(os.environ['DNATMPDIR'])

    SetMosflmInterfaceFile(os.environ['DNATMPDIR'] + '/fluffy-bunny.out')

    xml = "<?xml version='1.0'?><!DOCTYPE index_request><index_request><fileinfo><template>raimond_5_###.img</template><directory>/media/filestore/graeme/images/esrf/dna-dev/collect</directory><matrix_file></matrix_file></fileinfo><detector><type>adsc</type></detector><beam><x>93.3</x><y>93.5</y></beam><image>001</image><image>091</image></index_request>"

    index_request = XSD.Index_request()

    # this shouldn't be a method...
    dict = index_request.XMLStringToDict(xml)

    # print dict["index_request"]

    index_request.fromDict(dict["index_request"])

    index_response = do_index_request(index_request)

    # raise RuntimeError, 'Stopping now!'

    print index_response.marshal()

    do_refine = 0
    if do_refine == 1:

        # refine the unit cell next....
        document = "<?xml version='1.0'?><!DOCTYPE cell_refinement_request><cell_refinement_request><fileinfo><template>raimond_5_###.img</template><directory>/media/filestore/graeme/images/esrf/dna-dev/collect</directory><matrix_file></matrix_file></fileinfo><start_images><image>1</image><image>46</image></start_images><end_images><image>3</image><image>48</image></end_images></cell_refinement_request>"
        
        cell_refinement_request = XSD.Cell_refinement_request()
        cell_refinement_request.unmarshal(document)
        
        response = do_refine_cell_request(cell_refinement_request)

        print response.marshal()

    do_p1 = 0
    if do_p1 == 1:
        document = "<?xml version='1.0'?><!DOCTYPE cell_refinement_request><cell_refinement_request><symmetry>p1</symmetry><fileinfo><template>raimond_5_###.img</template><directory>/media/filestore/graeme/images/esrf/dna-dev/collect</directory><matrix_file></matrix_file></fileinfo><start_images><image>1</image><image>46</image></start_images><end_images><image>3</image><image>48</image></end_images></cell_refinement_request>"
        
        cell_refinement_request = XSD.Cell_refinement_request()
        cell_refinement_request.unmarshal(document)
        
        response = do_refine_cell_request(cell_refinement_request)



    single_integrate_request = XSD.Single_integrate_request()

    xml = "<?xml version='1.0'?><!DOCTYPE single_integrate_request><single_integrate_request><image>1</image><image>91</image><fileinfo><template>raimond_5_###.img</template><directory>/media/filestore/graeme/images/esrf/dna-dev/collect</directory><matrix_file></matrix_file></fileinfo></single_integrate_request>"

    single_integrate_request.unmarshal(xml)

    integrate_response = do_single_integrate_request(single_integrate_request)

    print integrate_response.marshal()

    strategy = 1
    if strategy == 1:
    
        strategy_request = XSD.Strategy_request();

        xml = "<?xml version='1.0'?><!DOCTYPE strategy_request><strategy_request><strategy_settings><resolution_limits><upper><isotropic>1.0</isotropic></upper></resolution_limits></strategy_settings></strategy_request>"
        
        
        print 'BEST'
        strategy_request.unmarshal(xml)
        strategy_response = do_best_strategy_request(strategy_request)
        
        print strategy_response.marshal()
        
        strategy_request = XSD.Strategy_request();
        
        xml = "<?xml version='1.0'?><!DOCTYPE strategy_request><strategy_request><strategy_settings><resolution_limits><upper><isotropic>1.0</isotropic></upper></resolution_limits></strategy_settings></strategy_request>"
        
                
        print 'Mosflm'
        strategy_request.unmarshal(xml)
        strategy_response = do_strategy_request(strategy_request)
        
        print strategy_response.marshal()

    big_integrate = 0
    if big_integrate == 1:

        xml = "<?xml version='1.0'?><!DOCTYPE integrate_request><integrate_request><start>1</start><end>91</end><fileinfo><template>raimond_5_###.img</template><directory>/media/filestore/graeme/images/esrf/dna-dev/collect</directory><matrix_file></matrix_file></fileinfo><number_of_batches>9</number_of_batches></integrate_request>"
        
        integrate_request = XSD.Integrate_request()
        integrate_request.unmarshal(xml)
        
        integrate_result = do_integrate_request(integrate_request)
        
        print integrate_result.marshal()
