#!/usr/bin/env python
######################################################
# MosflmDatabase.py
# Maintained By G.Winter
# 1st April 2004 
# A temporary results storage singleton to gather
# results from processing to build up statistics.
# 
# This is a temporary measure until we have a proper
# database back end.
#
# $Id: MosflmDatabase.py,v 1.2 2004/05/18 11:32:15 svensson Exp $
# 
######################################################

import math

class MosflmResult:
    # a class to carry around the results from mosflm
    # processing - only used internally

    def __init__(self, args):
        self.template = ""
        self.mosaicity = 0.0
        self.rms_deviation = 0.0
        self.beam_shift = 0.0
        self.rejected_fraction = 0.0

        if len(args) == 5:
            self.template = str(args[0])
            self.mosaicity = float(args[1])
            self.rms_deviation = float(args[2])
            self.beam_shift = float(args[3])
            self.rejected_fraction = float(args[4])

# a handy function! shouldn't make a habit of this

def divide(a, b):
    if a == 0.0 or b == 0.0:
        return 0.0
    else:
        return a / b

class MosflmDatabaseClass:

    def __init__(self):
        self.entries = []

    def add(self, template, mosaicity, rms_deviation, beam_shift, rejected_fraction):
        m = MosflmResult([template, mosaicity, rms_deviation, beam_shift, rejected_fraction])
        self.entries.append(m)

    def clear(self):
        self.entries = []

    def scores(self):

        if len(self.entries) == 0:
            return []

        # first go through and build up average values of things
        self.mosaicity = 0.0
        self.rms_deviation = 0.0
        self.beam_shift = 0.0
        self.rejected_fraction = 0.0

        for m in self.entries:
            self.mosaicity += m.mosaicity
            self.rms_deviation += m.rms_deviation
            self.beam_shift += m.beam_shift
            self.rejected_fraction += m.rejected_fraction

        self.mosaicity /= float(len(self.entries))
        self.rms_deviation /= float(len(self.entries))
        self.beam_shift /= float(len(self.entries))
        self.rejected_fraction /= float(len(self.entries))
        
        # now I have averages, build up standard deviation values

        self.mosaicity_sd = 0.0
        self.rms_deviation_sd = 0.0
        self.beam_shift_sd = 0.0
        self.rejected_fraction_sd = 0.0

        for m in self.entries:
            self.mosaicity_sd += (m.mosaicity - self.mosaicity) * \
                                 (m.mosaicity - self.mosaicity)
            self.rms_deviation_sd += (m.rms_deviation - self.rms_deviation) * \
                                     (m.rms_deviation - self.rms_deviation)
            self.beam_shift_sd += (m.beam_shift - self.beam_shift) * \
                                  (m.beam_shift - self.beam_shift)
            self.rejected_fraction_sd += \
                                      (m.rejected_fraction - \
                                       self.rejected_fraction) * \
                                       (m.rejected_fraction - \
                                        self.rejected_fraction)

        self.mosaicity_sd /= float(len(self.entries))
        self.rms_deviation_sd /= float(len(self.entries))
        self.beam_shift_sd /= float(len(self.entries))
        self.rejected_fraction_sd /= float(len(self.entries))

        self.mosaicity_sd = math.sqrt(self.mosaicity_sd)
        self.rms_deviation_sd = math.sqrt(self.rms_deviation_sd)
        self.beam_shift_sd = math.sqrt(self.beam_shift_sd)
        self.rejected_fraction_sd = math.sqrt(self.rejected_fraction_sd)

        # note - for the purposes of the scoring, the "ideal" values
        # of all of these properties are 0.0

        scores = []

        for m in self.entries:
            try:
                # in case of / 0.0
                score = divide(m.mosaicity, self.mosaicity_sd) + \
                        divide(m.rms_deviation, self.rms_deviation_sd) + \
                        divide(m.beam_shift, self.beam_shift_sd) + \
                        divide(m.rejected_fraction, self.rejected_fraction_sd)
                
                scores.append([m.template, score])
            except:
                scores.append([m.template, -1.0])

        return scores

MosflmDatabase = MosflmDatabaseClass()

if __name__ == '__main__':
    MosflmDatabase.clear()
    MosflmDatabase.add("beer", 1.0, 1.0, 1.0, 1.0)
    MosflmDatabase.add("lager", 2.0, 2.0, 2.0, 2.0)
    MosflmDatabase.add("tea", 3.0, 3.0, 3.0, 3.0)
    MosflmDatabase.add("coffee", 4.0, 4.0, 4.0, 4.0)
    MosflmDatabase.add("lemonade", 5.0, 5.0, 5.0, 5.0)

    for score in MosflmDatabase.scores():
        print "Score for %s is %f" % (score[0], score[1])
