#!/usr/bin/env python
# Mosflm.py
# Maintained by G.Winter
# Part of the second version of the scheduler code - version 1.1
# 19th January 2004
# 
# This section of the code will inherit from Driver to create a task to 
# control the program Mosflm.
# 
# 
# $Id: Mosflm.py,v 1.76 2006/06/20 08:22:50 gwin Exp $

# first import the system and check the environment

import os

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNA not configured'
if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 not configured'

dna = os.environ['DNAHOME']

# configure the python path to include the DNA bits
import sys, shutil, datetime

sys.path.append(dna + '/scheduler/Scheduler/Driver')

# include Graeme's expertise
sys.path.append(dna + '/expertise/python/graeme/Expert')

# and the DiffractionImage library screening bits
# this should be wrapped to tell if we have or don't
# have DiffractionImage ... with an include statement

sys.path.append(dna + '/scheduler/DiffractionImage/lib')

import Driver
import Output
import Translation
import Background
import Expert
import Exist
import Bits
import Config
from Somewhere import Somewhere
from Messenger import *
from SchedulerDNAConfig import SchedulerDNAConfig

import exceptions

# no longer try to do this - just do this
import Screen

# access to the DNA data transfer python objects
sys.path.append(dna + '/xsd/python')
import XSD

import copy, time, re, traceback

# add a singleton to store the binary information // bug 1590

class _MosflmFactory:
    def __init__(self):
        self.mosflm_binary = None
        self.mosflm_working_dir = None

    def get(self):
        return self.mosflm_binary

    def set(self, binary):
        self.mosflm_binary = binary

    def setWd(self, wd):
        self.mosflm_working_dir = wd
        return

    def getWd(self):
        return self.mosflm_working_dir

MosflmFactory = _MosflmFactory()

class Mosflm(Driver.Driver):
    '''a class to execute Mosflm'''

    def __init__(self):
        Driver.Driver.__init__(self, 1, 'mosflm')

        binary = 'ipmosflm'

        if MosflmFactory.get():
            binary = MosflmFactory.get()

        if MosflmFactory.getWd():
            wd = MosflmFactory.getWd()
            self.setWorkingDirectory(wd)
            self.path = wd
        
        self.setExecutable('%s DNA mosflm.stf' % binary)

        self.autoindexRequest = None
        self.autoindexResult = None
        self.kappaSettings = None
        self.cellRefineRequest = None
        self.cellRefineResult = None
        self.singleIntegrateRequest = None
        self.singleIntegrateResult = None

        self.headers = { }

        self.exclude_ice = False

        self.outputFiles = []

    def jpeg(self, directory, template, image):
        '''Write a jpeg image of the diffraction data'''

        # first take to pieces the filename
        prefix, hash, extension = Bits.UnTemplate(template)
        length = len(hash)

        code = Bits.Pad(image, length)

        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = '%s/failure' % here

        try:
            os.mkdir(workingDirectory)
        except:
            pass

        self.setWorkingDirectory(workingDirectory)

        self.start()
        
        self.input('xgui on')
        self.input('template %s' % template)
        self.input('directory %s' % directory)
        self.input('image ' + str(image))
        self.input('go')
        jpegfile = prefix + '_' + \
                   code + '.jpg'
        self.input('create_image binary true filename ' + jpegfile)
        jpegfile = prefix + '_' + \
                   code + '_small.jpg'
        self.input('create_image zoom -4 binary true filename ' + jpegfile)
        self.input('return')
        self.input('xgui off')

        self.close()

        while 1:
            try:
                line = self.output()
            except Driver.DriverException, e:
                raise MosflmException, e
            
            if not line:
                break
            
        self.kill()

        try:
            Messenger.log_write('Chmod-ing the jpeg images to mode 0644',
                                2)

            # chmod 0644 the images
            jpegfile = prefix + '_' + \
                       code + '.jpg'
            os.chmod(jpegfile, 0644)
            jpegfile = prefix + '_' + \
                       code + '_small.jpg'
            os.chmod(jpegfile, 0644)
        except:
            Messenger.log_write('... failed (files do not exist?)', 2)
        
        self.setWorkingDirectory(here)

        # there are no results from doing this - maybe a status should be
        # there ...

        return None

    def screen(self, input):
        '''Perform the screening step for Autoindex - this will ask the
        DiffractionImage module to screen each image in turn and will
        appropriately handle the error reporting etc.'''

        # This line should prevent the "sticky" ice flag reported
        # as bug 808.
        self.exclude_ice = False

        if input.name != 'index_request':
            raise MosflmException, 'bad input data'

        images = input.getImage()
        fileinfo = input.getFileinfo()

        directory = fileinfo.getDirectory()
        template = fileinfo.getTemplate()
        
        expression = r'(.*)_(\#*)\.(.*)'
        regexp = re.compile(expression)
        match = regexp.match(template)
        prefix = match.group(1)
        extension = match.group(3)
        length = len(match.group(2))
        
        format = '%0' + str(length) + 'd'
        
        for image in images:
            number = format % image
            filename = prefix + '_' + number + '.' + extension
            
            # then we may screen the image
            state = Screen.Screen(directory + '/' + filename)
            header = Screen.GetHeader(directory + \
                                      '/' + filename)
            self.headers[image] = header
            Somewhere.store('last header', header)
            
            if header['Message'] != ['']:
                Messenger.log_write(
                    'Warning recorded when opening ' + \
                    'image %s' % header['Message'], 2)
            Messenger.log_write("Pre-screened image %d - status = %s" % \
                                    (image, str(state)), 2)
                
            if state == 'blank':
                index_response = XSD.Index_response()
                status = XSD.Status()
                status.setCode('error')
                status.setMessage(directory + '/' + filename + ' is blank')
                index_response.setStatus(status)
                return index_response
            
            if state == 'icy':
                # record this for the Mosflm indexing
                self.exclude_ice = True
                
            if state == 'veryicy':
                # record this for the Mosflm indexing
                self.exclude_ice = True

                # Previously the processing stopped here but since we
                # agreed in the dna-dev meeting 27th-29th/09/2004 to
                # continue even if flagged as "veryicy" the following
                # code has been commented out (see bug 808):

                #index_response = XSD.Index_response()
                #status = XSD.Status()
                #status.setCode('error')
                #status.setMessage(directory + '/' + filename + \
                #                  ' is very icy!')
                #index_response.setStatus(status)
                #return index_response
            
            if state == 'backstop_problem':
                index_response = XSD.Index_response()
                status = XSD.Status()
                status.setCode('error')
                status.setMessage(directory + '/' + filename + \
                                  ' has a problem with backstop alignment')
                index_response.setStatus(status)
                return index_response

        # all is peachy - return the fact!
        index_response = XSD.Index_response()
        status = XSD.Status()
        status.setCode('ok')
        index_response.setStatus(status)
        if self.exclude_ice:
            # report the fact that ice rings were found
            index_response.setPowder_ice_rings('true')
        return index_response
            
    def autoindex(self, input, jpeg = 0):
        '''Perform a parallelised indexing task, indexing each of the images
        defined in the input individually and then all at once'''
        if input.name != 'index_request':
            raise MosflmException, 'bad input data'

        images = input.getImage()
        fileinfo = input.getFileinfo()

        # get the extra mosflm commands if they exist

        extra_commands = []
        
        extras = input.getExtra_commands()
        if extras != None:
            mosflm_commands = extras.getMosflm_commands()
            if mosflm_commands != None:
                commands = mosflm_commands.getCommand()
                if commands != None:
                    for command in commands:
                        extra_commands.append(command)
        
        # screen the images
        if len(images) > 1:
            directory = fileinfo.getDirectory()
            template = fileinfo.getTemplate()
            
            expression = r'(.*)_(\#*)\.(.*)'
            regexp = re.compile(expression)
            match = regexp.match(template)
            prefix = match.group(1)
            extension = match.group(3)
            length = len(match.group(2))
            
            format = '%0' + str(length) + 'd'

            # this needs to gather the screening results from a number
            # of images (shouldn't this actually be extracted from here???)
            # and do something clever with the screening results like
            # making jpeg images!
            
            index_response = self.screen(input)
            if index_response.getStatus().getCode() == 'error':
                return index_response


        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        if len(images) > 1:
            # then we need to perform a little parallel processing
            l = []
            for i in images:
                # s = copy of self
                # d = copy of data
                if self.drvOutputCopy:
                    temp = self.drvOutputCopy
                    self.drvOutputCopy = None
                else:
                    temp = None
                    
                if self.drvInputCopy:
                    temp2 = self.drvInputCopy
                    self.drvInputCopy = None
                else:
                    temp2 = None
                s = copy.deepcopy(self)
                d = copy.deepcopy(input)

                s.drvOutputCopy = temp
                self.drvOutputCopy = temp
                s.drvInputCopy = temp2
                self.drvInputCopy = temp2

                # hack around with the input data
                d.clearImage()
                d.addImage(i)

                # hack the configuration
                workingDirectory = here + '/autoindex_' + str(i)
                s.setWorkingDirectory(workingDirectory)

                try:
                    os.mkdir(workingDirectory)
                except:
                    pass

                b = Background.Background(s, 'autoindex', d)
                b.start()
                l.append(b)

            # now need to gather the results...

            # hack the configuration


            if self.getWorkingDirectory() == '':
                here = Driver.getcwd()
            else:
                here = self.getWorkingDirectory()

            directory_end = ''
            for img in images:
                directory_end += '_' + str(img)
            workingDirectory = here + '/autoindex_all'
            # + directory_end
            self.setWorkingDirectory(workingDirectory)
            
            try:
                os.mkdir(workingDirectory)
            except:
                pass

            try:
                self.realAutoindex(input, jpeg)
                

                self.results = { }

                # set results[-1] the overall result
                self.results[-1] = self.dnaResult
                
                j = 0
                for s in l:
                    # rejoin the thread
                    s.stop()
                    self.results[images[j]] = s.obj.dnaResult
                    j += 1

                # write out messages for which succeeded, which
                # did not

                Messenger.log_write('Indexing results by indexing task', 2)
                Messenger.log_write('Status for all images: %s' % \
                                    self.results[-1].getStatus().getCode(), 2)
                keys = self.results.keys()
                keys.sort()
                for key in keys[1:]:
                    Messenger.log_write(
                        'Status for image %d: %s' % \
                        (int(key),
                         self.results[key].getStatus().getCode()), 2)

                # finally need to merge the results - but only if
                # indexing all images together was successful...
                if self.results[-1].getStatus().getCode() != 'error':
                    self.dnaResult = \
                                   Expert.ConsiderIndexResponses(self.results)
                else:
                    self.dnaResult = Translation.Index_response_error(
                        'Error indexing all images together: %s ' % \
                        self.results[-1].getStatus().getMessage())
                    
                self.autoindexResult = self.dnaResult
                self.autoindexRequest = copy.deepcopy(input)
                
                # next ask the expert to look at the onconstrained orientation
                # matrix and have a guess at the lattice
                # switch this off for a moment
                if self.unconstrained_matrix and 1 == 0:
                    Expert.ConsiderUnconstrainedOrientation(self.unconstrained_matrix)


                # change the working directory back
                self.setWorkingDirectory(here)

                return self.dnaResult
            except MosflmException, e:
                # propogate this exception message to the caller as
                # an appropriate error response
                return Translation.Index_response_error('Error1! %s' % str(e))

            except exceptions.Exception, e:
                # something else went wrong!
                return Translation.Index_response_error('Error2! %s' % str(e))                

        else:
            # then actually perform some indexing on this image - no
            # parallelism required
            # do, or do not, there is no try
            try:
                return self.realAutoindex(input, jpeg)
            except MosflmException, e:
                return Translation.Index_response_error(e)

    def realAutoindex(self, input, jpeg):
        if input.name != 'index_request':
            raise MosflmException, 'bad input data'

        # wait for the files to exist before indexing
        time_waited = 0
        sleep_time = 15
        total_wait_time = int(Config.get('timeout'))
        while Exist.Fileinfo(input.getFileinfo(), input.getImage()) == 0:
            if time_waited > total_wait_time:
                raise MosflmException, \
                      'Waited for more %d seconds for image %s to appear' % \
                      (total_wait_time, str(input.getImage()))
            Messenger.log_write('Waiting for image %s' % str(input.getImage()),
                                2)
            time_waited += sleep_time
            time.sleep(sleep_time)

        self.start()

        # get the extra mosflm commands if they exist

        extra_commands = []
        
        extras = input.getExtra_commands()
        if extras != None:
            mosflm_commands = extras.getMosflm_commands()
            if mosflm_commands != None:
                commands = mosflm_commands.getCommand()
                if commands != None:
                    for command in commands:
                        extra_commands.append(command)

        for command in extra_commands:
            self.input(str(command))
        
        t = input.getTarget()
        if t != None:
            symmetry = t.getSymmetry()
            cell = t.getCell()
            if symmetry:
                if str(symmetry) != 'unknown':
                    self.input('symmetry ' + str(symmetry))
            if cell != None:
                if cell.getA():
                    self.input('cell ' + str(cell.getA()) + ' ' \
                               + str(cell.getB()) + ' ' \
                               + str(cell.getC()) + ' ' \
                               + str(cell.getAlpha()) + ' ' \
                               + str(cell.getBeta()) + ' ' \
                               + str(cell.getGamma()))
        
        f = input.getFileinfo()
        self.input('newmat autoindex.mat')
        self.input('directory ' + f.getDirectory())
        self.input('template ' + f.getTemplate())
        
        # detector type is now automatic - no need for the
        # detector command any longer!
        
        b = input.getBeam()
        if b:
            self.input('beam ' + str(b.getX()) + ' ' + str(b.getY()))

        i = input.getImage()
        # switch on the background writing output
        self.input('BEST ON')


        thresh = Somewhere.get('autoindex_i_over_sig')
        extra_input = ''
        if thresh:
            extra_input = ' thresh %f ' % thresh
        else:
            extra_input = ''

        max_cell = Somewhere.get('max_cell')
        if max_cell:
            extra_input += ' MAXCELL %f' % max_cell
        self.input('prerefine on')
        if self.exclude_ice:
            Messenger.log_write('Excluding spots in ice ring regions', 2)
            for image in i:
                if 'axis not phi' in self.headers[image]['Message']:
                    self.input(
                        'autoindex dps refine image %d phi %f %f exclude ice %s' % \
                        (image, self.headers[image]['PhiStart'], \
                         self.headers[image]['PhiEnd'], extra_input))
                else:
                    self.input('autoindex dps refine image ' + str(image) + \
                               ' exclude ice %s' % extra_input)
        else:
            for image in i:
                if 'axis not phi' in self.headers[image]['Message']:
                    self.input('autoindex dps refine image %d %s phi %f %f' % \
                               (image,
                                extra_input,
                                self.headers[image]['PhiStart'], \
                                self.headers[image]['PhiEnd']))
                else:
                    self.input('autoindex dps refine image ' + str(image) + \
                               extra_input)
        
        self.input('mosaic estimate')
        self.input('go')
        self.input('BEST OFF')

        if input.getJpeg() != None or jpeg != 0 or jpeg == 0:
            # make a jpeg image too...

            # first take to pieces the filename
            prefix, hash, extension = Bits.UnTemplate(f.getTemplate())
            length = len(hash)

            for image in i:
                code = Bits.Pad(image, length)
                self.input('xgui on')
                self.input('image ' + str(image))
                self.input('go')
                self.input('predict_spots')
                jpegfile = prefix + '_' + \
                           code + '.jpg'
                self.input('create_image binary true filename ' + jpegfile)
                jpegfile = prefix + '_' + code + \
                           '_pred.jpg'
                self.input('create_image prediction on ' \
                           + 'binary true filename ' + \
                           jpegfile)

                jpegfile = prefix + '_' + \
                           code + '_small.jpg'
                self.input('create_image zoom -4 binary true filename ' + jpegfile)
                jpegfile = prefix + '_' + code + \
                           '_pred_small.jpg'
                self.input('create_image zoom -4 prediction on ' \
                           + 'binary true filename ' + \
                           jpegfile)
                
                self.input('return')
                self.input('xgui off')
            
        self.close()

        while 1:
            try:
                line = self.output()
            except Driver.DriverException, e:
                raise MosflmException, e
            
            if not line:
                break
            
        self.kill()

        try:

            if self.getWorkingDirectory():
                self.result = Output.Output(self.getWorkingDirectory() + '/mosflm.stf')
            else:
                self.result = Output.Output('mosflm.stf')

        except exceptions.Exception, e:
            raise e
        
        # get from this result all of the things I may need
        # then translate the result to DNA speak
        
        # get out the unconstrained orientation matrix
        try:
            image = self.result.get('unconstrained_indexing_result', 'a_matrix', '11').keys()[0]
            self.unconstrained_matrix = { }
            for key in ['11', '12', '13', '21', '22', '23', '31', '32', '33']:
                self.unconstrained_matrix[key] = float(self.result.get('unconstrained_indexing_result', 'a_matrix', key)[image])
        except:
            self.unconstrained_matrix = None
        self.dnaResult = Translation.Index_response(self.result)

        return self.dnaResult

    def refine(self, input):
        '''Perform a cell refinement - this is assumed to be something where
        we will only perform one at a time, so there is no need to operate
        this job in parallel (i.e. split the task)'''

        # validate the input document
        if input.name != 'cell_refinement_request':
            raise MosflmException, 'bad input data'

        symmetry = input.getSymmetry()
        if symmetry != None:
            Messenger.log_write("Refining cell with symmetry %s" % symmetry, 2)
            directoryEnd = "_%s" % symmetry
        else:
            directoryEnd = ""

        # wait for the files to exist first need to construct a list of
        # used images
        starts = input.getStart_images().getImage()
        ends = input.getEnd_images().getImage()

        if len(starts) != len(ends):
            raise MosflmException, 'must have same number of start and ' + \
                  'end images'

        images = []
        for i in range(len(starts)):
            for j in range(starts[i], ends[i] + 1):
                images.append(j)
        while Exist.Fileinfo(input.getFileinfo(), images) == 0:
            time.sleep(1)
        # get the header information for each image
        for i in images:
            filename = Bits.Image(input.getFileinfo(), i)
            self.headers[i] = Screen.GetHeader(filename)

        # configure a working directory
        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = here + '/refine' + directoryEnd
        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir(workingDirectory)
        except:
            pass


        fileinfo = input.getFileinfo()

        beam = self.autoindexResult.getSolution().getRefinement().getBeam_shift().getNew_beam()
        mosaic = self.autoindexResult.getMosaicity_value()

        # that should be enough information to be going on, so should
        # be able to perform some processing now. Note well - this assumes
        # that there is a matrix file in the appropriate directory
        # named autoindex.mat which was the result of a previous round of
        # indexing.

        # also obtain the symmetry - unless already given!
        if symmetry == None:
            symmetry = str(self.autoindexResult.getSolution().getSymmetry())

        # start the mosflm job and get it doing some work
        self.start()

        # note well that the location of the autoindex matrix is a
        # funny one - should this file be somehow imported into the
        # data object so that it may be shipped around?

        # get the extra mosflm commands if they exist

        extra_commands = []
        
        extras = input.getExtra_commands()
        if extras != None:
            mosflm_commands = extras.getMosflm_commands()
            if mosflm_commands != None:
                commands = mosflm_commands.getCommand()
                if commands != None:
                    for command in commands:
                        extra_commands.append(command)

        for command in extra_commands:
            self.input(str(command))

        self.input("symmetry " + symmetry)
        self.input("matrix ../autoindex_all/autoindex.mat")
        self.input("newmat cellrefine.mat")
        self.input("mosaic " + str(mosaic))
        self.input("template " + str(fileinfo.getTemplate()))
        self.input("directory " + str(fileinfo.getDirectory()))
        self.input("beam " + str(beam.getX()) + " " + str(beam.getY()))

        segments = len(starts)
        
        self.input("postref segment " + str(segments))
        for i in range(segments):
            if 'axis not phi' in self.headers[starts[i]]['Message']:
                self.input('process %d to %d start %f angle %f' % \
                           (starts[i], ends[i], \
                            self.headers[starts[i]]['PhiStart'], \
                            self.headers[starts[i]]['PhiWidth']))
            else:
                self.input("process " + str(starts[i]) + " to " + str(ends[i]))
            self.input("run")

        # close and watch
        self.close()
        
        while 1:
            try:
                line = self.output()
            except Driver.DriverException, e:
                raise MosflmException, e

            if not line:
                break

        self.kill()
        
        if self.getWorkingDirectory():
            self.result = Output.Output(self.getWorkingDirectory() + '/mosflm.stf')
        else:
            self.result = Output.Output('mosflm.stf')


        self.dnaResult = Translation.Cell_refinement_response(self.result)
        self.cellRefineRequest = copy.deepcopy(input)
        self.cellRefineResult = self.dnaResult
        self.setWorkingDirectory(here)
        return self.dnaResult

    def strategy(self, input):
        '''Compute a strategy based on the previous indexing and the
        input information'''

        # note for me - I should be using the measured limit of the
        # integrated I/sigma as the resolution limit..

        if input.name != 'strategy_request':
            raise MosflmException, 'bad input data'

        # this is a bit of a pain really, because we will need to
        # index the images at the beginning of the strategy process to
        # properly configure the Mosflm common blocks, and so on

        # configure a working directory
        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = here + '/strategy'
        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir(workingDirectory)
        except:
            pass

        beam = self.autoindexResult.getSolution().getRefinement().getBeam_shift().getNew_beam()
        mosaic = self.autoindexResult.getMosaicity_value()

        if input.getSymmetry() == None:
            symmetry = str(self.autoindexResult.getSolution().getSymmetry())
        else:
            symmetry = str(input.getSymmetry())
        fileinfo = self.autoindexRequest.getFileinfo()

        # need to wade through a possibly non-existant data tree
        resolution = None
        overlap_limit = None
        if input.getStrategy_settings():
            if input.getStrategy_settings().getResolution():
                if input.getStrategy_settings().getResolution().getUpper():
                    resolution = str(input.getStrategy_settings().getResolution().getUpper())

            if input.getStrategy_settings().getOverlap_limit():
                overlap_limit = \
                              input.getStrategy_settings().getOverlap_limit()

        if not resolution:
            if hasattr(self, "resolutionForStrategy"):
                resolution = self.resolutionForStrategy

        Messenger.log_write("Strategy: Symmetry=" + symmetry)
        Messenger.log_write("Strategy: Resolution=%3.2f" % float(resolution))
        if overlap_limit:
            Messenger.log_write("Strategy: overlap limit is %2.1f" % \
                                overlap_limit)

        # the matrix files will be in either ../autoindex_all
        # or ../refine

        self.start()

        # get the extra mosflm commands if they exist

        extra_commands = []
        
        extras = input.getExtra_commands()
        if extras != None:
            mosflm_commands = extras.getMosflm_commands()
            if mosflm_commands != None:
                commands = mosflm_commands.getCommand()
                if commands != None:
                    for command in commands:
                        extra_commands.append(command)

        for command in extra_commands:
            self.input(str(command))


        if self.cellRefineResult:
            Messenger.log_write("NB: Using matrix from cell refinement", 2)
            Messenger.log_write("Using matrix ../refine/cellrefine.mat", 2)
            self.input("matrix ../refine/cellrefine.mat")
        else:
            Messenger.log_write("Using matrix ../autoindex_all/autoindex.mat",
                                2)
            self.input("matrix ../autoindex_all/autoindex.mat")

        self.input("symmetry " + symmetry)
        if resolution:
            self.input("resolution " + str(resolution))
        self.input("mosaic " + str(mosaic))
        self.input("template " + str(fileinfo.getTemplate()))
        self.input("directory " + str(fileinfo.getDirectory()))
        self.input("beam " + str(beam.getX()) + " " + str(beam.getY()))        

        # if we have done this stage, the the images must already exist
        # reindex first

        # now I don't do this - there is a better way to get the
        # image into memory

        use_autoindex = False

        if use_autoindex:
        
            for image in self.autoindexRequest.getImage():
                if 'axis not phi' in self.headers[image]['Message']:
                    self.input('autoindex dps image %d phi %f %f' % \
                               (image, self.headers[image]['PhiStart'], \
                                self.headers[image]['PhiEnd']))
                else:
                    self.input('autoindex dps image ' + str(image))
            self.input('go')

        else:
            self.input('xgui on')
            self.input('image %d' % \
                       self.autoindexRequest.getImage()[0])
            self.input('go')
            self.input('return')
            self.input('xgui off')

        # note well that use of this line requires Mosflm 6.2.5

        if overlap_limit:
            self.input('strategy testgen overlap %f' % overlap_limit)
        else:
            self.input('strategy testgen')
        self.input('go')

        self.close()

        while 1:
            line = self.output()
            if not line:
                break

        self.kill()

        if self.getWorkingDirectory():
            self.result = Output.Output(self.getWorkingDirectory() + '/mosflm.stf')
        else:
            self.result = Output.Output('mosflm.stf')

        self.dnaResult = Translation.Strategy_response(self.result)
        self.strategyRequest = copy.deepcopy(input)
        self.strategyResponse = self.dnaResult

        self.setWorkingDirectory(here)

        self.dnaResult = Expert.ConsiderStrategyResponse(self.autoindexResult,
                                                         self.dnaResult)
        
        return self.dnaResult

    def integrate(self, input):
        '''Integrate a set of data, in batches of n images, in parallel'''

        if input.name != 'integrate_request':
            raise MosflmException, 'bad input data'

        # take out the relevant data
        start = input.getStart()
        end = input.getEnd()

        if start == end:
            return self.realIntegrate(input)
        
        processors = input.getNumber_of_batches()

        left = end - start + 1
        first = start

        l = []

        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        for i in range(processors):
            batch = (end - first) / (processors - i)
            last = first + batch

            integrate_mtz = 'process_%d_%d.mtz' % (int(first), int(last))

            self.outputFiles.append(here + '/integrate_' + str(first) + \
                                    '_' + str(last) + '/' + \
                                    integrate_mtz)

            files = Somewhere.get('integrate_mtz')
            if not files:
                Somewhere.store('integrate_mtz', ['integrate_' + str(first) + \
                                                  '_' + str(last) + '/' + \
                                                  integrate_mtz])
            else:
                files.append('integrate_' + str(first) + \
                             '_' + str(last) + '/' + \
                             integrate_mtz)
                Somewhere.store('integrate_mtz', files)

            if self.drvOutputCopy:
                temp = self.drvOutputCopy
                self.drvOutputCopy = None
            else:
                temp = None
                    
            if self.drvInputCopy:
                temp2 = self.drvInputCopy
                self.drvInputCopy = None
            else:
                temp2 = None

            s = copy.deepcopy(self)
            d = copy.deepcopy(input)
            
            s.drvOutputCopy = temp
            self.drvOutputCopy = temp
            s.drvInputCopy = temp2
            self.drvInputCopy = temp2
            
            d.setStart(first)
            d.setEnd(last)
            d.setNumber_of_batches(1)

            b = Background.Background(s, 'realIntegrate', d)
            b.start()
            l.append(b)

            first = last + 1

        # self.realIntegrate(input)

        results = []

        for b in l:
            b.stop()
            results.append(b.obj.dnaResult)

        self.dnaResult = Expert.ConsiderIntegrateResponses(results)

        return self.dnaResult

    def single_integrate(self, input):

        if input.name != 'single_integrate_request':
            raise MosflmException, 'bad input data'

        # take out the relevant data
        images = input.getImage()

        Messenger.log_write('Single integrating images:' + str(images), 2)
        
        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        l = []

        for i in images:

            # this is wrong and probably broken!
            self.outputFiles.append(here + '/single_integrate_' + \
                                    str(i) + '/' + \
                                    'integrate.mtz')

            if self.drvOutputCopy:
                temp = self.drvOutputCopy
                self.drvOutputCopy = None
            else:
                temp = None
                    
            if self.drvInputCopy:
                temp2 = self.drvInputCopy
                self.drvInputCopy = None
            else:
                temp2 = None

            s = copy.deepcopy(self)
            
            s.drvOutputCopy = temp
            self.drvOutputCopy = temp
            s.drvInputCopy = temp2
            self.drvInputCopy = temp2

            d = XSD.Integrate_request()
            d.setFileinfo(input.getFileinfo())
            d.setExtra_commands(input.getExtra_commands())

            d.setStart(i)
            d.setEnd(i)

            b = Background.Background(s, 'realIntegrate', d)
            b.start()
            l.append(b)


        # Messenger.log_write('Have set off integration jobs')
        # Messenger.log_write('Here are the pointers:' + str(l))

        results = []

        try:
            for b in l:
                b.stop()
                results.append(b.obj.dnaResult)

            self.dnaResult = Expert.ConsiderIntegrateResponses(results)
            self.singleIntegrateResult = self.dnaResult

            return self.dnaResult
        except Driver.DriverException, e:
            # this probably results from a child process timing out!
            Messenger.log_write('Mosflm process timed out during' + \
                                ' of a single image')
            if str(e) != '':
                Messenger.log_write('Error %s' % str(e))
            return Translation.Integrate_response_error(str(e))
        
    def realIntegrate(self, input):
        '''Actually integrate a batch of images'''

        if input.name != 'integrate_request':
            raise MosflmException, 'bad input data'

        # take out the relevant data
        start = input.getStart()
        end = input.getEnd()
        processors = input.getNumber_of_batches()
        fileinfo = input.getFileinfo()

        while Exist.Fileinfo(fileinfo, [start]) == 0:
            time.sleep(5)

        # allow for writing to disk
        time.sleep(1)

        # get the header information from the first image
        filename = Bits.Image(fileinfo, start)
        self.headers[start] = Screen.GetHeader(filename)

        # that's all of the getting - now reconfigure our Mosflm
        # this will behave in a similar way to how it works for
        # indexing - this can take place all at once or one at a time
        
        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        if start == end:
            self.BEST = 'on'
            workingDirectory = here + '/integrate_' + str(start)

            # store this - for BEST! - key with "best" as a list...
            Somewhere.lock()
            bestinfo = Somewhere.get('best_hkl')
            if bestinfo:
                bestinfo.append('%s/bestfile.hkl' % workingDirectory)
            else:
                bestinfo = ['%s/bestfile.hkl' % workingDirectory]
            Somewhere.store('best_hkl', bestinfo)
            Somewhere.unlock()

            
            
        else:
            self.BEST = 'off'
            workingDirectory = here + '/integrate_' + str(start) + '_' + \
                               str(end)

        self.setWorkingDirectory(workingDirectory)
        
        integrate_mtz = 'process_%d_%d.mtz' % (int(start), int(end))

        try:
            os.mkdir(workingDirectory)
        except:
            pass

        # next see if we can access the results of cell refinement -
        # in particular the orientation matrix

        if self.cellRefineResult:
            # then we can get the matrix from ../refine
            Messenger.log_write('Using refined orientation', 2)
            matrix = '../refine/cellrefine.mat'
            mosaic = self.cellRefineResult.getRefined_mosaic_spread()
        else:
            matrix = '../autoindex_all/autoindex.mat'
            mosaic = self.autoindexResult.getMosaicity_value()

        beam = self.autoindexResult.getSolution().getRefinement().getBeam_shift().getNew_beam()
        symmetry = str(self.autoindexResult.getSolution().getSymmetry())

        self.start("HKLOUT %s" % integrate_mtz)
        # only write this message when we really start!
        if start != end:
            Messenger.log_write("Integrating image batch %d to %d" %
                                (start, end))
        else:
            Messenger.log_write("Integrating image batch %d to %d" %
                                (start, end), 2)

        # get the extra mosflm commands if they exist

        extra_commands = []
        
        extras = input.getExtra_commands()
        if extras != None:
            mosflm_commands = extras.getMosflm_commands()
            if mosflm_commands != None:
                commands = mosflm_commands.getCommand()
                if commands != None:
                    for command in commands:
                        extra_commands.append(command)

        for command in extra_commands:
            self.input(str(command))

        nullpix = Somewhere.get('NULLPIX')

        if nullpix:
            self.input('nullpix %d' % nullpix)

        self.input("symmetry " + symmetry)
        self.input("matrix " + matrix)
        self.input("newmat integrate.mat")
        if mosaic < 0.166:
            Messenger.log_write('Mosaic spread is %f - which is very low' % \
                                mosaic)
            default = Somewhere.get('default mosaic')
            if not default:
                default = 0.5
            Messenger.log_write('Will assume it is %f and not refine' % \
                                default)
            self.input('mosaic %f' % default)
            self.input('postref beam 0')
            Messenger.log_write('Note well - mosaic refinement switched off!')
            # cannot switch off postrefinement completely, since the
            # orientation and missetting angles need to be refined. if this
            # is not done, then the spot profiles get really nasty! - as
            # well as the profiles in phi
            # self.input('postref off')
        else:
            if self.cellRefineResult:
                # do not need to refine as we go
                self.input("postref fix all")
            self.input("mosaic " + str(mosaic))
        self.input("template " + str(fileinfo.getTemplate()))
        self.input("directory " + str(fileinfo.getDirectory()))
        self.input("beam " + str(beam.getX()) + " " + str(beam.getY()))

        try:
            total_wait_time = int(Config.get('timeout'))
        except:
            total_wait_time = 60

        # wait no minutes, total_wait_time seconds and add an extra 10
        # seconds on for good measure (see bug # 917)
        
        self.input('wait 0 %d 10' % total_wait_time)

        self.input("separation close")
        if 'axis not phi' in self.headers[start]['Message']:
            self.input('process %d to %d start %f angle %f' % \
                       (start, end, \
                        self.headers[start]['PhiStart'], \
                        self.headers[start]['PhiWidth']))
        else:
            self.input("process " + str(start) + " to " + str(end))
        if self.BEST == 'on':
            self.input('BEST ON')

        self.input("go")
        if self.BEST == 'on':
            self.input('BEST OFF')


        self.close()
        while 1:
            try:
                line = self.output()

                # record a little VERBOSE output
                
                if line[:17] == ' Processing Image':
                    Messenger.log_write('Processing image %d' % \
                                        int(line.split()[2]), 2)
                if line[:18] == ' Integrating Image':
                    Messenger.log_write('Integrating image %d' % \
                                        int(line.split()[2]), 2)
            except Driver.DriverException, e:
                raise MosflmException, e

            if not line:
                break

        self.kill()

        if self.getWorkingDirectory():
            self.result = Output.Output(self.getWorkingDirectory() + '/mosflm.stf')
            stf_file = self.getWorkingDirectory() + '/mosflm.stf'
        else:
            self.result = Output.Output('mosflm.stf')
            stf_file = 'mosflm.stf'

        if not start == end:
            # then we're doing a batch of integration, so try and store the
            # results - watch out for a race condition here!
            # add a mutex to the somewhere object then... :O)
            
            Messenger.log_write('Locking...', 2)
            Somewhere.lock()
            Messenger.log_write('Locked!', 2)
            try:
                output_bucket = Somewhere.get('integration_stuff')
                if not output_bucket:
                    output_bucket = Output.Output(stf_file)
                else:
                    output_bucket.append(stf_file)

                Somewhere.store('integration_stuff', output_bucket)
            except exceptions.Exception, e:
                Messenger.log_write('Error!!! %s' % str(e))
                
            Messenger.log_write('Unlocking...', 2)
            Somewhere.unlock()
            Messenger.log_write('Unlocked!', 2)

        self.dnaResult = Translation.Integrate_response(self.result)
        self.integrateRequest = input
        self.integrateResponse = self.dnaResult

        self.setWorkingDirectory(here)

        if start != end:
            Messenger.log_write("Finished integrating batch %d to %d" % \
                                (start, end))
        else:
            Messenger.log_write("Finished integrating batch %d to %d" % \
                                (start, end), 2)

        today = str(datetime.date.today())
        extra_prefix = str(fileinfo.getTemplate()).split('#')[0] + today + '_' 

        context = SchedulerDNAConfig.getDna_context()
        if context:
            logdir = context.getCurrent_log_dir()
        else:
            logdir = None
        if not logdir:
            if os.environ.has_key('DNALOGDIR'):
                logdir = dir = os.environ['DNALOGDIR']

        # copy the file to $DNALOGDIR
        if logdir and start != end:
            dir = logdir
            try:
                shutil.copyfile(here + '/integrate_' + str(start) + \
                                '_' + str(end) + '/' + \
                                integrate_mtz,
                                '%s/%s' % (dir, extra_prefix + integrate_mtz))
                Messenger.log_write('Copying mtz %s to %s' % \
                                    (here + '/integrate_' + str(start) + \
                                     '_' + str(end) + '/' + \
                                     extra_prefix + integrate_mtz, dir), 2)
            except:
                Messenger.log_write('Error copying mtz %s to %s' % \
                                    (here + '/integrate_' + str(start) + \
                                     '_' + str(end) + '/' + \
                                     extra_prefix + integrate_mtz, dir), 2) 
        # copy the log files to $DNALOGDIR

        name = here + '/integrate_' + str(start)
        if start != end:
            name += '_' + str(end) + '/'
        else:
            name += '/'
        if logdir:
            dir = logdir
            try:
                shutil.copyfile(name + 'mosflm.stf',
                                '%s/%s' % (dir, extra_prefix +
                                           'mosflm-%d-%d.stf' % (start, end)))
                shutil.copyfile(name + 'mosflm.log',
                                '%s/%s' % (dir, extra_prefix +
                                           'mosflm%d-%d.log' % (start, end)))
                Messenger.log_write('Copying log files to %s' % dir, 2)
            except exceptions.Exception, e:
                Messenger.log_write('Error copying log files to %s' % dir, 2)
                Messenger.log_write('Error: %s' % str(e), 2)

        return self.dnaResult



# the MosflmException, so that I may have an idea where things have gone
# wrong if I get an exception. This will be specific to those things which are
# Mosflm specific.

class MosflmException(exceptions.Exception):
    '''An exception for Mosflm'''

    def __init__(self, args):
        # a no brainer this one :o)
        self.args = args
        self.stack = traceback.extract_tb(sys.exc_traceback)

        # at this stage I also want to write an error report to the
        # standard error including all of the relevent information

        stderr = sys.stderr

        stderr.write('-------------------------------------------\n')
        stderr.write('MEX> %s\n' % str(args))
        for s in self.stack:
            stderr.write('MEXS> %s\n' % str(s))

        stderr.write('-------------------------------------------\n')

    def get(self):
        errors = self.args
        for s in self.stack:
            errors.append(s)
        return errors

        
if __name__ == '__main__':

    index_request = XSD.Index_request()
    
    document = "<?xml version='1.0'?><!DOCTYPE index_request><index_request><target><symmetry></symmetry><cell><a></a><b></b><c></c><alpha></alpha><beta></beta><gamma></gamma></cell></target><experiment><wavelength></wavelength><distance></distance></experiment><fileinfo><template>ref-random_3_###.img</template><directory>/data/graeme/dna-developer</directory><matrix_file></matrix_file></fileinfo><detector><type>adsc</type></detector><beam><x>93.2</x><y>93.6</y></beam><image>001</image><image>091</image></index_request>"
    index_request.unmarshal(document)
    
    m = Mosflm()
    index_response = m.autoindex(index_request)
    
    print 'Here is some solution information - the spacegroup'
    print index_response.getSolution().getSymmetry()
    print 'and the cell - as XML'
    print index_response.getSolution().getOrientation().getCell().marshal('cell')

    document = "<?xml version='1.0'?><!DOCTYPE cell_refinement_request><cell_refinement_request><fileinfo><template>postref-random_3_###.img</template><directory>/data/graeme/dna-developer</directory><matrix_file></matrix_file></fileinfo><start_images><image>1</image><image>4</image></start_images><end_images><image>3</image><image>6</image></end_images></cell_refinement_request>"

    cell_refinement_request = XSD.Cell_refinement_request()
    cell_refinement_request.unmarshal(document)
    cell_refinement_response = m.refine(cell_refinement_request)
    print cell_refinement_response.marshal('cell_refinement_response')

    strategy_response = m.strategy(XSD.Strategy_request())
    print strategy_response.marshal('strategy_response')

    integrate_request = XSD.Integrate_request()
    
    document = "<?xml version='1.0'?><!DOCTYPE integrate_request><integrate_request><fileinfo><template>random_3_###.img</template><directory>/data/graeme/dna-developer</directory><matrix_file></matrix_file></fileinfo><start>1</start><end>180</end><number_of_batches>6</number_of_batches></integrate_request>"
    
    integrate_request.unmarshal(document)
    integrate_response = m.integrate(integrate_request)
    print integrate_response.marshal('integrate_response')

    sys.path.append('../CCP4')

    import Sortmtz

    s = Sortmtz.Sortmtz()

    document = "<?xml version='1.0'?><!DOCTYPE sort_reflections_request><sort_reflections_request><sort_key>H K L M/ISYM BATCH</sort_key><input_reflections>"
    for hkl in m.outputFiles:
        document += "<hklin>" + hkl + "</hklin>"

    document += "</input_reflections><output_reflections><hklout>sorted.mtz</hklout></output_reflections></sort_reflections_request>"

    sort_reflections = XSD.Sort_reflections_request()
    sort_reflections.unmarshal(document)
    sort_response = s.sort(sort_reflections)

    print sort_response.marshal('status')

