#!/usr/bin/env python
############################################################
# LABELIT.py
# A wrapper for the program LABELIT developed by Nick Sauter
# and co at the lbl.
# Maintained by G.Winter
# 2nd August 2004
# 
# 
# $Id: LABELIT.py,v 1.4 2004/08/18 09:00:32 gwin Exp $
############################################################

# Note well - this ignores the fact that LABELIT is developed
# in Python and treats it as an external program

import os, sys, shutil

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

dna = os.environ['DNAHOME']

sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/scheduler/Scheduler/Driver')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')
sys.path.append(dna + '/xsd/python')

import Driver
import Bits
import XSD

from Messenger import Messenger

class LABELIT(Driver.Driver):

    def __init__(self):
        Driver.Driver.__init__(self, 1, 'labelit')
        self.setExecutable('labelit.screen')
        self.template = None
        self.directory = None
        self.images = []

    def autoindex(self, index_request):

        # first take the relevent information from the
        # index request

        self.template = index_request.getFileinfo().getTemplate()
        self.directory = index_request.getFileinfo().getDirectory()

        self.images = index_request.getImage()

        input = ''
        for i in self.images:
            input += ' ' + \
                     Bits.Image(index_request.getFileinfo(), i)


        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        # don't change this it works!
        # this should be /labelit
        workingDirectory = here + '/autoindex_all'

        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir(workingDirectory)
        except:
            pass
        
        self.start(input)

        # LABELIT takes no command-line input
        self.close()

        beam_x = 0.0
        beam_y = 0.0

        gather_labelit = False
        gather_mosflm = False

        labelit_answers = []
        mosflm_answers = []

        spots_used = 0
        rms_deviation = 0.0

        spacegroup = ''

        unit_cell = []
        orientation_matrix = []

        mosaic = 0.0

        no_spots = False
        mosflm_fail = False
        index_fail = False
        
        while 1:
            line = self.output()

            if not line:
                break

            if line[:22] == 'Too few unimodal peaks':
                no_spots = True
            if line[:23] == 'Too few candidate peaks':
                no_spots = True
            if line[:25] == 'No resolution bin profile':
                no_spots = True

            if line[:35] == 'MOSFLM logfile declares FATAL ERROR':
                mosflm_fail = True

            if line[:20] == 'No indexing solution':
                index_fail = True
                
            if gather_labelit:
                if line == '':
                    gather_labelit = False
                else:
                    Messenger.log_write(line[:-1])
                    labelit_answers.append(line.split())

            if line[0:25] == 'LABELIT Indexing results:':
                gather_labelit = True
                # eat some spare output
                for i in range(3):
                    line = self.output()
                    Messenger.log_write(line[:-1])
                    if line[0:11] == 'Beam center':
                        line = line.replace(',', '')
                        line = line.replace('mm', '')
                        list = line.split()
                        beam_x = float(list[3])
                        beam_y = float(list[5])


            if gather_mosflm:
                Messenger.log_write(line[:-1])
                mosflm_answers.append(line.split())

            if line[0:27] == 'MOSFLM Integration results:':
                gather_mosflm = True
                line = self.output()
            

        if no_spots:
            index_response = XSD.Index_response()
            status = XSD.Status()
            status.setCode('error')
            status.setMessage('No suitable spots on the image')
            index_response.setStatus(status)
            return index_response

        if mosflm_fail:
            index_response = XSD.Index_response()
            status = XSD.Status()
            status.setCode('error')
            status.setMessage('Mosflm integration of image failed')
            index_response.setStatus(status)
            return index_response

        if index_fail:
            index_response = XSD.Index_response()
            status = XSD.Status()
            status.setCode('error')
            status.setMessage('Indexing of diffraction pattern failed')
            index_response.setStatus(status)
            return index_response

        Messenger.log_write('Beam is %f %f' % (beam_x, beam_y))

        # pick the optimum solution - the first one in mosflm_answers which
        # begins with :)

        for m in mosflm_answers:
            if m[0] == ':)':
                # we have the best solution
                Messenger.log_write('Best solution is spacegroup - %s ' % \
                                    m[2])
                spacegroup = m[2]
                Messenger.log_write('Beam position %s %s' % \
                                    (m[3], m[4]))
                rms_deviation = float(m[-1])
                # look for the solution constants
                solution = m[1]
                mosaic = float(m[-2])
                for l in labelit_answers:
                    if len(l) == 0:
                        break
                    if l[0] == ':)':
                        number = l[1]
                        offset = 1
                    else:
                        number = l[0]
                        offset = 0


                    if number == solution:

                        if len(number) == 1:
                            matrix = 'index0' + number + '.mat'
                        else:
                            matrix = 'index' + number + '.mat'

                        spots_used = int(l[offset + 4])


                        # rename this something familiar to Mosflm.py
                        shutil.copyfile(self.getWorkingDirectory() + '/' +
                                        matrix, self.getWorkingDirectory()
                                        + '/autoindex.mat')

                        file = open(self.getWorkingDirectory() + '/' +
                                    matrix, 'r')
                        data = file.read()
                        file.close()

                        # amatrix is 0-8 of these
                        orientation_m = data.split()
                        orientation_matrix = []
                        for o in orientation_m:
                            orientation_matrix.append(float(o))

                        Messenger.log_write('Lattice is %s' % \
                                            l[offset + 6])
                        Messenger.log_write('Cell %s %s %s %s %s %s' % \
                                            (l[offset + 7], l[offset + 8],
                                             l[offset + 9], l[offset + 10],
                                             l[offset + 11], l[offset + 12]))

                        unit_cell = []
                        for i in range(offset + 7, offset + 13):
                            unit_cell.append(float(l[i]))

                        Messenger.log_write('Used %d spots' % spots_used)

        self.kill()

        index_response = XSD.Index_response()
        status = XSD.Status()

        status.setCode('ok')

        index_response.setStatus(status)

        spot_search_response = XSD.Spot_search_response()
        spot_search_response.setFound(spots_used)
        spot_search_response.setUsed(spots_used)
        spot_search_response.setRejected(0)

        index_response.setSpot_search_response(spot_search_response)

        solution = XSD.Solution()

        new = XSD.Beam()
        new.setX(float(beam_x))
        new.setY(float(beam_y))

        shift = XSD.Beam()
        shift.setX(0.0)
        shift.setY(0.0)
        
        refinement = XSD.Refinement()

        beam_shift = XSD.Beam_shift()
        beam_shift.setNew_beam(new)
        beam_shift.setShift(shift)
        
        refinement.setBeam_shift(beam_shift)

        reflections = XSD.Reflections()
        reflections.setUsed(spots_used)
        reflections.setUsed_in_indexing(spots_used)

        refinement.setReflections(reflections)

        solution.setRefinement(refinement)

        solution.setSpot_deviation(rms_deviation)
        
        initial_cell = XSD.Cell()
        initial_orientation_matrix = XSD.Matrix()
        refined_cell = XSD.Cell()

        properties = ['A', 'B', 'C', 'Alpha', 'Beta', 'Gamma']
        for i in range(6):
            property = properties[i]
            method = getattr(initial_cell, 'set%s' % property)
            method(unit_cell[i])
            method = getattr(refined_cell, 'set%s' % property)
            method(unit_cell[i])

        for property in ['11', '12', '13', '21', '22', '23', '31', '32', '33']:
            method = getattr(initial_orientation_matrix, 'setE%s' % property)
            method(0.0)

        a_matrix = XSD.Matrix()
        u_matrix = XSD.Matrix()
        
        # populate these from the matrix file data stored in
        # orientation_matrix

        a_matrix.setE11(orientation_matrix[0])
        a_matrix.setE12(orientation_matrix[1])
        a_matrix.setE13(orientation_matrix[2])
        a_matrix.setE21(orientation_matrix[3])
        a_matrix.setE22(orientation_matrix[4])
        a_matrix.setE23(orientation_matrix[5])
        a_matrix.setE31(orientation_matrix[6])
        a_matrix.setE32(orientation_matrix[7])
        a_matrix.setE33(orientation_matrix[8])

        u_matrix.setE11(orientation_matrix[12])
        u_matrix.setE12(orientation_matrix[12])
        u_matrix.setE13(orientation_matrix[12])
        u_matrix.setE21(orientation_matrix[12])
        u_matrix.setE22(orientation_matrix[12])
        u_matrix.setE23(orientation_matrix[12])
        u_matrix.setE31(orientation_matrix[12])
        u_matrix.setE32(orientation_matrix[12])
        u_matrix.setE33(orientation_matrix[12])
        
        initial = XSD.Initial()

        initial.setCell(initial_cell)
        initial.setOrientation_matrix(initial_orientation_matrix)
        solution.setInitial(initial)

        orientation = XSD.Orientation()
        orientation.setCell(refined_cell)
        orientation.setU_matrix(u_matrix)
        orientation.setA_matrix(a_matrix)
        
        solution.setOrientation(orientation)
        
        solution.setNumber(0)
        solution.setSpacegroup_number(0)
        solution.setSymmetry(spacegroup)

        index_response.setSolution(solution)
        index_response.setMosaicity_value(mosaic)

        return index_response
