#!/usr/bin/env python
# Driver.py
# Maintained by G.Winter
# Part of the second version of the Scheduler code - version 1.1
# 19th January 2004
# 
# This section of code will declare the generic class, which contains the 
# functionality required to start, control and end a child process being 
# controlled via standard io channels.
# 
# $Id: Driver.py,v 1.29 2006/03/30 08:59:53 gwin Exp $

# imports - system first

import sys, fcntl, string, time, shutil, select
import popen2, os, signal, thread, threading
import exceptions

# tie in the DNA messenger - this is a temporary measure
# if not os.environ.has_key('DNAHOME'):
#     raise RuntimeError, 'DNAHOME not defined'
#
# dna = os.environ['DNAHOME']
# sys.path.append(dna + '/scheduler/Scheduler/Mosflm')
from Messenger import Messenger

# this mutex will prevent damage to the starting of child
# processes - where the working directory must be set.
# a process may have at most one working directory.
driverStartMutex = thread.allocate_lock()

processTable = []
processesKilled = False
processTableMutex = thread.allocate_lock()

# access the processor stack
import Scheduler

import os

debug_scheduler = False

if os.environ.has_key('DNA_DEBUG_SCHEDULER'):
    if os.environ['DNA_DEBUG_SCHEDULER'] == 'on':
        debug_scheduler = True

def getcwd():
    '''A thread safe getcwd'''
    driverStartMutex.acquire()
    cwd = os.getcwd()
    driverStartMutex.release()
    return cwd

class Driver:
    '''A class to drive external binary applications'''

    def __init__(self, debug = 20, name = ''):
        '''Create a driver class - this doesnt do much'''
        self.drvExecutable = ''
        self.drvName = name
        self.drvInputCopy = None
        self.drvOutputCopy = None
        self.drvWorkingDirectory = ''
        self.drvPipe = None
        self.drvDebug = debug
        self.drvTimeout = 0
        self.drvCommandList = []

        self.drvLogFile = None

    # getters & setters - these are used from the inheriting application
    # class to configure the Driver aspect of itself. Note well - the
    # Driver specific attributes are prefixed drv to ensure that there is
    # less chance of a namespace overlap.

    def setExecutable(self, executableName):
        self.drvExecutable = executableName

    def getExecutable(self):
        return self.drvExecutable

    def setInputCopy(self, method):
        self.drvInputCopy = method

    def setOutputCopy(self, method):
        self.drvOutputCopy = method

    def setWorkingDirectory(self, directory):
        self.drvWorkingDirectory = directory

    def getWorkingDirectory(self):
        return self.drvWorkingDirectory

    def setTimeout(self, value):
        self.drvTimeout = value

    # real methods now

    def checkMakeWorkingDirectory(self):
        '''Check that the working directory (if set)
        exists - if not, mkdir() it
        is writable - if not raise.'''

        if not self.drvWorkingDirectory:
            return

        if not os.path.exists(self.drvWorkingDirectory):
            os.mkdir(self.drvWorkingDirectory)

        if not os.access(self.drvWorkingDirectory,
                         os.W_OK):
            raise RuntimeError, 'directory "%s" not writable' % \
                  self.drvWorkingDirectory
        
        # all is well...

        return

    def start(self, command = None):
        '''Make a connection to the child process - which involves opening
        a pipe to it'''
        if self.drvDebug > 0 and self.drvPipe:
            raise DriverException, 'pipe already open'

        if self.drvPipe:
            return

        if processesKilled == True:
            raise DriverException, 'reset abort'

        # Messenger.log_write('Waiting for a CPU')

        # try and grab a processor to work with
        Scheduler.grab()

        # Messenger.log_write('Have a CPU')

        # for this method the mutex should always be obtained, since the
        # situation could be confused if a task with no working directory
        # tried to do something.

        # a better way might be to raise an exception if no working directory
        # is configured - this way it would ensure that no sloppy programming
        # can take place - but add a "setWorkingDirectoryCWD()" type method -
        # for the lazy!

        # check the executable is set
        if self.drvExecutable == '':
            raise DriverException, ' no executable set'

        self.checkMakeWorkingDirectory()
        
        if self.drvWorkingDirectory != '':
            # then I will need to change directory

            # given that a process may have only one current working
            # directory, a mutex is required to ensure that the behaviour
            # of this system may be predicted.
            driverStartMutex.acquire()

            if self.drvDebug > 10:
                # sys.stderr.write('changing directory from ')
                pass
            cwd = os.getcwd()
            if self.drvDebug > 10:
                # sys.stderr.write(str(cwd) + ' to ')
                # sys.stderr.write(self.drvWorkingDirectory + '\n')
                pass
            try:
                os.chdir(self.drvWorkingDirectory)
            except:
                # free up resources
                # Messenger.log_write('Freeing CPU')
                Scheduler.release()
                driverStartMutex.release()
                raise DriverException, 'unable to change directory to %s' % \
                      self.drvWorkingDirectory

        # open the connection to the executable - this should be wrapped in
        # a try, except mechanism in case the executable does not actually
        # exist.

        self.drvCommandList = []
        if command:
            self.drvCommandList.append(command)
            

        

        if command:
            qrsh_command = "qrsh 'cd %s; %s %s'" % \
                           (self.drvWorkingDirectory, self.drvExecutable,
                            command)
        else:
            qrsh_command = "qrsh 'cd %s; %s'" % \
                           (self.drvWorkingDirectory, self.drvExecutable)
            
        try:
            # this should trap the case where it it not there
            # sge = Sun Grid Engine
            use_sge = False
            if os.environ.has_key('DNA_USE_SGE'):
                if os.environ['DNA_USE_SGE'] == 'on':
                    use_sge = True

            if use_sge:
                self.drvPipe = popen2.Popen4(qrsh_command)
            else:
                if command:
                    self.drvPipe = popen2.Popen4(self.drvExecutable + " " + \
                                                 command)
                else:
                    self.drvPipe = popen2.Popen4(self.drvExecutable)
                    
        except:
            if driverStartMutex.locked() and self.drvWorkingDirectory != '':
                driverStartMutex.release()
            Scheduler.release()

            raise DriverException, \
                  'unable to start child executable "%s" - may not exist!' % \
                  self.drvExecutable.split()[0]

        time.sleep(0.1)

        if debug_scheduler:
            print 'DEBUG: Starting process %s pid %d in %s' % \
                  (self.drvExecutable, self.drvPipe.pid, \
                   str(self.drvWorkingDirectory))
            

        if self.drvPipe.poll() != -1:
            if driverStartMutex.locked() and self.drvWorkingDirectory != '':
                os.chdir(cwd)
                driverStartMutex.release()
            Scheduler.release()
            
            raise DriverException, \
                  'failed to start child executable "%s" with command line %s in %s' % \
                  (self.drvExecutable.split()[0], command, self.drvWorkingDirectory)

        # next open a channel to store the output

        logfile = self.drvName
        if logfile == '':
            logfile = self.drvExecutable.split(' ')[0].split('/')[-1]

        try:
            self.drvLogFile = open('%s.log' % logfile, 'w')
        except exceptions.Exception, e:
            if debug_scheduler:
                print 'Using /dev/null for log file'
                print 'Got exception %s' % str(e)
            self.drvLogFile = open('/dev/null', 'w')

        if self.drvWorkingDirectory != '':
            if self.drvDebug > 10:
                pass
            os.chdir(cwd)
            driverStartMutex.release()

        # add the newly created driver object to the list of current
        # processes

        processTableMutex.acquire()
        processTable.append(self)
        processTableMutex.release()


    def kill(self):
        '''Terminate the child process - this is designed to implement the
        abort type method - so will be useful if the child process gets
        stuck in a tight loop, which may happen'''
        if self.drvPipe == None:
            if self.drvDebug > 0:
                raise DriverException, 'killing an already stopped process'
            return

        # release the allocated resources
        # Messenger.log_write('Freeing CPU')
        Scheduler.release()

        # remove me from the process table
        processTableMutex.acquire()
        if processTable.count(self):
            processTable.remove(self)
        processTableMutex.release()

        try:
            self.drvPipe.tochild.close()
        except:
            pass
        try:
            self.drvPipe.fromchild.close()
        except:
            pass
        try:
            self.drvPipe.childerr.close()
        except:
            pass
        
        try:
            # this can raise an exception if the child process is no longer
            # running - what else could cause this exception - permissions?
            os.kill(self.drvPipe.pid, signal.SIGKILL)
        except:
            # but we won't worry about that - because it is still dead
            pass


        self.return_code = self.drvPipe.poll()

        self.drvLogFile.close()
        self.drvPipe = None

    # some more sophisticated shell like job control

    def pause(self):
        '''Stop a process in its tracks, to allow restarting later on'''

        if self.drvPipe == None:
            raise DriverException, 'no child process to stop'

        try:
            os.kill(self.drvPipe.pid, signal.SIGSTOP)
        except:
            pass

    def unpause(self):
        '''Restart a stopped process'''

        if self.drvPipe == None:
            raise DriverException, 'no child process to restart'

        try:
            os.kill(self.drvPipe.pid, signal.SIGCONT)
        except:
            pass

    def input(self, line, newline = 1):
        '''Print a line to the standard input channel of the child process'''

        # record the command 
        self.drvCommandList.append(line)


        if debug_scheduler:
            print 'DEBUG COMMAND %d %s' % \
                  (self.drvPipe.pid, line)
        
        if self.drvPipe == None:
            raise DriverException, 'no open channel to child'
        try:
            if newline:
                self.drvPipe.tochild.write(line + '\n')
            else:
                self.drvPipe.tochild.write(line)
        except:
            raise DriverException, 'error writing to child'

        if self.drvInputCopy != None:
            try:
                self.drvInputCopy(line)
            except:
                raise DriverException, 'error copying line'

    def output(self):
        '''Read a line from the standard output of the child process'''
        if self.drvPipe == None:
            raise DriverException, 'no channel open to child'

        # if we have configured a time out on this channel, then we
        # have to fiddle around for a bit with things like select to
        # see if the child process is still alive - see above with the
        # kill method.
        
        if self.drvTimeout > 0:
            line = None
            while line == None:
                # this will select a list of (either 0 or 1) channels from
                # the list - and this is where the time out is used.
                channel = select.select([self.drvPipe.fromchild], [], [], \
                                        self.drvTimeout)
                if channel[0] == []:
                    # this means that there is nothing coming back from the
                    # child within the timeout duration, so kill it just in
                    # case it is in a tight loop.
                    # FIXME this is commented out
                    # to try and debug BUG # 1602
                    # self.kill()
                    # raise DriverException, 'child process timed out'
                    Messenger.log_write('Time out happened....')
                else:
                    try:
                        # to read a line from the standard output of the
                        # child process
                        line = self.drvPipe.fromchild.readline()
                        if self.drvOutputCopy:
                            self.drvOutputCopy(line)
                        self.drvLogFile.write(line)
                        return line
                    except:
                        # something must have come out, but not a whole line.
                        pass

        else:
            # just wait "forever" from output from the child - this could be
            # a long time in coming! there is no time out mechanism here.
            line = self.drvPipe.fromchild.readline()
            if self.drvOutputCopy:
                self.drvOutputCopy(line)
            self.drvLogFile.write(line)
            return line

    def close(self):
        '''Close the connection to the child process - this will simply
        close the input channels, so that we can be sure that this is
        flushed - the equivalent of ^D'''
        if self.drvPipe == None:
            if self.drvDebug > 0:
                raise DriverException, 'no channel to child open'
            return

        # note that the output pipes remain open for the duration
        self.drvPipe.tochild.close()
        # write a command log out
        if self.drvWorkingDirectory != '':
            file = open(self.drvWorkingDirectory + '/.commands', 'w')
        else:
            file = open('.commands', 'w')
        for command in self.drvCommandList:
            file.write(command + '\n')
        file.close()

    def debug(self):
        return self.drvCommandList

    def close_kill(self):
        '''Close the input channel, read all of the output and then kill
        the subprocess, tidying up as we go.'''

        # close stdin to child
        self.close()

        while 1:
            # read all of the output
            line = self.output()
            if not line:
                break

        # return exit status
        return self.kill()


def Abort():
    '''This will KILL all current running processes - which will in turn
    cause the object run methods to return'''

    processesKilled = True

    while len(processTable) > 0:
        for p in processTable:
            try:
                # try an old school killing
                os.kill(p.drvPipe.pid,
                        signal.SIGKILL)
            except:
                pass
        time.sleep(1.0)


def Reset():
    '''Reset the abort flag'''
    processTableMutex.acquire()
    processesKilled = False
    processTableMutex.release()

# next an exception for all of the "driver" software - which is used as the
# lower level not program specific kernel of the scheduler.

# this is pretty much a no-brainer, except for the fact that Python
# recognises this as an exception.

class DriverException(exceptions.Exception):
    '''a specific exception for the driver class'''

    def __init__(self, args):
        self.args = args

if __name__ == '__main__':
    # then run a test - this will use a Driver to mail the contents of the
    # file to me... which is a little recursive for my liking!

    # note that this does not inherit from Driver - I should really inherit
    # to a class Mail or something.

    d = Driver(0, 'mail')

    # under SuSE linux, the mail program is /usr/bin/nail - this could be a
    # problem on other systems.
    
    d.setExecutable('/usr/bin/nail -s "testing Driver.py" g.winter@dl.ac.uk')
    d.setWorkingDirectory('/tmp')
    d.start()

    f = open('Driver.py', 'r')
    for l in f.readlines():
        d.input(l, 0)
        d.pause()
        d.unpause()

    f.close()
    d.close()

    # I (graeme) should now receive another email with this file in!
