#!/usr/bin/env python
# Truncate.py
# Maintained by G.Winter
# 14th July 2004
# A part of the second generation scheduler.
# 
# This is a wrapper for the CCP4 program "Truncate", which is used for 
# correcting weak and negative intensities when calculating structure 
# factor amplitudes. The output is a set of SF amplitudes suitable, for
# instance, for running MR
# 
# $Id: Truncate.py,v 1.5 2004/10/22 07:00:38 gwin Exp $


import os

if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 not found'
if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

import sys

dna = os.environ['DNAHOME']
sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/scheduler/Scheduler/Driver')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')

import Driver
import Output
import Exist
import CCP4
import CCP4Translation
import XSD
from Somewhere import Somewhere

import Mattprob

# at the moment this is much simplified, so that the only options are
# to give HKLIN, HKLOUT and the sequence or number of residues

class Truncate(Driver.Driver):
    '''A class to execute truncate'''

    def __init__(self):
        Driver.Driver.__init__(self)

        self.setExecutable('truncate')
        self.hklin = ''
        self.hklout = ''
        self.sequence = ''
        self.residues = 0

    def setHklout(self, hklout):
        self.hklout = hklout

    def setHklin(self, hklin):
        self.hklin = hklin

    def setSequence(self, sequence):
        self.sequence = sequence
        self.residues = len(sequence)

    def setResidues(self, residues):
        self.sequence = ''
        self.residues = residues

    def run(self):
        '''Actually start the truncate procedure'''

        if not self.hklout:
            raise CCP4.CCP4Exception, 'no HKLOUT specified'

        if self.hklin == '':
            raise CCP4.CCP4Exception, 'no HKLIN specified'

        # do not need this any more! :O)
        # if self.residues == 0:
        # raise CCP4.CCP4Exception, 'number of residues not set'

        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = here + '/truncate'

        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir(workingDirectory)
        except:
            pass

        resolution = Somewhere.get('Complete resolution')
        if resolution and self.residues:
            # then we have a chance to determine nmol/asu
            # although this is the complete resolution not the
            # diffracting resolution
            nmol = Mattprob.calculate_nmol(self.hklin, resolution, \
                                           self.residues)

            self.residues = nmol * self.residues

        command = 'DNA truncate.stf HKLIN %s HKLOUT %s' % \
                  (self.hklin, self.hklout)

        self.start(command)

        label = Somewhere.get('HKLOUT_label')
        if label:
            line = 'LABOUT'
            for tag in ['F', 'SIGF', 'DANO', 'SIGDANO', 'F(+)', 'SIGF(+)', 'F(-)', 'SIGF(-)', 'IMEAN', 'SIGIMEAN', 'I(+)', 'SIGI(+)', 'I(-)', 'SIGI(-)', 'ISYM']:
                line += ' %s=%s_%s' % (tag, tag, label)
            self.input(line)
        
        if self.residues > 0:
            self.input('nres %d' % self.residues)
        self.close()

        while 1:
            try:
                line = self.output()
            except DriverException, e:
                raise CCP4.CCP4Exception, e

            if not line:
                break

        self.kill()

        if self.getWorkingDirectory():
            self.result = Output.Output(self.getWorkingDirectory() + \
                                        '/truncate.stf')
        else:
            self.result = Output.Output('truncate.stf')

        self.dnaResult = \
                       CCP4Translation.Truncate_reflections_response(
            self.result)


        self.setWorkingDirectory(here)
        return self.dnaResult


if __name__ == '__main__':
    # run a unit test
    hklin = '/data/graeme/images/scale/scaled.mtz'
    hklout = '/tmp/truncate.mtz'
    residues = 223

    t = Truncate()

    t.setHklin(hklin)
    t.setHklout(hklout)
    t.setResidues(residues)

    result = t.run()

    print result.marshal()
    
