#!/usr/bin/env python
# Reindex.py
# Maintained by G.Winter
# 22nd September 2004
# A part of the second generation scheduler
# 
# This is a wrapper for the CCP4 program "reindex" which will take 
# unmerged input reflections for one spacegroup and reindex them to 
# another asymmetric unit for another spacegroup. This will be useful
# if the reflections are integrated in P1 and the spacegroup is
# being searched for.
#
# $Id: Reindex.py,v 1.8 2005/11/22 13:38:15 svensson Exp $


import os

if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 not found'
ccp4 = os.environ['CCP4']
if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

import sys

dna = os.environ['DNAHOME']

sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/scheduler/Scheduler/Driver')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')
sys.path.append(dna + '/expertise/python/graeme/Expert')
import Expert

import Driver
import Output
import Exist
import CCP4
import XSD
import CCP4Translation
import Rebatch

from Messenger import Messenger

# nota bene - this will have to take account of the reciprocal cell axes,
# since for p3 the proper arrangement is to have gamma = 120 - and so on.
# this will have to be built in I guess.... bummer!

# Comment 2/2/05 Pierre LeG has something for this? There is cell constraint
# code in xupy.

class Reindex(Driver.Driver):
    '''A class to execute reindex'''

    def __init__(self):
        Driver.Driver.__init__(self)
        self.setExecutable('%s/bin/reindex' % ccp4)
        self.hklin = ''
        self.hklout = ''
        self.spacegroup = ''

    def setHklin(self, hklin):
        self.hklin = hklin    

    def setHklout(self, hklout):
        self.hklout = hklout

    def setSpacegroup(self, spacegroup):
        self.spacegroup = spacegroup

    def reindex(self):
        self.run()

    def run(self):
        if self.hklin == '':
            raise CCP4.CCP4Exception('No HKLIN set for reindex')
        if self.hklout == '':
            raise CCP4.CCP4Exception('No HKLOUT set for reindex')
        if self.spacegroup == '':
            raise CCP4.CCP4Exception('No SPACEGROUP set for reindex')

        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = here + '/reindex'
        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir (workingDirectory)
        except:
            pass

        self.start('HKLIN %s HKLOUT %s' % \
                   (self.hklin, self.hklout))
        self.input('SYMM %s' % self.spacegroup)
        # this will depend on the spacegroup and the current set of
        # cell parameters which will need to be obtained from the MTZ file
        
        # self.input('reindex axis b*, c*, a*')

        self.close()

        while 1:
            try:
                line = self.output()
            except DriverException, e:
                    raise CCP4.CCP4Exception, e

            if not line:
                break

        self.kill()

        self.setWorkingDirectory(here)

        return CCP4Translation.Reindex_reflections_response()

def subgroups_old(pointgroup):
    '''Return a list of sub-pointgroups for a given point group'''

    if pointgroup == '4':
        return ['4', '222', '2', '1']
    if pointgroup == '23':
        return ['23', '222', '2', '1']
    if pointgroup == '6':
        return ['6', '3', '1']
    if pointgroup == '222':
        return ['222', '2', '1']
    if pointgroup == '2':
        return ['2', '1']

    return [pointgroup]

def subgroups(spacegroup):
    '''Return a list of possible spacegroups (name, no)
    based on Expert.PossibleSpacegroups()'''

    return Expert.PossibleSpacegroups(spacegroup.lower())

import Sortmtz
import Scala

def DeterminePointGroup(hklin,
                        hklout_prefix,
                        current_space_group,
                        start_batch = None,
                        end_batch = None):
    '''Work through all of the subgroups of current_space_group
    and scale - recording the Rmerge value. This Rmerge value will
    then be used to decide if current_space_group is appropriate
    or if a lower symmetry would be appropriate. hklout_prefix
    is used to make temporary file names - this should be something
    like /tmp/pointless/proc5_1_000 (say)'''

    # no longer just working wirth "222"
    # pointgroup = current_space_group[1:]
    # pointgroups = subgroups(pointgroup)

    # spacegroups = []
    # for p in pointgroups:
    # spacegroups.append('p%s' % p)

    spacegroup_number = { }
    spacegroups = []
    subgroup_and_numbers = subgroups(current_space_group)
    for s in subgroup_and_numbers:
        spacegroups.append(s[0])
        spacegroup_number[s[0]] = s[1]

    # at this stage I should have a single MTZ file as input
    # for each spacegroup now sort/scale/merge and record Rmerge
    
    rmerges = { }

    if start_batch and end_batch:
        Messenger.log_write(
            'Using batches %d to %d for point group determination' % \
            (start_batch, end_batch))

        # Now use the new Rebatch object to sort this out...

        rebatch = Rebatch.Rebatch()

        new_hklin = '%s-rebatch.mtz' % hklout_prefix

        rebatch.setHklin(hklin)
        rebatch.setHklout(new_hklin)
        rebatch.range(start_batch, end_batch)
        rebatch.run()

        hklin = new_hklin
        start_batch = None
        end_batch = None

    for s in spacegroups:

        # reindex
        reindex = Reindex()
        reindex.setHklin(hklin)
        reindex.setHklout('%s-reindex-%s.mtz' % \
                          (hklout_prefix, s))
        reindex.setSpacegroup(s)
        reindex.reindex()

        # sort
        sortmtz = Sortmtz.Sortmtz()
        sortmtz.setHklin('%s-reindex-%s.mtz' % \
                         (hklout_prefix, s))
        sortmtz.setHklout('%s-sort-%s.mtz' % \
                          (hklout_prefix, s))
        sortmtz.sort()

        # scale
        scala = Scala.Scala()

        if start_batch and end_batch:
            scala.setStart_batch(start_batch)
            scala.setEnd_batch(end_batch)
        
        scala.setHklin('%s-sort-%s.mtz' % \
                       (hklout_prefix, s))
        scala.setHklout('%s-scale-%s.mtz' % \
                        (hklout_prefix, s))
        scala.setAnomalous_scattering('off')
        scala.setBfactor_refinement('on')
        scala.setCycle_limit(20)
        scala.setSpacing(5)
        scala.setSecondary(6)

        result = scala.scale()

        if result.getStatus() == 'error':
            Messenger.log_write('Error scaling data in spacegroup %s' % \
                                s)
        else:
            rmerges[s] = float(result.getRmerge())
            Messenger.log_write('Scaling data in spacegroup %s gave r=%2.2f' % \
                                (s, float(result.getRmerge())))

    # now I should have a list of spacegroups and an rmerge value for
    # each - so have a quick analyse in reverse (climbing the spacegroup
    # tree)

    benchmark = rmerges['p1']
    # spacegroups.reverse()
    correct = 'p1'

    # this is the initial best 
    best = spacegroup_number['p1']
    rmerge_best = benchmark
    for i in range(1,len(spacegroups)):
        s = spacegroups[i]
        # this should relate to the sqrt(order) of the spacegroup
        # for instance sqrt(12.0) for p23 or sqrt(4) for p222
        # since this should relate to how Rmerge increases
        # that is, compared to the p1 rmerge.
        if rmerges[s] < 2.0 * rmerge_best:
            # then this one is probably compatable with the truth!
            if best < spacegroup_number[s]:
                correct = s
                best = spacegroup_number[s]
                rmerge_best = rmerges[s]
        else:
            pass

    Messenger.log_write('Selected spacegroup %s as correct' % correct)

if __name__ == '__main__':
    spacegroup = 'p23'
    hklin = '/tmp/beer/integrate_1_45/integrate.mtz'
    hklout_prefix = 'foo'
    DeterminePointGroup(hklin,
                        hklout_prefix,
                        spacegroup)

if __name__ == '__moon__':
    # then test the code

    pointgroup = '23'
    pointgroups = subgroups(pointgroup)
    spacegroups = []
    for p in pointgroups:
        spacegroups.append('p%s' % p)

    hklin = '/tmp/demo/integrate_1_45/integrate.mtz'

    import Sortmtz
    import Scala

    rmerges = []

    for s in spacegroups:
        hklout = '/tmp/demo/reindex-%s.mtz' % s

        r = Reindex()
        r.setHklin(hklin)
        r.setHklout(hklout)
        r.setSpacegroup(s)

        r.reindex()

        # next also sort the files

        s2 = Sortmtz.Sortmtz()

        hklout2 = '/tmp/demo/sort-%s.mtz' % s

        s2.setHklin(hklout)
        s2.setHklout(hklout2)

        s2.sort()

        sc = Scala.Scala()
        
        hklout3 = '/tmp/demo/scale-%s.mtz' % s

        sc.setHklin(hklout2)
        sc.setHklout(hklout3)

        sc.setAnomalous_scattering('off')
        sc.setBfactor_refinement('on')
        sc.setCycle_limit(20)
        sc.setSpacing(5)
        sc.setSecondary(6)
        sc.setResolution(2.8)

        try:
            result = sc.scale()
            rmerge = result.getRmerge()
        except:
            # something went wrong - probably indicates wrong spacegroup
            # this should now be impossible with the pointgroups being
            # derived from the possibles based on the lattice...
            rmerge = 1.0

        rmerges.append(rmerge)

        print 'Spacegroup %s Rmerge %f' % (s, rmerge)

    for i in range(1, len(rmerges)):
        if rmerges[i] > rmerges[0] * 2.0:
            print 'Probable spacegroup %s' % spacegroups[i - 1]
            break
