#!/usr/bin/env python
# Rebatch.py
# Maintained by G.Winter
# 4th January 2005
# 
# This is a wrapper for the CCP4 program "rebatch" which offers the 
# functions of changing the batch numbers for measurements in unmerged
# MTZ files - useful for
#  - adding a constant value (e.g. 1000) to batch numbers for merging 
#    multiple files
#  - cropping a data set to say batches 1 to 90
# 
# $Id: Rebatch.py,v 1.3 2005/11/22 13:38:15 svensson Exp $

import os

if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 not found'
ccp4 = os.environ['CCP4']
if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

import sys

dna = os.environ['DNAHOME']

sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/scheduler/Scheduler/Driver')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')

import Driver
import Output
import CCP4
import XSD
import CCP4Translation

from Messenger import Messenger

class Rebatch(Driver.Driver):
    '''A class to execute rebatch'''

    def __init__(self):
        Driver.Driver.__init__(self)
        self.setExecutable('%s/bin/rebatch' % ccp4)
        self.hklin = ''
        self.hklout = ''
        self.batch_add = 0
        self.batch_start = 0
        self.batch_end = 0

    def setHklin(self, hklin):
        self.hklin = hklin    

    def setHklout(self, hklout):
        self.hklout = hklout

    def range(self, start, end):
        self.batch_start = start
        self.batch_end = end

    def add(self, add):
        self.batch_add = add

    def rebatch(self):
        self.run()

    def run(self):
        # check that the input is sensible and consistent
        if self.batch_add > 0 and \
               (self.batch_start > 0 or self.batch_end > 0):
            raise CCP4.CCP4Exception('Cannot increase batch numbers and ' + \
                                     'limit batches')
        if self.hklin == '':
            raise CCP4.CCP4Exception('No HKLIN set for reindex')
        if self.hklout == '':
            raise CCP4.CCP4Exception('No HKLOUT set for reindex')

        if self.getWorkingDirectory() == '':
            here = Driver.getcwd()
        else:
            here = self.getWorkingDirectory()

        workingDirectory = here + '/rebatch'
        self.setWorkingDirectory(workingDirectory)

        try:
            os.mkdir (workingDirectory)
        except:
            pass

        self.start('HKLIN %s HKLOUT %s' % \
                   (self.hklin, self.hklout))

        if self.batch_add > 0:
            Messenger.log_write('Adding %d to all batches in %s' % \
                                (self.batch_add, self.hklin))

            self.input('BATCH ALL ADD %d' % self.batch_add)

        else:
            Messenger.log_write('Limiting batches to %d to %d' % \
                                (self.batch_start, self.batch_end))

            if self.batch_start > 0:
                self.input('BATCH 0 TO %d REJECT' % (self.batch_start - 1))

            if self.batch_end > 0:
                self.input('BATCH %d to 10000 REJECT' % (self.batch_end + 1))

        self.close()

        while 1:
            try:
                line = self.output()
            except DriverException, e:
                    raise CCP4.CCP4Exception, e

            if not line:
                break

        self.kill()

        self.setWorkingDirectory(here)

if __name__ == '__main__':
    # test

    r = Rebatch()
    r.setHklin('../foo.mtz')
    r.setHklout('../bat.mtz')
    
    r.range(1, 45)

    r.run()

    
