#!/usr/bin/env python
# Mattprob.py
# Maintained by G.Winter
# 21st October 2004
# 
# Implementing the cell volume calculations of Kantardjieff and Rupp,
# Protein Science volume 12, 2002. Uses "mattprob_params.dat" from
# http://www-structure.llnl.gov/mattprob
# 
# 
# $Id: Mattprob.py,v 1.3 2005/11/22 13:38:15 svensson Exp $

import os, sys, math

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME must be defined'

dna = os.environ['DNAHOME']

sys.path.append('%s/expertise/python/graeme/Expert' % dna)
sys.path.append('%s/scheduler/Scheduler/Mosflm' % dna)

import Expert
from Messenger import Messenger
import Mtzq
import Matthews_coef

def to_float(list):
    result = []
    for l in list:
        result.append(float(l))
    return result

def calculate(volume, mass, resolution):
    file = open('%s/third_party/data/mattprob_params.dat' % dna, 'r')
    while 1:
        line = file.readline()
        if not line[0] == '#':
            break
    
    resolutions = to_float(line.split(','))[:13]
    P0_list = to_float(file.readline().split(','))[:13]
    Vm_bar_list = to_float(file.readline().split(','))[:13]
    w_list = to_float(file.readline().split(','))[:13]
    A_list = to_float(file.readline().split(','))[:13]
    s_list = to_float(file.readline().split(','))[:13]
    file.close()

    if resolution > resolutions[-1]:
        raise RuntimeError, 'Resolution outside useful range'
    if resolution < resolutions[0]:
        Messenger.log_write('Resolution higher than peak %f -> %f' % \
                            (resolution, resolutions[0]))
        resolution = resolutions[0]
                            

    start = 0

    for start in range(12):
        if resolution > resolutions[start] and \
           resolution < resolutions[start + 1]:
            break

    # can start at start - interpolate

    diff = (resolution - resolutions[start]) / \
           (resolutions[start + 1] - resolutions[start])

    P0 = P0_list[start] + diff * (P0_list[start + 1] - P0_list[start])
    Vm_bar = Vm_bar_list[start] + \
             diff * (Vm_bar_list[start + 1] - Vm_bar_list[start])
    w = w_list[start] + diff * (w_list[start + 1] - w_list[start])
    A = A_list[start] + diff * (A_list[start + 1] - A_list[start])
    s = s_list[start] + diff * (s_list[start + 1] - s_list[start])

    nmols = []
    pdfs = []
    
    nmol = 1
    Vm = volume / (mass * nmol)
    while 1:
        z = (Vm - Vm_bar) / w
        p = P0 + A * math.exp(- math.exp(-z) - z * s + 1)
        nmols.append(nmol)
        pdfs.append(p)
        nmol += 1
        Vm = volume / (mass * nmol)
        if Vm < 1.0:
            break

    return nmols, pdfs

# most of this information (apart from the number of residues) can
# come from the mtz file via Mtzq()

def nmol2(a, b, c, alpha, beta, gamma, spacegroup, resolution, mass):
    return nmol(a, b, c, alpha, beta, gamma, spacegroup, resolution,
                mass / 128.0)

def nmol(a, b, c, alpha, beta, gamma, spacegroup, resolution, residues):
    '''Calculate the most likely number of molecules of length
    residues in the asymmetric unit of cell and spacegroup diffracting
    to resolution.'''

    V = Expert.CellVolume(a, b, c, alpha, beta, gamma)
    order = Expert.SpacegroupOrder(spacegroup.lower())
    
    # we want nmol/asu so...
    mass = 128.0 * residues * order

    molecules, probabilities = calculate(V, mass, resolution)

    ptot = 0

    for i in range(len(molecules)):
        p = probabilities[i]
        ptot += p

    mbest = 0
    pbest = 0.0
    for i in range(len(molecules)):
        m = molecules[i]
        p = probabilities[i] / ptot
        if p > pbest:
            pbest = p
            mbest = m

    Messenger.log_write('Calculated nmol based on method by Kantartdjieff' + \
                        ' and Rupp (2002)')
    Messenger.log_write('Best nmol/asu found: %d (p = %f)' % \
                        (mbest, pbest))
    return mbest

def calculate_nmol(hklin, resolution, residues):
    m = Mtzq.Mtzq(hklin)
    spacegroup = m.getSpacegroup()
    cell = m.getCell()
    return nmol(cell[0], cell[1], cell[2], cell[3], cell[4], cell[5],
                spacegroup, resolution, residues)

def solvent(a, b, c, alpha, beta, gamma, spacegroup, nres, nmol):
    m = Matthews_coef.Matthews_coef()
    m.setCell(a, b, c, alpha, beta, gamma)

    # this needs to use the short name of the spacegroup - so make sure
    # that is the case...

    spacegroup = Expert.ShortSpacegroupName(spacegroup)

    m.setSymmetry(spacegroup)
    m.setMolweight(128.0 * nres)

    return m.matthews_coef()[nmol]

if __name__ == '__main__':
    calculate_nmol('/data/graeme/sfgh/peak.mtz', 1.7, 289)
