#!/usr/bin/env python
############################################################
# CCP4Interface.py
# Maintained by G.Winter
# 2nd April 2004
# 
# An interface to the classes which define the CCP4 programs.
# Note well, that these will probably NOT be stateful.
# 
# Programs catered for to date, and pending
# sortmtz
# scala
# 
# $Id: CCP4Interface.py,v 1.10 2005/11/22 13:38:15 svensson Exp $
############################################################

import os, sys

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

dna = os.environ['DNAHOME']

sys.path.append(dna + '/xsd/python')
sys.path.append(dna + '/expertise/python/graeme/Expert')
sys.path.append(dna + '/scheduler/Scheduler/Mosflm')

from Messenger import Messenger

import Sortmtz, CCP4, XSD, Scala, Expert, Truncate, Mtzutils
import exceptions

def sortmtz(input):
    '''Sort the reflections in an MTZ file'''

    if input.name != 'sort_reflections_request':
        raise CCP4.CCP4Exception, 'incorrect input type for sortmtz'

    hklin = input.getInput_reflections().getHklin()
    hklout = input.getOutput_reflections().getHklout()
    order = input.getSort_key()

    s = Sortmtz.Sortmtz()

    s.clearHklin()
    for h in hklin:
        s.addHklin(h)
    s.setHklout(hklout)
    s.setOrder(order)

    response = s.sort()

    return response

def scala(input):
    '''Scale the reflections in an MTZ file'''

    if input.name != 'scale_reflections_request':
        raise CCP4.CCP4Exception, 'incorrect input type for scala'

    hklin = input.getInput_reflections().getHklin()
    hklout = input.getOutput_reflections().getHklout()
    scaling_options = input.getScaling_options()

    polish = scaling_options.getUnmerged_polish_output()

    s = Scala.Scala()

    for h in hklin:
        s.setHklin(h)
    s.setHklout(hklout)
    s.setCycle_limit(scaling_options.getCycle_limit())
    s.setAnomalous_scattering(scaling_options.getAnomalous_scattering())
    s.setBfactor_refinement(scaling_options.getBfactor_refinement())
    s.setSpacing(scaling_options.getSpacing())
    s.setSecondary(scaling_options.getSecondary())

    if polish:
        s.setPolish(polish)

    start = scaling_options.getStart()
    end = scaling_options.getEnd()

    if start and end:
        s.setStart_batch(start)
        s.setEnd_batch(end)

    # try and get the resolution limits out
    upper = None
    lower = None
    
    try:
        resolution = scaling_options.getResolution()
        upper = resolution.getUpper()
        lower = resolution.getLower()
    except:
        pass
    
    s.setResolution(upper, lower)

    # next try and get the sdcorrection parameters
    try:
        sd = scaling_options.getStandard_deviation_parameters()
        sdfac_full = sd.getSdfac_full()
        sdadd_full = sd.getSdadd_full()
        sdfac_partial = sd.getSdfac_partial()
        sdadd_partial = sd.getSdadd_partial()
        s.setSD_full(sdfac_full, sdadd_full)
        s.setSD_partial(sdfac_partial, sdadd_partial)
    except exceptions.Exception, e:
        pass

    response = s.scale()

    response = Expert.ConsiderScaleReflectionsResponse(response)

    if polish:
        # call this again but this time without the instruction to
        # make the polish file - this should ensure that the log files
        # for the last scaling remain!
        Messenger.log_write(
            'Rescaling without producing unmerged polish file %s' % \
            polish)
        scaling_options.setUnmerged_polish_output(None)
        input.setScaling_options(scaling_options)
        return scala(input)

    return response

def truncate(input):
    '''Truncate the reflections in an MTZ file'''

    if input.name != 'truncate_reflections_request':
        raise CCP4.CCP4Exception, 'Incorrect input type for truncate'

    hklin = input.getInput_reflections().getHklin()
    hklout = input.getOutput_reflections().getHklout()
    truncate_options = input.getTruncate_options()

    # no longer need this test TRUNCATE can guess from volume
    if truncate_options:
        if (not truncate_options.getAa_sequence()) and \
               (not truncate_options.getResidue_count() and \
                False):
            raise CCP4.CCP4Exception, \
                  'must set either sequence or number of residues'

    t = Truncate.Truncate()
    t.setHklin(hklin[0])
    t.setHklout(hklout)

    if truncate_options:
        if truncate_options.getAa_sequence():
            t.setSequence(truncate_options.getAa_sequence())
        if truncate_options.getResidue_count():
            t.setResidues(truncate_options.getResidue_count())

    results = t.run()

    return results

def limit_resolution(hklin, hklout, resolution_limit):
    '''Limit the resolution of data in hklin when copying to hklout'''

    m = Mtzutils.Mtzutils()
    m.setHklin(hklin)
    m.setHklout(hklout)
    m.setResolution_limit(resolution_limit)
    m.process()

if __name__ == '__main__':

    s = XSD.Sort_reflections_request()

    xml = "<?xml version='1.0'?><!DOCTYPE sort_reflections_request><sort_reflections_request><input_reflections><hklin>graeme_1_001.mtz</hklin></input_reflections><output_reflections><hklout>test_ccp4_interface_sort.mtz</hklout></output_reflections><sort_key>H K L M/ISYM BATCH</sort_key></sort_reflections_request>"

    s.unmarshal(xml)

    result = sortmtz(s)

    print result.marshal()

    xml = "<?xml version='1.0'?><!DOCTYPE scale_reflections_request><scale_reflections_request><input_reflections><hklin>test_ccp4_interface_sort.mtz</hklin></input_reflections><output_reflections><hklout>test_ccp4_interface_scale.mtz</hklout></output_reflections><scaling_options><cycle_limit>20</cycle_limit><anomalous_scattering>off</anomalous_scattering><bfactor_refinement>off</bfactor_refinement><spacing>5</spacing><secondary>6</secondary><resolution><upper><isotropic>2.00</isotropic></upper></resolution></scaling_options></scale_reflections_request>"

    s = XSD.Scale_reflections_request()
    s.unmarshal(xml)

    result = scala(s)

    print result.marshal()
