// PeakPythonInterface.cpp
// maintained by G.Winter
// 6th January 2004
// 
// A Python interface to the PeakLists - this will be exported into the 
// DiffractionImage module along with the rest of the code, so the only
// difference will be that there are two different classes available in the
// one module.
// 
// This is designed to be rather more object orientated than before.
//
// $Id: PeakPythonInterface.cpp,v 1.5 2004/10/06 15:04:35 gwin Exp $

#ifdef INCLUDE_PYTHON

// this is only meaningful if we are working in Python land

#include "DiffractionImage.h"
#include "Peak.h"

// make all of the C++ code from the DiffractionImage and PeakList classes
// available to us

using namespace DI;

// these bits are important for the exception handling mechanisms and 
// point to things which are in DiffractionImagePythonInterface.cpp

extern PyObject * PyDiffractionImageException;

// note - the class itself will be exported into Python as a PeakList to
// maintain the exact same API - as far as possible. The only difference is
// that the Peaks in question will be returned as tuples into Python rather
// than as a different class, since the latter would be excessively expensive
// and a bit of a waste of time.

// prefix in this file will be pl, in the same way as the prefix in the
// DiffractionImagePythonInterface was di.

// here are a few forward references to allow the structures to be
// correctly configured later on in the file

// I think that these implement the len() and [] operators.

static PyObject * plLength(PeakList * self,
			   PyObject * args);
static PyObject * plSubscript(PeakList * self,
			      PyObject * args);

static PyObject * plGetAttribute(PeakList * self,
				 char * attributeName);

static void plDeAlloc(PyObject * self);

// this is to prevent memory leaks from the VARARGS parsing
static PyObject * plOtherLength(PeakList * self,
				PyObject * args)
{
  return PyInt_FromLong(self->length());
}

static PyMappingMethods plMappings = {
  (inquiry) plOtherLength,
  (binaryfunc) plSubscript,
  0
};

static PyTypeObject plType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "PeakList",
  sizeof(PeakList),
  0,
  (destructor) plDeAlloc,
  0,
  (getattrfunc) plGetAttribute,
  0,
  0,
  0,
  0,
  0,
  & plMappings,
  0
};

// now the actual implementation of the methods - for example the 
// Python constructor code and whatnot.

// with the constructor, the only possible argument is a pointer to an
// image, so that the PeakList can be initialised as a list of points from 
// that image - for this to work I will have to actually probe the 
// type of the arguments - this is probably something which will be 
// prone to going wrong, so check this.

PyObject * plPeakList(PyObject * self,
		      PyObject * args)
{
  // the DiffractionImage pointer is just in case we are performing 
  // a peak search as a part of the construction process.

  PyDiffractionImage * diffractionImage;
  PeakList * peakList;

  if (PyTuple_Size(args) > 1)
    {
      // then we have too many arguments to this function
      PyErr_SetString(PyDiffractionImageException,
		      "bad arguments");
      return NULL;
    }

  // first create the basic Python object

  peakList = PyObject_New(PeakList, & plType);

  if (peakList == NULL)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "error allocating storage for PeakList");
      return NULL;
    }

  // next have a look at the argument list - if there is an argument there
  // try and get at it as a DiffractionImage structure

  if (PyTuple_Size(args) == 0)
    {
      // then we have no argument - so just go ahead and
      // create the new Python object
      peakList->constructor();
      return (PyObject *) peakList;
    }

  // if we get to here then we must have an argument, which must in turn 
  // be tested for DiffractionImage-ness.

  diffractionImage = (PyDiffractionImage *) PyTuple_GetItem(args, 0);

  // check that this is indeed a DiffractionImage

  if (strcmp(diffractionImage->ob_type->tp_name,
	     "DiffractionImage") != 0)
    {
      // then this is not a DiffractionImage - clear up the mess
      // and raise an exception
      PyObject_Del((PyObject *) peakList);
      PyErr_SetString(PyDiffractionImageException,
		      "argument is not a DiffractionImage");
      return NULL;
    }

  // this must be a proper DiffractionImage structure - so go ahead
  // and call the constructor

  try
    {
      peakList->constructor(&(diffractionImage->diffractionImage));
    }
  catch (DiffractionImageException & e)
    {
      // an error occurred during the construction - so destruct
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      peakList->destructor();
      PyObject_Del((PyObject *) peakList);
      return NULL;
    }

  // if we reach this stage then all is peachy
  return (PyObject *) peakList;
}

static void plDeAlloc(PyObject * self)
{
  // free up this object
  PeakList * peakList = (PeakList *) self;

  // call the destructor
  peakList->destructor();

  PyObject_Del(self);
}

// that is the constructor and destructor handled - now for a couple of
// real methods - ones which do something proper.

// the following methods require implementation
// 
// int length(void) - tell the length of the list
// void clear(void) - clear the list
// IGNORE void add(float, float, float)
// IGNORE void remove(int)
// int find(DiffractionImage *) - return the number found
// (float, float, float) get(int) - get the info about a peak
// (float, float, float) [int] - likewise, as an array subscription operator
// (int, float, float, float) circle(int, float) - perform a circle search
// int remove(float, float, float, float) - remove those peaks on the circle
// 
// yes, there is some overlap in the name department, but this
// will have to be addressed by some type checking - or the removal of 
// fine grained access to the list (the add and remove operations)
//
// indeed, I think that this is the best way to proceed

// methods for the class implementation

static PyObject * plLength(PeakList * self,
			      PyObject * args)
{
  if (!PyArg_ParseTuple(args, ":PeakList"))
    {
      // should I raise an error here - this is something which will
      // need to be decided globally
      return NULL;
    }

  // this should always work - but wrap in a try/catch just to be sure

  try
    {
      return PyInt_FromLong(self->length());
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
    }

  return NULL;
}

static PyObject * plClear(PeakList * self,
			  PyObject * args)
{
  if (!PyArg_ParseTuple(args, ":PeakList"))
    {
      return NULL;
    }

  try
    {
      self->clear();
      Py_INCREF(Py_None);
      return Py_None;
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
    }

  return NULL;
}

// perform a peak search on the image - this will take the following argument 
// list:
// 
// p.find(image, max, thresh)
// image - DiffractionImage instance - required
// max - the maximum number of peaks to find - optional
// thresh - the I/sigma threshold to use - optional

static PyObject * plFind(PeakList * self,
			 PyObject * args)
{
  // these default values will be ignored - they are just to prevent compiler
  // warnings
  float thresh = 3.0;
  int max = 100000;
  PyDiffractionImage * diffractionImage;

  int size = PyTuple_Size(args);

  if (size == 0)
    {
      // raise an exception
      PyErr_SetString(PyDiffractionImageException,
		      "this requires at least one argument "
		      "(a DiffractionImage)");
      return NULL;
    }

  if (size > 3)
    {
      // then also be picky and raise an exception
      PyErr_SetString(PyDiffractionImageException,
		      "this takes at most three arguments, "
		      "DiffractionImage, max, threshold");
      return NULL;
    }

  // next get the ptr to the DiffractionImage
  diffractionImage = (PyDiffractionImage *) PyTuple_GetItem(args, 0);

  // check that this is a ptr to a DiffractionImage

  if (strcmp(diffractionImage->ob_type->tp_name,
	     "DiffractionImage") != 0)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "argument is not a DiffractionImage");
      return NULL;
    }

  if (size > 1)
    {
      // then get the max out
      if (!PyInt_Check(PyTuple_GetItem(args, 1)))
      {
	// then raise an exception
	PyErr_SetString(PyDiffractionImageException,
			"argument should be integer");
	return NULL;
      }

      max = PyInt_AsLong(PyTuple_GetItem(args, 1));
    }

  if (size > 2)
    {
      // then also get the I/sigma threshold out
      if (!PyFloat_Check(PyTuple_GetItem(args, 2)))
	{
	  PyErr_SetString(PyDiffractionImageException,
			  "argument should be a float");
	  return NULL;
	}

      thresh = PyFloat_AsDouble(PyTuple_GetItem(args, 2));


    }

  if (size == 1)
    {
      // then just perform the peak search and return
      try
	{
	  self->find(&(diffractionImage->diffractionImage));
	  self->reciprocal(diffractionImage->diffractionImage.getDistance(),
			   diffractionImage->diffractionImage.getWavelength());
	}
      catch (DiffractionImageException & e)
	{
	  PyErr_SetString(PyDiffractionImageException,
			  e.getMessage().c_str());
	  return NULL;
	}

      // return a helpful bit of information - the number of peaks found
      return PyInt_FromLong(self->length());
    }

  if (size == 2)
    {
      // then we have been passed in a maximum number of peaks
      try
	{
	  self->find(&(diffractionImage->diffractionImage), max);
	}
      catch (DiffractionImageException & e)
	{
	  PyErr_SetString(PyDiffractionImageException,
			  e.getMessage().c_str());
	  return NULL;
	}

      // return a helpful bit of information - the number of peaks found
      return PyInt_FromLong(self->length());
    }
  
  if (size == 3)
    {
      // then we have also passed in the threshold
      try
	{
	  self->find(&(diffractionImage->diffractionImage), max, thresh);
	}
      catch (DiffractionImageException & e)
	{
	  PyErr_SetString(PyDiffractionImageException,
			  e.getMessage().c_str());
	  return NULL;
	}

      // return a helpful bit of information - the number of peaks found
      return PyInt_FromLong(self->length());
    }

  // we should never get to here

  return NULL;
}

// next the methods to fetch bits of information from the 
// PeakList - for example the actual values of the points

static PyObject * plGet(PeakList * self,
			PyObject * args)
{
  PyObject * result;
  int offset;
  Peak peak;

  if (!PyArg_ParseTuple(args, "i", &offset))
    {
      return NULL;
    }

  try 
    {
      peak = self->get(offset);
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }

  result = PyTuple_New(3);

  if (result == NULL)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "error allocating result storage");
      return NULL;
    }

  PyTuple_SetItem(result, 0, PyFloat_FromDouble(peak.x));
  PyTuple_SetItem(result, 1, PyFloat_FromDouble(peak.y));
  PyTuple_SetItem(result, 2, PyFloat_FromDouble(peak.intensity));
  
  // Py_INCREF(result);

  return result;
}

static PyObject * plSubscript(PeakList * self,
			      PyObject * arg)
{
  // this is different from the above, because arg is not a tuple
  Peak peak;
  int offset;

  if (!PyInt_Check(arg))
    {
      PyErr_SetString(PyDiffractionImageException,
		      "argument should be an integer");
      return NULL;
    }

  offset = PyInt_AsLong(arg);

  try
    {
      peak = (*self)[offset];
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }

  PyObject * result = PyTuple_New(3);

  if (result == NULL)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "error allocating result storage");
      return NULL;
    }

  PyTuple_SetItem(result, 0, PyFloat_FromDouble(peak.x));
  PyTuple_SetItem(result, 1, PyFloat_FromDouble(peak.y));
  PyTuple_SetItem(result, 2, PyFloat_FromDouble(peak.intensity));
  
  // Py_INCREF(result);

  return result;
}

// now that is all of the "getting at things" methods wrapped it is 
// time to access some real functionality - like circle finding

// this will have the following argument parameters
// p.circle(iter, width)
// integer iter - optional - the number of iterations of circle finding to do,
//                           which will default to 1000.
// float width - optional - the error allowed in including a point within a
//                          a circle - this should be given a default value
//                          determined by the spread of points I guess.
//                          for the moment just use 5.0 :o)
//
// returns (count, x, y, r)

static PyObject * plCircle(PeakList * self,
			   PyObject * args)
{
  float width = 5.0;
  int iter = 1000;
  int size;

  // results from the circle finding - passed by reference
  float x, y, r;
  int count;

  size = PyTuple_Size(args);

  if (size > 0)
    {
      if (!PyInt_Check(PyTuple_GetItem(args, 0)))
	{
	  // then throw a wobbler
	  PyErr_SetString(PyDiffractionImageException,
			  "argument should be an integer");
	  return NULL;
	}
      
      iter = PyInt_AsLong(PyTuple_GetItem(args, 0));
    }

  if (size > 1)
    {
      if (!PyFloat_Check(PyTuple_GetItem(args, 1)))
	{
	  PyErr_SetString(PyDiffractionImageException,
			  "argument should be a float");
	  return NULL;
	}

      width = PyFloat_AsDouble(PyTuple_GetItem(args, 1));
    }

  try
    {
      count = self->circle(iter, width, x, y, r);
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }

  PyObject * result = PyTuple_New(4);

  if (result == NULL)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "error allocating result space");
      return NULL;
    }

  PyTuple_SetItem(result, 0, PyInt_FromLong(count));
  PyTuple_SetItem(result, 1, PyFloat_FromDouble(x));
  PyTuple_SetItem(result, 2, PyFloat_FromDouble(y));
  PyTuple_SetItem(result, 3, PyFloat_FromDouble(r));

  // Py_INCREF(result);

  return result;
}

// next the removal method - this will delete those spots found to be on
// a well defined circle

// This takes the following arguments, all non optional, all floats:
// 
// width, x, y, r
// 
// where x and y define the central coordinates of the circle and r
// is the radius. width has the same definition as above.
//
// this returns a count of the number of spots removed, and is designed to
// work in tandem with circle() above.

static PyObject * plRemove(PeakList * self,
			   PyObject * args)
{
  float width, x, y, r;
  int count;

  if (!PyArg_ParseTuple(args, "ffff", &width, &x, &y, &r))
    {
      return NULL;
    }

  try
    {
      count = self->remove(width, x, y, r);
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }

  return PyInt_FromLong(count);
}

// that should be all of the methods together, so now I just need to glue
// them all together

static PyMethodDef plMethods[] = 
  {
    {"length", (PyCFunction) plLength, METH_VARARGS, "Get length of PeakList"},
    {"clear", (PyCFunction) plClear, METH_VARARGS, "Clear PeakList"},
    {"find", (PyCFunction) plFind, METH_VARARGS, "Find peaks"},
    {"get", (PyCFunction) plGet, METH_VARARGS, "Get a peak"},
    {"circle", (PyCFunction) plCircle, METH_VARARGS,
     "Look for rings"},
    {"remove", (PyCFunction) plRemove, METH_VARARGS, 
     "Remove peaks on a circle"},
    {NULL, NULL, 0, NULL}
  };

static PyObject * plGetAttribute(PeakList * self,
				 char * name)
{
  return Py_FindMethod(plMethods, (PyObject *) self, name);
}

// that's all folks!

#endif
