// Peak.cpp
// maintained by G.Winter
// 6th January 2004
// 
// The implementation of the "PeakList" class, for performing peak searches
// on images.
// 
// 
// 
// 
// $Id: Peak.cpp,v 1.12 2004/05/18 11:32:14 svensson Exp $

#include "Peak.h"

// this is to allow some very peculiar things to be done in the 
// Python allocator
#include <new>
#include <list>
// this is only required if not GCC
#ifdef NOT_USING_GCC
#include <math>
#endif
using namespace std;

struct DPS_Peak {
  float x;
  float y;
  float isigma;
};

extern "C" {
  extern int dps_peaksearch_cn(unsigned short *data,
			       int nx,
			       int ny,
			       int npeaks_out,
			       float min_isigma,
			       int min_spacing,
			       DPS_Peak *pptr);
};

namespace DI
{
  // the default constructor
  PeakList::PeakList(void)
  {
    constructor();
  }

  PeakList::PeakList(DiffractionImage * diffractionImage)
  {
    constructor(diffractionImage);
  }

  PeakList::PeakList(const PeakList & peakList)
  {
    constructor(peakList);
  }

  PeakList::~PeakList(void)
  {
    destructor();
  }

  // next a little real code - with the actual constructors and 
  // destructors. this will be pretty simple as most of the work
  // will be handled by the STL.

  // however, the constructors are not called as a part of malloc,
  // which is what python allocation uses. this means that I can't
  // combine Python with STL. bummer. even trying to explicitly call
  // the constructor fails... -> change _list to a ptr to the list?

  void PeakList::constructor(void)
  {

#ifdef INCLUDE_PYTHON
    // then we need to manually call the new operator
    // to construct the STL list
    
    (void) new (&_list) list<Peak>;

#endif

    _length = 0;
  }

  void PeakList::constructor(DiffractionImage * diffractionImage)
  {
    constructor();
    find(diffractionImage);
  }

  void PeakList::constructor(const PeakList & peakList)
  {
    constructor();
    list<Peak>::iterator i;
    // this constructor is currently broken and needs to be repaired
  }

  void PeakList::destructor(void)
  {
    // call the destructor manually - is this kosher? hope so

#ifdef INCLUDE_PYTHON

    _list.~list<Peak>();
#endif

  }

  int PeakList::length(void)
  {
#ifdef TEST_CODE
    // then we will check that the structure of the list is in
    // tact - but _list.size() can be an O(n) operation so it is
    // not really desirable
    if (_list.size() != _length)
      {
	throw (DiffractionImageException("list structure is broken"));
      }
#endif
    return _length;
  }
  
  void PeakList::clear(void)
  {
    // call the default constructor to do this job
    constructor();
  }

  void PeakList::add(float x,
		     float y,
		     float intensity)
  {
    Peak peak;
    peak.x = x;
    peak.y = y;
    peak.intensity = intensity;

    _list.push_back(peak);
    _length ++;
  }

  void PeakList::add(Peak peak)
  {
    _list.push_back(peak);
    _length ++;
  }

  void PeakList::remove(int offset)
  {
    // this will be an expensive operation
    if (offset >= _length)
      {
	throw (DiffractionImageException("offset greater than length"));
      }

    int n;
    list<Peak>::iterator i;

    n = 0;
    for (i = _list.begin(); i != _list.end(); i++)
      {
	if (n == offset)
	  {
	    _list.erase(i);
	    _length --;
	    return;
	  }
	n ++;
      }

    // should not be able to reach here
    throw (DiffractionImageException("cannot reach this point"));
  }

  void PeakList::find(DiffractionImage * diffractionImage)
  {
    // this will perform a DPS peaksearch on the image (which 
    // must first be validated as being ok) and then return the
    // results of this search

    int maxPeaks = 100000;
    find(diffractionImage, maxPeaks);
  }

  void PeakList::find(DiffractionImage * diffractionImage,
		      int maxPeaks)
  {
    float intensityThreshold = 2.0;
    find(diffractionImage, maxPeaks, intensityThreshold);
  }

  void PeakList::find(DiffractionImage * diffractionImage,
		      int maxPeaks,
		      float intensityThreshold)
  {
    int i;
    DPS_Peak * peaks;

    // validate the input
    if (diffractionImage == NULL)
      {
	throw (DiffractionImageException("NULL image"));
      }

    if (maxPeaks <= 0)
      {
	throw (DiffractionImageException("bad input"));
      }

    if ((diffractionImage->getWidth() == 0) ||
	(diffractionImage->getHeight() == 0))
      {
	throw (DiffractionImageException("bad image dimensions"));
      }

    try
      {
	peaks = new DPS_Peak[maxPeaks];
      }
    catch (bad_alloc & e)
      {
	throw (DiffractionImageException("error allocating temporary "
					 "spot storage"));
      }

    if (peaks == NULL)
      {
	// belt + braces - I am not sure if new will throw an exception
	// or return NULL - according to the C++ book either may happen
	// integer comparisons are not expensive anyway!
	throw (DiffractionImageException("error allocating temporary "
					 "spot storage"));
      }

    // using 5 as the default interspot separation, in pixels
    // everything else will come from the image information and the 
    // input to the routine

    int found = dps_peaksearch_cn(diffractionImage->getImage(),
				  diffractionImage->getWidth(),
				  diffractionImage->getHeight(),
				  maxPeaks,
				  intensityThreshold,
				  3,
				  peaks);

    Peak peak;

    float phi = 0.5 * (diffractionImage->getPhiStart() + diffractionImage->getPhiEnd());

    float scalex, scaley, beamx, beamy;

    scalex = diffractionImage->getPixelX();
    scaley = diffractionImage->getPixelY();

    beamx = diffractionImage->getBeamX();
    beamy = diffractionImage->getBeamY();

    for (i = 0; i < found; i++)
      {
	peak.x = scalex * peaks[i].x - beamx;
	peak.y = scaley * peaks[i].y - beamy;
	// store the phi value from the centre of this image
	peak.phi = phi;
	peak.theta = 0;
	peak.kappa = 0;
	for (int k = 0; k < 3; k++)
	{
	  peak.p[k] = 0;
	}
	peak.intensity = peaks[i].isigma;
	add(peak);
      }

    delete [] peaks;

  }

  Peak & PeakList::operator[] (int offset)
  {
    int n;
    list<Peak>::iterator i;

    if (offset > _length)
      {
	throw (DiffractionImageException("offset greater than length"));
      }

    n = 0;
    for (i = _list.begin(); i != _list.end(); i++)
      {
	if (n == offset)
	  {
	    return (*i);
	  }
	n ++;
      }

    // this is just to keep the compiler happy

    return (* (_list.end()));

  }

  Peak & PeakList::get(int offset)
  {
    // nasty use of the this pointer here
    return (*this)[offset];
  }

  Peak PeakList::pop(int offset)
  {
    Peak peak;
    peak = get(offset);
    remove(offset);
    return peak;
  }

  vector<Peak> PeakList::getPeaks(void)
  {
    vector<Peak> result;
    list<Peak>::iterator i;

    for (i = _list.begin(); i != _list.end(); i++)
      {
	result.push_back((*i));
      }

    return result;
  }


  // I have found a problem with this - there is a fair need for some
  // refinement of the circle against the list of peaks, so that we can 
  // be sure that we have the best fit - how would the best way to do this
  // be - refine with some kind of kernel function for the displacement from
  // the edge?

  // transform all of the reflection coordinates to reciprocal space

  void PeakList::reciprocal(float distance,
			    float wavelength)
  {
    int l = length();
    for (int i = 0; i < l; i++)
      {
	float X, Y, D;
	float x, y, z;
	X = (*this)[i].x;
	Y = (*this)[i].y;
	D = sqrt(X * X + Y * Y + distance * distance);
	x = X / (wavelength * D);
	y = Y / (wavelength * D);
	z = distance / (wavelength * D);
	
	// want to perform the inverse rotation so
	float phi = - (*this)[i].phi;

	float c = cos(phi * 3.141592654 / 180.0);
	float s = sin(phi * 3.141592654 / 180.0);


	// using the reference frame in Rossmann (1979)
	(*this)[i].p[0] = x * c + z * s;
	(*this)[i].p[1] = y;
	(*this)[i].p[2] = -s * x + c * z;

      }
  }

  void PeakList::detector(float distance,
			  float wavelength)
  {
    // convert from reciprocal to detector coordinates


  }

  int PeakList::circle(int iterations,
		       float width,
		       float & x,
		       float & y,
		       float & r)
  {
    // since this will probably involve some fairly hefty traversing of
    // linked lists of spots, I'll copy them into a good ol' array first.

    // should this weight the importance of spots in some way, or should
    // all spots be considered equally?

    float * _x, * _y;

    int i, j, n;
    int best = 0;

    list<Peak>::iterator iterator;

    if (_length == 0)
      {
	// then we have a problem!
	throw (DiffractionImageException("trying to find circles "
					 "without any spots"));
      }

    // here's a C++ question - does new throw an exception if it
    // fails, or does it just return NULL?
    try
      {
	_x = new float[_length];
      }
    catch (bad_alloc & e)
      {
	throw (DiffractionImageException("error allocating workspace"));
      }

    if (_x == NULL)
      {
	// throw an exception
	throw (DiffractionImageException("error allocating workspace"));
      }

    try
      {
	_y = new float[_length];
      }
    catch (bad_alloc & e)
      {
	delete [] _x;
	throw (DiffractionImageException("error allocating workspace"));
      }

    if (_y == NULL)
      {
	delete [] _x;
	// also throw an exception
	throw (DiffractionImageException("error allocating workspace"));
      }

    // now copy from the list into the workspace
    i = 0;
    for (iterator = _list.begin(); iterator != _list.end(); iterator ++)
      {
	_x[i] = (*iterator).x;
	_y[i] = (*iterator).y;

	i ++;
      }

    // that should be the list copied, so we're getting there
    // now the main working loop of this - finding the circles!
    // caveat user - at some point it would peobably be a good idea
    // to call srand48() to seed the random number generator which I 
    // shall be using here!

    for (i = 0; i < iterations; i++)
      {
	// stage one - pick three random reflections
	int zero, one, two;
	float x0, y0, x1, y1, x2, y2;
	float cx, cy, cr;

	zero = (int) (drand48() * _length);
	one = (int) (drand48() * _length);
	two = (int) (drand48() * _length);

	x0 = _x[zero];
	y0 = _y[zero];
	x1 = _x[one];
	y1 = _y[one];
	x2 = _x[two];
	y2 = _y[two];
	
	if (fitCircle(x0, y0, x1, x1, x2, x2, cx, cy, cr) == 0)
	  {
	    // go through and count the number of points which are "close"
	    // to this circle
	    n = 0;
	    for (j = 0; j < _length; j++)
	      {
		if (fabs(sqrt((_x[j] - cx) * (_x[j] - cx) + 
			      (_y[j] - cy) * (_y[j] - cy)) - cr) < width)
		  {
		    n ++;
		  }
	      }

	    // now see if this is better than best - and if it is, 
	    // record the values
	    if (n > best)
	      {
		best = n;
		r = cr;
		x = cx;
		y = cy;


	      }
	  }
      }


    // that should be all of the important work done now so
    delete [] _x;
    delete [] _y;

    return best;
  }
	       
  // the next method is one to work with the previous one - in which the 
  // objective is to remove those spots found to be on the circle - this
  // can be used iteratively to find all of the circles on an image

  int PeakList::remove(float width,
		       float x,
		       float y,
		       float r)
  {
    list<Peak>::iterator i;
    list<Peak> l;
    int n = 0;
    Peak peak;

    // this will work manually though the linked list, as the direct 
    // assess if required. this is where a list container is good, 
    // because the iterators are preserved on removal of elements

    // are the list containers preserved? I get a segmentation violation
    // at this stage which suggests that the structure of the 
    // list must be coming appart.. Bummer. :o(

    // workaround implemented! - I copy across to a new list and then delete
    // the old one - I guess that this should be implemented using the swap()
    // method

    for (i = _list.begin(); i != _list.end(); i++)
      {
	peak = (*i);
	if (fabs(sqrt((peak.x - x) * (peak.x - x) +
		      (peak.y - y) * (peak.y - y)) - r) < width)
	  {
	    // then remove this point - NOT
	    // _list.erase(i);
	    // and record the fact in a total
	    // in fact, keep this if we're wanting to
	    n ++;
	  }
	else
	  {
	    l.push_back(*i);
	  }
      }

    _list.swap(l);
    _length -= n;

    // that all done now so return the number removed for examination
    return n;
  }


  // those other namespace sharing functions I was talking about

  // this function will take the three planar points (x0, y0) ...
  // (x2, y2) and fit a circle to them. clearly an identity between any 
  // two points will result in a degenerate circle - in this case the result
  // of the function will be 1. otherwise this should be successful, with 
  // return value 0 and the values r, x, y set to useful values.

  int fitCircle(float x0,
		float y0,
		float x1,
		float y1,
		float x2,
		float y2,
		float & x,
		float & y,
		float & r)
  {
    float a1, a2, b1, b2, c1, c2;

    // check for degeneracy - in the old code this used
    // x1 - x0 but this is a bad comparison function - change this to
    // fabs (x1 - x0) < epsilon, where the latter is a tiny number.

    // a useful point for scaling of the problem - these are coordinates
    // in pixels, so ought to be of order 1000. a suitable value for epsilon
    // will therefore be around 1.0

    const float epsilon = 1.0;

    if ((fabs(x0 - x1) < epsilon) ||
	(fabs(x1 - x2) < epsilon) ||
	(fabs(x2 - x0) < epsilon) ||
	(fabs(y0 - y1) < epsilon) ||
	(fabs(y1 - y2) < epsilon) ||
	(fabs(y2 - y0) < epsilon))
      {
	return 1;
      }
    
    // if we have reached this stage then we should be safe from degenerate 
    // points

    a1 = 2.0 * (x0 - x1);
    a2 = 2 * (x1 - x2);
    b1 = 2 * (y0 - y1);
    b2 = 2 * (y1 - y2);
    c1 = x0 * x0 - x1 * x1 + y0 * y0 - y1 * y1;
    c2 = x1 * x1 - x2 * x2 + y1 * y1 - y2 * y2;
    
    // get the centres
    x = (c1 * b2 - c2 * b1) / (a1 * b2 - a2 * b1);
    y = (c2 * a1 - c1 * a2) / (a1 * b2 - a2 * b1);

    // compute the radius from the first point - I am not sure that an
    // average would achieve anything here, other than highlight the 
    // rounding errors of floating point values

    r = sqrt((x0 - x) * (x0 - x) + (y0 - y) * (y0  - y));
    
    // all parameters computed, so job done

    return 0;
  }

};
