// DiffractionImagePythonInterface.cpp
// maintained by G.Winter
// 5th January 2004
// 
// A Python wrapper for the DiffractionImage library - second version.
// 
// 
// 
// 
// $Id: DiffractionImagePythonInterface.cpp,v 1.15 2005/11/22 13:38:14 svensson Exp $

#ifdef INCLUDE_PYTHON

#include "DiffractionImage.h"

#ifdef INCLUDE_JASPER
extern "C" {
#include "jasper/jasper.h"
}
#endif

#include <vector>

using namespace DI;

// Things for the Python exception handling - which requires a global 
// static pointer to be available. This is being made unstatic so that it
// is avilable outside the file - although this does mean that it will 
// pollute the namespace, I don't think that this is a big problem.

PyObject * PyDiffractionImageException;

// a few forward references, to allow a structure to be constructed
// these will become methods of the Python object, so to keep things
// simple they are prefixed with di..

static PyObject * diLength(PyDiffractionImage * self,
			   PyObject * args);
static PyObject * diSubscript(PyDiffractionImage * self,
			      PyObject * args);

static PyObject * diGetAttribute(PyDiffractionImage * self,
				 char * attributeName);

static void diDeAlloc(PyObject * self);

static PyMappingMethods diMappings = {
    (inquiry) diLength,
    (binaryfunc) diSubscript,
    0
};

static PyTypeObject diType = {
    PyObject_HEAD_INIT(NULL)
    0,
    "DiffractionImage",
    sizeof(PyDiffractionImage),
    0,
    (destructor) diDeAlloc,
    0,
    (getattrfunc) diGetAttribute,
    0,
    0,
    0,
    0,
    0,
    & diMappings,
    0
};

// now some concrete methods - first the constructor
// add an option to the constructor to load only the 
// header if the second argument is 0 or False

static PyObject * diPyDiffractionImage(PyObject * self,
				       PyObject * args)
{
    char * filename;
    PyDiffractionImage * diffractionImage;

    if (PyTuple_Size(args) > 2)
    {
	PyErr_SetString(PyDiffractionImageException,
			"bad arguments");
	return NULL;
    }

    // ignore the arguments until the code would diverge
    diffractionImage = PyObject_New(PyDiffractionImage, &diType);
    
    if (diffractionImage == NULL)
    {
	// then there was an error in allocating the
	// memory space - propogate this error
	
	PyErr_SetString(PyDiffractionImageException,
			"error allocating PyDiffractionImage object");
	
	return NULL;
    }

    if (PyTuple_Size(args) == 1)
    {
	// we are being passed a filename in the constructor - 
	// so try and create a new image and load the file in
	// passing the filename to the PyDiffractionImage constructor
	// method will achieve this aim
	if (!PyArg_ParseTuple(args, "s", &filename))
	{
	    // we should raise an exception here
	    return NULL;
	}

	try
	{
	    diffractionImage->diffractionImage.constructor(string(filename));
	}
	catch (DiffractionImageException & e)
	{
	    // an error occurred in calling the constructor - so the 
	    // error must propogate
	
	    PyErr_SetString(PyDiffractionImageException,
			    e.getMessage().c_str());
	    PyObject_Del(diffractionImage);
	    return NULL;
	}

	return (PyObject *) diffractionImage;
    }
    else if (PyTuple_Size(args) == 2)
      {
	// we are being passed the flag for maybe don't read the image
	int flag;
	if (!PyArg_ParseTuple(args, "si", &filename, &flag))
	{
	    // we should raise an exception here
	    return NULL;
	}

	if (flag != 0)
	  {
	    try
	      {
		diffractionImage->diffractionImage.constructor(string(filename));
	      }
	    catch (DiffractionImageException & e)
	      {
		// an error occurred in calling the constructor - so the 
		// error must propogate
		
		PyErr_SetString(PyDiffractionImageException,
				e.getMessage().c_str());
		PyObject_Del(diffractionImage);
		return NULL;
	      }
	  } 
	else
	  {
	    try
	      {
		diffractionImage->diffractionImage.constructor();
		diffractionImage->diffractionImage.loadHeader(string(filename));
	      }
	    catch (DiffractionImageException & e)
	      {
		// an error occurred in calling the constructor - so the 
		// error must propogate
		
		PyErr_SetString(PyDiffractionImageException,
				e.getMessage().c_str());
		PyObject_Del(diffractionImage);
		return NULL;
	      }
	  }

	return (PyObject *) diffractionImage;
    }

    else
    {
	try
	{
	    diffractionImage->diffractionImage.constructor();
	}
	catch (DiffractionImageException & e)
	{
	    // an error occurred in calling the constructor - so the 
	    // error must propogate
	
	    PyErr_SetString(PyDiffractionImageException,
			    e.getMessage().c_str());
	    PyObject_Del(diffractionImage);
	    return NULL;
	}

	return (PyObject *) diffractionImage;
    }
}

static void diDeAlloc(PyObject * self)
{
    // this is easy
    PyDiffractionImage * diffractionImage = (PyDiffractionImage *) self;

    // call the destructor
    diffractionImage->diffractionImage.destructor();

    // then free the allocated memory
    PyObject_Del(self);
}

// this object will require in Python all of the same methods which are
// provided in the C++ API. Wouldn't it be nice to be able to write a 
// macro to handle all of this? :o)

// I have, but it is very ugly!

// all of these methods are getters - which should make matters easier!

#define GETTER_METHOD_INT(capitalArg) \
static PyObject * diGet ## capitalArg(PyDiffractionImage * self, \
PyObject * args) \
{ \
    if (!PyArg_ParseTuple(args, ":DiffractionImage")) { return NULL; } \
return PyInt_FromLong(self->diffractionImage.get ## capitalArg()); \
}

#define GETTER_METHOD_FLOAT(capitalArg) \
static PyObject * diGet ## capitalArg(PyDiffractionImage * self, \
PyObject * args) \
{ \
    if (!PyArg_ParseTuple(args, ":DiffractionImage")) { return NULL; } \
return PyFloat_FromDouble(self->diffractionImage.get ## capitalArg()); \
}

#define SETTER_METHOD_FLOAT(capitalArg) \
static PyObject * diSet ## capitalArg(PyDiffractionImage * self, \
PyObject * args) \
{ \
float value; \
  if (!PyArg_ParseTuple(args, "f", &value)) { return NULL; } \
  self->diffractionImage.set ## capitalArg(value); \
  Py_INCREF(Py_None); \
  return Py_None; }

GETTER_METHOD_INT(Width);
GETTER_METHOD_INT(Height);
GETTER_METHOD_FLOAT(Distance);
GETTER_METHOD_FLOAT(Wavelength);
GETTER_METHOD_FLOAT(PhiStart);
GETTER_METHOD_FLOAT(PhiEnd);
GETTER_METHOD_FLOAT(PixelX);
GETTER_METHOD_FLOAT(PixelY);
GETTER_METHOD_FLOAT(BeamX);
GETTER_METHOD_FLOAT(BeamY);
GETTER_METHOD_FLOAT(ExposureTime);

SETTER_METHOD_FLOAT(Distance);
SETTER_METHOD_FLOAT(Wavelength);
SETTER_METHOD_FLOAT(PhiStart);
SETTER_METHOD_FLOAT(PhiEnd);
SETTER_METHOD_FLOAT(PixelX);
SETTER_METHOD_FLOAT(PixelY);
SETTER_METHOD_FLOAT(BeamX);
SETTER_METHOD_FLOAT(BeamY);
SETTER_METHOD_FLOAT(ExposureTime);

// next a couple of manual methods to access image loading 
// warnings

static PyObject * diGetMessage(PyDiffractionImage * self,
			       PyObject * args)
{
  PyObject * result;
  if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
      return NULL;
    }
  
  try
    {
      result = 
	PyString_FromString(self->diffractionImage.getMessage().c_str());
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }
  
  return result;
}

static PyObject * diGetWarning(PyDiffractionImage * self,
			       PyObject * args)
{
  PyObject * result;
  if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
      return NULL;
    }
  
  try
    {
      if (self->diffractionImage.getWarning() == true)
	{
	  result = PyInt_FromLong(1);
	}
      else
	{
	  result = PyInt_FromLong(0);
	}
    }
  catch (DiffractionImageException & e)
    {
      PyErr_SetString(PyDiffractionImageException,
		      e.getMessage().c_str());
      return NULL;
    }
  
  return result;
}


// next a manual method to get a pointer to the image

static PyObject * diGetImage(PyDiffractionImage * self,
			     PyObject * args)
{
    PyObject * result;

    if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
	return NULL;
    }

    try
    {
	result = PyInt_FromLong((long int) self->diffractionImage.getImage());
    }
    catch (DiffractionImageException & e)
    {
	PyErr_SetString(PyDiffractionImageException,
			e.getMessage().c_str());
	return NULL;
    }

    return result;
}

// next the image reading methods - all rolled into a single "load"
// method

static PyObject * diLoad(PyDiffractionImage * self,
			 PyObject * args)
{
    char * filename;
    string type;
    PyObject * result;

    if (!PyArg_ParseTuple(args, "s", &filename))
    {
	return NULL;
    }

    try
    {
	type = self->diffractionImage.load(string(filename));
    }
    catch (DiffractionImageException & e)
    {
	PyErr_SetString(PyDiffractionImageException,
			e.getMessage().c_str());
	return NULL;
    }

    // pass the image type back...

    result = PyString_FromString(type.c_str());
    
    return result;
}

static PyObject * diLoadHeader(PyDiffractionImage * self,
			       PyObject * args)
{
    char * filename;

    if (!PyArg_ParseTuple(args, "s", &filename))
    {
	return NULL;
    }

    try
    {
	self->diffractionImage.loadHeader(string(filename));
    }
    catch (DiffractionImageException & e)
    {
	PyErr_SetString(PyDiffractionImageException,
			e.getMessage().c_str());
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

// this will want to increase in sophistication with time - 
// returning all of the original functionality which was 
// available

static PyObject * diJpeg(PyDiffractionImage * self,
			 PyObject * args)
{
    char * filename;

    if (!PyArg_ParseTuple(args, "s", &filename))
    {
	return NULL;
    }

    try
    {
	self->diffractionImage.jpeg(string(filename));
    }
    catch (DiffractionImageException & e)
    {
	PyErr_SetString(PyDiffractionImageException,
			e.getMessage().c_str());
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * diHistogram(PyDiffractionImage * self,
			      PyObject * args)
{
  if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
        return NULL;
    }

  PyObject * result = PyList_New(17);

  vector<int> histogram = self->diffractionImage.histogram();

  for (int i = 0; i < 17; i++)
    {
      PyList_SetItem(result, i, PyInt_FromLong(histogram[i]));
    }

  return result;
}


static PyObject * diRadial(PyDiffractionImage * self,
			   PyObject * args)
{

  int bins = 16;
  
  if (PyTuple_Size(args) == 1)
    {
      // we have an argument - the number of bins to use
      if (!PyArg_ParseTuple(args, "i", &bins))
	{
	  return NULL;
	}
    }
  else
    {
      if (!PyArg_ParseTuple(args, ":DiffractionImage"))
	{
	  return NULL;
	}
    }

  PyObject * result = PyList_New(bins + 1);

  vector<float> radial = self->diffractionImage.radial(bins);

  for (int i = 0; i < bins + 1; i++)
    {
      PyList_SetItem(result, i, PyFloat_FromDouble(radial[i]));
    }

  return result;
}
    
static PyObject * diAsymmetry(PyDiffractionImage * self,
			      PyObject * args)
{

  if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
      return NULL;
    }

  PyObject * result = PyList_New(18);

  vector<float> asymmetry = self->diffractionImage.asymmetry();

  for (int i = 0; i < 18; i++)
    {
      PyList_SetItem(result, i, PyFloat_FromDouble(asymmetry[i]));
    }

  return result;
}
    

  

// finally we need the subscription operator methods
// not worth trying to write a macro for these

static PyObject * diLength(PyDiffractionImage * self,
			   PyObject * args)
{
    if (!PyArg_ParseTuple(args, ":DiffractionImage"))
    {
        return NULL;
    }
    
    return PyInt_FromLong(self->diffractionImage.getWidth() * self->diffractionImage.getHeight());
}

static PyObject * diSubscript(PyDiffractionImage * self,
			      PyObject * args)
{
    int offset;
    unsigned short * image;

    try
    {
	image = self->diffractionImage.getImage();
    }
    catch (DiffractionImageException & e)
    {
	PyErr_SetString(PyDiffractionImageException,
			e.getMessage().c_str());
	return NULL;
    }

    if (PyInt_Check(args))
    {
	offset = PyInt_AsLong(args);
	try
	{
	    return PyInt_FromLong(self->diffractionImage[offset]);
	}
	catch (DiffractionImageException & e)
	{
	    PyErr_SetString(PyDiffractionImageException, 
			    e.getMessage().c_str());
	    return NULL;
	}
    }

    // we shouldn't be able to reach this point

    return NULL;
}

// finally we have the code to set this up as a proper Python object -
// to create a loadable module

// some external symbols to allow these tables to be built

extern PyObject * plPeakList(PyObject * self,
			     PyObject * args);


static PyObject * diMin(PyDiffractionImage * self,
			PyObject * args)
{
  PyDiffractionImage * diffractionImage;

  if (PyTuple_Size(args) != 1)
    {
      PyErr_SetString(PyDiffractionImageException,
		      "takes a DiffractionImage");
      return NULL;
    }

  diffractionImage = (PyDiffractionImage *) PyTuple_GetItem(args, 0);
  
  // check that this is indeed a DiffractionImage
  
  if (strcmp(diffractionImage->ob_type->tp_name,
	     "DiffractionImage") != 0)
    {
      // then this is not a DiffractionImage - clear up the mess
      // and raise an exception
      PyErr_SetString(PyDiffractionImageException,
		      "argument is not a DiffractionImage");
      return NULL;
    }

  self->diffractionImage.min(diffractionImage->diffractionImage);
  
  Py_INCREF(Py_None);
  return Py_None;
}

// module methods

static PyMethodDef diModuleMethods[] = 
{
    {"DiffractionImage", diPyDiffractionImage, METH_VARARGS, 
    "Create a DiffractionImage object"},
    {"PeakList", plPeakList, METH_VARARGS,
     "Create a PeakList object"},
    {NULL, NULL, 0, NULL}
};
    
// create class instance methods

static PyMethodDef diMethods[] = 
{
    {"getWidth", (PyCFunction) diGetWidth, METH_VARARGS, "Get image width"},
    {"getHeight", (PyCFunction) diGetHeight, METH_VARARGS, "Get image height"},
    {"getImage", (PyCFunction) diGetImage, METH_VARARGS, "Get image bytes"},
    {"getBeamX", (PyCFunction) diGetBeamX, METH_VARARGS, "Get image beamX"},
    {"getBeamY", (PyCFunction) diGetBeamY, METH_VARARGS, "Get image beamY"},
    {"getPixelX", (PyCFunction) diGetPixelX, METH_VARARGS, "Get image pixelX"},
    {"getPixelY", (PyCFunction) diGetPixelY, METH_VARARGS, "Get image pixelY"},
    {"getDistance", (PyCFunction) diGetDistance, METH_VARARGS, 
     "Get image distance"},
    {"getWavelength", (PyCFunction) diGetWavelength, METH_VARARGS, 
     "Get image wavelength"},
    {"getExposureTime", (PyCFunction) diGetExposureTime, METH_VARARGS, 
     "Get image exposure time"},
    {"getPhiStart", (PyCFunction) diGetPhiStart, METH_VARARGS, 
     "Get phi start"},
    {"getPhiEnd", (PyCFunction) diGetPhiEnd, METH_VARARGS, "Get phi end"},
    {"getMessage", (PyCFunction) diGetMessage, METH_VARARGS, 
     "Get messages from loading"},
    {"getWarning", (PyCFunction) diGetWarning, METH_VARARGS, 
     "Get warning flag from loading"},

    {"setBeamX", (PyCFunction) diSetBeamX, METH_VARARGS, "Set image beamX"},
    {"setBeamY", (PyCFunction) diSetBeamY, METH_VARARGS, "Set image beamY"},
    {"setPixelX", (PyCFunction) diSetPixelX, METH_VARARGS, "Set image pixelX"},
    {"setPixelY", (PyCFunction) diSetPixelY, METH_VARARGS, "Set image pixelY"},
    {"setDistance", (PyCFunction) diSetDistance, METH_VARARGS, 
     "Set image distance"},
    {"setWavelength", (PyCFunction) diSetWavelength, METH_VARARGS, 
     "Set image wavelength"},
    {"setExposureTime", (PyCFunction) diSetExposureTime, METH_VARARGS, 
     "Set image exposure time"},
    {"setPhiStart", (PyCFunction) diSetPhiStart, METH_VARARGS, 
     "Set phi start"},
    {"setPhiEnd", (PyCFunction) diSetPhiEnd, METH_VARARGS, "Set phi end"},
    {"load", (PyCFunction) diLoad, METH_VARARGS, "Load an image"},
    {"jpeg", (PyCFunction) diJpeg, METH_VARARGS, "Create a jpeg"},
    {"min", (PyCFunction) diMin, METH_VARARGS, 
     "Create a minimum image of two images"},
    {"histogram", (PyCFunction) diHistogram, 
     METH_VARARGS, "Create a histogram of pixel intensities"},
    {"radial", (PyCFunction) diRadial,
     METH_VARARGS, "Create a radial profile"},
    {"asymmetry", (PyCFunction) diAsymmetry,
     METH_VARARGS, "Define areas of asymmetry"},
    {NULL, NULL, 0, NULL}
};

// export all of this functionality

extern "C" {
    DL_EXPORT(void)
	initDiffractionImage(void)
    {
	PyObject * module, * dict;
	diType.ob_type = & PyType_Type;
	module = Py_InitModule("DiffractionImage", diModuleMethods);
	dict = PyModule_GetDict(module);

	// register the exception class

	PyDiffractionImageException = 
	  PyErr_NewException("DiffractionImage.error",
			     NULL, NULL);

	PyDict_SetItemString(dict, "error", PyDiffractionImageException);

#ifdef INCLUDE_JASPER
	// initialise the Jpeg 2000 library
	jas_init();
#endif
    }
}

// next the implementation of the class methods 

static PyObject * diGetAttribute(PyDiffractionImage * self,
				 char * name)
{
    return Py_FindMethod(diMethods, (PyObject *) self, name);
}

// here endeth the class
    

#endif

