// DiffractionImageMarCCD.cpp
// maintained by G.Winter
// 17th December 2003
// 
// The bits of the DiffractionImage library which are specific to the 
// MarCCD image type.
// 
// 
// 
// $Id: DiffractionImageMarCCD.cpp,v 1.9 2004/09/10 13:28:52 gwin Exp $

// C++ things
#include <iostream>
#include <fstream>
#include <map>


// core C things - required for using open() and read() below
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "DiffractionImage.h"

// the marccd data structure

// #include "mar.h"
extern "C" {
#include "../../../third_party/MAR/marcvt/io/marccd_header.h"
}

// if we are in development mode, then include things pertaining to
// assert.

#ifdef TEST_CODE
#define NDEBUG
#include <assert.h>
#endif

using namespace std;

namespace DI
{
    void DiffractionImage::loadMarCCD(string filename)
    {
	// this will just call the children

	loadMarCCDHeader(filename);
	allocateImage();

#ifdef TEST_CODE
	assert(image != NULL);
#endif

	// a MarCCD image header takes 4k bytes - so the offset is
	// 4096 - 0x1000 in hex - and using hex to limit typos
	loadRawImage(filename, 0x1000);

    }

    void DiffractionImage::loadMarCCDHeader(string filename)
    {

#ifdef TEST_CODE
	cout << "marccd: " << filename << endl;
#endif

	// notes:
	// 
	// A marccd header consists of 1k TIFF header, then
	// 160 unsigned int records (equivalent - 640 bytes), then
	// 'the rest' (608 signed int records - most of this is 
	// rubbish) - this is 2432 bytes.
	// 
	
	// the easiest way to handle this is using a data structure from 
	// the web... - see "mar.h"

	// this is the marccd data structure
	// frame_header_type f;
	MARCCD_HEADER f;

	// this is somewhere to read the structure into - 
	// which will be memcpy'd into f
	unsigned char buffer[0x1000];

	int fd, amount;

	fd = open(filename.c_str(), O_RDONLY);

	if (fd == -1)
	{
	    throw(DiffractionImageException("error opening image"));
	}

	amount = read(fd, buffer, 0x1000);

	if (amount < 0x1000)
	{
	    close(fd);
	    throw(DiffractionImageException("error reading from file"));
	}

	// if we have reached this stage, then all is peachy

	close(fd);

	// check that this is a marccd image - the first two bytes of
	// buffer should be 0x49
	
	if ((buffer[0] != 0x49) ||
	    (buffer[1] != 0x49))
	{
	    throw(DiffractionImageException("not a marccd header"));
	}

	// store the header as in the file

	originalHeaderLength = 4096;
	memcpy(originalHeader, buffer, 4096);

	// copy the data from the buffer area into the
	// data structure
	memcpy((void *) &f, buffer + 1024, sizeof(f));


#ifdef TEST_CODE
	cout << f.data_byte_order << endl;
	cout << sizeof(f) << endl;
#endif

	// next check the byte order - for the image
	if (f.data_byte_order == BIG_ENDIAN)
	{
	    bigEndian = true;
	}
	else
	{
	    bigEndian = false;
	}

	// if the header is big endian and the machine little endian, or 
	// vice versa, invert it.

	if ((bigEndian && !machineBigEndian()) ||
	    (!bigEndian && machineBigEndian()))
	{
	    for (int i = 0; i < 768; i++)
	    {
		// this is horrible
		byteSwap((unsigned int *) (buffer + 1024 + 4 * i));
	    }

	    // now recopy the data...
	    memcpy((void *) &f, buffer + 1024, sizeof(f));
	}

	// now the header in memory should be in the appropriate format
	// N.B. my definition of the image orientation is that the index
	// for the pixel changes fastest in the WIDTH axis.

	// the loops are therefore for (i < height) for (j < width)
	// WIDTH ALWAYS INSIDE HEIGHT

	// first however validate the information in the image

	if (f.depth != 2)
	{
	    // throw an exception
	    throw(DiffractionImageException("image not short"));
	}

	// assume for the moment that f.data_type is always unsigned

	width = f.nfast;
	height = f.nslow;

	beamX = 0.001 * f.beam_x;
	beamY = 0.001 * f.beam_y;

	if (f.xtal_to_detector != 0)
	  {
	    distance = 0.001 * f.xtal_to_detector;
	  }
	else if (f.start_xtal_to_detector != 0)
	  {
	    distance = 0.001 * f.start_xtal_to_detector;
	  }
	else if (f.end_xtal_to_detector != 0)
	  {
	    distance = 0.001 * f.end_xtal_to_detector;
	  }
	else
	  {
	    // we have an error condition
	    distance = -1.0;
	  }

	twoTheta = 0.001 * f.start_twotheta;

	exposureTime = 0.001 * f.exposure_time;

	// another place where the exposure time can be kept then...
	if (exposureTime < 0.0)
	  {
	    exposureTime = 0.001 * f.integration_time;
	  }

	// at this stage I will need to verify that this is
	// how the information is really stored - caveat user
	// f.phi_end isn't actually written

	phiStart = 0.001 * f.start_phi;
	phiEnd = 0.001 * (f.start_phi + f.rotation_range);

	// want the pixel size in microns
	pixelX = 0.000001 * f.pixelsize_x;
	pixelY = 0.000001 * f.pixelsize_y;

	wavelength = 0.00001 * f.source_wavelength;

	// that should be everything so 
    }

    // that's all we need here..

};
