// DiffractionImageJpeg.cpp
// maintained by G.Winter
// 17th December 2003
// 
// The Jpeg (old school) handling routines for the DiffractionImage
// module. In addition, all of the utility code for handling the image
// is contained herein.
// 
// 
// $Id: DiffractionImageJpeg.cpp,v 1.7 2004/01/06 16:11:56 gwin Exp $

#include "DiffractionImage.h"

#include <iostream>
#include <fstream>
#include <string>

// only include the following if we are explicitly including the 
// jpeg functionality. since this depends on an external library
// (jpeg-6b) this must be optional.

#ifdef INCLUDE_JPEG
extern "C" {
#include "jinclude.h"
#include "jpeglib.h"
#include "jerror.h"
}
#endif

#ifdef TEST_CODE
#define NDEBUG
#include <assert.h>
#endif

namespace DI 
{
    // I guess for this that I should recode all of the jpeg handling bits -
    // and this time make it possible to draw things onto the images too.
    // this could be fun. fortunately I have made the code above flexible
    // enough to allow this
    
    // next some real meaty methods - involving JPEG images
    // the first will call the second, which will call the third
    
    // filename is the name of the output file

    void DiffractionImage::jpeg(string filename)
    {
	jpeg(filename, 85);
    }
    
    // make the image 1:1 with the original - i.e. zoom = 0
    
    void DiffractionImage::jpeg(string filename,
				int quality)
    {
	jpeg(filename, quality, 0);
    }
    
    // zoom the image - if zoom > 0 enlarge, else shrink
    // -1 == 0 == 1 => 1:1

    // this will create a jpeg file named $filename, with the quality
    // and zoom configured manually. this will produce an 8 bit 
    // representation of the image, which will be calculated using
    // bit shifting as this is far cheaper than the alternative

    // this will call the FUNCTION writeJpegImage to do the "dirty 
    // work". 

    // n.b. the traditional way to display these images is as a negative - 
    // so areas of high intensity are darker.

    void DiffractionImage::jpeg(string filename,
				int quality,
				int zoom)
    {
	DiffractionImage::jpeg(filename, quality, zoom, false);
    }

    void DiffractionImage::jpeg(string filename,
				int quality,
				int zoom,
				bool optimise)
    {
	
	if ((zoom == 0) ||
	    (abs(zoom) == 1))
	{
	    writeJpegImage(image, width, height, quality,
			   optimise, filename.c_str());
	}
	else if (zoom > 1)
	{
	    // we wish to enbiggen the image
	    unsigned short * big;
	    
	    // allocate the image
	    try
	      {
		big = new unsigned short[width * height * zoom * zoom];
	      }
	    catch (bad_alloc & e)
	      {
		throw (DiffractionImageException("not enough memory to "
						 "allocate new image"));
	      }
	    
	    if (big == NULL)
	      {
		throw (DiffractionImageException("not enough memory to "
						 "allocate new image"));
	      }
	    
	    // this should be wrapped in a try, catch to enable the
	    // allocated memory to be freed

	    try 
	    {
		enlargeImage(image, width, height, zoom, big);
		writeJpegImage(big, width * zoom, height * zoom, quality,
			       optimise, filename.c_str());
	    }
	    catch (DiffractionImageException e) 
	    {
		// tidy up and rethrow the exception
		delete [] big;
		throw (e);
	    }

	    // tidy up anyway
	    delete [] big;
	}
	else
	{
	    // we wish to make the image nice and small
	    unsigned short * small;
	    int factor = -1 * zoom;
	    
	    // note that there are two layers of error trapping here, 
	    // since I am not sure exactly what will happen in C++ if an
	    // allocation fails - better to be sure.

	    try
	      {
		small = new unsigned short[width * height / (factor * factor)];
	      }
	    catch (bad_alloc & e)
	      {
		throw (DiffractionImageException("not enough memory to "
						 "allocate new image"));
	      }
	    
	    if (small == NULL)
	    {
		throw (DiffractionImageException("not enough memory to "
						 "allocate new image"));
	    }
	    

	    // try and catch again
	    try
	    {
		reduceImage(image, width, height, factor, small);
		writeJpegImage(small, width / factor, height / factor, quality,
			       optimise, filename.c_str());
	    }
	    catch (DiffractionImageException e)
	    {
		delete [] small;
		throw (e);
	    }
	    delete [] small;
	}
	
    }

    // this is a utility function so is not contained within the class 
    // structure - perhaps this should be exported in a different module

    void enlargeImage(unsigned short * original,
		      int width,
		      int height,
		      int zoom,
		      unsigned short * copy)
    {
	// this is pretty easy
	int i, j, k, l;
	int position;
	
	if (original == NULL)
	{
	    throw (DiffractionImageException("original image NULL"));
	}
	
	if (copy == NULL)
	{
	    throw (DiffractionImageException("copy image NULL"));
	}
	
	if ((width <= 0) ||
	    (height <= 0))
	{
	    throw (DiffractionImageException("bad size"));
	}
	
	// just a hack to prevent funny things going on
	
	if (zoom < 1)
	{
	    zoom = 1;
	}
	
	// check for special case zoom == 1 => duplicate
	if (zoom == 1)
	{
	    for (i = 0; i < width * height; i++)
	    {
		copy[i] = original[i];
	    }
	    
	    return;
	    
	}
	
	for (i = 0; i < height; i++)
	{
	    for (j = 0; j < width; j++)
	    {
		for (k = 0; k < zoom; k++)
		{
		    for (l = 0; l < zoom; l++)
		    {
			// simply copy one pixel over a larger area
			position = l +
			    k * width * zoom +
			    j * zoom +
			    i * zoom * width * zoom ;
			copy[position] = original[i * width + j];
		    }
		}
	    }
	}
    }
    
    // see above statement

    void reduceImage(unsigned short * original,
		     int width,
		     int height,
		     int zoom,
		     unsigned short * copy)
    {
	int i, j, k, l;
	int position, accumulator;
	    
	if (original == NULL)
	{
	    throw (DiffractionImageException("original image NULL"));
	}
	
	if (copy == NULL)
	{
	    throw (DiffractionImageException("copy image NULL"));
	}
	
	if ((width <= 0) ||
	    (height <= 0))
	{
	    throw (DiffractionImageException("bad size"));
	}
	
	// check the zoom value
	
	if (zoom < 1)
	{
	    zoom = 1;
	}
	
	// check for special case zoom == 1 => duplicate
	if (zoom == 1)
	{
	    for (i = 0; i < width * height; i++)
	    {
		copy[i] = original[i];
	    }
	    
	    return;
	    
	}
	
	// reset the input values to allow for the scaling - the above code 
	// will now work except it is now backwards
	
	// these routines will really bash the memory - probably a good
	// benchmark for memory bandwidth these!

	width /= zoom;
	height /= zoom;
	
	for (i = 0; i < height; i++)
	{
	    for (j = 0; j < width; j++)
	    {
		// for this, apply the average value over zoom x zoom
		// pixels as the new pixel value
		accumulator = 0;
		for (k = 0; k < zoom; k++)
		{
		    for (l = 0; l < zoom; l++)
		    {
			position = l +
			    k * width * zoom +
			    j * zoom +
			    i * zoom * width * zoom ;
			
			accumulator += original[position];
		    }
		}
		
		accumulator /= (zoom * zoom);
		
		copy[i * width + j] = accumulator;
	    }
	}
    }
    
    // this will convert a 16 bit unsigned short image to an unsigned 
    // char (byte) representation. theree are two options for these -
    // optimise and not optimise. the optimised version will perform
    // some floating point calculations to make a nicer image, but will
    // probably be quite a bit slower. the results from both should be 
    // tested.

    void downSample(unsigned short * original,
		    int width,
		    int height,
		    bool optimise,
		    unsigned char * copy)
    {
	// check that the input is ok
	
	if (original == NULL)
	{
	    throw (DiffractionImageException("original image NULL"));
	}
	
	if (copy == NULL)
	{
	    throw (DiffractionImageException("copy image NULL"));
	}
	
	if ((width <= 0) ||
	    (height <= 0))
	{
	    throw (DiffractionImageException("bad image size"));
	}
	
	// if not optimise then determine the appropriate left shiting to
	// do for each element - keeping integer arithmatic should make
	// this nice and fast
	
#ifdef TEST_CODE
	if (optimise)
	{
	    cout << "downsample: optmisation on" << endl;
	}
	else
	{
	    cout << "downsample: optimisation off" << endl;
	}
#endif

	if (optimise == false)
	{
	    int i, left, value;
	    
#ifdef TEST_CODE
	    cout << sizeof(long long) << endl;
#endif

	    // a long long should be big enough to store the total
	    // 8 bytes -> 18446744073709551616

	    long long total = 0;
	    

	    left = 1;
	    
	    // work over the image and work out, to the nearest power of
	    // 2, the maximum value found in the image

	    for (i = 0; i < width * height; i++)
	    {
		if (original[i] > (2 << left))
		{
		    left = left ++;
		}
		total += original[i];
	    }
	    
	    total /= (width * height);

	    // want the average to be about "half" the byte - but this
	    // may want to be tweaked

	    while (total < (2 << (left - 4)))
	    {
		left --;
	    }

#ifdef TEST_CODE
	    assert(left <= 16);
	    cout << "downsample: shifting " << left << " bits" << endl;
	    cout << "average: " << total << endl;
#endif
	    // recall that we want to shift from 16 to 8 bit
	    // so the amount to shift by is whatever is greater
	    // than 8
	    
	    if (left < 8)
	    {
		left = 0;
	    }
	    else
	    {
		left -= 8;
	    }

#ifdef TEST_CODE
	    cout << "downsample: shifting " << left << " bits" << endl;
#endif

	    // go through and copy byte shifted values into the new array
	    
	    for (i = 0; i < width * height; i++)
	    {
		value = original[i] >> left;

		// check that the values will fit into a byte - which 
		// we know they will by design

		if (value > 0xff)
		{
		    value = 0xff;
		}

		if (value < 0)
		{
		    value = 0;
		}

		// perform the assignment - with the typing
		// to prevent complaints
		copy[i] = (unsigned char) value;
	    }
	}
	else
	{
	    // we do floating point calculations to get a nice spread 
	    // of values over this image

	    // these are declared to be double as the average pixel value
	    // over the entire image will be gathered up - and we don't 
	    // want rounding errors, do we? :o)
	    
	    double min, max, average, total, current;
	    int value, i;
	    
	    // determine the average pixel value - this could 
	    // probably be applied over a smaller subset of the image to 
	    // good effect.

	    total = 0;
	    for (i = 0; i < width * height; i++)
	    {
		if (original[i] < 0x8000)
		{
		    total += original[i];
		}
		else
		{
		    total += 0x8000;
		}
	    }
	    
	    // this is found to give a nice image for X-ray 
	    // diffraction images - a dynamic range of sixteen seems
	    // to bring out both the diffraction and background nicely.
	    
	    average = total / (width * height);
	    
	    min = 0;
	    max = average * 4.0;
	 
#ifdef TEST_CODE
	    cout << "downsample: min " << min << " max " << max << endl;
#endif

	    // now rescale the original pixel values into the new ranges
	    // and store as bytes

	    for (i = 0; i < width * height; i++)
	    {
		current = original[i];
		value = (int) (0xff * (current - min) / (max - min));

		// this checking will definately be needed, as there
		// will by design be a large number of pixels which 
		// exceed the maximum value

		if (value > 0xff)
		{
		    value = 0xff;
		}
		else if (value < 0)
		{
		    value = 0;
		}
		
		// assign finally

		copy[i] = (unsigned char) value;
	    }
	}
    }
    
    // next a method to invert the image, since we will probably want that
    // x-ray diffraction images are normally displayed as a negative, 
    // with the areas of high x-ray intensity (i.e. spots) shown as black.
    
    void invertImage(unsigned char * image,
		     int width,
		     int height)
    {
	// value checking
	
	if (image == NULL)
	{
	    throw (DiffractionImageException("NULL image"));
	}
	
	if ((width <= 0) ||
	    (height <= 0))
	{
	    throw (DiffractionImageException("Bad size"));
	}
	
	int i;
	
	// simply compute the additive inverse

	for (i = 0; i < width * height; i++)
	{
	    image[i] = 0xff - image[i];
	}
    }
    
    // this function is the one which performs most of the actual
    // spade work, and will be calling all of the aboce functions to
    // convert the unsigned short non-negative image into unsigned
    // char (byte) negative image.

    void writeJpegImage(unsigned short * image,
			int width,
			int height,
			int quality,
			bool optimise,
			const char * filename)
    {
	unsigned char * copy;
	
	// validate the input
	
	if (filename == NULL)
	{
	    throw (DiffractionImageException("NULL image filename"));
	}
	
	if (image == NULL)
	{
	    throw (DiffractionImageException("NULL image"));
	}
	
	if ((width <= 0) ||
	    (height <= 0))
	{
	    throw (DiffractionImageException("bad size"));
	}
	
	if ((quality < 5) || 
	    (quality > 100))
	{
	    throw (DiffractionImageException("bad quality"));
	}
	
	// if we haven't included JPEG support then abort now - but
	// raise an exception to make debugging more possible.
	
#ifndef INCLUDE_JPEG
	throw (DiffractionImageException("JPEG support not included"
					 "define INCLUDE_JPEG"));
#endif
	
	// now we can get on with it!

	try
	  {
	    copy = new unsigned char[width * height];
	  }
	catch (bad_alloc & e)
	  {
	    throw (DiffractionImageException("error allocating image copy"));
	  }


	if (copy == NULL)
	{
	    throw (DiffractionImageException("error allocating image copy"));
	}

	// I am not sure how these could fail, but wrap in a try/catch 
	// clause just to be sure.
	
	try
	{
	    downSample(image, width, height, optimise, copy);
	    invertImage(copy, width, height);
	}
	catch (DiffractionImageException e)
	{
	    delete [] copy;
	    throw (e);
	}

#ifdef INCLUDE_JPEG
	
	// now to create the actual jpeg image
	// most of this code is taken from the example in the 
	// jpeg-6b distribution.

	// note well - that this will write to a C file stream - so 
	// opening a channel to the standard output is not at this
	// moment an option - but could perhaps be so in the future.

	// perhaps writing the jpeg to memory will be useful too.

	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	FILE * outfile;
	JSAMPROW row_pointer[1];
	int row_stride;
	
	outfile = fopen(filename, "w");
	
	if (outfile == NULL)
	{
	    delete [] copy;
	    throw (DiffractionImageException("error opening output file"));
	}
	
	// this may need to be fixed, as the default behaviour on 
	// error is to exit()
	
	// sed 's/may/will'

	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	
	jpeg_stdio_dest(&cinfo, outfile);
	
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 1;
	cinfo.in_color_space = JCS_GRAYSCALE;
	
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo, quality, TRUE);
	
	jpeg_start_compress(&cinfo, TRUE);
	
	row_stride = width;
	
	// iterate over the scanlines in the image and write the
	// output to the jpeg file.

	while (cinfo.next_scanline < cinfo.image_height)
	{
	    row_pointer[0] = & (copy[cinfo.next_scanline * row_stride]);
	    jpeg_write_scanlines(&cinfo, row_pointer, 1);
	}
	
	jpeg_finish_compress(&cinfo);
	fclose(outfile);
	jpeg_destroy_compress(&cinfo);
	
#endif
	
	delete [] copy;
	
    }
  














};
