// DiffractionImageADSC.cpp
// maintained by G.Winter
// 17th December 2003
// 
// The bits of the DiffractionImage library which pertain to ADSC
// images.
// 
// 
// 
// $Id: DiffractionImageADSC.cpp,v 1.7 2004/06/03 10:35:39 gwin Exp $

// C++ things
#include <iostream>
#include <fstream>
#include <map>


// core C things - required for using open() and read() below
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "DiffractionImage.h"

// if we are in development mode, then include things pertaining to
// assert.

#ifdef TEST_CODE
#define NDEBUG
#include <assert.h>
#endif

using namespace std;

namespace DI 
{

    // odds and sods

    // strip whitespace, make lower case

    string tidyKey(string key)
    {
	string result = "";
	int i, length;
	length = key.length();

	char littleString[2];
	littleString[1] = '\0';

	for(i = 0; i < length; i++)
	{
	    if ((key[i] > 32) &&
		(key[i] != '='))
	    {
		littleString[0] = (char) tolower(key[i]);
		result.append(littleString);
	    }
	}

	return result;
    }

    // strip whitespace from the ends

    string tidyValue(string value)
    {
	string result = "";
	int i, j, length;

	char littleString[2];
	littleString[1] = '\0';

	length = value.length();

	for(i = 0; i < length; i++)
	{
	    if (value[i] > 32)
	    {
		break;
	    }
	}

	for(j = i; j < length; j++)
	{
	    if (value[j] != ';')
	    {
		littleString[0] = value[j];
		result.append(littleString);
	    }
	}

	return result;
    }

    
    // read in an ADSC image - this will read the header first, then 
    // reopen the file to read the unsigned shorts out - byte 
    // swapping if needed. NOTE WELL - this is hacky as there is 
    // an assumption here that the images will be represented as
    // unsigned shorts, which could well be wrong!

    void DiffractionImage::loadADSC(string filename)
    {
	// read the header
	loadADSCHeader(filename);

	// allocate the image structure

	allocateImage();

#ifdef TEST_CODE
	assert(image != NULL);
#endif

	// after this stage, the rest of the image is just a big pile
	// of unsigned shorts - another chance to share code! :o)

	// this routine will be defined in DiffractionImageRaw.cpp

	loadRawImage(filename, originalHeaderLength);

    }
    
    void DiffractionImage::loadADSCHeader(string filename)
    {
	// how shall I read the header? probably using a hash table 
	// (map) is best..
	map <string, string> header;
	fstream fin;
	int equals, semicolon;
	string line, key, value;
	bool headerComplete;

	// an ADSC header begins with { and ends with } alone 
	// on a line

	headerComplete = false;

	// open the file for input

	fin.open(filename.c_str(), ios::in);
	
	if (!fin.good())
	{
	    throw(DiffractionImageException("error opening file"));
	}

	getline(fin, line);

	if (line != "{")
	{
	    // this is not an ADSC image
	    fin.close();
	    throw(DiffractionImageException("not an ADSC header"));
	}

	// now work through the header

	while (headerComplete == false)
	{
	    if (fin.eof())
	    {
		throw(DiffractionImageException("EOF before header end"));
	    }

	    getline(fin, line);

	    if (line[0] == '}')
	    {
		// then the header is finished
		headerComplete = true;
		break;
	    }

#ifdef TEST_CODE_SHOW_HEADER
	    cerr << line << endl;
#endif

	    equals = line.find('=');

	    if (equals == (int) string::npos)
	    {
		throw(DiffractionImageException("'=' not found in header"));
	    }

	    semicolon = line.find(';');

	    if (semicolon == (int) string::npos)
	    {
		throw(DiffractionImageException("';' not found in header"));
	    }

	    // delete these things from the line now - or just 
	    // select the limits carefully
	    key = line.substr(0, equals);
	    // advance equals to miss the sign
	    equals += 1;
	    value = line.substr(equals, semicolon - equals);
	    
	    // these are defined locally

	    key = tidyKey(key);
	    value = tidyValue(value);

#ifdef TEST_CODE
	    cout << "header key = \"" << key << "\" value = \"" << value 
		 << "\"" << endl;
#endif

	    header[key] = value;
	}

	// now parse what we stored in the map

	// verify that the header structure is ok - this needs to be 
	// done via a macro - and now is

	// here is a prototype of what I want

	if (header.find("header_bytes") == header.end())
	{
	    // header_bytes not found in map
	    throw(DiffractionImageException("header_bytes" 
					    " not found in header"));
	}

#ifndef TEST_CODE
#define ASSERT_IN_HEADER(key) \
if (header.find(key) == header.end()) \
{ \
    throw(DiffractionImageException(key " not found in header")); \
}
#else
#define ASSERT_IN_HEADER(key) \
if (header.find(key) == header.end()) \
{ \
    throw(DiffractionImageException(key " not found in header")); \
} else { cout << "found key \"" << key << "\"" << endl; }
#endif

	ASSERT_IN_HEADER("size1");
	ASSERT_IN_HEADER("size2");
	ASSERT_IN_HEADER("byte_order");
	
	ASSERT_IN_HEADER("distance");
	ASSERT_IN_HEADER("wavelength");
	ASSERT_IN_HEADER("beam_center_x");
	ASSERT_IN_HEADER("beam_center_y");
	ASSERT_IN_HEADER("pixel_size");
        ASSERT_IN_HEADER("time");

	ASSERT_IN_HEADER("osc_start");

	// now check that the rest of the oscillation 
	// information is in there

	if ((header.find("osc_end") == header.end()) &&
	    (header.find("osc_range") == header.end()))
	{
	    // then there isn't enough information in there
	    throw(DiffractionImageException("oscillation information "
					    "not inheader"));
	}

	// if we get here then everything we want is in the header
	// now fetch it all out into the header data structure

	// this will also be done via macro, with the following
	// prototype

        sscanf(header["header_bytes"].c_str(), "%d", 
	       &originalHeaderLength);

#ifndef TEST_CODE
#define GET_FROM_HEADER(key, format, value) \
sscanf(header[key].c_str(), format, &value);
#else
#define GET_FROM_HEADER(key, format, value) \
sscanf(header[key].c_str(), format, &value); \
cout << key << "=" << value << endl;
#endif	

        GET_FROM_HEADER("size1", "%d", width);
	GET_FROM_HEADER("size2", "%d", height);

        GET_FROM_HEADER("beam_center_x", "%f", beamX);
        GET_FROM_HEADER("beam_center_y", "%f", beamY);

	GET_FROM_HEADER("wavelength", "%f", wavelength);
	GET_FROM_HEADER("distance", "%f", distance);

	GET_FROM_HEADER("pixel_size", "%f", pixelX);	
	GET_FROM_HEADER("pixel_size", "%f", pixelY);

        GET_FROM_HEADER("time", "%f", exposureTime);

	if (header["byte_order"] == "big_endian")
	{
	    bigEndian = true;
	}
	else
	{
	    bigEndian = false;
	}

	float range;
	GET_FROM_HEADER("osc_start", "%f", phiStart);

if (header.find("phi") != header.end())
{
  float phiStartTemp;
  phiStartTemp = phiStart;
  GET_FROM_HEADER("phi", "%f", phiStart);
  if (fabs(phiStartTemp - phiStart) > 0.01)
    {
      // something dodgy is going on
      warning = true;
      message = "ambiguous phi values;";
    }
}

if (header.find("axis") != header.end())
{
  string axis = header["axis"];
  if (axis != "phi")
    {
      // something dodgy is going on here too!
      if (warning == true)
	{
	  message = message + "axis not phi;";
	}
      else
	{
	  warning = true;
	  message = "axis not phi;";
	}
    }
}

	if (header.find("osc_end") == header.end())
	{
	    // then we have to figure it out from the range
	    GET_FROM_HEADER("osc_range", "%f", range);
	    phiEnd = phiStart + range;
	}
	else
	{
	    // we can just get it straight from the header
	    GET_FROM_HEADER("osc_end", "%f", phiEnd);
	}

	// that's all folks!


    }

};
