// DiffractionImage.cpp
// maintained by G.Winter
// 16th December 2003
// 
// A replacement for the old DiffractionImage module - this time developed
// more carefully so that it should be much easier to extend and more
// portable.
// 
// 
// $Id: DiffractionImage.cpp,v 1.14 2005/11/22 13:38:14 svensson Exp $

#include "DiffractionImage.h"

#include <iostream>
#include <fstream>
#include <string>
#include <new>

// core C things - required for using open() and read() below
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

// including the jpeg and jasper stuff in the same file causes arguments -
// so liberate one from the file - jpeg.

#ifdef INCLUDE_JASPER
extern "C" {
#include "jasper/jasper.h"
}
#endif

using namespace std;
namespace DI 
{

    // first start on the constructors
    
    DiffractionImage::DiffractionImage(void)
    {
	constructor();
    }
    
    DiffractionImage::DiffractionImage(string filename)
    {
	constructor(filename);
    }
    
    DiffractionImage::~DiffractionImage(void)
    {
	destructor();
    }
    
    void DiffractionImage::constructor(void)
    {
	// initalise all of the data in this object
	memset(originalHeader, 0, 4096);
	originalHeaderLength = 0;
	
	// assume that we are setting everything to false || 0
	bigEndian = false;
	
	width = 0;
	height = 0;
	
	// these values could be set to NaN or something to 
	// raise an exception if they are used
	
	beamX = 0;
	beamY = 0;
	distance = 0;
	wavelength = 0;
	twoTheta = 0;
	phiStart = 0;
	phiEnd = 0;
	pixelX = 0;
	pixelY = 0;
	exposureTime = 0.0;

	warning = false;

	// need to call the constructor on message then
	(void) new (&message) string;
	message = "";
	
	image = NULL;
    }
    
    void DiffractionImage::constructor(string filename)
    {
	// simply initialise and then read an image - this is more
	// a convenience function than anything else
	
	constructor();
	load(filename);
    }
    
    void DiffractionImage::destructor(void)
    {
	// if there is space allocated for an image, delete it
	if (image != NULL)
	{
	    delete [] image;
	    image = NULL;
	}
	
	// the advertised behaviour is to reset so
	constructor();
    }
    
    // next the data access methods - yawn!
    // define a couple of C preprocessor macros to make this easier
    
#define GETTER_METHOD(type, arg, capitalArg) \
type DiffractionImage::get ## capitalArg(void) { return arg; }
    
#define SETTER_METHOD(type, arg, capitalArg) \
void DiffractionImage::set ## capitalArg(type new ## capitalArg) \
{ arg = new ## capitalArg; }
    
    // now the methods - appologies for the macros

    GETTER_METHOD(int, height, Height);
    GETTER_METHOD(int, width, Width);
    GETTER_METHOD(float, beamX, BeamX);
    GETTER_METHOD(float, beamY, BeamY);
    GETTER_METHOD(float, wavelength, Wavelength);
    GETTER_METHOD(float, distance, Distance);
    GETTER_METHOD(float, twoTheta, TwoTheta);
    GETTER_METHOD(float, phiStart, PhiStart);
    GETTER_METHOD(float, phiEnd, PhiEnd);
    GETTER_METHOD(float, pixelX, PixelX);
    GETTER_METHOD(float, pixelY, PixelY);
    GETTER_METHOD(float, exposureTime, ExposureTime);

  GETTER_METHOD(bool, warning, Warning);
  GETTER_METHOD(string, message, Message);
    
    SETTER_METHOD(float, beamX, BeamX);
    SETTER_METHOD(float, beamY, BeamY);
    SETTER_METHOD(float, wavelength, Wavelength);
    SETTER_METHOD(float, distance, Distance);
    SETTER_METHOD(float, twoTheta, TwoTheta);
    SETTER_METHOD(float, phiStart, PhiStart);
    SETTER_METHOD(float, phiEnd, PhiEnd);
    SETTER_METHOD(float, pixelX, PixelX);
    SETTER_METHOD(float, pixelY, PixelY);
    SETTER_METHOD(float, exposureTime, ExposureTime);
    
    // this is MUCH quicker than typing all of the methods by hand
    
    // for accessing the image I think I will have to write the code by
    // hand - Oh what a shame

    unsigned short * DiffractionImage::getImage(void)
    {
	if (image == NULL)
	{
	    throw (DiffractionImageException("image is NULL"));
	}
	
	return image;
    }

    unsigned short & DiffractionImage::operator[] (int offset)
    {
	// check before we go
	if (image == NULL)
	{
	    throw(DiffractionImageException("NULL image"));
	}

	if ((offset < 0) ||
	    (offset >= width * height))
	{
	    throw(DiffractionImageException("out of range"));
	}

	// if we get to here then

	return image[offset];
    }

    DiffractionImage & DiffractionImage::operator= (DiffractionImage & source)
    {
	// first call the destructor to reinitialise the structure
	destructor();

	// then don't copy the header info

	// now copy the elements from the source image
	width = source.getWidth();
	height = source.getHeight();
	beamX = source.getBeamX();
	beamY = source.getBeamY();
	distance = source.getDistance();
	wavelength = source.getWavelength();
	twoTheta = source.getTwoTheta();	
	phiStart = source.getPhiStart();
	phiEnd = source.getPhiEnd();
	pixelX = source.getPixelX();
	pixelY = source.getPixelY();

	try
	{
	    unsigned short * otherImage = source.getImage();
	    // allocate some space
	    allocateImage();
	    // copy
	    for (int i = 0; i < width * height; i++)
	    {
		image[i] = otherImage[i];
	    }
	}
	catch (DiffractionImageException & e)
	{
	    // do nothing - the image is NULL;
	}

	return *this;

    }


    string DiffractionImage::load(string filename)
    {
	string format = guessFormat(filename);
	try
	  {
	    load(filename, format);
	  }
	catch(DiffractionImageException & e)
	  {
	    throw(e);
	  }

	// if we get this far with an unknown image type it is a mar
	if (format == "unknown")
	  {
	    format = "mar";
	  }

	return format;
    }

    void DiffractionImage::load(string filename,
				string format)
    {
	if (format == "adsc")
	{
	    loadADSC(filename);
	}
	else if (format == "marccd")
	{
	    loadMarCCD(filename);
	}
	else
	{
	  try
	    {
	      loadMAR(filename);
	    }
	  catch(DiffractionImageException & e)
	    {
	      throw(DiffractionImageException("unknown image format"));
	    }
	}

	// now have a look at the beam position, and if it looks like it
	// is really small numbers translate it to be relative to the 
	// centre of the detector

	float image_width = width * pixelX;
	float image_height = height * pixelY;

	float r = sqrt(beamX * beamX + beamY * beamY);

	if (r < 0.1 * image_width)
	  {
	    beamX += 0.5 * image_width;
	    beamY += 0.5 * image_height;
	  }


    }	

    string DiffractionImage::loadHeader(string filename)
    {
	string format = guessFormat(filename);
	try
	  {
	    loadHeader(filename, format);
	  }
	catch(DiffractionImageException & e)
	  {
	    throw(e);
	  }

	// if we get this far with an unknown image type it is a mar
	if (format == "unknown")
	  {
	    format = "mar";
	  }

	// now have a look at the beam position, and if it looks like it
	// is really small numbers translate it to be relative to the 
	// centre of the detector

	float image_width = width * pixelX;
	float image_height = height * pixelY;

	float r = sqrt(beamX * beamX + beamY * beamY);

	if (r < 0.1 * image_width)
	  {
	    beamX += 0.5 * image_width;
	    beamY += 0.5 * image_height;
	  }

	return format;
    }

    void DiffractionImage::loadHeader(string filename,
				      string format)
    {
	if (format == "adsc")
	{
	    loadADSCHeader(filename);
	}
	else if (format == "marccd")
	{
	    loadMarCCDHeader(filename);
	}
	else
	{
	  try
	    {
	      loadMARHeader(filename);
	    }
	  catch(DiffractionImageException & e)
	    {
	      throw(DiffractionImageException("unknown image format"));
	    }
	}
    }

    void DiffractionImage::loadCompressed(string filename)
    {
	string headerName = filename + ".hdr";
	string imageName = filename + ".jp2";
	loadCompressed(headerName, imageName);
    }

    void DiffractionImage::loadCompressed(string headerName,
					  string imageName)
    {
	loadHeader(headerName);
	loadCompressedImage(imageName);
    }

    void DiffractionImage::loadCompressedImage(string imageName)
    {

#ifndef INCLUDE_JASPER
	throw(DiffractionImageException("Jpeg2000 support not included"));
#endif

#ifdef INCLUDE_JASPER


	// first check header
	if ((width  <= 0) ||
	    (height <= 0))
	{
	    throw(DiffractionImageException("bad size"));
	}

	int newWidth, newHeight;

	// the image will be allocated when it is needed - and
	// not before!

	jas_stream_t * in = jas_stream_fopen(imageName.c_str(), "r");
	if (in == NULL)
	{
	    // there was an error opening the file
	    throw(DiffractionImageException("error opening file"));
	}

	jas_image_t * jp2 = jas_image_decode(in, jas_image_strtofmt("jp2"), 0);

	if (jp2 == NULL)
	{
	    jas_stream_close(in);
	    throw(DiffractionImageException("error reading jp2 image"));
	}

	jas_stream_close(in);

	// now get the image out - this should be relatively easy

	newWidth = jas_image_cmptwidth(jp2, 0);
	newHeight = jas_image_cmptheight(jp2, 0);

	if ((newHeight != height) ||
	    (newWidth != width))
	{
	    // then this is the wrong image!
	    jas_image_destroy(jp2);
	    throw(DiffractionImageException("bad size in image"));
	}

	// if we get to here then all's well
	jas_matrix_t * data = jas_matrix_create(width, height);

	if (data == NULL)
	{
	    jas_image_destroy(jp2);
	    throw(DiffractionImageException("error allocating image "
					    "component"));
	}
	
	jas_image_readcmpt(jp2, 0, 0, 0, width, height, data);

	// next copy this information out of the matrix and into the 
	// image

	// allocate or reallocate the image if necessary
	if (image != NULL)
	{
	    delete [] image;
	    image = NULL;
	}

	// again two layers of memory error trapping - since we 
	// don't want any unhandled exceptions.

	try
	  {
	    image = new unsigned short[width * height];
	  }
	catch (bad_alloc & e)
	  {
	    jas_image_destroy(jp2);
	    jas_matrix_destroy(data);
	    throw(DiffractionImageException("error allocating image"));
	  }

	if (image == NULL)
	{
	    jas_image_destroy(jp2);
	    jas_matrix_destroy(data);
	    throw(DiffractionImageException("error allocating image"));
	}

	for (int i = 0; i < height; i++)
	{
	    for (int j = 0; j < width; j++)
	    {
		image[i * width + j] = jas_matrix_get(data, i, j);
	    }
	}

	// free the data
	jas_image_destroy(jp2);
	jas_matrix_destroy(data);

#endif

    }

    void DiffractionImage::regenerateHeader(void)
    {
	// do nothning
    }

    // try and guess the format of an image - based on the first two
    // bytes. adsc images begin {\n and marccd images begin 0x4949
  
    string guessFormat(string filename)
    {
	int fd, amount;
	unsigned char buffer[4];
	fd = open(filename.c_str(), O_RDONLY);
	if (fd < 0)
	{
	    throw(DiffractionImageException("error opening file"));
	}
	amount = read(fd, buffer, 4);
	if (amount != 4)
	{
	    close(fd);
	    throw(DiffractionImageException("error reading file"));
	}
	close(fd);

	if ((buffer[0] == 0x49) &&
	    (buffer[1] == 0x49))
	{
	    // this is a marccd image
	    return string("marccd");
	}

	if (((buffer[0] == 0x0a) &&
	     (buffer[1] == 0x7b)) ||
	    ((buffer[0] == 0x7b) &&
	     (buffer[1] == 0x0a)))
	{
	    // this is an ADSC image
	    return string("adsc");
	}

	if ((buffer[0] == 'R') &&
	    (buffer[1] == 'A') && 
	    (buffer[2] == 'X') && 
	    (buffer[3] == 'I')) 
	  {
	    return string("raxis");
	  }
	  
	return string("unknown");
    }
  
    ostream & operator<< (ostream & out,
			  DiffractionImage & d)
    {
	const string tab = "\t";

	out << "width" << tab << d.width << tab 
	    << "height" << tab << d.height << endl;
	out << "X" << tab << d.beamX << tab 
	    << "Y" << tab << d.beamY << endl;
	out << "distance" << tab << d.distance << tab 
	    << "wavelength" << tab << d.wavelength << endl;
	out << "start" << tab << d.phiStart << tab 
	    << "end" << tab << d.phiEnd << endl;

	return out;
    }
   
    // DiffractionImageException stuff
    
    DiffractionImageException::DiffractionImageException(void)
    {
	message = "";
    }
    
    DiffractionImageException::DiffractionImageException(string newMessage)
    {

#ifdef TEST_CODE
	cerr << newMessage << endl;
#endif

	message = newMessage;
    }
    
    DiffractionImageException::~DiffractionImageException(void)
    {
	// do nothing
    }
    
    string DiffractionImageException::getMessage(void)
    {
	return message;
    }
 
};
