#!/usr/bin/env python
# Screen.py
# Maintained by G.Winter
# 2nd february 2004
# 
# A part of the second generation scheduler - this will screen images.
# In this there will be the following abilities:
# 
# 1. Check if an image is okay:
# This may return one of the following strings:
# 'ok'
# 'blank'
# 'icy'
# 
# $Id: Screen.py,v 1.32 2006/03/13 13:03:53 gwin Exp $

import sys, math, os, re, exceptions

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

dna = os.environ['DNAHOME']

verbose = False

try:
    import DiffractionImage
except:
    raise RuntimeError, 'DiffractionImage library not available'

sys.path.append('%s/scheduler/Scheduler/Mosflm' % dna)
sys.path.append('%s/xsd/python' % dna)
from Messenger import Messenger
from Somewhere import Somewhere

sys.path.append('%s/expertise/python/graeme/Expert' % dna)
import Expert

# PROPER SCREENING THINGS WHICH BELONG IN HERE

def Screen(filename):
    '''Perform a properly structured screening of the image filename'''

    d = DiffractionImage.DiffractionImage(filename)
    p = DiffractionImage.PeakList()

    # perform a peak search on this image - this is the basis of the
    # screening

    if verbose:
        print 'Image: %s' % filename
        print 'Beam %f %f' % (d.getBeamX(), d.getBeamY())
        print 'Wavelength, distance %f %f' % \
              (d.getWavelength(), d.getDistance())

    count = p.find(d, 10000, 5.0)

    # check for a misaligned back stop (this looks at the balance of
    # intensity on the image)

    if asymmetry_search(d, p) == 'backstop_problem':
        return 'backstop_problem'

    # next see if the image is 'blank'
    if blank_search(d, p) == 'blank':
        return 'blank'

    # store some information for future reference at the
    # autoindexing stage
    i_over_sig = Expert.GuessIOverSig(p)
    Messenger.log_write('Screening recommended I/Sigma %f' % i_over_sig, 2)
    Somewhere.store('autoindex_i_over_sig', i_over_sig)

    # finally see if there are ice rings on the image
    if ice_search(d, p) > 0.2:
        return 'icy'

    if ring_search(d, p) > 0.1:
        Messenger.log_write('Rings found but not in ice regions')
        Messenger.log_write('I suggest you inspect these images')
        return 'rings'

    # if we get to here then it should be ok
    return 'ok'

def ScreenOLD(filename):
    try:
        d = DiffractionImage.DiffractionImage(filename)
    except:
        raise RuntimeError, 'Error opening image %s' % filename

    p = DiffractionImage.PeakList(d)

    Messenger.log_write('Found %d peaks' % p.length(), 2)
    top = -1
    for i in range(p.length()):
        if p[i][2] < 20.0:
            top = i
            break
    
    Messenger.log_write('%d peaks with I/sigma > 20.0' % top, 2)

    # next guess an appropriate i/sigma value to use in indexing
    i_over_sig = Expert.GuessIOverSig(p)

    Messenger.log_write('Screening recommended I/Sigma %f' % i_over_sig, 2)
    
    Somewhere.store('autoindex_i_over_sig', i_over_sig)

    top = -1
    for i in range(p.length()):
        if p[i][2] < i_over_sig:
            top = i
            break

    Messenger.log_write('%d peaks with I/sigma > %f' % (top, i_over_sig), 2)
    
    radial = d.radial(32)
    histogram = d.histogram()

    total = 0
    for h in histogram:
        total += h

    subtotal = 0
    i = 0
    for h in histogram:
        subtotal += h
        i += 1
        if subtotal > (0.95 * total):
            break

    for r in radial:
        # print r
        pass

    status = LookForIce(radial)
    if status != 'ok':
        return status

    # check how many peaks we have with an I/sigma > 10 - if this is fewer
    # than 10 say that this is blank.

    for i in range(p.length()):
        peak = p[i]
        if peak[2] < 10:
            break

    if i < 20:
        return 'blank'


    # at this stage I want to look for asymmetry in the diffracted intensity -
    # that is a misaligned back stop

    asymmetry = d.asymmetry()
    for a in asymmetry:
        if math.fabs(a) > 0.33:
            return 'backstop_problem'

    # next search for ice rings - use I/sigma = 20 for this
    # assert: I/sigma of 20 is too large for this ...
    # need a calibration data set

    # 6/10/04 change i/sigma to 10.0
    
    p = DiffractionImage.PeakList()
    count = p.find(d, 10000, 10.0)

    if count < 20:
        # then we are very unlikely to have ice rings!
        # although this image could be described as being a
        # bit blank!
        return 'ok'

    # use 1000 iterations of the search and a search radius of 1mm
    # 6/10/04 extend this to 10k iterations to allow for a better search
    circle = p.circle(10000, 1.0)
    fraction = float(circle[0]) / float(count)

    # a note pertaining to bug 839 - this does not handle well the problem
    # where you have a number of ice rings. in this instance a more subtle
    # method of looking for ice rings may be needed, for instance if I
    # had a search among the spot list for large numbers of spots with
    # resolutions recognisable as "ice".

    print fraction

    if fraction > 1.33:
        # then the image is probably very icy - this can't happen
        # has been overruled by LookForIce
        return 'veryicy'
    # 6/10/04 change this to 0.1 from 0.2 to better trap ice rings -
    # since a false positive is better than a false negative
    elif fraction > 0.10:
        # then we probably have slight ice rings
        return 'icy'
    
    return 'ok'

def Histogram(filename):
    try:
        d = DiffractionImage.DiffractionImage(filename)
    except:
        raise RuntimeError, 'Error opening image %s' % filename

    histogram = d.histogram()
    return histogram

def Radial(filename):
    try:
        d = DiffractionImage.DiffractionImage(filename)
    except:
        raise RuntimeError, 'Error opening image %s' % filename

    radial = d.radial(32)
    return radial

# a much better ice ring search!

def Ice(filename):
    '''A search for ice rings based on the numbers of spots in given
    resolution ranged - note well that this depends on the header
    information being correct'''

    # read in the image
    d = DiffractionImage.DiffractionImage(filename)

    # create a peak list
    p = DiffractionImage.PeakList()

    # search for spots - this will return a list of (x, y, i) tuples
    count = p.find(d, 10000, 5.0)

    return ice_search(d, p)

def asymmetry_search(d, p):
    '''Decide if an image is "asymmetric" - that is is the beam stop
    correctly aligned with the beam - p is not used'''

    asymmetry = d.asymmetry()

    for a in asymmetry:
        if math.fabs(a) > 0.33:
            return 'backstop_problem'

    return 'ok'

def blank_search(d, p):
    '''Decide if an image is "blank" from a peaklist (already found)
    and the image (not used)'''

    # an image is also "blank" if it has fewer than 10 peaks with I/sigma
    # of 10.0

    if p.length() > 10:
        if p[10][2] > 10.0:
            return 'ok'

    # an image is blank if it's radial profile is (i) flat and (ii)
    # full of small numbers

    radial = d.radial(16)[:-1]

    mean = 0.0
    for r in radial:
        mean += r
    mean /= 16.0

    ok = False
    for r in radial:
        # if |mean - bin| > 2.0 then this is "not blank" 
        if math.fabs(r - mean) > 2.0:
            ok = True
            break

    if not ok:
        # image was blank because all pixel values uniform
        return 'blank'
        
    return 'blank'

def ice_search(d, p):
    '''Count the number of peaks in list p which fall withing ice
    rings according to the header information in p'''

    count = p.length()

    if count < 20:
        return 0.0

    wavelength = d.getWavelength()
    distance = d.getDistance()

    ice_regions = 0
    
    for i in range(count):
        d = math.sqrt(p[i][0] * p[i][0] + p[i][1] * p[i][1])
        resolution = wavelength * 0.5 / \
                     (math.sin(0.5 * math.atan(d / distance)))
        if resolution < 3.94 and resolution > 3.86:
            ice_regions += 1
        elif resolution < 3.71 and resolution > 3.63:
            ice_regions += 1
        elif resolution < 3.48 and resolution > 3.40:
            ice_regions += 1
        elif resolution < 2.71 and resolution > 2.63:
            ice_regions += 1
        elif resolution < 2.29 and resolution > 2.21:
            ice_regions += 1
        elif resolution < 2.09 and resolution > 2.01:
            ice_regions += 1


    fraction = float(ice_regions) / float(count)

    return fraction

def ring_search(d, p):
    '''Look for rings - may be slow!!'''

    count = p.length()
    circle = p.circle(10000, 1.0)
    fraction = float(circle[0]) / float(count)

    return fraction

def DecideIfImageIsIcy(filename):
    '''The routine which will actually decide if an image is icy based
    on the number of "spots" in ice ring regions'''

    fraction = Ice(filename)

    if fraction < 0.3:
        return 'ok'
    else:
        return 'icy'

def LookForIce(radial):
    '''Look for "ice rings" in a radial profile plot - this should be
    deprecated'''

    total = 0.0
    max = -65536.0
    min = 65536.0

    # get a few stats from the first half

    count = len(radial) - 1
    
    for i in range(1, count / 2):
        total += radial[i]
        if radial[i] > max:
            max = radial[i]
        if radial[i] < min:
            min = radial[i]
            
    mean = total / float(int(count / 2) - 1)
    variation = max - min

    # and work over the second half - if a point in here is
    # much higher than the first batch we may have ice!
    
    for i in range(count / 2, count):
        di = radial[i] - radial[i - 1]
        if di > variation:
            return 'veryicy'

    # next take an average of all of the data and see if di > average
    # at some stage... => icy probably

    # this is a bit weak - it describes things as icy which aren't!

    # comment this out for the moment!

    if 1 == 2:

        total = 0.0
        for i in range(count):
            total += radial[i]
        mean /= count

        for i in range(1, count):
            di = radial[i] - radial[i - 1]
            if di > mean:
                return 'icy'

    return 'ok'

# NOT PROPER SCREENING THINGS - THESE DON'T BELONG HERE

def GetHeader(filename):
    '''Get back a dictionary with the header information for file filename'''
    try:
        d = DiffractionImage.DiffractionImage(filename, 0)
    except exceptions.Exception, e:
        Messenger.log_write('Error getting header %s' % str(e))
        raise RuntimeError, 'Error opening image %s:%s' % (filename, str(e))

    header = { }
    header['ExposureTime'] = d.getExposureTime()
    header['BeamX'] = d.getBeamX()
    header['BeamY'] = d.getBeamY()
    header['Distance'] = d.getDistance()
    header['Wavelength'] = d.getWavelength()
    header['PixelX'] = d.getPixelX()
    header['PixelY'] = d.getPixelY()
    header['Width'] = d.getWidth()
    header['Height'] = d.getHeight()
    messages = d.getMessage()
    header['Message'] = messages.split(';')
    header['PhiStart'] = d.getPhiStart()
    header['PhiEnd'] = d.getPhiEnd()
    header['PhiWidth'] = d.getPhiEnd() - d.getPhiStart()
    r = 0.5 * d.getPixelX() * d.getWidth()
    A = d.getDistance()

    if A > 0.0:
        theta = math.atan(r / A)
        
        wave = d.getWavelength()
        resolution = wave / math.sin(theta)
        header['EdgeResolution'] = resolution

    else:
        header['EdgeResolution'] = 0.0

    # determine the image format
    bytes = open(filename, 'r').read(4)

    if bytes == '{\nHE':
        header['Format'] = 'smv'
    elif bytes[:2] == 'II':
        header['Format'] = 'tiff'
    elif bytes[:2] == 'RA':
        header['Format'] = 'raxis'
    elif bytes == '\xd2\x04\x00\x00':
        header['Format'] = 'marip'
    else:
        header['Format'] = 'unknown'

    del(d)
    return header

def Type(filename):
    d = DiffractionImage.DiffractionImage()
    type = d.load(filename)
    del(d)
    return type

def findNULLPIX(radial):
    diffmax = -100;
    nullpix = 0
    for i in range(len(radial) - 1):
        diff = radial[i + 1] - radial[i]
        if diff > diffmax:
            nullpix = radial[i] - radial[0]
            diffmax = diff

    return nullpix

def Filename(directory, template, image):
    expression = r'(.*)_(\#*)\.(.*)'
    regexp = re.compile(expression)
    match = regexp.match(template)
    prefix = match.group(1)
    extension = match.group(3)
    length = len(match.group(2))
    
    format = '%0' + str(length) + 'd'
    
    number = format % image
    filename = directory + '/' + prefix + '_' + number + '.' + extension

    return filename

def Nullpix(image1, image2):
    d1 = DiffractionImage.DiffractionImage(image1)
    d2 = DiffractionImage.DiffractionImage(image2)
    
    d1.min(d2)

    radial = d1.radial(d1.getWidth() / 2)

    nullpix = findNULLPIX(radial)

    Messenger.log_write('NULLPIX determined to be %d' % int(nullpix))

    return int(nullpix)

def Print(filename):
    d = DiffractionImage.DiffractionImage(filename)
    print 'Header information'
    print 'image %d %d' % (d.getWidth(), d.getHeight())
    print 'pixel %f %f' % (d.getPixelX(), d.getPixelY())
    print 'beam %f %f' % (d.getBeamX(), d.getBeamY())
    print 'distance %f' % d.getDistance()
    print 'geometry %f %f' % \
          (d.getWidth() * d.getPixelX(),
           d.getHeight() * d.getPixelY())
    print 'exposure time %f' % d.getExposureTime()
    print 'state %s' % Screen(filename)

if __name__ == '__main__':

    import sys

    filenames = []

    if len(sys.argv) > 1:
        for i in range(1, len(sys.argv)):
            filenames.append(sys.argv[i])

    else:
        raise RuntimeError, '%s image' % sys.argv[0]

    if filenames[0][:2] == '-v':
        verbose = True
        filenames = filenames[1:]

    for filename in filenames:
        try:
            status = Screen(filename)
            print filename, status
            # d = DiffractionImage.DiffractionImage(filename)
            # print d.getBeamX(), d.getBeamY()
        except exceptions.Exception, e:
            print str(e)
            print filename, ' failed'

