##############################################################################
#
#   File:      dr_.py
#
#   Project:   DataRanking Project
#
#   Author(s): Romeu A. Pieritz [pieritz@esrf.fr]
#              Olof Svensson [svensson@esrf.fr]
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Data Ranking Project Project
#   Copyright (c) 2005, Data Ranking Project developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of DataRanking's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################
   
   
from dr_system           import *
from dr_kernel           import *
from dr_engine           import *
from dr_engine_plugin    import *

from dr_dna_screening_output_lattice    import DRDnaScreeningOutputLattice


##############################################################################
# DRCrystal

class UnitTestDRCrystal( DRTestCase ): 
    
    def testDRCrystal( self ):
        DRVerbose.unitTest( "UnitTest DRCrystal:" )
           
        odrCrystal = DRCrystal()
        odrCrystal.setName( "bio-crystal" )
        
        # add only the last...to not repeat factors = controled by the MagicNumber
        for i in range( 0, 10 ):
            odrFactorResolution = DRFactorResolution( i )
            odrFactorResolution.setName( "resolution-" + DRString( i ) )
            odrCrystal.addFactor( odrFactorResolution )
       
        odrCrystal.outputScreen()
        
        DRVerbose.unitTest()
     
     
    # DNA BioXHIT Compatibility Layer   
    def testDRCrystalDNA( self ):
        DRVerbose.unitTest( "UnitTest DRCrystalDNA:" )
           
        odrCrystal = DRCrystal()
 #       odrCrystal.importXmlDna( "data-crystal-dna.xml" )
        odrCrystal.importXmlDna( "data-dna-project-drank.xml" )
        odrCrystal.setResolution( DRRandom.randFloat( 1.0, 12.0 ) )
        odrCrystal.exportXmlDna( "DATA-dna-project-drank-COPY.xml" )
        odrCrystal2 = odrCrystal.duplicate()
        odrCrystal2.exportXmlDna( "DATA-dna-project-drank-COPY-DUPLICATED.xml" )       
        odrCrystal.outputScreen()
        
        DRVerbose.unitTest()
        
    
    def testDRPythonStuffs( self ):
        DRVerbose.unitTest( "UnitTest PythonStuffs:" )
        odrDnaScreeningOutputLattice = DRDnaScreeningOutputLattice()
        DRVerbose.unitTest()


##############################################################################
# DRProject

class UnitTestDRProject( DRTestCase ): 
    
    def testDRProject( self ):
        DRVerbose.unitTest( "UnitTest DRProject:" )
           
        odrProject = DRProject()
        odrProject.setName( "Project-Test-Crystal" )
        
        for i in range( 0, 10 ):
            odrFactorResolution = DRFactorResolution( i )
            odrFactorResolution.setName( "resolution-" + DRString( i ) )
            odrCrystal = DRCrystal()
            odrCrystal.setName( "Crystal-" + DRString( i ) )
            odrCrystal.addFactor( odrFactorResolution )
            odrProject.addCrystal( odrCrystal )
       
        odrProject.outputScreen()
        
        DRVerbose.unitTest()
        

    # DNA BioXHIT Compatibility Layer 
    def testDRProjectDNA( self ):
        DRVerbose.unitTest( "UnitTest DRProject:" )
           
           
        DRVerbose.unitTest( "DRProject::Creating" )
        odrProject = DRProject()
        odrProject.setName( "Project-Test-Crystal-ImportExport-DNA" )
        
        for i in range( 0, 10 ):
            odrCrystal = DRCrystal()
            odrCrystal.importXmlDna( "data-crystal-dna.xml" )
            odrProject.addCrystal( odrCrystal )
            
            
        DRVerbose.unitTest( "DRProject::OutputScreen" )
        odrProject.outputScreen()
        
        DRVerbose.unitTest( "DRProject::Export" )
        odrProject.exportXmlDna( "dataRankingProject-dna.xml" )
        
        
        odrProjectIN = DRProject()
        DRVerbose.unitTest( "DRProject::Import" )
        odrProjectIN.importXmlDna( "dataRankingProject-dna.xml" )
        
        DRVerbose.unitTest( "DRProject::Export" )
        odrProjectIN.exportXmlDna( "dataRankingProject-dna-COPY.xml" )
        
        DRVerbose.unitTest( "DRProject::Export HARD Clone" )
        odrProjectIN2 = DRProject( odrProjectIN )
        odrProjectIN2.exportXmlDna( "dataRankingProject-dna-COPY-HARDCLONE.xml" )
        
        DRVerbose.unitTest( "DRProject::Export duplicate" )
        odrProjectIN3 = odrProjectIN.duplicate()
        odrProjectIN3.exportXmlDna( "dataRankingProject-dna-COPY-DUPLICATE.xml" )
        
        DRVerbose.unitTest()
        
##############################################################################
# DRDataRanking

class UnitTestDRDataRanking( DRTestCase ): 
    
    def testDRDataRanking( self ):
        DRVerbose.unitTest( "UnitTest DRDataRanking:" )
           
        odrProject = DRProject()
        odrProject.setName( "Project-Test-Crystal" )
        
        for i in range( 0, 10 ):
            odrFactorResolution = DRFactorResolution( i )
            odrFactorResolution.setName( "resolution-" + DRString( i ) )
            odrCrystal = DRCrystal()
            odrCrystal.setName( "Crystal-" + DRString( i ) )
            odrCrystal.addFactor( odrFactorResolution )
            odrProject.addCrystal( odrCrystal )
       
        odrDataRanking = DRDataRanking()
        odrDataRanking.setProject( odrProject )
        odrDataRanking.setProjectName( "Test-Setting-Name-Project-BioCrystal" )      
        odrDataRanking.setEngine( DRToken( "DREngineResolution" ) )
        odrDataRanking.execute()
        odrDataRanking.outputProjectScreen() 
        odrDataRanking.outputProject( "test-output-project.pdrp" )
       
        #odrProjectResult = odrDataRanking.getProject()
        #odrProjectResult.outputScreen()
          
        DRVerbose.unitTest()
        
        
        
              
##############################################################################
# DRManagerDataRanking

class UnitTestDRManagerDataRanking( DRTestCase ): 
    
    def testDRManagerDataRanking( self ):
        DRVerbose.unitTest( "UnitTest DRManagerDataRanking:" )
           
        odrProject = DRProject()
        odrProject.setName( "Project-Test-Crystal" )
        
        #Generate the project to input
        for i in range( 0, 10 ):
            odrFactorResolution = DRFactorResolution( DRRandom.randFloat( 0.0, 5.0 ) )
            odrFactorResolution.setName( "resolution-" + DRString( i ) )
            odrCrystal = DRCrystal()
            odrCrystal.setName( "Crystal-" + DRString( i ) )
            odrCrystal.addFactor( odrFactorResolution )
            if i!=9:
                odrProject.addCrystal( odrCrystal )
            #Input crystal to test
            if i==9:
                odrCrystal.setName( "Crystal-CommeFromOutside" )
                odrStreamSerial   = DRStreamSerial( "crystal.crys", "w" )
                odrStreamSerial.output( odrCrystal )
                odrStreamSerial.close()
       
        odrDataRanking0 = DRDataRanking()
        odrDataRanking0.setProject( odrProject )
        odrDataRanking0.setProjectName( "Test-Setting-Name-Project-BioCrystal" )      
        odrDataRanking0.outputProject( "project-input.pdrp" )
        odrDataRanking0.outputProject( "project-test.pdrp" )
       
       
        # Application Framework definition
        odrApplication = DRApplication( "DataRankingTesting", "0.9.0" )
 
        # Workaround to have an command line to test the stuffs
        odrCommandLineArgument = DRCommandLineArgument( [ "--ip",
                                                         "ERRORproject-input.pdrp",  # error simulation
     #                                                    "project-input.pdrp",
                                                         "--op",
                                                         "project-output.pdrp",
                                                         "--rcrystal",
                                                         "crystal.crys",
                                                         "--rproject",
                                                         "project-test.pdrp",
                                                         "--rengine",
                                                         "DREngineResolution",
                                                         "--verbose" ])
     
        odrApplication.addListArgument( odrCommandLineArgument ) 


        # DataRanking Kernel
        odrDataRanking = DRDataRanking()    
 
         
        # Adding PLUGINS: ie. PERSONAL Ranking Engines  #####################
        odrDataRanking.addEngine( DRPluginEnginePerso01( "EnginePerso01" ) )
        odrDataRanking.addEngine( DRPluginEnginePerso02( "EnginePerso02" ) )
        # ###################################################################
           
       
        # Adding Standart Ranking Engines
        odrDataRanking.addEngine( DREngineMosaicity( "DREngineMosaicity" ) )
        odrDataRanking.addEngine( DREngineResolution( "DREngineResolution" ) )
        odrDataRanking.addEngine( DREngineResolutionMosaicity( "DREngineResolutionMosaicity" ) )
        #.......
        #.......
        
    
    
        # Manager
        odrManagerDataRanking = DRManagerDataRanking( odrDataRanking )
        odrManagerDataRanking.m_odrSlotError.connect( odrApplication.exit )
        
        # Command Definitions
        odrApplication.addCommand( "--ip", odrManagerDataRanking.inputProject )
        odrApplication.addCommand( "--op", odrManagerDataRanking.outputProject )
        odrApplication.addCommand( "--rcrystal", odrManagerDataRanking.inputCrystalToRank )
        odrApplication.addCommand( "--rproject", odrManagerDataRanking.inputProjectToRank )
        odrApplication.addCommand( "--rengine",  odrManagerDataRanking.setRankEngine )

        
        # Execution
        odrApplication.executeConnect( odrManagerDataRanking.execute )
        odrApplication.execute()

           
        DRVerbose.unitTest()
        

    def testDRManagerDataRankingDNACompatibility( self ):
        DRVerbose.unitTest( "UnitTest DRManagerDataRanking:" )
                
        # Generates the data-crystal to analyse
        odrCrystal = DRCrystal()
        odrCrystal.importXmlDna( "data-crystal-dna-v2.xml" )
    #    odrCrystal.importXmlDna( "data-crystal-dna-v3.xml" )
        odrCrystal.setResolutionObtained( DRRandom.randFloat( 1.0, 12.0 ) )
        odrCrystal.setMosaicity( DRRandom.randFloat( 0.1, 10.0 ) )
        odrCrystal.setRotation( DRRandom.randFloat( 0.5, 2.0 ) )
        odrCrystal.setExposureTime( DRRandom.randFloat( 0.5, 2.0 ) )
        odrCrystal.setSpotsUsed( DRRandom.randInt( 500, 10000 ) )
        odrCrystal.setSpotsFound( DRRandom.randInt( 500, 10000 ) )
        odrCrystal.setSpotsRejected( DRRandom.randInt( 50, 200 ) )
        odrCrystal.setSpotDeviationR( DRRandom.randFloat( 0.1, 1.0 ) )
        odrCrystal.exportXmlDna( "data-crystal-dna-COPY.xml" )
       # odrCrystal.exportXmlDna( "data-crystal-dna-COPY-v3.xml" )
        odrCrystal.outputScreen()
        
    #    return 
       

        # DNA BioXHIT Compatibility Layer
        DRVerbose.unitTest( "DRProject::Creating" )
        odrProject = DRProject()
        odrProject.setName( "Project-Test-Crystal-ImportExport-DNA" )     
        for i in range( 0, 10 ):
            odrCrystal = DRCrystal()
            odrCrystal.importXmlDna( "data-crystal-dna-COPY.xml" )
            odrCrystal.setResolutionObtained( DRRandom.randFloat( 1.0, 12.0 ) )
            odrCrystal.setMosaicity( DRRandom.randFloat( 0.1, 10.0 ) )
            odrCrystal.setRotation( DRRandom.randFloat( 0.5, 2.0 ) )
            odrCrystal.setExposureTime( DRRandom.randFloat( 0.5, 2.0 ) )
            odrCrystal.setSpotsUsed( DRRandom.randInt( 500, 10000 ) )
            odrCrystal.setSpotsFound( DRRandom.randInt( 500, 10000 ) )
            odrCrystal.setSpotsRejected( DRRandom.randInt( 50, 200 ) )
            odrCrystal.setSpotDeviationR( DRRandom.randFloat( 0.1, 1.0 ) )
            odrProject.addCrystal( odrCrystal )        
        DRVerbose.unitTest( "DRProject::Export" )
        odrProject.exportXmlDna( "project-dna.xml" )
        odrProject.exportXmlDna( "project-test.xml" )
 
       
       
        # Application Framework definition
        odrApplication = DRApplication( "DataRankingTesting", "0.9.0" )
 
  
  
        # Workaround to have an command line to test the stuffs related to DNA BioXHIT stuff
        odrCommandLineArgument = DRCommandLineArgument( [ #"--ipDNA", "ERRORproject-dna.xml",  # error simulation
                    #  "--ipDNA", "project-dna.xml",
                    "--opDNA", "project-dna-output.xml",
                    "--rcrystalDNA", "data-crystal-dna-COPY.xml",
                    #"--rcrystalDNA", "data-crystal-dna-error.xml", # simulate an error
                    "--rprojectDNA", "project-test.xml",
                    "--rengine", "DREngineResolutionObtained",
                   # "--rengine", "DREngineDataCollectionTime",
                    #"--rengine", "DREngineDataCollectionNumberImage",
                    #"--rengine", "DREngineSpotsUsed",
                    #"--rengine", "DREngineSpotsFound",
                    #"--rengine", "DREngineSpotsRejected",
                    #"--rengine", "DREngineSpotDeviationR",
                    #"--rengine", "DREngineMosaicity",
                    #"--rengine", "DREngineResolutionMosaicity",
                    #"--rengine", "EnginePerso01", # Simulate an Error
                    "--verbose" ])
                    #"--verbose", "--DEBUG" ])
                    # "--quiet" ])
              
              
# command line to test the program features and errors
#--opDNA project-dna-output.xml --rcrystalDNA  data-crystal-dna.xml --rprojectDNA  project-test.xml --rengine  DREngineResolutionMosaicity --DEBUG

        odrApplication.addListArgument( odrCommandLineArgument ) 


        # DataRanking Kernel
        odrDataRanking = DRDataRanking()    
    
        # Adding PLUGINS: ie. PERSONAL Ranking Engines  #####################
        odrDataRanking.addEngine( DRPluginEnginePerso01( "EnginePerso01" ) )
        odrDataRanking.addEngine( DRPluginEnginePerso02( "EnginePerso02" ) )
        # ###################################################################
           
       
        # Adding Standart Ranking Engines
        odrDataRanking.addEngine( DREngineMosaicity( "DREngineMosaicity" ) )
        odrDataRanking.addEngine( DREngineResolutionObtained( "DREngineResolutionObtained" ) )
        odrDataRanking.addEngine( DREngineResolutionMosaicity( "DREngineResolutionMosaicity" ) )
        odrDataRanking.addEngine( DREngineSpotsUsed( "DREngineSpotsUsed" ) )
        odrDataRanking.addEngine( DREngineSpotsFound( "DREngineSpotsFound" ) )
        odrDataRanking.addEngine( DREngineSpotsRejected( "DREngineSpotsRejected" ) )
        odrDataRanking.addEngine( DREngineSpotDeviationR( "DREngineSpotDeviationR" ) )
        odrDataRanking.addEngine( DREngineDataCollectionTime( "DREngineDataCollectionTime" ) )
        odrDataRanking.addEngine( DREngineDataCollectionNumberImage( "DREngineDataCollectionNumberImage" ) )
        #.......
        #.......
        
    
        # Manager
        odrManagerDataRanking = DRManagerDataRanking( odrDataRanking )
 
 
       # Command Definitions Compatibility with DNA BioXHIT Project
        odrApplication.addCommand( "--ipDNA", odrManagerDataRanking.inputProject )
        odrApplication.addCommand( "--opDNA", odrManagerDataRanking.outputProject )
        odrApplication.addCommand( "--rcrystalDNA", odrManagerDataRanking.inputCrystalToRank )
        odrApplication.addCommand( "--rprojectDNA", odrManagerDataRanking.inputProjectToRank )
        odrApplication.addCommand( "--rengine",  odrManagerDataRanking.setRankEngine )
    
    
        # Execution
        odrApplication.executeConnect( odrManagerDataRanking.execute )
        odrApplication.execute()

           
        DRVerbose.unitTest()
        
##############################################################################
       
class DRUnitTestKernel( DRUnitTest ):
    def preProcessor( self ):
        #self.addTestMethod( UnitTestDRCrystal().testDRCrystal )
        #self.addTestMethod( UnitTestDRCrystal().testDRCrystalDNA )
       # self.addTestMethod( UnitTestDRCrystal().testDRPythonStuffs )
        #self.addTestMethod( UnitTestDRProject().testDRProject )
        #self.addTestMethod( UnitTestDRProject().testDRProjectDNA )
        #self.addTestMethod( UnitTestDRDataRanking().testDRDataRanking )
        #self.addTestMethod( UnitTestDRManagerDataRanking().testDRManagerDataRanking )
        self.addTestMethod( UnitTestDRManagerDataRanking().testDRManagerDataRankingDNACompatibility )
       
    
##############################################################################

if __name__ == '__main__':

    odrUnitTestKernel = DRUnitTestKernel( "Test DataRanking Kernel" )
    odrUnitTestKernel.execute()


     
     
     
     
     



       
    # Commandline DEMO:
    # drank.exe  -ip  project-input.pdrp  -op  project-output.pdrp
    #            -rcrystal  crystal.crys  -rproject project.pdrp
    #            -rengine  standart  -verbose
    

    
    
    # -ip  project-input.pdrp  -op  project-output.pdrp -rcrystal  crystal.crys  -rproject project.pdrp -rengine  standart  -verbose