##############################################################################
#
#   File:      dr_.py
#
#   Project:   DataRanking Project
#
#   Author(s): Romeu A. Pieritz [pieritz@esrf.fr]
#              Olof Svensson [svensson@esrf.fr]
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Data Ranking Project Project
#   Copyright (c) 2005, Data Ranking Project developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of DataRanking's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from dr_system             import  *
from dr_kernel             import  *
from dr_data_ranking       import  DRDataRanking
from dr_crystal            import  DRCrystal
from dr_project            import  DRProject



class DRManagerDataRanking( DRManager ):
    
    def __init__( self, _odrDataRanking ):
        DRManager.__init__( self )
        self.m_odrDataRanking               = _odrDataRanking
        self.m_odrInputProject              = self.m_odrDataRanking.getProject()
        self.m_odrStringOutputProjectName   = None
        self.m_odrTokenEngine               = None
        self.m_odrCrystalToRank             = None
        self.m_odrProjectToRank             = None
    
        self.m_odrDataRanking.m_oalSlotEnd.connect( self.outputData )
          
         
    def inputCrystalToRank( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.inputCrystalToRank()" )
        DRVerbose.screen( "DataRankingManager::Reading Input Crystal to Ranking")
        self.m_odrCrystalToRank = self.inputObject( _odrObject )
        
   
    def inputProjectToRank( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.inputProjectToRank()" )      
        DRVerbose.screen( "DataRankingManager::Reading Input Project to Ranking")
        self.m_odrProjectToRank = self.inputObject( _odrObject )
            
         
    def inputProject( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.inputProject()" )      
        DRVerbose.screen( "DataRankingManager::Reading Input Project")
        self.m_odrInputProject = self.inputObject( _odrObject )
                 
           
    def setRankEngine( self, _odrObject = None ):     
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.setRankEngine()" )      
        odrToken = DRToken( _odrObject )
        odrCmdLineArg = DRApplication.m_oalCommandLineArgument           
        if odrCmdLineArg.existArgument( odrToken ):
            self.m_odrTokenEngine = DRToken( odrCmdLineArg.getArgument( odrToken ) )
            DRVerbose.screen( "DataRankingManager::Setting Rank Engine = " + self.m_odrTokenEngine )
                 
       
    def outputProject( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.outputProject()" )      
        DRVerbose.screen( "DataRankingManager::Setting Output Project Name")
        odrToken = DRToken( _odrObject )
        odrCmdLineArg = DRApplication.m_oalCommandLineArgument           
        if odrCmdLineArg.existArgument( odrToken ):
            self.m_odrStringOutputProjectName = DRString( odrCmdLineArg.getArgument( odrToken ) )
                       
             
    def inputObject( self, _odrToken ):      
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.inputObject()" )      
        odrObject = None
        odrToken = DRToken( _odrToken )
        odrCmdLineArg = DRApplication.m_oalCommandLineArgument           
        if odrCmdLineArg.existArgument( odrToken ):
            odrStreamName   = DRString( odrCmdLineArg.getArgument( odrToken ) )
            if ( DRDiskExplorer.existFile( odrStreamName ) == False ):
                DRApplicationCommunicator.setSignal( DRSignal.error( odrStreamName + "::File do not exist" ) )
            elif ( odrToken==DRToken("--rcrystalDNA" ) ):
                odrObject = DRCrystal()
                odrObject.importXmlDna( odrStreamName )
                if (odrObject.testIntegrity()==False):
                    DRApplicationCommunicator.setSignal( DRSignal.error( odrStreamName + "::Crystal File Corrupted" ) )
                    odrObject = None
            elif ( odrToken==DRToken("--rprojectDNA" ) or odrToken==DRToken("--ipDNA" ) ):
                odrObject = DRProject()
                odrObject.importXmlDna( odrStreamName )
                if (odrObject.testIntegrity()==False):
                    DRApplicationCommunicator.setSignal( DRSignal.error( odrStreamName + "::Project File Corrupted" ) )
                    odrObject = None
            # To be deprecated
            else :  
                odrStreamSerial = DRStreamSerial( odrStreamName, "r" )
                odrObject = odrStreamSerial.input()
                odrStreamSerial.close()
        return odrObject         
       
       
    def outputData( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.outputData()" )      
        # DNA compatibility
        if DRApplication.m_oalCommandLineArgument.existArgument( DRToken( "--opDNA" ) ):
            self.m_odrDataRanking.exportProjectDna( self.m_odrStringOutputProjectName )  
        else:
            self.m_odrDataRanking.outputProject( self.m_odrStringOutputProjectName )
       
       
    def execute( self, _odrObject = None ):
        DRVerbose.DEBUG( "Execution: DRManagerDataRanking.execute()" )      
        DRVerbose.screen( "DataRankingManager::Execute" )
        # Setting
        self.m_odrInputProject.addCrystal( self.m_odrCrystalToRank )
        self.m_odrInputProject.addProject( self.m_odrProjectToRank )
        
        # Setting DataRanking
        self.m_odrDataRanking.setProject(  self.m_odrInputProject  )
        self.m_odrDataRanking.setEngine(   self.m_odrTokenEngine   )
        
        # Execution
        self.m_odrDataRanking.executeAction()
        self.m_odrDataRanking.waitActionEnd()
                   

##############################################################################

