##############################################################################
#
#   File:      dr_.py
#
#   Project:   DataRanking Project
#
#   Author(s): Romeu A. Pieritz [pieritz@esrf.fr]
#              Olof Svensson [svensson@esrf.fr]
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Data Ranking Project Project
#   Copyright (c) 2005, Data Ranking Project developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of DataRanking's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from dr_system                    import *
from dr_kernel                    import *
from dr_engine_class              import DREngine
from dr_project                   import DRProject


class DRDataRanking( DRAction ):
    
    def __init__( self ):
        DRAction.__init__( self )
        self.m_odrEngine        = None
        self.m_odrProject       = DRProject()
        self.m_odrFactoryEngine = DRFactoryPlugin()
        

    def addEngine( self, _odrObject ):
        self.synchronizeOn()
        if ( _odrObject != None ):
            self.m_odrFactoryEngine.addPlugin( _odrObject )
        self.synchronizeOff()
        
    
    def getEngine( self, _odrString ):
        self.synchronizeOn()
        odrEngine = self.m_odrFactoryEngine.getPlugin( _odrString )
        if odrEngine == None :
            odrEngine = DREngine()
        self.synchronizeOff()
        return odrEngine


    def setProject( self, _odrProject ):
        self.synchronizeOn()
        if ( _odrProject != None ):
            self.m_odrProject = DRProject( _odrProject )
        if self.m_odrEngine != None:
            self.m_odrProject.setEngineType( self.m_odrEngine.getName() )
        self.synchronizeOff()
        
        
    def getProject( self ):
        return self.m_odrProject.duplicate()
        

    def setProjectName( self, _odrString ):
        self.synchronizeOn()
        self.m_odrProject.setName( DRString( _odrString ) )
        self.synchronizeOff()
      
      
    def setEngine( self, _odrToken ):
        self.synchronizeOn()
        odrToken = DRToken( _odrToken )
        self.m_odrEngine = self.engineFactory( odrToken )
        self.m_odrProject.setEngineType( odrToken )
        self.synchronizeOff()
        
  
    def outputProjectScreen( self ):
        self.synchronizeOn()
        self.m_odrProject.outputScreen()
        self.synchronizeOff()
        
  
    # Already Synchronized by the stream
    def outputProject( self, _odrString ):
        odrStreamName = DRString( _odrString  )
        odrStreamSerial = DRStreamSerial( odrStreamName, "w" )
        odrStreamSerial.output( self.m_odrProject )
        odrStreamSerial.close()
       
          
    def engineFactory( self, _odrToken ):
        return self.getEngine( _odrToken )

          
           
    # Already Synchronized by the stream
    def exportProjectDna( self, _odrString ):     
        self.m_odrProject.exportXmlDna( _odrString )
           
    
    def rank( self ):
        return self.m_odrEngine.rank( self.getProject() )
        
        
    # Method called by Execute
    def process( self, _oalObject = None ):
        self.synchronizeOn()
        self.setProject( self.rank() )  
        self.synchronizeOff()
  
  
##############################################################################
