##############################################################################
#
#   File:      al_xml.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from al_python                  import *
from al_object                  import ALObject
from al_verbose                 import ALVerbose
from al_token                   import ALToken
from al_string                  import ALString
from al_slot                    import ALSlot
from al_stream_file             import ALStreamFile
from al_synchronized            import ALSynchronized
from al_xml_node                import ALXmlNode
from al_threading_condition     import ALThreadingCondition
from al_list                    import ALList
from al_trace                   import ALTrace

     
class ALXml( ALObject, ALSynchronized, PyXmlDocument ):
    """
    Interface for ALXml objects.
    ATTENTION:: must be transformed to thread safety
    """
    
    m_oalThreadingConditionXml = ALThreadingCondition()
    
    
    def __init__( self, _oalString = None ):
        ALObject.__init__( self )
        PyXmlDocument.__init__( self )
        ALSynchronized.__init__( self )
        if _oalString == None :
            _oalString = "document" # self.m_oalStringName
        opyElement = self.createElementNS( None, _oalString )
        self.appendChild( opyElement )
        self.doctype = None
        self.implementation = PyDOMImplementationLS()


    def createXml( _oalStrFileName ):
        ALXml.m_oalThreadingConditionXml.synchronizeOn()
        oXml = None
        # used to bypass the error inside C routine from Python
        try:
            oXml = PyXmlParser( _oalStrFileName )
        except:
            ALVerbose.error( "ALXml::createXml() = parser Failed" )
            ALVerbose.error( "File Parser Error :: " + ALString( _oalStrFileName ) )
            ALTrace.trace()
        ALXml.m_oalThreadingConditionXml.synchronizeOff()
        return oXml
    createXml = staticmethod( createXml )
    
    
        
    def addComment( self, _oalString ):
        self.synchronizeOn()
        oalXmlNodeString = PyXml.Comment( ALString( _oalString ) )     
        self.documentElement.appendChild( oalXmlNodeString )     
        self.synchronizeOff()
        
         
         
    def close( self ):
        self.synchronizeOn()
        self.unlink()
        self.synchronizeOff()
    
    
    def destroy( self ):
        self.close()
        
    
    
  
    def addTokenValue( self, _oalToken, _oalString ):
        self.synchronizeOn()
        if ( _oalString!=None ):
            oalXmlNodeString = self.createTextNode( ALString( _oalString ) )
            oalXmlNode = self.createElement( ALToken( _oalToken ) )
            oalXmlNode.appendChild( oalXmlNodeString )
            self.documentElement.appendChild( oalXmlNode )
        self.synchronizeOff()
    
   
    
    # Only in the main branch of the tree
    # WARNING: returns every time the first object!!!
    def getTokenValue( self, _oalToken ):
        return ALXml.getElementTokenValue( self.documentElement, _oalToken )
    
  
  
    
    # Looking for ONE object XmlElements in the main branch of the tree
    # WARNING: returns every time the first object!!!
    def getElement( self, _oalToken ):
        self.synchronizeOn()
        oalXmlElement = None       
        # Trick to bypass bad Xml documents
        if ALToken( self.documentElement.tagName ) == ALToken( _oalToken ):
            oalXmlElement = self.documentElement
        else :
            self.synchronizeOff()
            return ALXml.getElementObject( self.documentElement, _oalToken )
        self.synchronizeOff()
        return oalXmlElement
    
   
   
    def removeElement( self, _oalToken ):
        self.synchronizeOn()
        oalXmlElement = self.getElement( _oalToken )
        if ( oalXmlElement != None ):
            self.documentElement.removeChild( oalXmlElement )
        self.synchronizeOff()
        
        
    
    # Looking for a list of XmlElements in the main branch of the tree
    def getElementList( self, _oalToken ):
        self.synchronizeOn()
        oalListXmlElement = None   
        # Trick to bypass bad Xml documents
        if ALToken( self.documentElement.tagName ) == ALToken( _oalToken ):
            oalListXmlElement = ALList()
            oalListXmlElement.add( self.documentElement )
        else :
            self.synchronizeOff()
            return ALXml.getElementListObject( self.documentElement, _oalToken )           
        self.synchronizeOff()
        return oalListXmlElement
    
    
    
    
    # Looking for VALUE of XmlElement from any start node (sub-points)
    # WARNING: returns every time the first object!!!
    def getElementTokenValue( _oalXmlElement, _oalToken ):
        ALXml.m_oalThreadingConditionXml.synchronizeOn()
        oalStringValue = None
        for inode in _oalXmlElement.childNodes:
            if inode.nodeType==ALXmlNode.ELEMENT_NODE:
                if ( ALToken( inode.nodeName ) == ALToken( _oalToken ) ):
                    #if ( inode.nodeType==ALXmlNode.TEXT_NODE ):
                    #    oalStringValue = ALString( inode.data )
                    #    # cleaning the xml errors from the python code!!!!!!
                    #    oalStringValue = oalStringValue.cleanXml()
                    #    # BAD....but runs
                    #    ALXml.m_oalThreadingConditionXml.synchronizeOff()
                    #    return oalStringValue
                    #else:
                    #    for jnode in inode.childNodes:
                    #        if ( jnode.nodeType==ALXmlNode.TEXT_NODE ):
                    #            oalStringValue = ALString( jnode.data )
                    #            # cleaning the xml errors from the python code!!!!!!
                    #            oalStringValue = oalStringValue.cleanXml()
                    #            # BAD....but runs
                    #            ALXml.m_oalThreadingConditionXml.synchronizeOff()
                    #            return oalStringValue
                    
                    for jnode in inode.childNodes:
                        if ( jnode.nodeType==ALXmlNode.TEXT_NODE ):
                            oalStringValue = ALString( jnode.data )
                            # cleaning the xml errors from the python code!!!!!!
                            oalStringValue = oalStringValue.cleanXml()
                            # BAD....but runs
                            ALXml.m_oalThreadingConditionXml.synchronizeOff()
                            return oalStringValue
        ALXml.m_oalThreadingConditionXml.synchronizeOff()
        return oalStringValue
    getElementTokenValue = staticmethod( getElementTokenValue )
    
    
    # Looking for a list of XmlElements from any start node (sub-points)      
    def getElementListObject( _oalXmlElement, _oalToken ):
        ALXml.m_oalThreadingConditionXml.synchronizeOn()
        oalListXmlElement = None
        for inode in _oalXmlElement.childNodes:
            if inode.nodeType==ALXmlNode.ELEMENT_NODE:
                if ( ALToken( inode.nodeName ) == ALToken( _oalToken ) ):
                    if ( oalListXmlElement==None ):
                        oalListXmlElement = ALList()
                    oalListXmlElement.add( inode )
        ALXml.m_oalThreadingConditionXml.synchronizeOff()
        return oalListXmlElement
    getElementListObject = staticmethod( getElementListObject )
    
    
    
    # Looking for ONE XmlElement from any start node (sub-points)
    # WARNING: returns every time the first object!!!
    def getElementObject( _oalXmlElement, _oalToken ):
        ALXml.m_oalThreadingConditionXml.synchronizeOn()
        oalXmlElement = None
        for inode in _oalXmlElement.getElementsByTagName( ALToken( _oalToken ) ):
            oalXmlElement = inode
            # Bad....
            ALXml.m_oalThreadingConditionXml.synchronizeOff()
            return oalXmlElement
        ALXml.m_oalThreadingConditionXml.synchronizeOff()
        return oalXmlElement
    getElementObject = staticmethod( getElementObject )
    
    
    
    #Used to fix the "m_"  of AALib class atributte squeme
    def getAtributeToken( _oalToken ):
        oalToken = ALToken( _oalToken )
        if oalToken[0]=="m" and oalToken[1]=="_" :
            #oalToken.replace( "m_", "" ) # It can generate a problem in complex names
            oalTokenNew = oalToken[2:]
            oalToken = oalTokenNew
        return oalToken
    getAtributeToken = staticmethod( getAtributeToken )
    
    
    
    def addObject( self, _oalObject ):
        self.synchronizeOn()
        _oalObject.outputXml( self, self.documentElement )
        self.synchronizeOff()
    
    
    
    def rootNode( self ):
        return self.documentElement
    
    
    
    def output( self, _oalStringName, _bPrintType = False, _oalStringEncodingType = "ISO-8859-1" ): # or using "utf-8"  
        self.synchronizeOn()
        oalStreamFile = ALStreamFile( _oalStringName, "w" )
        if ( _bPrintType == False ) :
            oalStreamFile.write( self.toxml( _oalStringEncodingType ) )
        else:
            oalStreamFile.write( self.toprettyxml( "\t", "\n", _oalStringEncodingType ) ) # it creates a bug in komodo kpf file - tests
        oalStreamFile.close()
        self.synchronizeOn()
        
        
        
    def input( self, _oalStrFileName ):
        self.synchronizeOn()
        oXml = ALXml.createXml( _oalStrFileName )         
        if ( oXml!=None ) :
            # Hard copy of base node to bypass Python BUG!!!!!
            baseNode = oXml.documentElement                                
            self.documentElement.tagName  = baseNode.tagName
            self.documentElement.nodeName = baseNode.nodeName
            self.documentElement.prefix   = baseNode.prefix
            self.documentElement.namespaceURI = baseNode.namespaceURI
            if baseNode.hasAttributes():
                for ikey in baseNode._attrs:
                     self.documentElement.setAttribute( ikey, baseNode.getAttribute( ikey ) ) 
                                                   
            # copy of all nodes                
            for inode in oXml.documentElement.childNodes:
                if inode.nodeType==ALXmlNode.COMMENT_NODE:
                    oalComment = PyXml.Comment( ALString( inode.data ) )     
                    self.documentElement.appendChild( oalComment )     
                else:
                    self.documentElement.appendChild( self.importNode( inode, True ) )
                
            # DOM garbage collector
            oXml.unlink()  
        self.synchronizeOff()
        
        

##############################################################################