##############################################################################
#
#   File:      al_thread_control.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################



from al_definition                  import *
from al_string                      import ALString
from al_verbose                     import ALVerbose
from al_slot                        import ALSlot
from al_signal                      import ALSignal
from al_thread                      import ALThread
from al_threading_condition         import ALThreadingCondition
from al_synchronized                import ALSynchronized
from al_trace                       import ALTrace
from al_application_communicator    import ALApplicationCommunicator


class ALThreadControl( ALThread ):
    """
    Interface for AL thread control objects.
    """
    def __init__( self ):
        ALThread.__init__( self )
        self.m_oalSlotInit      = ALSlot()
        self.m_oalSlotRun       = ALSlot()
        self.m_oalSlotEnd       = ALSlot()
        self.m_oalSlotPause     = ALSlot()
        self.m_oalSlotResume    = ALSlot()
        self.m_oalSlotAbort     = ALSlot()
        self.m_oalSlotException = ALSlot()
        self.m_oalSlotError     = ALSlot() # self.m_oalSlotAbort # to be improved
 
 
        # Basic Event Callback Connections
        self.m_oalSlotEnd.connect( self.endOfAction )       
        self.m_oalSlotAbort.connect( self.verboseAbort, ALSignal.abort() )
        self.m_oalSlotException.connect( self.verboseException, ALSignal.exception() )
        
        
        self.m_oalThreadingCondition  = ALThreadingCondition()
        self.m_bIsEnded = False
        self.m_bIsAbort = False
        self.m_bIsPause = False
   
        
    
    def synchronizeOn( self ):
        self.m_oalThreadingCondition.synchronizeOn()


    def synchronizeOff( self ):
        self.m_oalThreadingCondition.synchronizeOff()


    def synchronizeWait( self, _fTimeOut = None ):
        self.m_oalThreadingCondition.synchronizeWait( _fTimeOut )


    def synchronizeNotify( self ):
        self.m_oalThreadingCondition.synchronizeNotify()


        
    def pause( self, _oalObject = None ):
        self.synchronizeOn()
        self.m_oalSlotPause.call()
        self.m_bIsPause = True
        self.synchronizeOff()
        
        
    def resume( self, _oalObject = None ):
        self.m_oalSlotResume.call()
        self.synchronizeNotify()
        self.m_bIsPause = False
       
        
    def abort( self, _oalObject = None ):
        self.synchronizeOn()
        self.m_bIsAbort = True
        self.synchronizeOff()
        
        
    def isAbort( self ):
        return self.m_bIsAbort
        
        
    def isPause( self ):
        return self.m_bIsPause
    
    
    def isEnded( self ):
        return self.m_bIsEnded
        
        
    def threadControl( self ):
        if self.isAbort():
            raise AL_ACTION_SIGNAL_ABORT
        if self.isPause():
            oalString = ALString( "Thread PAUSED under control: " + self.getName() + " " + self.getID() )  
            ALVerbose.screen( oalString )
            self.synchronizeWait()

        # Experimental           
        oalApplicationSignal = ALApplicationCommunicator.getSignal()
        if oalApplicationSignal != None:
            oalSignal = oalApplicationSignal.getSignal()
            oalString = oalApplicationSignal.getInformation()  
            if oalSignal.equals( AL_ACTION_SIGNAL_ERROR ) or oalSignal.equals( AL_ACTION_SIGNAL_ABORT ) :
                ALVerbose.error( oalSignal + "::" + oalString )
                raise AL_ACTION_SIGNAL_ABORT
        

    
    def verboseAbort( self, _oalObject = None ):
        self.synchronizeOn()
        self.m_bIsAbort = True
        oalString = ALString( "Action::ABORTED under control: " + self.getName() + " " + self.getID() )  
        ALVerbose.screen( oalString )
        self.synchronizeOff()
        self.synchronizeNotify()   
    
    
    def verboseException( self, _oalObject = None ):
        self.synchronizeOn()
        self.m_bIsAbort = True # spike to bypass wait in the main thread
        ALApplicationCommunicator.setSignal( ALSignal( AL_ACTION_SIGNAL_ERROR, ALString( "Thread Exception" ) ) )
        ALTrace.trace( self )
        self.synchronizeOff()
        self.synchronizeNotify()
        
 

    # experimental to block the execution....TESTED but it must be improved
    def waitActionEnd( self, _floatTimeout = 1000.0 ):
        self.join( _floatTimeout )
    
   
    def endOfAction( self, _oalObject = None ):
        self.m_bIsEnded = True
        self.synchronizeNotify()


    #
    # # Warning: Serious crash in Python interpreter when an Action found a Exception
    ##          after this function is called at the first time: Do NOT use it!!!
    #def waitEndOfAction( self, _oalObject = None ):
    #    if self.m_bIsAbort!=True :
    #        if self.m_bIsEnded!=True :
    #            self.synchronizeWait()
    #            
                
##############################################################################

