##############################################################################
#
#   File:      al_ressource.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from al_sys                   import  ALSystem
from al_object                import  ALObject
from al_string                import  ALString
from al_token                 import  ALToken
from al_dict                  import  ALDict
from al_threading_condition   import  ALThreadingCondition
from al_xml                   import  ALXml
from al_xml_node              import  ALXmlNode
from al_stream_file           import  ALStreamFile
from al_stream_serial         import  ALStreamSerial
from al_stream_xml            import  ALStreamXml
from al_disk_explorer         import  ALDiskExplorer
from al_file                  import  ALFile
from al_verbose               import  ALVerbose



class ALResource( ALObject ):
    """
    Parent class for all AL
    ressource objects.
    """
    
    m_oalThreadingConditionResource    = ALThreadingCondition()
    
        
    def __init__( self, _oalString = "resource.res", _bUserPath = True, _bRead = True ):
        ALObject.__init__( self )
        self.m_oalDict = ALDict()
        self.m_oResourceFilePath = ALDiskExplorer.getCurrentFolder()
        self.m_oResourceFileName = ALString( self.m_oResourceFilePath + ALString( "/" ) + ALString( _oalString ) )      
        if _bUserPath == True:
            self.m_oResourceFilePath = ALSystem.getUserPath() + ALString( "/." ) + ALFile.getBaseName( _oalString )
            self.m_oResourceFileName = ALString( self.m_oResourceFilePath + ALString( "/" ) + ALString( _oalString ) )
        if _bRead == True:
            self.input()
        



    def getResource( self, _oalToken ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()
        oalObject = self.m_oalDict.getValue( ALToken( _oalToken ) )
        ALResource.m_oalThreadingConditionResource.synchronizeOff()  
        return oalObject
    
    
    def setResource( self, _oalToken, _oalTokenObject ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()
        self.m_oalDict.add( ALToken( _oalToken ), _oalTokenObject )
        ALResource.m_oalThreadingConditionResource.synchronizeOff()  

 
    def reset( self ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()
        self.m_oalDict = ALDict()
        ALResource.m_oalThreadingConditionResource.synchronizeOff()
        self.output()





    def output( self ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()       
        ALDiskExplorer.createFolder( self.m_oResourceFilePath )
        
        # Old version
        #oalXml = ALXml( self.getMagicNumber() )
        #oalXml.addComment( "AALib Resource File::DO NOT EDIT - TO BE DEPRECATED" )
        #self.outputXml( oalXml )
        #oalXml.output( self.m_oResourceFileName )
        #oalXml.close()
        
        oalXml = ALXml()
        oalXml.addComment( "AALib Resource File::DO NOT EDIT" )
        oalXml.addObject( self )
        oalXml.output( self.m_oResourceFileName, True )
        oalXml.close()
 
 
        #self.outputSerial() # TO BE DEPRECATED
        
        ALVerbose.log( "Resource::Saved   = " + self.m_oResourceFileName  )
        ALResource.m_oalThreadingConditionResource.synchronizeOff()
    
    def input( self ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()    
        if ALDiskExplorer.existFile( self.m_oResourceFileName ) :
            # Old to be deleted
            #oalXml = ALXml()
            #oalXml.input( self.m_oResourceFileName )         
            #self.inputXml( oalXml )      
            #oalXml.close()
            #
            
            oalXml = ALXml()
            oalXml.input( self.m_oResourceFileName )         
            oalXmlElement = oalXml.getElement( self.getMagicNumber() )
            if (oalXmlElement!=None):
                self.inputXml( oalXmlElement )    
            oalXml.close()
            
            #self.inputSerial() # TO BE DEPRECATED
            
            ALVerbose.log( "Resource::Loaded  = " + self.m_oResourceFileName )
        else :
            ALVerbose.log( "Resource::CREATED = " + self.m_oResourceFileName )    
        ALResource.m_oalThreadingConditionResource.synchronizeOff()
        
        
        
        
             
    ###########################################################################
    # XML Output - To be modified in the next version with correct xml model
    # Must add a node constructed lovcaly to the main node of the xml
    
    def outputXml( self, _oalXml, _oalXmlElement ):
        oalXmlElementObject = self.outputXmlObject( _oalXml, _oalXmlElement )       
        for iToken in self.m_oalDict: 
            oalXmlElement = _oalXml.createElement( ALToken( iToken ) )
            oalXmlElement.appendChild( _oalXml.createTextNode( ALToken( self.getResource( iToken ) ) ) )      
            oalXmlElementObject.appendChild( oalXmlElement )
    
        
      
    def inputXml( self, _oalXmlElement  ):
        self.inputXmlObject( _oalXmlElement )
        for inode in _oalXmlElement.childNodes:
            if inode.nodeType==ALXmlNode.ELEMENT_NODE:
                for jnode in inode.childNodes:
                    if jnode.nodeType==ALXmlNode.TEXT_NODE:
                        oalToken  = ALToken( inode.nodeName )
                        oalString = ALString( jnode.data )
                        if ( oalToken!=ALToken("objectID") and oalToken!=ALToken("objectVersion") and oalToken!=ALToken( "objectName" ) ): 
                            self.setResource( oalToken.cleanXml(), oalString.cleanXml() )  
       
      
    #  
    #         
    ############################################################################
    ## XML Output - To be modified in the next version with correct xml model
    ## Must add a node constructed lovcaly to the main node of the xml
    #
    #def outputXml( self, _oalXml ):
    # #   ALObject.outputXml( _oalXml, _oalXmlNode ) 
    #    for iToken in self.m_oalDict: 
    #        _oalXml.addTokenValue( ALToken( iToken ), self.getResource( iToken ) )     # to change  
    #    
    #  
    #  
    #def inputXml( self, _oalXml ):
    #    #ALObject.inputXml( _oalXml )
    #    #
    #    #oalXmlNode = _oalXml.getNodeObject( self.getMagicNumber() )
    #    #for inode in oalXmlNode.childNodes:
    #    #    if inode.nodeType==ALXmlNode.ELEMENT_NODE:
    #    #        for jnode in inode.childNodes:
    #    #            if jnode.nodeType==ALXmlNode.TEXT_NODE:
    #    #                self.setResource( ALToken( inode.nodeName ), ALString( jnode.data ) )  # to change
    #    
    #    for inode in _oalXml.documentElement.childNodes:
    #        #if inode.nodeName == self.getMagicNumber(): 
    #        if inode.nodeType==ALXmlNode.ELEMENT_NODE:
    #            for jnode in inode.childNodes:
    #                if jnode.nodeType==ALXmlNode.TEXT_NODE:
    #                    self.setResource( ALToken( inode.nodeName ), ALString( jnode.data ) )  # to change
    #
    #    
        
        
    ###########################################################################
    # Serial Output - TO BE DEPRECATED
    def outputSerial( self ):
        oalStreamSerial = ALStreamSerial( self.m_oResourceFileName, "wb" )
        oalStreamSerial.output( self.m_oalDict )
        oalStreamSerial.close()
        
    

    def inputSerial( self ):
        if ALDiskExplorer.existFile( self.m_oResourceFileName ) :
            oalStreamSerial = ALStreamSerial( self.m_oResourceFileName, "rb" )
            self.m_oalDict = oalStreamSerial.input()
            oalStreamSerial.close()
            ALVerbose.log( "Resource::Loaded  = " + self.m_oResourceFileName )
        else :
            ALVerbose.log( "Resource::CREATED = " + self.m_oResourceFileName )
        
     
    # Only to import old resouce files - STRINGS - To Be Deprecated ==========
    def outputSerialASCII( self, _oalObject = None ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn()
        oalStreamFile = ALStreamFile( "ascii-resource.res", "w" )
        for ikey in self.m_oalDict:
            oalStreamFile.write( ALString( ikey ) + ' ' + ALString( self.m_oalDict[ikey] ) + '\n')    
        oalStreamFile.close()
        ALVerbose.log( "Resource::Saved   = " + "ascii-resource.res"  )
        ALResource.m_oalThreadingConditionResource.synchronizeOff()
    

    def inputSerialASCII( self, _oalObject = None ):
        ALResource.m_oalThreadingConditionResource.synchronizeOn() 
        if ALDiskExplorer.existFile( "ascii-resource.res" ) :
            oalStreamFile = ALStreamFile( "ascii-resource.res", "r" )
            #self.m_oalDict = oalStreamFile.read()                  
            for iline in oalStreamFile.readLines():
                key, token  = ALString.split( iline )
                self.setResource( ALString( key ), ALString( token ) )
            oalStreamFile.close()
            ALVerbose.log( "Resource::Loaded  = " + "ascii-resource.res" )
        else :
            ALVerbose.log( "Resource::CREATED = " + "ascii-resource.res" )
        ALResource.m_oalThreadingConditionResource.synchronizeOff()
          

##############################################################################
