##############################################################################
#
#   File:      al_object.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from    al_definition   import  *
from    al_python       import  PyOS
from    al_python       import  PyCopy
from    al_python       import  PyXml
from    al_math         import  ALRandom
from    al_string       import  ALString


class ALObject:
    """
    Virtual base class for all AL classes. AL = Asynchronous Action Library
    class namespace definition.
    
    The self.m_strMagicNumber variable (constant) is used to define
    the name of the class ini.e.:XML files.
    
    The self.m_strID is used to assign a unique identity string
    to each object in order to distinguish between different
    objects and copies of objects.
    
    WARNING: the method getId() is deprecated!!
    Must use the getID() instead...  
    """
    
    
    def __init__( self ):
        """
        Constructor of the main pure virtual class.
        This constructor implements:
        the MagicNumber and the unique ID of the objects.
        
        @type  _oalString: ALString
        @param _oalString: used to define the MagicNumber
        """
        #self.m_strID = self.generateRandomID()
        #oalString = self.getMagicNumber()
        #self.m_oalStringName = ALString( "obj"+ oalString[3:]  )
       
        self.m_objectID = self.generateRandomID()
        self.m_objectMagicNumber = self.getMagicNumber()
        self.m_objectName = ALString( "obj"+ self.m_objectMagicNumber[3:]  )
        self.m_objectVersion = ALString( AL_VERSION )
    
    
    
    def getName( self ):
        return self.m_objectName.duplicate()
    
    
    def setName( self, _oalStringName ):
        self.m_objectName = ALString( _oalStringName )


    def generateRandomID( self ):
        #Trick to bypass system dependance
        pid = PyOS.getpid() % 10000
        oalStringVersion = ALString( float( AL_VERSION ) * 10 )
        oalString = "%s-%04i-%04i-%04i-%04i-%04i-%04i" % ( AL_OBJECT_ID, int( oalStringVersion.atof() ),
                                                      ALRandom.randInt( pid,9999 ),
                                                      ALRandom.randInt( 0, 9999 ),
                                                      ALRandom.randInt( pid,9999 ),
                                                      ALRandom.randInt( 0, pid ),
                                                      ALRandom.randInt( pid, 9999 ))
        return ALString( oalString )
    
    def duplicate( self ):
        #Trick to bypass system dependance
        return PyCopy.deepcopy( self )
    
    
    def getID( self ):
        return self.m_objectID.duplicate()
    
    
    def getVersion( self ):
        return self.m_objectVersion.duplicate() 
    
   
    def getMagicNumber( self ):
        oalString = ALString( self.__class__ )
        oalString = oalString.replace( "<class '", "" )
        oalString = oalString.replace( "'>", "" )
        self.m_objectMagicNumber = ALString( "mag"+ ALString( oalString.split( "." )[1] ) )
        return self.m_objectMagicNumber.duplicate()
    
   
    
    # Think about ############################################################
    def output( self, _oalStream = None ):
        if  _oalStream != None :
            _oalStream.output( self )
    
    
    def input( self, _oalStream = None ):
        pass 



    def outputXsdObject( self,  _oalXml, _oalXmlElement ):
        #oalXmlElementObject = _oalXml.createElement( self.getMagicNumber() )
        #_oalXmlElement.appendChild( oalXmlElementObject )
        #
        #for ikey in self.__dict__:
        #    if ( self.__dict__[ ikey ]!= None ):
        #        oalXmlElement = _oalXml.createElement( ikey[2:] )
        #        oalXmlElement.appendChild( _oalXml.createTextNode( ALString(  self.__dict__[ ikey ] ) ) )      
        #        oalXmlElementObject.appendChild( oalXmlElement )
        #
        #return  oalXmlElementObject
        pass
    
   

    def outputXmlObject( self,  _oalXml, _oalXmlElement ):
        oalXmlElementObject = _oalXml.createElement( self.getMagicNumber() )
        _oalXmlElement.appendChild( oalXmlElementObject )
        
        for ikey in self.__dict__:
            if ( self.__dict__[ ikey ]!= None ):
                oalXmlElement = _oalXml.createElement( ikey[2:] )
                oalXmlElement.appendChild( _oalXml.createTextNode( ALString(  self.__dict__[ ikey ] ) ) )      
                oalXmlElementObject.appendChild( oalXmlElement )
        
        return  oalXmlElementObject
    
       

            
            
    def inputXmlObject( self, _oalXmlElement ):
        for inode in _oalXmlElement.childNodes:
            if inode.nodeType==PyXml.Node.ELEMENT_NODE:
                if ( ALString( inode.nodeName ) == ALString( "objectID" ) ):
                    for jnode in inode.childNodes:
                        if jnode.nodeType==PyXml.Node.TEXT_NODE:
                            oalStringValue = ALString( jnode.data )
                            self.m_strID = oalStringValue.cleanXml()
                if ( ALString( inode.nodeName ) == ALString( "objectVersion" ) ):
                    for jnode in inode.childNodes:
                        if jnode.nodeType==PyXml.Node.TEXT_NODE:
                            oalStringValue = ALString( jnode.data )
                            self.m_version = oalStringValue.cleanXml()
                if ( ALString( inode.nodeName ) == ALString( "objectName" ) ):
                    for jnode in inode.childNodes:
                        if jnode.nodeType==PyXml.Node.TEXT_NODE:
                            oalStringValue = ALString( jnode.data )
                            self.m_oalStringName = oalStringValue.cleanXml()

        #Not possible to include here to change the old fashion style
        #for ikey in self.__dict__:
        #    odrValue = DRXml.getElementTokenValue( _odrXmlElement, DRXml.getAtributeToken( ikey ) )
        #    if (odrValue!=None):
        #        self.__dict__[ ikey ] = odrValue






    # Only to be virtual
    def outputXsd( self,  _oalXml, _oalXmlElement ):
        oalXmlElementObject = self.outputXsdObject( _oalXml, _oalXmlElement )
        
    def outputXml( self,  _oalXml, _oalXmlElement ):
        oalXmlElementObject = self.outputXmlObject( _oalXml, _oalXmlElement )
            
    def inputXml( self, _oalXmlElement ):
        self.inputXmlObject( _oalXmlElement )


    
##############################################################################
 
    
    