#############################################################################
#
#   File:      al_disk_explorer.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################



from al_python                  import *
from al_sys                     import ALSystem
from al_object                  import ALObject
from al_verbose                 import ALVerbose
from al_string                  import ALString
from al_list                    import ALList
from al_threading_condition     import ALThreadingCondition

     
class ALDiskExplorer( ALObject ):
    """
    Interface for AL disk folder and file manager.
    """
    
    
    m_oalThrCondDiskExplorer    = ALThreadingCondition()
    
    
    def __init__( self ):
        ALObject.__init__( self )
       
    
    def existFile( _oalString ):
        """
        Verify the existence of the file
        """
        return ALDiskExplorer.isFile( _oalString  )
    existFile = staticmethod( existFile )
    
    
    def isFile( _oalString ):
        """
        Verify the file
        """
        return PyOS.path.isfile( ALDiskExplorer.normPath( _oalString ) )
    isFile = staticmethod( isFile )
    
    
    def isDirectory( _oalString ):
        """
        Verify the directory
        """     
        return PyOS.path.isdir( ALDiskExplorer.normPath( _oalString ) )   
    isDirectory  = staticmethod( isDirectory )
    
    
    def isFolder( _oalString ):   
        return ALDiskExplorer.isDirectory( _oalString ) 
    isFolder  = staticmethod( isFolder )
      
      
    def existFolder( _oalString ):
        return ALDiskExplorer.isDirectory( _oalString )
    existFolder  = staticmethod( existFolder )
    
    
    def createFolder( _oalString ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        if _oalString != None:
            if ALDiskExplorer.existFolder( _oalString ) is False:
                PyOS.mkdir( ALDiskExplorer.normPath( _oalString ) )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
    createFolder  = staticmethod( createFolder )
    
    
    def deleteFolder( _oalString ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        if _oalString != None:
            if ALDiskExplorer.existFolder( _oalString ) is True:
                PyOS.rmdir( ALDiskExplorer.normPath( _oalString ) )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
    deleteFolder  = staticmethod( deleteFolder )
    
   
    def deleteFile( _oalString ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        if _oalString != None:
            if ALDiskExplorer.existFile( _oalString ) is True:              
                PyOS.remove( ALDiskExplorer.normPath( _oalString ) )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
    deleteFile  = staticmethod( deleteFile )
    
    
    def isLink( _oalString ):
        """
        Verify the link
        """
        return PyOS.path.islink(  ALDiskExplorer.normPath( _oalString ) )
    isLink  = staticmethod( isLink )
    
    
    def getFileList( _oalStringPath = "./", _oalStringMask = "*" ):
        """
        This metthod get the file list of a folder by
        a mask
        """
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalStringFind = PyOS.path.join( _oalStringPath, _oalStringMask )
        oalListFile = ALList( PyGlob.glob( oalStringFind ) )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
        for i in range( 0, len( oalListFile )):
            oalListFile[i] = ALDiskExplorer.normPath( oalListFile[i] )
        return oalListFile
    getFileList = staticmethod( getFileList )


    def getFolderList( _oalStringPath = "./", _oalStringMask = "*" ):
        return ALDiskExplorer.getFileList( _oalStringPath, _oalStringMask )
    getFolderList = staticmethod( getFolderList )
    
 
    def folderName( _oalStringPath ):
        return PyOS.path.dirname( ALDiskExplorer.normPath( _oalStringPath ) )
    folderName = staticmethod( folderName )
 
 
    def getFolderName( _oalStringPath ):
        return PyOS.path.dirname( ALDiskExplorer.normPath( _oalStringPath ) )
    getFolderName = staticmethod( getFolderName )
 
    
    def absPath( _oalStringPath ):
        return ALDiskExplorer.folderName( _oalStringPath )
    absPath = staticmethod( absPath )
    
    
    def absolutePath( _oalStringPath ):
        return ALDiskExplorer.absPath( _oalStringPath )
    absolutePath = staticmethod( absolutePath )
    
   
    def normPath( _oalStringPath ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalStrPath = PyOS.path.normpath( _oalStringPath )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
        return ALDiskExplorer.normDrive( oalStrPath )
    normPath = staticmethod( normPath )


  
    def normalize( _oalStringPath ):
        return ALDiskExplorer.normPath( ALString( _oalStringPath ) )
    normalize = staticmethod( normalize )



    def normDrive( _oalStringPath ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalStringPath = ALString( _oalStringPath )
        oalStringPath = oalStringPath.replace( "\\", "/" )
        if ALSystem.getOSType()==ALString( "Cygwin" ):
            if oalStringPath.count("cygdrive") == True :
                lstr = oalStringPath[10]           
                oalStringPath = ALString.replace( oalStringPath, "/cygdrive/" + lstr + "/", lstr + ":/" )
        #if PyOS.name == "posix" :
        #    oalStrSystem = PyOS.uname()
        #    if oalStrSystem[0] != "Linux" :
        #        if oalStringPath.count("cygdrive") == True :
        #            lstr = oalStringPath[10]           
        #            oalStringPath = PyString.replace( oalStringPath, "/cygdrive/" + lstr + "/", lstr + ":/" )
        #        #else : # fix to olof...must put it ??? for old cygwin
        #        #    tstr = "c:/cygwin" + ostr
        #        #    ostr = tstr
        #        
        ## To Think = bypass the system....        
        #if oalStringPath.count("cygdrive") == True :
        #    lstr = oalStringPath[10]           
        #    oalStringPath = PyString.replace( oalStringPath, "/cygdrive/" + lstr + "/", lstr + ":/" )
            
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
        return oalStringPath
    normDrive = staticmethod( normDrive )


    def iteractPath( _oalPath, _oalObjectMethod, _oalArgument) :
        """
        Execute the object method iteractivetily with alll elements
        in the folder
        """
        PyOS.path.walk( _oalPath, _oalObjectMethod, _oalArgument )
    iteractPath = staticmethod( iteractPath )


    def mergePath( _oalPath0, _oalPath1 ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalPath0 = ALDiskExplorer.normPath( _oalPath0 )
        oalPath1 = ALDiskExplorer.normPath( _oalPath1 )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
        return ALDiskExplorer.normPath( PyOS.path.join( oalPath0, oalPath1 ) )
    mergePath = staticmethod( mergePath )


    def addPath( _oalPath0, _oalPath1 ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalPath0 = ALDiskExplorer.normPath( _oalPath0 )
        oalPath1 = ALDiskExplorer.normPath( _oalPath1 )
        oalPath = oalPath0 + "/" + oalPath1
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
        return oalPath
    addPath = staticmethod( addPath )


    def getCwd():
        return ALDiskExplorer.normPath( PyOS.getcwd() )
    getCwd = staticmethod( getCwd )


    def getCurrentFolder():
        return ALDiskExplorer.getCwd()
    getCurrentFolder = staticmethod( getCurrentFolder )


    def setCurrentFolder( _oalPath ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalStrPath = ALDiskExplorer.normPath( _oalPath )
        PyOS.chdir( oalStrPath )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
    setCurrentFolder = staticmethod( setCurrentFolder )


    def setMode( _oalPath, _pyMode ):
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOn()
        oalStrPath = ALDiskExplorer.normPath( _oalPath )
        PyOS.chmod( oalStrPath, _pyMode )
        ALDiskExplorer.m_oalThrCondDiskExplorer.synchronizeOff()
    setMode = staticmethod( setMode )


##############################################################################