##############################################################################
#
#   File:      al_client_http.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################



from al_client           import ALClient
from al_message          import ALMessage
from al_string           import ALString
from al_http             import ALHTTP


class ALClientHTTP( ALClient ):
    """
    Parent class for all AL client
    HTTP communication objects.
    """
    
    
    def __init__( self, _ostrHostName, _iPortNumber ):
        ALClient.__init__( self )
        self.m_oalStringHostName = ALString( _ostrHostName ) 
        self.m_iPortNumber  = _iPortNumber
        
        
    def sendMessage( self, _oalMessageCommand ):

        errorCode    = "ok"
        errorMessage = None

        if self.m_oalStringHostName is None:
            errorCode    = "error"
            errorMessage = "ALClientHTTP: Host name is None." 
        elif self.m_iPortNumber is None:
            errorCode    = "error"
            errorMessage = "ALClientHTTP: Port number is None." 

        slMessage          = ALMessage( None )
        strPath            = _oalMessageCommand.getCommand()
        strMessage         = _oalMessageCommand.getMessage()
        strMessageResponse = None
        
        if errorCode == "ok" :
            try:
                h = ALHTTP( "%s:%d" % ( self.m_oalStringHostName, self.m_iPortNumber ) )
                h.putrequest( "POST",           strPath )
                h.putheader(  "Content-type",   "text/xml" )
                h.putheader(  "Content-length", "%d" % len( strMessage ) )
                h.putheader(  "Accept",         'text/plain' )
                h.putheader(  "Host",           self.m_oalStringHostName )
                h.endheaders( )
                h.send( strMessage )
            except:
                errorCode    = "error"
                errorMessage = "ALClientHTTP: Send failed."
                #raise
                
        reply = None
        msg   = None
        hdrs  = None
        
        if errorCode == "ok" :
            try:        
                reply, msg, hdrs = h.getreply()
            except:
                errorCode    = "error"
                errorMessage = "HTTP client reply failed."
                
        oslStringContentLength = None
        if errorCode == "ok" :
            oslStringContentLength = ALString( hdrs.getrawheader( "Content-Length" ) )
            if oslStringContentLength is None:
                errorCode    = "error" 
                errorMessage = "ALClientHTTP: No header returned"

        if errorCode == "ok" :
                try:
                    oslStringContentLength = oslStringContentLength.replace( "\n", "") 
                    iContentLength         = oslStringContentLength.atoi( )
                except:
                    errorCode    = "error"
                    errorMessage = "ALClientHTTP: Cannot convert Content-Length"
                    raise

        if errorCode == "ok" :
            try:
                strMessageResponse = h.getfile( ).read( iContentLength )
            except:
                errorCode    = "error"
                errorMessage = "ALClientHTTP: Cannot read response"
                
        slMessage = ALMessage( strMessageResponse )
        slMessage.setErrorCode( errorCode )
        slMessage.setErrorMessage( errorMessage )
        return slMessage


##############################################################################
