##############################################################################
#
#   File:      al_application.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################


from al_definition                  import *
from al_object                      import ALObject
from al_string                      import ALString
from al_token                       import ALToken
from al_verbose                     import ALVerbose
from al_time                        import ALTime
from al_list                        import ALList
from al_sys                         import ALSystem
from al_slot                        import ALSlot
from al_signal                      import ALSignal
from al_command_line_argument       import ALCommandLineArgument
from al_command_line                import ALCommandLine
from al_resource                    import ALResource
from al_log_file                    import ALLogFile
from al_threading_condition         import ALThreadingCondition
from al_trace                       import ALTrace
from al_disk_explorer               import ALDiskExplorer
from al_application_communicator    import ALApplicationCommunicator



class ALApplicationClass( ALObject ):
    """
    Interface for Application objects.
    
    """
    m_oalThreadingConditionApplication  = ALThreadingCondition()
    m_oalApplicationName                = None
    m_oalApplicationVersion             = None
    m_oalCommandLineArgument            = None
    m_oalCommandLine                    = None        
    m_oalResource                       = None
    m_oalHelp                           = None
    m_oalMan                            = None
    m_oalDebug                          = False
    m_oalSlotSignal                     = ALSlot()   
    
    def __init__( self, _oalString = AL_FRAMEWORK_APPLICATION_NAME, _dversion = AL_VERSION ):
        ALObject.__init__( self )
        ALApplicationClass.m_oalApplicationName     = ALString( _oalString )
        ALApplicationClass.m_oalApplicationVersion  = ALString( _dversion )
        self.create()
     
     
    def create( self ):
        ALApplicationClass.m_oalCommandLineArgument  = ALCommandLineArgument()
        ALApplicationClass.m_oalCommandLine          = ALCommandLine( ALApplicationClass.m_oalCommandLineArgument )        
        ALVerbose.m_oalLogFile                       = ALLogFile( ALApplicationClass.m_oalApplicationName + ".log" )
        ALTime.init()
        ALApplicationClass.m_oalResource             = ALResource( ALApplicationClass.m_oalApplicationName + ".res" )
 
        self.m_oalSlotInit          = ALSlot()
        self.m_oalSlotPreProcess    = ALSlot()
        self.m_oalSlotProcess       = ALSlot()
        self.m_oalSlotPostProcess   = ALSlot()
        self.m_oalSlotEnd           = ALSlot()
        self.m_oalSlotExecute       = self.m_oalSlotProcess
        
        # End the application in the error of the commandline
        ALApplicationClass.m_oalSlotSignal.connect( self.exit, ALSignal.error() )
        ALApplicationClass.m_oalSlotSignal.connect( self.exit, ALSignal.exception() )
        ALApplicationClass.m_oalSlotSignal.connect( self.exit, ALSignal.exit() )
        ALApplicationClass.m_oalCommandLine.m_oalSlotException.connect( self.exit, ALSignal.exception() ) # not used??
    
 
 
    def getApplicationName():
        ALApplicationClass.synchronizeOn()
        oalString =  ALApplicationClass.m_oalApplicationName
        ALApplicationClass.synchronizeOff()
        return oalString
    getApplicationName = staticmethod( getApplicationName )
    

    def getApplicationLogFileName():
        ALApplicationClass.synchronizeOn()
        oalString =  ALVerbose.m_oalLogFile.getFileName()
        ALApplicationClass.synchronizeOff()
        return oalString
    getApplicationLogFileName = staticmethod( getApplicationLogFileName )
    
    
    
    def synchronizeOn():
        ALApplicationClass.m_oalThreadingConditionApplication.synchronizeOn()
    synchronizeOn = staticmethod( synchronizeOn )


    def synchronizeOff():
        ALApplicationClass.m_oalThreadingConditionApplication.synchronizeOff()
    synchronizeOff = staticmethod( synchronizeOff )

     
    def outputVersion():
        ALApplicationClass.synchronizeOn()
        ALVerbose.screen( ALApplicationClass.m_oalApplicationName + "::Version " + ALApplicationClass.m_oalApplicationVersion )
        ALApplicationClass.synchronizeOff()
    outputVersion = staticmethod( outputVersion )
    

    def ouputOption():
        ALApplicationClass.synchronizeOn()
        ALVerbose.screen( ALApplicationClass.m_oalApplicationName + "::Options :" )
        ALVerbose.screen( "                   --help or -h    : to show help if avaiable" )
        ALVerbose.screen( "                   --man  or -m    : to show manual if avaiable" )
        ALVerbose.screen( "                   --verbose       : to iteractive output" )
        ALVerbose.screen( "                   --quiet         : NO output" )
        ALVerbose.screen( "                   --nolog         : NO log file output" )
        ALVerbose.screen( "                   --version or -v : executable version info" )
        ALApplicationClass.synchronizeOff()
    ouputOption = staticmethod( ouputOption )
    
    
    
    def outputHelp():
        ALApplicationClass.synchronizeOn()
        if ALApplicationClass.m_oalHelp is None:
            ALVerbose.screen( ALApplicationClass.m_oalApplicationName + "::No Help Avaiable" )
            ALApplicationClass.ouputOption()
        else:
            ALApplicationClass.m_oalHelp.outputScreen()
        ALApplicationClass.synchronizeOff()
    outputHelp = staticmethod( outputHelp )
    

    
    def outputMan():
        ALApplicationClass.synchronizeOn()
        if ALApplicationClass.m_oalMan is None:
            ALVerbose.screen( ALApplicationClass.m_oalApplicationName + "::No Manual Avaiable" )
            ALApplicationClass.ouputOption()
        else:
            ALApplicationClass.m_oalMan.outputScreen()
        ALApplicationClass.synchronizeOff()
    outputMan = staticmethod( outputMan )
    


    def setHelp( _oalHelp ):
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalHelp = _oalHelp
        ALApplicationClass.synchronizeOff() 
    setHelp = staticmethod( setHelp )



    def setMan( _oalMan ):
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalMan = _oalMan
        ALApplicationClass.synchronizeOff() 
    setMan = staticmethod( setMan )


    # Signal =================================================================
    def getSignal():
        ALApplicationClass.synchronizeOn()
        oalSignal = ALSignal.ok()
        oalSignalExt = ALApplicationCommunicator.getSignal()
        if oalSignalExt != None :
            oalSignal = oalSignalExt.duplicate()
        ALApplicationClass.synchronizeOff()
        return oalSignal
    getSignal = staticmethod( getSignal )


    def setSignal( _oalSignal ):
        ALApplicationClass.synchronizeOn() 
        if _oalSignal != None:
            ALApplicationCommunicator.setSignal( _oalSignal.duplicate() )
            ALApplicationClass.m_oalSlotSignal.callSignal( _oalSignal.duplicate() )
        ALApplicationClass.synchronizeOff()
    setSignal = staticmethod( setSignal )


    # Command Line ===========================================================
    def getCommandLineArguments():
        ALApplicationClass.synchronizeOn() 
        oalArguments = ALApplicationClass.m_oalCommandLineArgument.duplicate()
        ALApplicationClass.synchronizeOff()
        return oalArguments
    getCommandLineArguments = staticmethod( getCommandLineArguments )
    

    def addCommand( self, _oalStringToken, _oalObjectMethod ):
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalCommandLine.addCommand( _oalStringToken, _oalObjectMethod )
        ALApplicationClass.synchronizeOff() 
        
    
    def addCommandLineArguments( _oalListArgument ):
        ALApplicationClass.synchronizeOn()
        oalListArgument = ALList( _oalListArgument )
        for i in oalListArgument:
            ALApplicationClass.m_oalCommandLineArgument.append( i )
            ALApplicationClass.m_oalCommandLine.addCommandArgument( i )
        ALApplicationClass.synchronizeOff()
    addCommandLineArguments = staticmethod( addCommandLineArguments )
      
     
    def addListArgument( self, _oalListArgument ):
        ALApplicationClass.synchronizeOn()
        oalListArgument = ALList( _oalListArgument )
        for i in oalListArgument:
            ALApplicationClass.m_oalCommandLineArgument.append( i )
            ALApplicationClass.m_oalCommandLine.addCommandArgument( i )
        ALApplicationClass.synchronizeOff()
    
   
    def addCommandLineArgument( _oalStringToken ):
        ALApplicationClass.synchronizeOn()
        oalStringToken = ALToken( _oalStringToken )
        ALApplicationClass.m_oalCommandLineArgument.append( oalStringToken )
        ALApplicationClass.m_oalCommandLine.addCommandArgument( oalStringToken )
        ALApplicationClass.synchronizeOff()
    addCommandLineArgument = staticmethod( addCommandLineArgument )
    
 
    def getCommandLineArgument( _oalStringToken ):
        ALApplicationClass.synchronizeOn()
        oalCommandLineArg = ALApplicationClass.m_oalCommandLine.getArgument( _oalStringToken )
        ALApplicationClass.synchronizeOff()
        return oalCommandLineArg
    getCommandLineArgument = staticmethod( getCommandLineArgument )
    
 
      
    def getCommandLine():
        ALApplicationClass.synchronizeOn()
        oalCommandLine = ALApplicationClass.m_oalCommandLine.duplicate()
        ALApplicationClass.synchronizeOff()
        return oalCommandLine
    getCommandLine = staticmethod( getCommandLine )
    
 
    def existCommand( _oalStringCommand ):
        ALApplicationClass.synchronizeOn()
        bExist = ALApplicationClass.m_oalCommandLine.existCommand( _oalStringCommand )
        ALApplicationClass.synchronizeOff()
        return bExist
    existCommand = staticmethod( existCommand )
 
      
    # Ressource ================================================================
    def getResource( _oalToken ):
        ALApplicationClass.synchronizeOn() 
        oalRessourceValue = ALApplicationClass.m_oalResource.getResource( _oalToken )
        ALApplicationClass.synchronizeOff()
        return oalRessourceValue
    getResource = staticmethod( getResource )
    
  
    def setResource( _oalToken, _oalTokenObject ):
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalResource.setResource( _oalToken, _oalTokenObject )
        ALApplicationClass.synchronizeOff() 
    setResource = staticmethod( setResource )



    def resetResource():
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalResource.reset()
        ALApplicationClass.m_oalResource  = ALResource( ALApplicationClass.m_oalApplicationName + ".res" )
        ALApplicationClass.synchronizeOff() 
    resetResource = staticmethod( resetResource )


    def inputResource():
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalResource.input()
        ALApplicationClass.synchronizeOff() 
    inputResource = staticmethod( inputResource )


    def outputResource():
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalResource.output()
        ALApplicationClass.synchronizeOff() 
    outputResource = staticmethod( outputResource )
    
    
    # Output control ========================================================== 
    def setDebugMode():
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalDebug = True
        ALVerbose.setVerboseDebugOn()
        ALApplicationClass.synchronizeOff() 
    setDebugMode = staticmethod( setDebugMode )
    
  
    def setDebugModeON():
        ALApplicationClass.setDebugMode()
    setDebugModeON = staticmethod( setDebugModeON )
    
  
    def setDebugModeOFF():
        ALApplicationClass.synchronizeOn() 
        ALApplicationClass.m_oalDebug = False
        ALVerbose.setVerboseDebugOff()
        ALApplicationClass.synchronizeOff() 
    setDebugModeOFF = staticmethod( setDebugModeOFF )
    
    
      
    def isDebugMode():
        ALApplicationClass.synchronizeOn() 
        bDebug = ALApplicationClass.m_oalDebug
        ALApplicationClass.synchronizeOff()
        return bDebug
    isDebugMode = staticmethod( isDebugMode )
    
    
    
    # SlotConnection =========================================================    
    def executeConnect( self, _oMethod ):
        ALApplicationClass.synchronizeOn()
        if _oMethod != None :
            self.m_oalSlotExecute.connect( _oMethod )
        ALApplicationClass.synchronizeOff() 
  

  
  



    def init( self, _bInit = True ):
        """
        Any processing that must occur before the action should go into
        this method.
        """
        oalString = ALString( ALApplicationClass.m_oalCommandLineArgument )
        ALVerbose.log( oalString )
        ALVerbose.setVerboseOff()
        
        if ALApplicationClass.m_oalCommandLine.existCommand( "--verbose" ):
            ALVerbose.setVerboseOn()

        if ALApplicationClass.m_oalCommandLine.existCommand( "--quiet" ):
            ALVerbose.setVerboseOff()
            
        if ALApplicationClass.m_oalCommandLine.existCommand( "--DEBUG" ):
            ALApplicationClass.setDebugMode()
            ALVerbose.DEBUG( "Debug mode" )
            
                                
        if _bInit :
            ALVerbose.screen( self.m_oalApplicationName + "::Starting" )
            if ALApplicationClass.m_oalCommandLine.existCommand( "--help" ) or ALApplicationClass.m_oalCommandLine.existCommand( "-h" ):
                ALApplicationClass.outputHelp()
                self.exit( ALSignal.exit() )
   
            if ALApplicationClass.m_oalCommandLine.existCommand( "--version" ) or ALApplicationClass.m_oalCommandLine.existCommand( "-v" ):
                ALApplicationClass.outputVersion()
                self.exit( ALSignal.exit() )
                
            if ALApplicationClass.m_oalCommandLine.existCommand( "--man" ) or ALApplicationClass.m_oalCommandLine.existCommand( "-m" ):
                ALApplicationClass.outputMan()
                self.exit( ALSignal.exit() )
    
            if ALApplicationClass.m_oalCommandLine.existCommand( "--nolog" ):
                ALVerbose.setLogFileOff()
            
            ALApplicationClass.inputResource()
            oalObject = ALApplicationClass.getResource( "application_last_run" )
            if (oalObject is not None):
                oalStringLastRun = ALString( oalObject );
                ALVerbose.screen( "Last Run: " + oalStringLastRun )
                           
        
        
        
        
    def __forceExit( self, _iArg = 1 ):    
        raise AL_FRAMEWORK_FORCE_CLOSE_APPLICATION



    def exit( self, _oalObject = None ):
        if _oalObject != None and isinstance( _oalObject, ALSignal ):
            oalSignal = _oalObject.getSignal()
            oalString = _oalObject.getInformation()          
            if oalSignal.equals( AL_ACTION_SIGNAL_ERROR ) or oalSignal.equals( AL_ACTION_SIGNAL_ABORT ) :
                ALVerbose.error( "Application::" + oalSignal + "::" + oalString )
            else:
                ALVerbose.log( "Application::" + oalSignal + "::" + oalString )
        self.__forceExit()
   
         

    def end( self ):
        """
        Similar to preProcess, any processing
        """
        ALApplicationClass.setResource( "application_last_run", ALTime.timeString() )
        ALApplicationClass.outputResource()
        ALVerbose.screen( self.m_oalApplicationName + "::Closing" )
        ALVerbose.screen( self.m_oalApplicationName + "::Runtime: " + ALString( ALTime.dif() ) + " [s]" )


    
    def processCommandLine( self ):
        """
        This is the command line method of the application.
        """
        ALVerbose.DEBUG( "Execution: ALApplication.processCommandLine()" )
        ALVerbose.screen( self.m_oalApplicationName + "::Processing" )
        ALApplicationClass.m_oalCommandLine.executeAction()     
        ALApplicationClass.m_oalCommandLine.waitActionEnd()
         
 


    def __signalControl( self ):
        """
        This application Signal control.
        """
        ALVerbose.DEBUG( "Execution: ALApplication.__signalControl()" )
        ALApplicationClass.synchronizeOn()
        oalSignalExt = ALApplicationCommunicator.getSignal()
        if oalSignalExt != None :
            oalSignal = oalSignalExt.duplicate()
            ALApplicationClass.m_oalSlotSignal.callSignal( oalSignal )
        ALApplicationClass.synchronizeOff()

       
    
 
    def execute( self, _bInit = True ):
        """
        This is the method started by the application.
        """
        ALVerbose.DEBUG( "Execution: ALApplication.execute()" )
        try:
            # Init
            self.init( _bInit )
            self.__signalControl()
            self.m_oalSlotInit.call( self )
              
            # PreProcess Sequence
            self.__signalControl()
            self.preProcess()
            self.__signalControl()
            self.m_oalSlotPreProcess.call( self )

            # Process Sequence
            self.__signalControl()
            self.processCommandLine()
            
            # Process Sequence
            self.__signalControl()
            self.process()
            self.__signalControl()
            self.m_oalSlotProcess.call( self )
                   
            # PostProcess Sequence
            self.__signalControl()
            self.postProcess()
            self.__signalControl()
            self.m_oalSlotPostProcess.call( self )
            
            # End
            self.__signalControl()
            self.m_oalSlotEnd.call( self )
            self.end()
        
        except AL_FRAMEWORK_FORCE_CLOSE_APPLICATION:
            if (ALApplicationClass.isDebugMode()):
                ALTrace.trace( self )
            # raise # to allow use the Komodo interface
            self.end()
            
        except:
            ALTrace.trace( self )
            #raise # to allow use the Komodo interface
            self.end()

            

############################################################################
# To be modified by the user
        
    def preProcess( self ):
        """
        Any processing that must occur before the action should go into
        this method.
        """
        ALVerbose.DEBUG( "Execution: ALApplication.preProcess()" )
    
    
    def process( self ):
        """
        This is the main method of the application.
        """
        ALVerbose.DEBUG( "Execution: ALApplication.process()" )

    
    def postProcess( self ):
        """
        Similar to preProcess, any processing
        """
        ALVerbose.DEBUG( "Execution: ALApplication.postProcess()" )

    
##############################################################################
   