##############################################################################
#
#   File:      al_action_class.py
#
#   Project:   Asynchronous Action Library Project
#              http://aalib.sourceforge.net/
#
#   Author(s): Romeu A. Pieritz & Olof Svensson
#   Contact  : aalib_info@yahoo.com
#
#   Date:      2005/06/24
#
#   License:
#   --------------------------------------------------------------------------
#   Asynchronous Action Library Project
#   Copyright (c) 2005, AALib developers (see above AUTHORS)
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
#   Neither the names of AALib's copyright owner nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
#   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
#   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
#   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################



from al_definition          import  *
from al_string              import  ALString
from al_verbose             import  ALVerbose
from al_manager_signal      import  ALManagerSignal
from al_slot                import  ALSlot
from al_signal              import  ALSignal
from al_thread_control      import  ALThreadControl


class ALAction( ALThreadControl ):
    """
    Parent class for all AL action objects.
    """ 
    def __init__( self ):
        """
        Constructor for ALAction.
        
        """
        ALThreadControl.__init__( self )
        self.m_oalSlotPreProcess    = ALSlot()
        self.m_oalSlotProcess       = ALSlot()
        self.m_oalSlotPostProcess   = ALSlot()
        
       
    def executeAction( self ):
        """
        This method starts the thread.
        Only here to remember the thread action in the framework
        """
        self.execute()
        
    
    def run( self ):
        """
        This is the method started by the thread.
        """
        try:
            # Thread Init
            self.m_oalSlotInit.call( self )
              
            # PreProcess Sequence
            self.threadControl()
            self.preProcess()
            self.m_oalSlotPreProcess.call( self )

            # Process Sequence  
            self.threadControl()      
            self.process()
            self.m_oalSlotProcess.call( self )
                   
            # PostProcess Sequence
            self.threadControl()   
            self.postProcess()
            self.m_oalSlotPostProcess.call( self )
            
            #Thread end
            self.m_bIsEnded = True
            self.m_oalSlotEnd.call( self )
            
        except AL_ACTION_SIGNAL_ABORT:
            self.m_oalSlotAbort.callSignal( ALSignal.abort() )
            
        except AL_ACTION_SIGNAL_ERROR:        
            self.m_oalSlotError.callSignal( ALSignal.error() )         
        
        except:
            self.m_oalSlotException.callSignal( ALSignal.exception() )
            
 


############################################################################
# To be modified by the user

    def preProcess( self, _oalObject = None ):
        """
        Any processing that must occur before the action should go into
        this method.
        """
        ALVerbose.DEBUG( "Execution: ALAction.preProcess()" )

    
    def process( self, _oalObject = None ):
        """
        This is the main method of the action.
        """
        ALVerbose.DEBUG( "Execution: ALAction.process()" )

    
    def postProcess( self, _oalObject = None ):
        """
        Similar to preProcess, any processing
        """
        ALVerbose.DEBUG( "Execution: ALAction.postProcess()" )



##############################################################################

    