package dna.util;

import java.io.FileInputStream;
import java.util.Enumeration;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * A utility class which allows the getting of Java properties from a local
 * source file or standard System properties.
 */
public class LocalProperties
{
   private static Properties properties = new Properties(System.getProperties());

   static
   {
      int i = 1;
      boolean displayError = true;
      boolean error = false;

      String propertiesFiles = properties.getProperty("dl.propertiesFiles");
      String defaultPropertiesFile = System.getProperty("user.home")
         + System.getProperty("file.separator") + "java.properties";
      defaultPropertiesFile = System.getProperty("dl.propertiesFile",
         defaultPropertiesFile);
      String propertiesFile;

      if (propertiesFiles != null)
      {
         StringTokenizer st = new StringTokenizer(propertiesFiles, " ");
         while (st.hasMoreTokens())
         {
            propertiesFile = st.nextToken();
            loadPropertiesFile(propertiesFile, displayError);
         }
      }
      else
      {
         propertiesFile = defaultPropertiesFile;
         do
         {
            error = loadPropertiesFile(propertiesFile, displayError);
            displayError = false;
            propertiesFile = defaultPropertiesFile + "_" + i++;
         }
         while (!error);
      }

      doPropertyInterpolation();
   }

   private static boolean loadPropertiesFile(String file, boolean displayError)
   {
      FileInputStream in = null;
      boolean error = false;
      try
      {
         in = new FileInputStream(file);
         properties.load(in);
         in.close();
      }
      catch (Exception ex)
      {
         if (displayError)
            Debug.out("Error reading properties file " + file);
         error = true;
      }

      return error;
   }

   /**
    * Scan through Java Properties for ${} patterns. Each instance should
    * contain a reference to a named java property (e.g. "${gda.src.java}"). If
    * any found, look for a matching java property of same name and replace the
    * pattern instance with the value of that property using string
    * interpolation.
    * 
    * N.B. Should handle multiple pattern instances per property value, but
    * doesnt handle recursive interpolation yet (i.e. a pattern containing a
    * property with a value containing further patterns).
    * 
    */
   protected static void doPropertyInterpolation()
   {
      /*
      properties.clear();
      
      properties.setProperty("gda.fooroot", "d:/GDA/fubar");
      // next one fails due to nonexistant key specified - to test fail path
      properties.setProperty("gda.foo.root1", "${gda.fooroot_nonexistent}/blah");
      properties.setProperty("gda.foo.root", "${gda.fooroot}/blah");

      properties.setProperty("gda.src.java", "D:/GDA");
      properties.setProperty("gda.station", "stn7_6");
   
      properties.setProperty("mytest.path", "${gda.src.java}/${gda.station}/params/properties");
      properties.setProperty("mytest.path1", " #${gda.src.java}/foo/${gda.station}/bar");

      properties.setProperty("gda.src.java1", "${gda.src.java}/blah");
      // next one fails due to double interpolation - not yet supported
      properties.setProperty("mytest.path2", "foo/${gda.src.java1}/bar/");
      
      properties.setProperty("gda.src.java2", "/foo/${gda.src.java}");
      properties.setProperty("gda.src.java3", "$/fo$o/${gda.src.java}");
      
      // N.B. - change Enumeration line below to the following:- 
      Enumeration e = properties.keys(); // fetches just user keys, not default keys
       */
      // properties.list(System.out);
      // N.B. fetches both keys and default keys (from system properties)
      Enumeration e = properties.propertyNames();

      // check each property to find ${} patterns
      while (e.hasMoreElements())
      {
         // get next key and its value
         String key = (String) e.nextElement();
         String value = properties.getProperty(key);

         int index = 0;
         int count = 0;

         // scan for multiple pattern instances per value
         while ((index = value.indexOf("${", count)) >= 0)
         {
            // find end of current pattern
            int end = value.indexOf("}", index);

            // extract the referenced property from the pattern
            String propertyName = value.substring(index + 2, end);

            // try to fetch the property
            String propertyValue = properties.getProperty(propertyName);

            // look in System properties if not found
            if (propertyValue != null && propertyValue != "null")
            {
               Debug.out("", 3);
               Debug.out("key=" + key + " value=" + value, 3);
               Debug.out("found key=" + propertyName, 3);
               Debug.out("inserting its value=" + propertyValue, 3);

               // replace matched pattern with referenced property value
               value = value.substring(0, index) + propertyValue
                  + value.substring(end + 1, value.length());

               // reset original property with new interpolated value
               properties.setProperty(key, value);

               Debug.out("result=" + value, 3);
               Debug.out("", 3);

               // Found match so increment past interpolated value.
               // start next search at end of pattern instance
               // N.B. next line could just be "count = index;" and it might
               // handle "recursive" evaluation of properties!
               count = index + propertyValue.length();
            }
            else
            {
               // Failed to find interpolated value
               // so increment past original pattern.
               // start next search at end of interpolated value
               // N.B. next line could just be "count = index;" and it might
               // handle "recursive" evaluation of properties!
               count = index + propertyName.length() + 3;
            }
         }
      }
   }

   public static void parseProperties(String[] propertyPairs)
   {
      StringTokenizer st;
      String propertyName;
      String propertyValue;

      for (int i = 0; i < propertyPairs.length; i++)
      {
         propertyName = null;
         propertyValue = null;
         st = new StringTokenizer(propertyPairs[i], "=");

         if (st.hasMoreTokens())
            propertyName = st.nextToken();
         if (st.hasMoreTokens())
            propertyValue = st.nextToken();

         if ((propertyName != null) && (propertyValue != null))
            properties.setProperty(propertyName, propertyValue);
      }
   }

   public static String get(String propertyName)
   {
      String propertyValue = null;

      propertyValue = properties.getProperty(propertyName);

      // README - we're outlawing backslashes - since no distinction between
      // string properties and URL/URI/path properties in GDA code
      // so have to do this for all property strings.
      if (propertyValue != null)
      {
         propertyValue = propertyValue.replace('\\', '/');
      }

      return propertyValue;
   }

   public static boolean check(String propertyName)
   {
      return check(propertyName, false);
   }

   public static boolean check(String propertyName, boolean defaultCheck)
   {
      String propertyValue = get(propertyName);
      boolean check = defaultCheck;

      if (propertyValue != null)
      {
         if (propertyValue.equals("true"))
            check = true;
         else if (propertyValue.equals("false"))
            check = false;
      }

      return check;
   }

   public static int getInt(String propertyName, int defaultValue)
   {
      String propertyValue = get(propertyName);
      int value = defaultValue;

      if (propertyValue != null)
      {
         try
         {
            value = Integer.parseInt(propertyValue);
         }
         catch (NumberFormatException nfex)
         {
         }
      }

      return value;
   }

   public static double getDouble(String propertyName, double defaultValue)
   {
      String propertyValue = get(propertyName);
      double value = defaultValue;

      if (propertyValue != null)
      {
         try
         {
            value = Double.parseDouble(propertyValue);
         }
         catch (NumberFormatException nfex)
         {
         }
      }

      return value;
   }

   public static String get(String propertyName, String defaultValue)
   {
      String propertyValue = get(propertyName);
      String value = defaultValue;

      if (propertyValue != null)
      {
         value = propertyValue;
      }

      // README - we're outlawing backslashes - since no distinction between
      // string properties and URL/URI/path properties in GDA code
      // so have to do this for all property strings.
      if (value != null)
      {
         value = value.replace('\\', '/');
      }

      return value;
   }

   public static void set(String propertyName, String value)
   {
      properties.put(propertyName, value);
   }
}
