package dna;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import dna.observable.IObservable;
import dna.observable.IObserver;
import dna.observable.ObservableComponent;
import dna.util.Debug;
import dna.xml.Collect_data_request;
import dna.xml.Collect_request;
import dna.xml.Fileinfo;
import dna.xml.Oscillation_sequence;
import dna.xml.Quick_scale_request;
import dna.xml.Resolution;
import dna.xml.Strategy_interpretation;
import dna.xml.Strategy_request;
import dna.xml.Strategy_response;
import dna.xml.Strategy_settings;

public class StrategyPanel
   extends JPanel
   implements IObservable, IObserver, ActionListener, Runnable
{
   private static final int STRATEGY = 0;
   private static final int COLLECT = 1;
   private static final int INTEGRATE = 2;
   private static final int SCALE = 3;

   private ExpertDataModel edm = ExpertDataModel.getInstance();

   private StrategyHelper strategy = new StrategyHelper();
   private Strategy_request strategy_request = new Strategy_request();
   private Strategy_settings settings = new Strategy_settings();

   private StrategyDataPanel strategyDataPanel = new StrategyDataPanel();
   private JButton calculate = new JButton("Calculate Strategy");
   private JButton collect = new JButton("Collect Data");
   private JButton integrate = new JButton("Collect & Integrate Data");
   private JButton scale = new JButton("Quick Scaling Statistics");
   private JScrollPane pane;
   private JPanel buttonPanel = new JPanel();
   private static int operation;
   private int runNumber = 1;
   private boolean dataIntegrated = false;
   private Fileinfo fileinfo = new Fileinfo();
   private Oscillation_sequence oscillation_sequence = new Oscillation_sequence();
   private Collect_request collect_request = new Collect_request();
   private Collect_data_request collect_data_request = new Collect_data_request();
   private CollectHelper collectHelper = CollectHelper.getInstance();
   private Resolution collect_resolution = new Resolution();
   private Quick_scale_request quick_scale_request = new Quick_scale_request();
   private ScaleHelper scaleHelper = new ScaleHelper();

   private ObservableComponent observableComponent = new ObservableComponent();

   public StrategyPanel()
   {
      GridBagConstraints c = new GridBagConstraints();

      setLayout(new GridBagLayout());

      populateButtonPanel();

      c.gridx = 1;
      c.gridy = GridBagConstraints.RELATIVE;
      c.insets = new Insets(0, 5, 0, 5); // Top, Left, Bottom, Right

      c.anchor = GridBagConstraints.CENTER;
      strategyDataPanel.addIObserver(this);
      add(strategyDataPanel, c);
      c.gridx = 1;
      add(buttonPanel, c);
   }

   private void populateButtonPanel()
   {
      GridBagConstraints c = new GridBagConstraints();
      buttonPanel.setLayout(new GridBagLayout());
      c.insets = new Insets(5, 5, 5, 5); // Top, Left, Bottom, Right
      calculate.setActionCommand("Calculate");
      calculate.addActionListener(this);
      collect.setActionCommand("Collect");
      collect.addActionListener(this);
      integrate.setActionCommand("Integrate");
      integrate.addActionListener(this);
      scale.setActionCommand("Scale");
      scale.addActionListener(this);
      deactivate(scale);

      c.gridx = GridBagConstraints.RELATIVE;
      c.gridy = 1;
      c.anchor = GridBagConstraints.EAST;

      buttonPanel.add(calculate, c);
      buttonPanel.add(collect, c);
      buttonPanel.add(integrate, c);
      buttonPanel.add(scale, c);
   }

   private void activate(JButton button)
   {
      button.setEnabled(true);
   }

   private void deactivate(JButton button)
   {
      button.setEnabled(false);
   }

   // IObservable

   public void addIObserver(IObserver anIObserver)
   {
      observableComponent.addIObserver(anIObserver);
   }

   public void deleteIObserver(IObserver anIObserver)
   {
      observableComponent.deleteIObserver(anIObserver);
   }

   public void deleteIObservers()
   {
      observableComponent.deleteIObservers();
   }

   // IObserver

   public void update(Object iObservable, Object arg)
   {
      Resolution resolution = new Resolution();
      
      if (arg instanceof ExpertState)
      {
         ExpertState state = (ExpertState) arg;
         switch (state.getState())
         {
            case ExpertState.ABORTED:
               dataIntegrated = false;
            case ExpertState.IDLE:
               activate(calculate);
               activate(collect);
               activate(integrate);
               if (dataIntegrated)
               {
                  activate(scale);
                  dataIntegrated = false;
               }                 
             break;
            case ExpertState.COLLECTING_REFERENCE_IMAGES:
            case ExpertState.INDEXING:
            case ExpertState.CALCULATING_STRATEGY:
            case ExpertState.CHARACTERIZING:
               deactivate(calculate);
               deactivate(collect);
               deactivate(integrate);
               deactivate(scale);
               break;
            case ExpertState.INTEGRATING_DATA:
               dataIntegrated = true;
            case ExpertState.COLLECTING_DATA:
               deactivate(calculate);
               deactivate(collect);
               deactivate(integrate);
               deactivate(scale);
               break;
         }
      }
      else if (arg instanceof Resolution)
      {
         resolution = (Resolution) arg;
         deactivate(collect);
         deactivate(integrate);
         edm.setClearStrategyResolution(false);
         if (strategy_request.getStrategy_settings() == null)
         {
            settings.setResolution(resolution);
            strategy_request.setStrategy_settings(settings);
         }
         else
         {
            settings = strategy_request.getStrategy_settings();
            settings.setResolution(resolution);
            strategy_request.setStrategy_settings(settings);
         }

      }
      else if (arg == null)
      {
         deactivate(collect);
         deactivate(integrate);
      }
      
   }

   // ActionListener

   public void actionPerformed(ActionEvent ae)
   {
      if (ae.getActionCommand().equals("Calculate"))
      {
         operation = STRATEGY;
         new Thread(this).start();
      }
      else if (ae.getActionCommand().equals("Collect"))
      {
         operation = COLLECT;
         new Thread(this).start();
      }
      else if (ae.getActionCommand().equals("Integrate"))
      {
         operation = INTEGRATE;
         new Thread(this).start();
      }
      else if (ae.getActionCommand().equals("Scale"))
      {
         System.out.println("Scale button pressed.");
         operation = SCALE;
         new Thread(this).start();
      }
   }

   // Runnable

   public void run()
   {
      StrategyData expt;
      Oscillation_sequence sequence;

      collect_data_request = new Collect_data_request();

      switch (operation)
      {
         case STRATEGY:
            try
            {
               /*
                * If the resolution hasn't been altered clear it in the settins.
                * Don't want values hanging over from previous strategy
                * calculations.
                */
               if (edm.getClearStrategyResolution())
               {
                  settings.setResolution(null);
               }
               settings.setAnomalous(edm.getAnomalous());
               strategy_request.setStrategy_settings(settings);
               
               observableComponent.notifyIObservers(this, new ExpertState(
                  ExpertState.CALCULATING_STRATEGY));
               Strategy_response strategy_response = strategy.sendStrategy_requestSynchronous(
                  strategy_request, edm.getStrategyTimeout());

               if (strategy_response != null)
               {
                  updateStrategyDataPanel(strategy_response);
               }
            }
            catch (StrategyException ex)
            {
               MessagePanel messagePanel = MessagePanel.getInstance();
               messagePanel.appendText("Error caught in Strategy");
               messagePanel.appendText(ex.getMessage());
               new ErrorDialog(this, ex, "Strategy");
            }
            finally
            {
               observableComponent.notifyIObservers(this, new ExpertState(
                  ExpertState.IDLE));
            }
            break;

         case INTEGRATE:
            collect_data_request.setIntegrate(true);
            observableComponent.notifyIObservers(this, new ExpertState(
               ExpertState.INTEGRATING_DATA));
        case COLLECT:
            try
            {
               fileinfo.setDirectory(edm.getDirectory());
               fileinfo.setPrefix(edm.getPrefix());
               if (edm.isRunNumberOn())
               {
                  runNumber = edm.getValueOfRunNumber();
               }
               else
               {
                  runNumber = 1;
               }
               fileinfo.setRun_number(runNumber);
               collect_request = new Collect_request();
               collect_request.setFileinfo(fileinfo);

               Oscillation_sequence[] oscillation_sequence = new Oscillation_sequence[strategyDataPanel.getDataArraySize()];
               for (int i = 0; i < strategyDataPanel.getDataArraySize(); i++)
               {
                  expt = strategyDataPanel.getStrategyData(i);

                  oscillation_sequence[i] = new Oscillation_sequence();
                  oscillation_sequence[i].setStart(expt.getValueOfStartPhi());
                  oscillation_sequence[i].setRange(expt.getValueOfIncrement());
                  oscillation_sequence[i].setOverlap(expt.getValueOfOverlap());
                  oscillation_sequence[i].setExposure_time(expt.getValueOfTime());
                  oscillation_sequence[i].setNumber_of_images(expt.getValueOfNImages());
                  oscillation_sequence[i].setNumber_of_passes(expt.getValueOfPasses());
                  oscillation_sequence[i].setStart_image_number(expt.getValueOfTemplateStart());
                  Resolution tmp_resolution = new Resolution();
                  tmp_resolution.setUpper(expt.getValueOfResolution());
                  collect_request.setResolution(tmp_resolution);
               }

               collect_request.setOscillation_sequence(oscillation_sequence);
               collect_resolution.setUpper(edm.getValueOfResolution());

               // Set the session id if available...
               if (edm.getSession() != null)
               {
                  collect_request.setSessionId((edm.getSession()).getSessionId());
               }

               collect_data_request.setCollect_request(collect_request);
               observableComponent.notifyIObservers(this, new ExpertState(
                  ExpertState.COLLECTING_DATA));
              collectHelper.sendCollect_data_request(collect_data_request);
            }
            catch (NumberFormatException ex)
            {
               System.out.println("Error converting reference image data");
            }
            catch (CollectException ex)
            {
               System.out.println("CollectException caught: " + ex.getMessage());
            }
            break;
        case SCALE:
            {
               try
               {
                  quick_scale_request.setFileinfo(fileinfo);
                  quick_scale_request.setAnomalous(edm.getAnomalous());
                  scaleHelper.sendQuick_scale_request(quick_scale_request);
               }
               catch (Quick_scaleException ex)
               {
                  System.out.println("Quick_scaleException caught: "
                     + ex.getMessage());
               }
            }
            break;

      }
   }

   public void updateStrategyDataPanel(Strategy_response strategy_response)
   {
      StrategyData expt;
      Oscillation_sequence sequence;

      Strategy_interpretation strategy_interpretation = strategy_response.getStrategy_interpretation();
      if (strategy_interpretation != null)
      {
         Oscillation_sequence[] oscillation_sequence = strategy_interpretation.getOscillation_sequence();

         strategyDataPanel.setDataArraySize(oscillation_sequence.length);
         for (int i = 0; i < oscillation_sequence.length; i++)
         {
            sequence = oscillation_sequence[i];
            expt = strategyDataPanel.getDnaData(i);
            updateTableData(expt, sequence, runNumber + i);
            expt = strategyDataPanel.getStrategyData(i);
            updateTableData(expt, sequence, runNumber + i);
         }
      }
      else
         Debug.out("NULL strategy interpretation");
   }

   private void updateTableData(StrategyData expt,
      Oscillation_sequence sequence, int runNumber)
   {
      expt.setStartPhi(sequence.getStart());
      expt.setIncrement(sequence.getRange());
      expt.setOverlap(sequence.getOverlap());
      expt.setTime(sequence.getExposure_time());
      expt.setNImages(sequence.getNumber_of_images());
      expt.setPasses(sequence.getNumber_of_passes());
      expt.setTemplateStart(sequence.getStart_image_number());
      expt.setRunNumber(runNumber);
      expt.setResolution(edm.getCalculatedResolution().getUpper());
   }

}