package dna;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.DefaultCellEditor;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableColumn;

import dna.observable.IObservable;
import dna.observable.IObserver;
import dna.observable.ObservableComponent;
import dna.util.DecimalField;
import dna.util.LeftRenderer;
import dna.xml.Resolution;

public class StrategyDataPanel
   extends JPanel
   implements IObserver, IObservable, KeyListener
{
   private ExpertDataModel edm = ExpertDataModel.getInstance();
   private UserTableModel userTableModel = new UserTableModel();
   private DnaTableModel dnaTableModel = new DnaTableModel();
   private JTable dnaTable = null;
   private JTable userTable = null;
   private DefaultTableColumnModel dnaColumnModel = new DefaultTableColumnModel();
   private DefaultTableColumnModel strategyColumnModel = new DefaultTableColumnModel();
   private TableColumn column = null;
   private LeftRenderer leftRenderer = new LeftRenderer();
   private ObservableComponent observableComponent = new ObservableComponent();

   private static final int RESOLUTION = 5;

   public StrategyDataPanel()
   {
      dnaTable = new JTable(dnaTableModel);
      userTable = new JTable(userTableModel);
      // Ensures table updates data model when cells are exited, EVEN when
      // tabbed to!
      userTable.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);

      dnaTable.setAutoCreateColumnsFromModel(false);
      userTable.setAutoCreateColumnsFromModel(false);

      createTableColumnModel(dnaTableModel, dnaColumnModel);
      createTableColumnModel(userTableModel, strategyColumnModel);

      setLayout(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();

      c.fill = GridBagConstraints.BOTH;
      c.insets = new Insets(2, 5, 2, 5);
      c.gridy = GridBagConstraints.RELATIVE;
      add(setupPanel(dnaTable, "DNA Reference Strategy"), c);
      add(setupPanel(userTable, "Data Collection Strategy"), c);
      userTableModel.addIObserver(this);

      userTable.addKeyListener(this);
   }

   private JPanel setupPanel(JTable table, String title)
   {

      JPanel panel = new JPanel();
      panel.setLayout(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();

      table.setPreferredScrollableViewportSize(new Dimension(375, 200));
      table.setDefaultRenderer(String.class, leftRenderer);
      table.setDefaultRenderer(Double.class, leftRenderer);
      table.setDefaultRenderer(Integer.class, leftRenderer);

      c.fill = GridBagConstraints.BOTH;
      c.gridx = 1;

      c.gridy = GridBagConstraints.RELATIVE;
      c.insets = new Insets(2, 5, 2, 5);

      setUpIntegerEditor(table);
      setUpDoubleEditor(table);
      panel.add(new JLabel(title), c);
      panel.add(new JScrollPane(table), c);
      return panel;
   }

   private void createTableColumnModel(AbstractTableModel tableModel,
      DefaultTableColumnModel columnModel)
   {

      for (int modelColumnNumber = 0; modelColumnNumber < StrategyData.NUMBER_OF_COLUMNS; modelColumnNumber++)
      {
         switch (modelColumnNumber)
         {
            case StrategyData.STARTPHI:
            case StrategyData.INCREMENT:
            case StrategyData.TIME:
            case StrategyData.NIMAGES:
            case StrategyData.TEMPLATESTART:
            case StrategyData.RESOLUTION:
               column = new TableColumn(modelColumnNumber);
               column.setHeaderValue(tableModel.getColumnName(modelColumnNumber));
               columnModel.addColumn(column);
               break;
            case StrategyData.OVERLAP:
               break;
            case StrategyData.PASSES:
               if (edm.isPassesOn())
               {
                  column = new TableColumn(modelColumnNumber);
                  column.setHeaderValue(tableModel.getColumnName(modelColumnNumber));
                  columnModel.addColumn(column);
               }
               break;
            case StrategyData.RUNNUMBER:
               break;

         }
      }

      dnaTable.setColumnModel(columnModel);
      userTable.setColumnModel(columnModel);
   }

   public StrategyData getDnaData(int index)
   {
      return dnaTableModel.getStrategyData(index);
   }

   public StrategyData getStrategyData(int index)
   {
      return userTableModel.getStrategyData(index);
   }

   public void setDataArraySize(int count)
   {
      userTableModel.setDataArraySize(count);
      dnaTableModel.setDataArraySize(count);
   }

   public int getDataArraySize()
   {
      return userTableModel.getRowCount();
   }

   private void setUpDoubleEditor(JTable table)
   {
      final DecimalField doubleField = new DecimalField(0.0, 8, 1);

      DefaultCellEditor doubleEditor = new DefaultCellEditor(doubleField)
      {
         // Override DefaultCellEditor's getCellEditorValue method
         // to return a Double, not a String:
         public Object getCellEditorValue()
         {
            Double d = null;
            try
            {
               String contents = doubleField.getText();
               if (contents.equals(""))
               {
                  userTable.editingCanceled(null);
               }
               else
               {
                  d = new Double(contents);
               }
            }
            catch (Exception e)
            {
               userTable.editingCanceled(null);
            }
            return d;
         }

         public boolean stopCellEditing()
         {
            return super.stopCellEditing();
         }
      };

      table.setDefaultEditor(Double.class, doubleEditor);
   }

   private void setUpIntegerEditor(JTable table)
   {
      final DecimalField integerField = new DecimalField(8, 8, 1);
      DefaultCellEditor integerEditor = new DefaultCellEditor(integerField)
      {
         // Override DefaultCellEditor's getCellEditorValue method
         // to return an Integer, not a String:
         public Object getCellEditorValue()
         {
            Integer i = null;
            try
            {
               String contents = integerField.getText();
               if (contents.equals(""))
               {
                  userTable.editingCanceled(null);
               }
               else
               {
                  i = new Integer(contents);
               }
            }
            catch (Exception e)
            {
               userTable.editingCanceled(null);
            }
            return i;
         }

         public boolean stopCellEditing()
         {
            return super.stopCellEditing();
         }
      };

      table.setDefaultEditor(Integer.class, integerEditor);
   }

   private void initColumnSizes(JTable table, StrategyTableModel model)
   {
      TableColumn column = null;
      Component comp = null;
      int headerWidth = 0;
      int cellWidth = 0;

      for (int i = 0; i < StrategyData.getColumnCount(); i++)
      {
         column = table.getColumnModel().getColumn(i);

         try
         {
            comp = column.getHeaderRenderer().getTableCellRendererComponent(
               null, column.getHeaderValue(), false, false, 0, 0);
            headerWidth = comp.getPreferredSize().width;
         }
         catch (NullPointerException e)
         {
         }

         cellWidth = 8;
         column.setPreferredWidth(Math.max(headerWidth, cellWidth));
      }
   }

   private Resolution refreshResolution()
   {
      double tolerance = 0.005;
      Resolution resolution = null;

      for (int i = 0; i < userTableModel.getDataArraySize(); i++)
      {
         resolution = edm.getCalculatedResolution();
         StrategyData data = userTableModel.getStrategyData(i);
         if (Math.abs(resolution.getUpper() - (data.getValueOfResolution())) > tolerance)
         {
            resolution.setUpper(data.getValueOfResolution());
            break;
         }
      }
      return resolution;
   }

   // IObservable

   public void addIObserver(IObserver anIObserver)
   {
      observableComponent.addIObserver(anIObserver);
   }

   public void deleteIObserver(IObserver anIObserver)
   {
      observableComponent.deleteIObserver(anIObserver);
   }

   public void deleteIObservers()
   {
      observableComponent.deleteIObservers();
   }

   // IObserver

   public void update(Object iObservable, Object arg)
   {
      if (arg instanceof ExpertState)
      {
         ExpertState state = (ExpertState) arg;

         switch (state.getState())
         {
            case ExpertState.CALCULATING_STRATEGY:
            {
               /*
                * Someone has altered the resolution field for the strategy. The
                * following flag ensures that this is passed on as part of the
                * strategy request, rather than being cleared.
                */
               edm.setClearStrategyResolution(false);
               setDataArraySize(0);
               break;
            }
            case ExpertState.COLLECTING_REFERENCE_IMAGES:
            case ExpertState.INDEXING:
            case ExpertState.CHARACTERIZING:
            {
               edm.setClearStrategyResolution(true);
               setDataArraySize(0);
               break;
            }
         }
      }
      if (arg instanceof Double)
      {
         Resolution resolution = refreshResolution();
         observableComponent.notifyIObservers(this, resolution);
      }
   }

   // KeyListener

   public void keyPressed(KeyEvent e)
   {
   }

   public void keyReleased(KeyEvent e)
   {
   }

   public void keyTyped(KeyEvent e)
   {
      if (userTable.getSelectedColumn() == RESOLUTION)
      {
         observableComponent.notifyIObservers(this, null);
      }
   }
}