package dna;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Iterator;
import java.util.Vector;

import dna.util.Debug;

public class ExpertClientHTTPServer
   implements Runnable
{
   private static ExpertClientHTTPServer httpGUIServer = null;
   private static ExpertClientHTTPServer httpBCMServer = null;
   private Vector expertListeners = new Vector();
   private static ExpertDataModel edm = ExpertDataModel.getInstance();
   private int serverPort;

   private static int bufferLength = 15000;
   private static char[] buffer = new char[bufferLength];
   private Socket socket;

   private ExpertClientHTTPServer(int serverPort)
   {
      this.serverPort = serverPort;
      new Thread(this).start();
   }

   public static synchronized ExpertClientHTTPServer getGUIServer()
   {
      if (httpGUIServer == null)
      {
         httpGUIServer = new ExpertClientHTTPServer(edm.getGUIPortNumber());
      }

      return httpGUIServer;
   }

   public static synchronized ExpertClientHTTPServer getBCMServer()
   {
      if (httpBCMServer == null)
      {
         httpBCMServer = new ExpertClientHTTPServer(edm.getBCMPortNumber());
      }

      return httpBCMServer;
   }

   private static synchronized ExpertXMLMessage extractXMLString(
      BufferedReader reader) throws IOException
   {
      String headerLine, xml_message = null, path = null;
      int bytes = 0, pos;
      ExpertXMLMessage expertXMLMessage = null;

      while (!((headerLine = reader.readLine()).equals("")))
      {
         Debug.out("Header => " + headerLine, 2);
         if (headerLine.startsWith("POST"))
         {
            path = headerLine.substring(5);
            pos = path.indexOf(' ');
            path = path.substring(0, pos);
            Debug.out("Path " + headerLine + " => **" + path + "**", 2);
         }
         else if (headerLine.startsWith("Content-Length")
            || headerLine.startsWith("Content-length"))
         {
            bytes = Integer.parseInt(headerLine.substring(16));
            Debug.out("Header " + headerLine + " => byte read " + bytes, 2);
         }
      }

      if (bytes > bufferLength)
      {
         bufferLength = bytes;
         buffer = new char[bufferLength];
      }

      reader.read(buffer, 0, bytes);
      xml_message = new String(buffer).substring(0, bytes);
      // Replace encoded newlines.
      xml_message = xml_message.replaceAll("CRLF", "\n");
      Debug.out("xml_message: " + xml_message, 2);
      Debug.out("xml_message.len: " + xml_message.length(), 2);

      if (path != null && xml_message != null)
      {
         expertXMLMessage = new ExpertXMLMessage(path, xml_message);
      }

      return expertXMLMessage;
   }

   public void run()
   {
      ServerSocket serverSocket;
      BufferedReader reader;
      String xml_message;

      try
      {
         serverSocket = new ServerSocket(serverPort);
      }
      catch (IOException ex)
      {
         Debug.out("Fatal error in ExpertClientHTTPServer - "
            + "failed to bind socket on port " + serverPort, 2);
         return;
      }

      while (true)
      {
         try
         {
            socket = serverSocket.accept();
            reader = new BufferedReader(new InputStreamReader(
               socket.getInputStream()));

            ExpertXMLMessage expertXMLMessage = extractXMLString(reader);

            if (expertXMLMessage != null)
            {
               xml_message = expertXMLMessage.getXMLMessage();
               if (!isSynchronous(xml_message))
               {
                  String response = "<?xml version=\"1.0\"?><status><code>ok</code></status>";
                  replyToExpert(response);
               }

               this.fireExpertXMLMessage(expertXMLMessage);
            }
            else
            {
               String response = "<status>Error</status>";
               replyToExpert(response);
            }

            reader.close();
         }
         catch (IOException ioex)
         {
            System.out.println("IOException caught in "
               + "ExpertClientHTTPServer");
         }
      }
   }

   public boolean isSynchronous(String xml_message)
   {
      boolean synchronous = false;
      int start, end;
      String content;

      xml_message = xml_message.toLowerCase();

      if ((start = xml_message.indexOf("<synchronous>")) > -1)
      {
         if ((end = xml_message.indexOf("</synchronous>")) > -1)
         {
            content = xml_message.substring((start + 13), end);
            if (content.indexOf("true") > -1)
               synchronous = true;
         }
      }

      return synchronous;
   }

   public void replyToExpert(String message)
   {
      try
      {
         BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
            socket.getOutputStream()));
         writer.write("HTTP/1.0 200 OK\n");
         writer.write("Host: " + edm.getExpertSystemHostName() + "\n");
         writer.write("Content-Type: text/xml\n");
         writer.write("Content-Length: " + message.length() + "\n\n");
         writer.write(message);
         writer.newLine();
         writer.flush();
      }
      catch (IOException ioex)
      {
         System.out.println("IOException caught in " + "ExpertClientHTTPServer");
      }
   }

   public synchronized void addExpertListener(ExpertListener expertListener)
   {
      if (!expertListeners.contains(expertListener))
         expertListeners.add(expertListener);
   }

   public synchronized void removeExpertListener(ExpertListener expertListener)
   {
      boolean tmp = expertListeners.remove(expertListener);
   }

   public synchronized void fireExpertXMLMessage(ExpertXMLMessage the_message)
   {
      Iterator iterator = expertListeners.iterator();
      while (iterator.hasNext())
      {
         ExpertListener expertListener = (ExpertListener) iterator.next();
         expertListener.expertMessage(the_message);
      }
   }
}
