package dna;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;

import dna.observable.IObservable;
import dna.observable.IObserver;
import dna.observable.ObservableComponent;
import dna.util.LocalProperties;
import dna.xml.Beam;
import dna.xml.Detector;
import dna.xml.Fileinfo;
import dna.xml.Index_request;
import dna.xml.Index_response;
import dna.xml.Resolution;
import dna.xml.Target;
import dna.xml.types.Spacegroup;

public class AutoIndexPanel
   extends JPanel
   implements IObservable, IObserver, ActionListener, Runnable, MouseListener
{
   private ExpertDataModel edm = ExpertDataModel.getInstance();

   private AutoIndexHelper autoIndex = AutoIndexHelper.getInstance();
   private Index_request index_request = new Index_request();
   private Fileinfo fileinfo = new Fileinfo();
   private DefaultValues collectionParametersPanel = new DefaultValues();
   private Beam beam = new Beam();
   private Target target = new Target();
   private Detector detector = new Detector();

   private AutoIndexSolution solutionPanel = AutoIndexSolution.getReference();
   private JButton start = new JButton("Start Auto Index and Integrate");

   private BasicFileData fileData;
   private String down1File = LocalProperties.get("dna.down1");
   private String down2File = LocalProperties.get("dna.down2");
   private JLabel down1 = null;
   private JLabel down2 = null;

   private boolean fileEditable = false;
   private boolean collectionEditable = false;
   private JPopupMenu popup;
   private JCheckBox[] checkBox;
   private int popupOffset = -20;

   private ObservableComponent observableComponent = new ObservableComponent();

   public AutoIndexPanel()
   {
      String indexFilesEditable = LocalProperties.get("dna.indexFilesEditable");
      if (indexFilesEditable != null && indexFilesEditable.equals("true"))
         fileEditable = true;
      fileData = new BasicFileData("Source Reference Images", fileEditable);

      String collectionParametersEditable = LocalProperties.get("dna.collectionParametersEditable");
      if (collectionParametersEditable != null
         && collectionParametersEditable.equals("true"))
         collectionEditable = true;

      createPopup();
      addMouseListener(this);
      popup.addMouseListener(this);

      edm.addIObserver(this);

      GridBagConstraints c = new GridBagConstraints();
      setLayout(new GridBagLayout());

      c.gridx = 1;
      c.gridy = GridBagConstraints.RELATIVE;
      c.insets = new Insets(0, 5, 0, 5); // Top, Left, Bottom, Right
      c.fill = GridBagConstraints.HORIZONTAL;
      c.weightx = 1.0;

      add(createTopPanel(), c);
      add(createButtonPanel(), c);
      add(solutionPanel, c);
   }

   private JPanel createButtonPanel()
   {
      boolean guideButtons = false;
      JPanel buttonPanel = new JPanel();
      GridBagConstraints c = new GridBagConstraints();
      buttonPanel.setLayout(new GridBagLayout());

      if (down1File != null && down2File != null)
      {
         guideButtons = true;
         down1 = new JLabel(new ImageIcon(down1File));
         down2 = new JLabel(new ImageIcon(down2File));
      }

      c.gridx = GridBagConstraints.RELATIVE;
      c.gridy = 1;
      c.insets = new Insets(5, 5, 5, 5); // Top, Left, Bottom, Right
      c.fill = GridBagConstraints.HORIZONTAL;
      c.weightx = 1.0;

      if (guideButtons)
      {
         buttonPanel.add(Box.createGlue(), c);

         c.insets = new Insets(2, 5, 15, 5); // Top, Left, Bottom, Right
         c.weightx = 0.0;
         buttonPanel.add(down1, c);

         c.insets = new Insets(2, 5, 2, 5); // Top, Left, Bottom, Right
      }

      start.setActionCommand("Start");
      start.addActionListener(this);
      buttonPanel.add(start, c);

      if (guideButtons)
      {
         c.weightx = 0.0;
         c.insets = new Insets(15, 5, 2, 5); // Top, Left, Bottom, Right
         buttonPanel.add(down2, c);

         c.insets = new Insets(2, 5, 2, 5); // Top, Left, Bottom, Right
         c.weightx = 1.0;
         buttonPanel.add(Box.createGlue(), c);
      }

      return buttonPanel;
   }

   private JPanel createTopPanel()
   {
      JPanel topPanel = new JPanel();
      GridBagConstraints c = new GridBagConstraints();

      topPanel.setLayout(new GridBagLayout());

      c.gridx = GridBagConstraints.RELATIVE;
      c.gridy = 1;
      c.gridwidth = 3;
      c.insets = new Insets(0, 5, 0, 5); // Top, Left, Bottom, Right
      c.fill = GridBagConstraints.HORIZONTAL;
      c.weightx = 1.0;

      topPanel.add(fileData, c);
      c.gridwidth = 1;
      c.weightx = 0.0;
      topPanel.add(collectionParametersPanel, c);
      setupCollectionParameterEdits();

      return topPanel;
   }

   private void createPopup()
   {
      popup = new JPopupMenu();
      checkBox = new JCheckBox[2];

      checkBox[0] = new JCheckBox("Allow File Edits", fileEditable);
      checkBox[1] = new JCheckBox("Enter Collection Parameters",
         collectionEditable);
      popup.add(checkBox[0]);
      popup.add(checkBox[1]);

      checkBox[0].addActionListener(new ActionListener()
      {
         public void actionPerformed(ActionEvent ev)
         {
            fileData.setEditable(checkBox[0].isSelected());
         }
      });

      checkBox[1].addActionListener(new ActionListener()
      {
         public void actionPerformed(ActionEvent ev)
         {
            collectionEditable = checkBox[1].isSelected();
            setupCollectionParameterEdits();
         }
      });
   }

   private void setupCollectionParameterEdits()
   {
      collectionParametersPanel.setVisible(collectionEditable);
   }

   // IObservable

   public void addIObserver(IObserver anIObserver)
   {
      observableComponent.addIObserver(anIObserver);
   }

   public void deleteIObserver(IObserver anIObserver)
   {
      observableComponent.deleteIObserver(anIObserver);
   }

   public void deleteIObservers()
   {
      observableComponent.deleteIObservers();
   }

   // IObserver

   public void update(Object iObservable, Object arg)
   {
      if (arg instanceof ExpertState)
      {
         ExpertState state = (ExpertState) arg;
         switch (state.getState())
         {
            case ExpertState.IDLE:
            case ExpertState.ABORTED:
               start.setEnabled(true);
               break;
            case ExpertState.COLLECTING_REFERENCE_IMAGES:
            case ExpertState.INDEXING:
            case ExpertState.CALCULATING_STRATEGY:
            case ExpertState.COLLECTING_DATA:
            case ExpertState.CHARACTERIZING:
               start.setEnabled(false);
               break;
         }
      }
      else if (arg instanceof ExpertDataModel)
      {
         fileData.update();
      }
   }

   public synchronized void updateSolutionPanel(Index_response index_response)
   {
      solutionPanel.update(index_response);
   }

   // ActionListener

   public void actionPerformed(ActionEvent ae)
   {
      if (ae.getActionCommand().equals("Start"))
      {
         try
         {
            if (solutionPanel.hasSymetryChanged())
            {
               solutionPanel.clear(AutoIndexSolution.CELL);
            }
            else
            {
               solutionPanel.clear(AutoIndexSolution.ALL);
            }
            edm.setFileinfo(fileData.getDirectory(), fileData.getPrefix(),
               Integer.parseInt(fileData.getRunNumber()), fileData.getSuffix());
         }
         catch (NumberFormatException nef)
         {
         }

         observableComponent.notifyIObservers(this, new ExpertState(
            ExpertState.INDEXING));

         fileinfo.setDirectory(edm.getDirectory());
         fileinfo.setRun_number(edm.getValueOfRunNumber());
         fileinfo.setSuffix(edm.getSuffix());
         String prefix = edm.getPrefix();
         if (edm.isPrefixingOn())
            fileinfo.setPrefix(edm.getPrefixPrefix() + prefix);
         else
            fileinfo.setPrefix(prefix);
         index_request = new Index_request();
         index_request.setFileinfo(fileinfo);

         if (edm.getSpaceGroup() != null)
            target.setSymmetry(Spacegroup.valueOf(edm.getSpaceGroup()));
         else
            target.setSymmetry(Spacegroup.valueOf("unknown"));
         index_request.setTarget(target);
         index_request.addImage(1);
         index_request.addImage(2);

         if (collectionEditable)
         {
            if (collectionParametersPanel.saveDataModel())
            {
               index_request.setDetector(edm.getDetector());
               index_request.setBeam(edm.getBeam());
               new Thread(this).start();
               start.setEnabled(false);
            }
            else
            {
               observableComponent.notifyIObservers(this, new ExpertState(
                  ExpertState.IDLE));
            }
         }
         else
         {
            new Thread(this).start();
            start.setEnabled(false);
         }
      }
   }

   // Runnable

   public void run()
   {
      try
      {
         observableComponent.notifyIObservers(this, new ExpertState(
            ExpertState.INDEXING));
         Index_response index_response = autoIndex.sendIndex_requestSynchronous(
            index_request, edm.getAutoIndexTimeout());
         updateSolutionPanel(index_response);
         Resolution resolution = index_response.getCalculated_resolution();
         if (resolution != null)
            edm.setCalculatedResolution(resolution);
      }
      catch (IndexException ex)
      {
         MessagePanel messagePanel = MessagePanel.getInstance();
         messagePanel.appendText("Error caught in AutoIndexing");
         messagePanel.appendText(ex.getMessage());
         new ErrorDialog(this, ex, "AutoIndex");
      }
      finally
      {
         observableComponent.notifyIObservers(this, new ExpertState(
            ExpertState.IDLE));
      }
   }

   // MouseListener

   public void mousePressed(MouseEvent me)
   {
      if (me.isPopupTrigger())
         popup.show(this, me.getPoint().x + popupOffset, me.getPoint().y
            + popupOffset);
   }

   public void mouseReleased(MouseEvent me)
   {
      if (me.isPopupTrigger())
         popup.show(this, me.getPoint().x + popupOffset, me.getPoint().y
            + popupOffset);
   }

   public void mouseClicked(MouseEvent me)
   {
      if (me.isPopupTrigger())
         popup.show(this, me.getPoint().x + popupOffset, me.getPoint().y
            + popupOffset);
   }

   public void mouseEntered(MouseEvent me)
   {
   }

   public void mouseExited(MouseEvent me)
   {
   }
}
