#!/usr/bin/env python
# Expert.py
# Maintained by G.Winter
# A component of the Expert system for DNA.
# 20th January 2004
# 
# This part is the beginnings of the expertise for DNA.
# 
# 
# 
# $Id: Expert.py,v 1.58 2006/02/23 12:06:04 gwin Exp $

# system imports
import os, sys, math

# DNA related imports
if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

dna = os.environ['DNAHOME']

if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 must be defined'

sys.path.append(dna + '/xsd/python')

import XSD

# this is small bits of expertise - more like checks really..
import Checks

import Crystallographer, Vector, Statistics

from ExpertException import ExpertException

# ConsiderIndexResponses
# This will look at the differences between the combined response and the 
# 'n' individual responses. This is looking for characteristic differences
# for instance if all 'n' agree, but disagree with the combined, this 
# suggests that the rotation is "going wrong."

# add the Messenger system into here...

sys.path.append(dna + '/scheduler/Scheduler/Mosflm')
from Messenger import *
from Somewhere import Somewhere

sys.path.append(dna + '/scheduler/StatisticalTools/lib')

import Outlier

# get hold of the ES configuration proxy
from SchedulerDNAConfig import *

def ConsiderIndexResponses(index_response_dict):
    '''A method to compare a dictionary of index responses indexed by the
    image number from which they were determined, and -1 for the
    result of the combined indexing'''
    
    for key in index_response_dict.keys():
        if index_response_dict[key].name != 'index_response':
            raise ExpertException, 'argument in list not an index_response'

    image_list = index_response_dict.keys()

    # sort the list, so that I can separate out the results of the indexing
    # of single images (key >= 0) from the combined result (key = -1)
    image_list.sort()

    # check for warnings or errors from the single image indexings
    for i in image_list[1:-1]:
        if index_response_dict[i].getStatus().getCode() == 'error':
            # then return this as an error
            return index_response_dict[i]


    # next look at the lattices selected for the individual indexings -
    # if these all agree BUT the overall disagrees this suggests that the
    # rotation angles are bad - could also look at the reciprocal lattice
    # volume from indexing

    consensus = index_response_dict[image_list[1]].getSolution().getSymmetry()
    for i in image_list[1:-1]:
        other = index_response_dict[i].getSolution().getSymmetry()
        if consensus != other:
            Messenger.log_write("Solutions from %d and %d disagree" % \
                                (image_list[1], i))
            consensus = ''

    if consensus:
        if index_response_dict[-1].getSolution().getSymmetry() != consensus:
            Messenger.log_write("All individual image solutions agree " + \
                                "but the combined does not")
            Messenger.log_write("Check the rotations are correct")
                                

    combined = index_response_dict[-1]
    combinedP1 = combined.getLattice_character_response().getLattice_character()[0].getCell()

    # perform some analysis of the data - according to the DNA
    # specifications - this requires getting the parameters from the
    # ES through the proxy

    user_defaults = SchedulerDNAConfig.getUser_defaults()
    if user_defaults:
        Messenger.log_write("Obtained DNA configuration", 2)
    
        # analyse the combined result
        index_response = index_response_dict[-1]
    
        Messenger.log_write("Begin interpretation", 2)
        status = index_response_interpretation(user_defaults.getIndex_parameters(), index_response)
    
        for s in status:
            if s.getCode() == 'error':
                # cascade this error back!
                
                Messenger.log_write('Error found')
                Messenger.log_write(s.getMessage())
                
                index_response = XSD.Index_response()
                index_response.setStatus(s)
                return index_response
    
    
    else:
        Messenger.log_write("Failed to obtain DNA configuration", 2)

        # also, attempt to get some user parameters, and if successful
        # use these to decide about the quality of the results

        try:
            user_defaults = XSD.User_defaults()
            user_defaults_xml = open(os.environ['HOME'] + '/.dna/user_defaults.xml').read()
            user_defaults.unmarshal(user_defaults_xml)
            Messenger.log_write('Picked up used defaults from %s/.dna/user_defaults.xml' % os.environ['HOME'], 2)
            index_parameters = user_defaults.getIndex_parameters()

            # analyse the combined result
            index_response = index_response_dict[-1]
            
            Messenger.log_write("Begin interpretation", 2)
            status = index_response_interpretation(index_parameters, \
                                                   index_response)

            for s in status:
                if s.getCode() == 'error':
                    # cascade this error back!

                    Messenger.log_write('Error found')
                    Messenger.log_write(s.getMessage())
                    
                    index_response = XSD.Index_response()
                    index_response.setStatus(s)
                    return index_response
        except:

            Messenger.log_write('No user defaults found in ~/.dna/user_defaults.xml', 2)
        

    # at this stage check that there are no errors BEFORE going on to the
    # comparison stuff - just return "combined"

    for key in index_response_dict.keys():
        if index_response_dict[key].getStatus().getCode() == 'error':
            Messenger.log_write('Error reported for indexing of %s' % str(key))
            Messenger.log_write('Stopping solution analysis - just using ' \
                                + 'combined (all images) indexing solution')
            return combined

    


    # see if this is anything commonplace
    Checks.NameThatProtein(combined.getSolution().getOrientation().getCell())

    mosaic = { }

    groovy = 1

    # n (n - 1) / 2 check
    for i in range(1, len(image_list)):
        solution_i = index_response_dict[image_list[i]]
        p1_i = solution_i.getLattice_character_response().getLattice_character()[0].getCell()
        for j in range(1, i):
            if i != j:
                Messenger.log_write('Comparing solutions from '  + \
                                    str(image_list[i]) + \
                                    ' and ' + str(image_list[j]), 2)
                
                solution_j = index_response_dict[image_list[j]]
                p1_j = solution_j.getLattice_character_response().getLattice_character()[0].getCell()
                if Checks.CompareCells(p1_i, p1_j) == 1:
                    groovy = 0
                    Messenger.log_write('Incompatible solutions for ' + \
                                        str(image_list[i]) + \
                                        ' and ' + str(image_list[j]))

                # compare orientation matrices
                matrix_i = solution_i.getSolution().getOrientation().getA_matrix()
                matrix_j = solution_j.getSolution().getOrientation().getA_matrix()
                if Checks.DotAMatrix(matrix_i, matrix_j) == 1:
                    Messenger.log_write("Incompatible orientation matrices", 2)
                else:
                    Messenger.log_write("Orientation matrices agree", 2)
                


    # now check the 'n' with the overall
    for i in range(1, len(image_list)):
        Messenger.log_write('Comparing unit cells between overall and image ' + \
                            str(image_list[i]), 2)
        solution = index_response_dict[image_list[i]]

        mosaic[image_list[i]] = solution.getMosaicity_value()
        
        solutionP1 = solution.getLattice_character_response().getLattice_character()[0].getCell()
        if Checks.CompareCells(combinedP1, solutionP1) == 1:
            Messenger.log_write('Solutions incompatible between ' + \
                                'all images and ' + \
                                'image ' + str(image_list[i]), 2)
            if groovy == 1:
                Messenger.debug_write('Check the rotation - ' + \
                                      'perhaps the phi value has ' + \
                                      'not been registered')

    # by default return just the combined result, on the logic that this
    # will be the best

    if groovy == 1:
        Messenger.log_write('Solution comparson passed, nicely', 2)

    if Checks.MosaicSpread(mosaic) == 1:
        Messenger.log_write('Mosaic spreads varied by > 25%', 2)
    else:
        Messenger.log_write('Mosaic spreads varied by < 25%', 2)
    
    return combined

def ConsiderStrategyResponse(index_response, strategy_response):
    '''Digest the strategy response into something a little more manageable'''
    Messenger.log_write("Optimum rotation gives %4.2f%% of unique data."%(strategy_response.getCompleteness().getStandard()))
    Messenger.log_write("%9s%9s%8s%10s%10s%10s" % \
                     ("Phi start","Phi end","no images", \
                      "osc angle", "% overlaps",  "% fulls"))
    osc_start = None
    osc_end = None
    osc_range = None
    for segment in strategy_response.getStrategy_summary()[0].getSegment():
        start = segment.getOscillation_sequence().getStart()
        no_images = segment.getOscillation_sequence().getNumber_of_images()
        range = segment.getOscillation_sequence().getRange()
        end = start + no_images*range
        if osc_start is None:
            osc_start = start
            osc_end = end
            osc_range = range
        if start < osc_start:
            osc_start = start
        if end > osc_end:
            osc_end = end
        if range < osc_range:
            osc_range = range
        full = segment.getPredicted_spots().getFull()
        overlaps = segment.getPredicted_spots().getOverlap()
        Messenger.log_write("%9.1f%9.1f%8d%10.2f%10.1f%10.1f"%(start, end, no_images, range, overlaps, full))
    
    segment = strategy_response.getSegment()[0]
    start = segment.getOscillation_sequence().getStart()
    no_images = segment.getOscillation_sequence().getNumber_of_images()
    range = segment.getOscillation_sequence().getRange()
    
    end = start + range
    no_images = int(float(range) / float(osc_range) + 0.5)
    osc_start = start
    osc_end = end
    
    if index_response.getMosaicity_value() < 0.4:
        if osc_range > 0.5:
            osc_range = 0.5
        else:
            pass
    else:
        if osc_range > 1.0:
            osc_range = 1.0
    osc_number_of_images = int((osc_end-osc_start)/osc_range+0.9999999999999)
    Messenger.log_write("Suggested strategy: start = %6.2f, end = %6.2f"%(osc_start, osc_end))
    Messenger.log_write("Suggested strategy: oscillation range = %6.2f, number of images = %d"%(osc_range, osc_number_of_images))
    strategy_interpretation = XSD.Strategy_interpretation() 
    oscillation_sequence = XSD.Oscillation_sequence()
    oscillation_sequence.setStart(osc_start)
    oscillation_sequence.setRange(osc_range)
    oscillation_sequence.setNumber_of_images(osc_number_of_images)
    oscillation_sequence.setOverlap(0.0)
    # get the exposure time from where?
    header = Somewhere.get('last header')
    if header:
        exposure_time = header['ExposureTime']
    else:
        exposure_time = 1.0
    oscillation_sequence.setExposure_time(exposure_time)
    oscillation_sequence.setStart_image_number(1)
    oscillation_sequence.setNumber_of_passes(1)
    strategy_interpretation.addOscillation_sequence(oscillation_sequence)

    # Return the strategy_response

    status = XSD.Status()
    status.setCode("ok")      
    strategy_response2 = XSD.Strategy_response()

    # copy across the competeness etc

    strategy_response2.setCompleteness(
        strategy_response.getCompleteness())
    
    strategy_response2.setStrategy_interpretation(strategy_interpretation)
    strategy_response2.setStatus(status)

    return strategy_response2

def ConsiderBestStrategy(mosflm_strategy, best_strategy):
    '''Combine the results from the Mosflm strategy and the BEST strategy'''

    Messenger.log_write('Now combining data collection strategies', 2)

    segment = best_strategy.getSegment()[0]

    wedge = segment.getOscillation_sequence()
    width = float(wedge.getRange()) / float(wedge.getNumber_of_images())

    interpretation = mosflm_strategy.getStrategy_interpretation()
    oscillations = interpretation.getOscillation_sequence()[0]
    best_exposure_time = float(Somewhere.get('BEST exposure time'))
    original_exposure_time = oscillations.getExposure_time()

    new_exposure_time = best_exposure_time * \
                        float(oscillations.getRange()) / \
                        width

    # changed the definition so that the new time has to be both small
    # and much smaller than before...
    
    if new_exposure_time < 0.25 * original_exposure_time and \
           new_exposure_time < 1.000:
        Messenger.log_write('BEST has suggested a very small exposure time')
        Messenger.log_write('Sticking to the original exposure time of %3.2f' \
                            % original_exposure_time)
        new_exposure_time = original_exposure_time

    oscillations.setExposure_time(new_exposure_time)
    interpretation.clearOscillation_sequence()
    interpretation.addOscillation_sequence(oscillations)
    strategy_response = mosflm_strategy
    strategy_response.setStrategy_interpretation(interpretation)
    return strategy_response


def ConsiderIntegrateResponses(response_dict):
    '''Combine the results from many integrate responses'''
    integrate_response = XSD.Integrate_response()

    status = XSD.Status()
    status.setCode('ok')
    integrate_response.setStatus(status)

    total_images = 0
    total_images_bad_spots = 0

    total_bad_spots = 0
    total_spots = 0

    overall_intensity = []

    for i in response_dict:
        # I do not think that this trap is correct!!!
        if i.getStatus().getCode() != 'ok':
            status.setCode('error')
            status.setMessage(i.getStatus().getMessage())
            integrate_response.setStatus(status)
            return integrate_response

        if not i.getIntegrated_image():
            status.setCode('error')
            status.setMessage('No images were integrated!')
            integrate_response.setStatus(status)
            return integrate_response

        for j in i.getIntegrated_image():
            total_images += 1
            # add the resolution limit of the integrated image
            resolution = XSD.Resolution()
            r = EstimateResolution(j)
            if r == 0.0:
                status.setCode("warning")
                status.setMessage("image " + str(j) + " had resolution 0")
                integrate_response.setStatus(status)
            u = float(r)
            resolution.setUpper(u)
            resolution.setLower(0)
            j.setResolution(resolution)
            integrate_response.addIntegrated_image(j)

            # now look for bad spots
            spots = j.getIntegration_summary().getSpot_information()
            signal = j.getIntegration_summary().getOverall_signal()

            # make this only available for "verbose" output now...
            Messenger.log_write('Overall I/sig for image %d is %3.1f' % \
                                (j.getImage(), signal), 2)
            
            overall_intensity.append((j.getImage(), signal))
            bad_spots = spots.getBad_spots()
            partial = spots.getPartial_spots()
            full = spots.getFull_spots()
            fraction = float(bad_spots) / float(full + partial)
            total_spots += (full + partial)
            total_bad_spots += bad_spots
            if fraction > 0.1:
                Messenger.log_write(
                    'Image %d had a high fraction of bad spots' % \
                    j.getImage())
            if fraction > 0.05:
                # count this image
                total_images_bad_spots += 1

    intensity_list = []
    for o in overall_intensity:
        intensity_list.append(o[1])

    outliers = Outlier.Outlier(intensity_list)
    if len(outliers) == 0:
        Messenger.log_write('Looking for duff images - none found!', 2)
    else:
        Messenger.log_write('Looking for duff images:')
        duff_batches = []
        for o in outliers:
            Messenger.log_write('Image %d' % overall_intensity[o][0])
            duff_batches.append(overall_intensity[o][0])
        Somewhere.store('duff_batches', duff_batches)

    # at this stage I need to analyse the "overall_intensity" array
    # for signs of deterioration in the crystal - which I am pretty
    # sure are there. could block the data in say batches of ten
    # and see if the average falls off...

    Messenger.log_write('%d out of %d images had "too many" bad spots' % \
                        (total_images_bad_spots, total_images), 2)

    Messenger.log_write('That is %d spots out of %d were bad' % \
                        (total_bad_spots, total_spots), 2)

    # this can't give / 0.0 error otherwise indexing would have failed.
    fraction_bad = float(total_bad_spots) / float(total_spots)

    if fraction_bad > 0.33:
        # something is wrong if 1/3 of spots are bad!
        integrate_response = XSD.Integrate_response()

        status = XSD.Status()
        status.setCode('error')
        status.setMessage('Integration error: high fraction of spots (%3.2f)'
                          % fraction_bad)
        integrate_response.setStatus(status)
        return integrate_response

    # change here to ignore resolutions of 0.0

    bestResolution = 100.0

    # next need to build a model of the mosaic spread

    mosaic = { }
    image_list = [ ]

    for i in integrate_response.getIntegrated_image():
        image_list.append(i.getImage())
        mosaic[i.getImage()] = i.getRefined_mosaic_spread()
        
        r = EstimateResolution(i)
        if r < bestResolution and r > 0.0:
            bestResolution = r

    image_list.sort()
    mosaic_list = [ ]
    for i in image_list:
        mosaic_list.append(mosaic[i])

    # probably the best way to analyse the mosaic spread would be to
    # fit a sinusoidal function to it - since I would expect that this
    # varies with theta
    # although I don't know how many images you need to make 360 degrees -
    # could this information be in the integrated image result?

    # another (possibly better) option would be o smooth this data with
    # some kind of spline or smoothing function, and then just look at the
    # extremes.

    # first just look at the raw extremes!

    mosaic2 = mosaic_list
    mosaic2.sort()

    Messenger.log_write('Extremes of mosaic spread %f %f' % \
                        (mosaic2[1], mosaic2[-1]), 2)

    if bestResolution >= 100.0:
        status = XSD.Status()
        status.setCode('error')
        status.setMessage('No resolution found!')
        integrate_response.setStatus(status)

    resolution = XSD.Resolution()
    resolution.setLower(0.0)
    resolution.setUpper(bestResolution)

    integrate_response.setCalculated_resolution(resolution)
    
    return integrate_response

def ConsiderScaleReflectionsResponse(scale_reflections_response):

    scales = scale_reflections_response.getScale_factor_list()

    data = { }
    xvals = []
    for scale in scales.getScale_factor():
        x = float(scale.getBatch())
        y = float(scale.getMean_k())
        xvals.append(x)
        data[x] = y

    # now compute a best fit to these
    xbar = 0
    ybar = 0
    
    for x in xvals:
        xbar += x
        ybar += data[x]
        
    xbar /= len(xvals)
    ybar /= len(xvals)
    
    sumxx = 0
    sumxy = 0
    sumyy = 0
    
    for x in xvals:
        sumxx += (x - xbar) * (x - xbar)
        sumxy += (x - xbar) * (data[x] - ybar)

    m = sumxy / sumxx
    c = ybar - (xbar * m)

    # note here that I have normalized by the average scale factor - to get
    # the deviation on a fixed scale.
    
    # From here, what should I measure? The good (trypsin) and bad (lysozyme)
    # data clearly differ - maybe an RMS deviation from the linear best fit
    # (R value?) is the best way to go...
    #
    # Would knowing an idea of the errors on the scale factors help much?
    # probably not...
    
    rms = 0
    for x in xvals:
        rms += ((data[x] - (m * x + c)) / ybar) * \
               ((data[x] - (m * x + c)) / ybar)

    rms /= len(xvals)

    rms = math.sqrt(rms)

    # this value appears to be 0.010 for "good" data and 0.177 for "bad" data
    # can I covert this into a real value?

    Messenger.log_write('RMS scaling variation was %3.2f' % rms, 2)
    rmerge = float(scale_reflections_response.getRmerge())
    Messenger.log_write('The overall RMerge was %3.2f' % rmerge, 2)
    if rmerge < 0.10:
        Messenger.log_write('This is probably OK', 2)
    else:
        Messenger.log_write('This is rather high - check the space group', 2)


    # now look at Rmerge as a function of batch - this should be rather
    # interesting...

    per_batch_results = scale_reflections_response.getPer_batch_results()
    batch_results = per_batch_results.getBatch_results()
    rmerge_dict = { }
    for b in batch_results:
        rmerge_dict[b.getBatch()] = b.getRmerge()
    batches = rmerge_dict.keys()
    batches.sort()

    # analysis - look to find a limit for which it is a not false
    # hypothesis that the average rmerge is a fixed value

    Messenger.log_write('Batches in scaling [%d, %d]' % \
                        (batches[0], batches[-1]), 2)

    rmerges = []
    for b in batches:
        rmerges.append(rmerge_dict[b])

    # I want to use a chi squared test on the mean value

    quarter = len(rmerges) / 4
    mean_quarter = 0
    for r in rmerges[:quarter]:
        mean_quarter += r
    mean_quarter /= quarter

    sigma_quarter = 0
    for r in rmerges[:quarter]:
        diff = r - mean_quarter
        sigma_quarter += diff * diff
    sigma_quarter /= quarter
    
    Messenger.log_write('From the first %d batches get Rmerge statistics:' \
                        % quarter, 2)
    Messenger.log_write('Mean %f Variance %f' %
                        (mean_quarter, sigma_quarter), 2)

    chi_sq_list = Statistics.mean_chi_sq_list(rmerges,
                                              mean_quarter,
                                              sigma_quarter)
    for i in range(len(chi_sq_list)):
        chi_sq = chi_sq_list[i]
        Messenger.log_write('For %d images get chi squared = %f' % \
                            (i + 1, chi_sq), 2)

    sample_size = 10

    radiation_damage = False

    for i in range(0, (len(chi_sq_list) - sample_size), sample_size):
        if Statistics.mean_significantly_over_limit(
            chi_sq_list[i:i + sample_size], 1.5):
            radiation_damage = True
            Messenger.log_write(
                'Radiation damage starting to bite at image %d' % i)
            Messenger.log_write('Over a batch of %d images' % sample_size)
            Messenger.log_write(
                'this has an average chi square value of 1.5')
            Messenger.log_write(
                'Will truncate data set at this batch in later scaling')
            Somewhere.store('Radiation_Damage_Limit', i)
            break

    if not radiation_damage:
        Messenger.log_write(
            'No radiation damage found in analysis of Rmerge vs. batch')

    # next look at the intensity of reflections in the highest couple of
    # bins and see if the associated sigma is not 1.0 - in which
    # case issue some kind of warning at the moment

    intensity_bins = []
    full_sigma = { }
    partial_sigma = { }
    n_full = { }
    n_partial = { }

    # get the existing sd correctikon values

    sdcorrection = scale_reflections_response.getStandard_deviation_parameters()

    Messenger.log_write('Existing sdadd parameters: %f %f' % \
                        (sdcorrection.getSdadd_full(),
                         sdcorrection.getSdadd_partial()))

    for b in scale_reflections_response.getPer_intensity_bin_results(
        ).getIntensity_bin_results():
        bin = b.getBin()
        intensity_bins.append(bin)
        full_sigma[bin] = b.getSigma_full()
        n_full[bin] = b.getN_ref_full()
        partial_sigma[bin] = b.getSigma_partial()
        n_partial[bin] = b.getN_ref_partial()
        
    intensity_bins.sort()

    Messenger.log_write('Statistical analysis', 2)
    for b in intensity_bins:
        Messenger.log_write('Bin: %d %d %f %d %f' % \
                            (b, n_full[b], full_sigma[b],
                             n_partial[b], partial_sigma[b]), 2)

    # next want to calculate what to do with these numbers - they need to
    # be used to calculate appropriate values for sdadd...

    # use the top half of the intensity bins...

    full_total = 0.0
    full_count = 0
    partial_total = 0.0
    partial_count = 0

    for b in range((len(intensity_bins) - 1) / 2, len(intensity_bins)):
        full_total += n_full[b] * full_sigma[b]
        full_count += n_full[b]
        partial_total += n_partial[b] * partial_sigma[b]
        partial_count += n_partial[b]

    if full_count:
        sdadd_full = (full_total / full_count) / 1.0
    else:
        sdadd_full = 1.0
    sdadd_partial = (partial_total / partial_count) / 1.0

    sdadd_full *= sdcorrection.getSdadd_full()
    sdadd_partial *= sdcorrection.getSdadd_partial()

    Messenger.log_write('Corrections to sdadd %f %f' % \
                        (sdadd_full, sdadd_partial))

    Somewhere.store('sdadd_full', sdadd_full + sdcorrection.getSdadd_full())
    Somewhere.store('sdadd_partial', sdadd_partial +
                    sdcorrection.getSdadd_partial())
    
    # next have a look for systematic absences - e.g. two fold axes.
    # this will depend on the lattice - e.g. P1 can't have these, but
    # p222 could have 3 x two fold screw.

    # this is acrually built into the Scala output - someone elses problem!

    # Next have a stab at the resolution
    sig_vs_resol = { }
    fractional_bias = { } 
    for b in scale_reflections_response.getPer_resolution_bin_results(
        ).getResolution_bin_results():
        sig_vs_resol[b.getResolution().getUpper()] = b.getMean_i_over_sigma()
        fractional_bias[b.getResolution().getUpper()] = \
                                                      b.getFractional_bias()
    resol = sig_vs_resol.keys()
    resol.sort()

    mean_fractional_bias = 0

    for r in resol:
        mean_fractional_bias += fractional_bias[r]

    mean_fractional_bias /= len(resol)

    Messenger.log_write('Mean fractional bias is %f' % mean_fractional_bias)
    if mean_fractional_bias > -0.025:
        Messenger.log_write('This is probably OK')
    else:
        Messenger.log_write('This is a bit high - check your mosaic spread')
        
    if sig_vs_resol[resol[0]] > 1.0:
        Messenger.log_write('Data are good to highest resolution (%f)' % \
                            resol[0])
    else:
        i = 0
        good_resol = 0
        while i < len(resol):
            if sig_vs_resol[resol[i]] > 1.0:
                good_resol = resol[i]
                break
            i += 1
        if i == len(resol):
            # this is very unlikely
            Messenger.log_write('No good data found!')
        else:
            Messenger.log_write('Data good to %f' % good_resol)
            Somewhere.store('data_good_to', good_resol)
    

    return scale_reflections_response


def DetectScrewAxes(axial_reflections):
    '''Detect axial reflections from the autocorrelation of axial
    reflections'''

    for axis in axial_reflections.keys():
        Messenger.log_write('Checking axis %s for systematic absences' % \
                            axis, 2)
        keys = axial_reflections[axis].keys()
        keys.sort()

        offsets = { }
        for offset in [0, 1, 2, 3, 4, 6]:
            sum = 0.0
            for k in keys:
                if k + offset in keys:
                    sum += axial_reflections[axis][k] * \
                           axial_reflections[axis][k + offset]
            # Messenger.log_write('Offset %d sum %f' % (offset, sum))
            offsets[offset] = sum

        for offset in [1, 2, 3, 4, 6]:
            if offsets[offset] > 0.5 * offsets[0]:
                if offset != 1:
                    Messenger.log_write('Detected %d fold screw on axis %s' \
                                        % (offset, axis))
                break

    
    return None
        

def EstimateResolution(integrated_image):
    '''Guess the resolution of the data from an integrated image'''

    resolutions = []
    signals = { }
    image = int(integrated_image.getImage())
    bins = integrated_image.getIntegration_bin()
    for b in bins:
        r = b.getResolution()
        u = r.getUpper()
        l = r.getLower()
        # change 19 November 2004 AGWL recommends using profile
        # fitting at all times
        m = b.getMeasured_spots_profile()
        f = float(m.getFull().getSignal_to_noise())
        p = float(m.getPartial().getSignal_to_noise())
        # need to know nspots to build up a weighhted mean
        fc = float(m.getFull().getSpot_count())
        pc = float(m.getPartial().getSpot_count())

        intensity = (f * fc + p * pc) / (fc + pc)
        
        if math.fabs(u) > 0.01:
            resolutions.append(float(u))
            # if partials are stronger then fulls use partials
            # no now use a weighted average
            signals[float(u)] = intensity

    # change 19 November 2004 definition of "edge" is where I/sigma drops
    # below 2.0 now not 3.0

    resolutions.sort()
    s = []
    for r in resolutions:
        s.append(signals[r])

    if len(resolutions) > 1:
        record = 0
    else:
        record = 0
    if record:
        file = open('/tmp/resolution%d' % image, 'w')
        for i in range(len(resolutions)):
            file.write('%f %f\n' % (float(resolutions[i]), \
                                    float(signals[resolutions[i]])))
        file.close()

    # len -1 is the last element - and len - 1 - 1 is the one before,
    # so the interpolation below may work, in the event that ALL
    # bins have s < 2.0

    # in fact better even to check that there is at least one > 2.0

    # this method needs to be fixed, since it is at the moment
    # not 100 percent reliable

    all_gt_2 = True
    found_gt_2 = False
    for _s in s:
        if _s > 2.0:
            found_gt_2 = True
        if _s < 2.0:
            all_gt_2 = False

    if all_gt_2:
        Messenger.log_write('i/sigma > 2 at edge - taking resolution %2.2f' % \
                            resolutions[0], 2)
        return resolutions[0]

    if found_gt_2:
        bin = len(s) - 2
        while s[bin] > 2.0:
            if bin == 0:
                break
            bin -= 1

        if math.fabs(s[bin + 1] - s[bin]) < 0.01:
            resolution = resolutions[bin]
        else:
            resolution = resolutions[bin + 1] - \
                         (resolutions[bin + 1] - resolutions[bin]) * \
                         (s[bin + 1] - 2.0) / \
                         (s[bin + 1] - s[bin])

        estimatedResolution = 0.0
        
        if bin == 0 and s[bin] > 2.0:
            estimatedResolution = resolutions[bin]
        else:
            estimatedResolution = resolution

        header = Somewhere.get('last header')
        if header:
            if resolution < header['EdgeResolution']:
                estimatedResolution = header['EdgeResolution']
                Messenger.log_write(
                    'Resolution better than detector edge - using %2.2f' % \
                    header['EdgeResolution'], 2)
            else:
                estimatedResolution = resolution
                Messenger.log_write('Resolution is %2.2f' % resolution, 2)
        resolution = estimatedResolution

    else:
        Messenger.log_write("No resolution bin had (i/sigma) > 2.0", 2)
        resolution = 0.0

    if resolution < 0.0:
        Messenger.log_write('Error calculating resolution for image %d' % \
                            image)
        Messenger.log_write('Probably a good idea to inspect the ' + \
                            'integration log')
        resolution = 0.0
    return resolution
        

def AnalyseIntegrateResponse(integrate_response):
    '''Analyse an integrate response to get information out for instance
    the resolution of the data or the number of reflections'''

    if not integrate_response.getIntegrated_image():
        return

    for i in integrate_response.getIntegrated_image():
        image_number = i.getImage()
        Messenger.log_write("Integration results - image %d" % image_number, 2)
        resolution = EstimateResolution(i)

        if resolution == 0.0:
            Messenger.log_write('Error calculating resolution for image %d' % \
                                image_number)
        

    return

# for this function we should allow something through if it
# almost passes one criterion and passes all of the others - that
# is, have the "mean fitness" (e.g. real/limit) of at most 1.0
# and the absolute fitness of any one criterion < 2.0
# therefore define two new variables - may overload and average overload
# criteria are:
# fraction rejected
# beam shift
# rms deviation
# therefore sum (these/max) should be less than 3.0

def index_response_interpretation(index_parameters, index_response):
    # Returns thee different status - RMS, beam centre and
    # fraction of spots rejected from the cell refinement.
    
    # Interpret the response
    
    solution = index_response.getSolution()

    # check that these data items actually exist - if they don't then there
    # was a substantial problem!

    mosaicity = index_response.getMosaicity_value()
    cell1 = solution.getInitial().getCell()
    cell2 = solution.getOrientation().getCell()
    refinement = solution.getRefinement()
    reflections = refinement.getReflections()
    spots = index_response.getSpot_search_response()
    deltabeam = refinement.getBeam_shift().getShift()

    max_overload = 0.0
    max_overload_property = ''
    total_overload = 0.0

    Messenger.log_write("Refined solution %d with %s symmetry imposed." % \
                        (solution.getNumber(),solution.getSymmetry()))
    
    Messenger.log_write("Initial cell (before refinement) is  %8.3f%8.3f%8.3f%8.3f%8.3f%8.3f" % \
                        (cell1.getA(),cell1.getB(),cell1.getC(), \
                         cell1.getAlpha(),cell1.getBeta(),cell1.getGamma()))
    
    Messenger.log_write("Refined cell parameters:             %8.3f%8.3f%8.3f%8.3f%8.3f%8.3f" % \
                        (cell2.getA(),cell2.getB(),cell2.getC(), \
                         cell2.getAlpha(),cell2.getBeta(),cell2.getGamma()))
    
    Messenger.log_write("Using %d indexed reflections for refinement (out of %d spots used in indexing)" % \
                        (reflections.getUsed(), reflections.getUsed_in_indexing()))
    
    # Check fraction of spots rejected
    
    status_fraction_rejected = XSD.Status()
    status_fraction_rejected.setCode("ok")
    status_fraction_rejected.setMessage("")
    refinement_spot_frac_rejected = 1.0 - 1.0*reflections.getUsed()/reflections.getUsed_in_indexing()
    Messenger.log_write("Fraction rejected reflections in refinement / used reflections in indexing: %.3f" % \
                        refinement_spot_frac_rejected)
    warning_index_spot_frac_rejected = index_parameters.getWarning_index_spot_frac_rejected()
    max_index_spot_frac_rejected = index_parameters.getMax_index_spot_frac_rejected()

    # total_overload += refinement_spot_frac_rejected / max_index_spot_frac_rejected
    if refinement_spot_frac_rejected / max_index_spot_frac_rejected > max_overload:
        max_overload = refinement_spot_frac_rejected / max_index_spot_frac_rejected
        max_overload_property = 'fraction of rejected spots in refinement'
    # This is the fix for bug 840:
    # The increment of total_overload has been commented out here and moved to
    # the " if refinement_spot_frac_rejected > max_index_spot_frac_rejected:" block
    # below. The same is done for the RMS error and the beam shift.
    #    total_overload += 1.0

    # Is this error handling intelligent enough? Having hard limits 
    # may always be a problem, since something which is 10% over may
    # be OK - I have examples of this.
    # 
    # Should an error occur only when something actually FAILS?
    # What constitutes a real failure?
    # 
    # This is a point for discussion - maybe this will be a neural
    # network in DNA 2.0...
    
    if warning_index_spot_frac_rejected is None:
        warning_index_spot_frac_rejected = (2.0/3.0)*max_index_spot_frac_rejected
    if refinement_spot_frac_rejected > max_index_spot_frac_rejected:
        message = "Fraction of rejected spots greater than the max allowed (%.3f)" % max_index_spot_frac_rejected
        Messenger.log_write("SIGNIFICANT WARNING!! %s" % message)
        # this is being demoted to "warning" to allow processing to proceed - caveat user!!
        status_fraction_rejected.setCode("warning")
        status_fraction_rejected.setMessage(message)
        Messenger.log_write("This error relates to the setting of the " + \
                            "maximum number of rejected reflections")
        Messenger.log_write("and may be an indication that your data " + \
                            "are not ideal")
        total_overload += 1.0
    elif refinement_spot_frac_rejected > warning_index_spot_frac_rejected:
        message = "Fraction of rejected spots is high: %.3f" % refinement_spot_frac_rejected
        Messenger.log_write("WARNING! %s" % message)
        status_fraction_rejected.setCode("warning")
        status_fraction_rejected.setMessage(message)

    # Check RMS spot deviation

    status_RMS = XSD.Status()
    status_RMS.setCode("ok")
    status_RMS.setMessage("")
    warning_rms_error = index_parameters.getWarning_index_spot_rms_error()
    max_rms_error = index_parameters.getMax_index_spot_rms_error() 
    if warning_rms_error is None:
        warning_rms_error = 2.0/3.0*max_rms_error
    rms_error = solution.getSpot_deviation()
    min_sep = Somewhere.get('minimum_separation')

    # total_overload += rms_error / max_rms_error
    if rms_error / max_rms_error > max_overload:
        max_overload = rms_error / max_rms_error
        max_overload_property = 'rms deviation in position'
    #    total_overload += 1.0
    
    if rms_error > max_rms_error:
        message = "RMS spot deviation: %.3f mm greater than max allowed (%.3f mm)" % \
                  (rms_error, max_rms_error)
        Messenger.log_write("ERROR! %s" % message)
        status_RMS.setCode("warning")
        status_RMS.setMessage(message)

        if min_sep:
            Messenger.log_write('RMS/minsep = %2.2f' % (rms_error/min_sep), 2)
            if rms_error/min_sep > 0.5:
                message = 'Relative RMS error is %2.2f' % (rms_error/min_sep)
                status_RMS.setCode("error")
                status_RMS.setMessage(message)
        total_overload += 1.0

    elif rms_error > warning_rms_error:
        message = "RMS spot deviation is big: %.3f mm" % rms_error
        Messenger.log_write("WARNING! %s" % message, 2)
        status_RMS.setCode("warning")
        status_RMS.setMessage(message)
    else:
        Messenger.log_write("RMS spot deviation: %.3f" % rms_error)

    # Check beam shift

    status_beam_shift = XSD.Status()
    status_beam_shift.setCode("ok")
    status_beam_shift.setMessage("")
    delta_beam_x = deltabeam.getX()
    delta_beam_y = deltabeam.getY()
    warning_beam_shift = index_parameters.getWarning_beam_shift()
    max_beam_shift     = index_parameters.getMax_beam_shift()

    # total_overload += math.fabs(delta_beam_x + delta_beam_y) / (2.0 * max_beam_shift)
    if math.fabs(delta_beam_x + delta_beam_y) / (2.0 * max_beam_shift) > max_overload:
        max_overload = math.fabs(delta_beam_x + delta_beam_y) / (2.0 * max_beam_shift)
        max_overload_property = 'beam shift'
    #    total_overload += 1.0
    
    
    if warning_beam_shift is None:
        warning_beam_shift = 2.0/3.0*max_beam_shift
    if (abs(delta_beam_x) > max_beam_shift) or (abs(delta_beam_y) > max_beam_shift):
        message = "Beam coordinates has been shifted by %.3f %.3f which is more than the max allowed shift (%.3f)" % \
                  (delta_beam_x, delta_beam_y, max_beam_shift)
        Messenger.log_write("ERROR! %s" % message)
        status_beam_shift.setCode("warning")
        status_beam_shift.setMessage(message)
        total_overload += 1.0
    elif (abs(delta_beam_x) > warning_beam_shift) or (abs(delta_beam_y) > warning_beam_shift):
        message = "Beam coordinates shift is big: %.3f %.3f" % \
                  (delta_beam_x, delta_beam_y)
        Messenger.log_write("WARNING! %s" % message)
        status_beam_shift.setCode("warning")
        status_beam_shift.setMessage(message)
    else:
        Messenger.log_write("Beam coordinates has been shifted by %7.3f%7.3f" % \
                            (delta_beam_x, delta_beam_y))

    # Mosaicity
    Messenger.log_write("Mosaicity: %.3f" % mosaicity)

    # now digest the overload values

    status_overall = XSD.Status()
    status_overall.setCode("ok")

    # propogate the index_response status here
    if index_response.getStatus().getCode() == "error":
        status_overall = index_response.getStatus()

    if max_overload > 2.0:
        status_overall.setCode("warning")
        status_overall.setMessage("A parameter (%s) was a factor of two over the limit!" % max_overload_property)
        Messenger.log_write("A parameter (%s) was a factor of two over the limit!" % max_overload_property)
    if total_overload > 2.5:
        status_overall.setCode("error")
        status_overall.setMessage("All parameters were over the limit!")
        Messenger.log_write("All parameters were over the limit!")

    if status_RMS.getCode() == 'error':
        status_overall.setCode("error")
        status_overall.setMessage(status_RMS.getMessage())

    return (status_RMS, status_beam_shift, status_fraction_rejected, status_overall)

def ConsiderUnconstrainedOrientation(a_matrix):
    '''This will consider the contents of an orientation matrix (a*, b*, c*)
    from indexing and have a stab at the lattice. The matrix is a dictionary
    keyed by 11, 12, 13, 21, 22, 23, 31, 32, 33.'''

    # this is loosely based on the method for lattice
    # penalty calculation described in
    # Otwinowski & Minor, IUCR International Tables for Crystallography 'F'
    # Ed. Michael G. Rossmann & Eddy Arnold
    # Kluwer academic publishers, 2001

    a_star = [a_matrix['11'], a_matrix['21'], a_matrix['31']]
    b_star = [a_matrix['12'], a_matrix['22'], a_matrix['32']]
    c_star = [a_matrix['13'], a_matrix['23'], a_matrix['33']]

    Amatrix = Vector.Matrix()
    for i in range(3):
        Amatrix[i][0] = a_star[i]
        Amatrix[i][1] = b_star[i]
        Amatrix[i][2] = c_star[i]
        

    A, B, C = Crystallographer.Invert(a_star, b_star, c_star)

    a, b, c, alpha, beta, gamma = Crystallographer.Cell(A, B, C)

    lattices = ['aP', 'mP', 'mC', 'oP', 'oC', 'oI', 'oF',
                'tP', 'tI', 'hR', 'hP', 'cP', 'cI', 'cF']

    penalties = { }

    for lattice in lattices:
        try:
            _a, _b, _c, _alpha, _beta, _gamma = \
                Crystallographer.Constrain(lattice, \
                                           [a, b, c, alpha, beta, gamma])
            
            print_constraint = False
            
            if print_constraint:
                
                Messenger.log_write('Constraining cell for lattice %s' % \
                                    lattice)
                Messenger.log_write('Original:    %4.2f %4.2f %4.2f %4.2f %4.2f %4.2f' % \
                                    (a, b, c, alpha, beta, gamma))
                Messenger.log_write('Constrained: %4.2f %4.2f %4.2f %4.2f %4.2f %4.2f' % \
                                    (_a, _b, _c, _alpha, _beta, _gamma))
                
            oldcell = Crystallographer.InvertCell(a, b, c, alpha, beta, gamma)
            cell = Crystallographer.InvertCell(_a, _b, _c, _alpha, _beta, _gamma)

            BmatrixOld = Crystallographer.BMatrix(oldcell[0], oldcell[1], \
                                                  oldcell[2], oldcell[3], \
                                                  oldcell[4], oldcell[5])

            Bmatrix = Crystallographer.BMatrix(cell[0], cell[1], \
                                               cell[2], cell[3], \
                                               cell[4], cell[5])
            
            BmatrixInv = Vector.Invert(BmatrixOld)

            # next need to calculate a 'U' matrix, and invert it, and
            # then see how unitary it is...
            
            
            RealA = Vector.Matrix()
            RealA[0] = A
            RealA[1] = B
            RealA[2] = C

            U = Vector.MultiplyMatrix(Bmatrix, BmatrixInv)
            
            Ui = Vector.Invert(U)
            Ut = Vector.Transpose(U)
            
            penalty = 0.0
            for i in range(3):
                for j in range(3):
                    diff = Ui[i][j] - Ut[i][j]
                    penalty += diff * diff

            penalty = math.sqrt(penalty) / 6.0

            penalties[lattice] = penalty

        except:
            # assume we have a mathematical error/domain problem
            # assumption is:
            # mathematical error => not a suitable lattice => make penalty
            # hurt

            penalty = 1.0e9
            
            penalties[lattice] = 1.0e9

        if print_constraint:

            Messenger.log_write('Penalty for lattice %s is %f' % \
                                (lattice, penalty))

    # next attempt to determine the correct lattice based on the
    # penalties and the crystallographic relationships between the
    # lattices e.g. cF -> oF -> aP and cF -> hR -> mP -> aP

    up = { }
    up['aP'] = ['oF', 'oI', 'mP', 'mC']
    up['oF'] = ['cF']
    up['oI'] = ['tI', 'cI']
    up['mP'] = ['oP']
    up['mC'] = ['hP', 'oC', 'hR']
    up['tI'] = []
    up['cF'] = []
    up['oP'] = ['tP']
    up['hP'] = []
    up['oC'] = []
    up['hR'] = ['cI', 'cF']
    up['cI'] = []
    up['tP'] = ['cP']
    up['cP'] = []

    stepMax = 2.0

    start = penalties['aP']
    next = up['aP']
    lattice = ClimbTree(penalties['aP'], up['aP'], 'aP', \
                                penalties, up)
        
    Messenger.log_write('Selected lattice %s based on geometry' % lattice)
    message = 'The space group should be one of: '
    for sg in Crystallographer.Spacegroups(lattice):
        message += '%s ' % sg
    Messenger.log_write(message)

# this needs to be revised substantially to perform a proper
# search in the case of high symmetry spacegroups - for instance, we
# want to know how high up the tree we got - the best should be
# the highest up the tree...

def Score(lattice):
    scores = { }
    scores['aP'] = 0.0
    scores['mC'] = 1.0
    scores['mP'] = 1.0
    scores['oP'] = 2.0
    scores['oC'] = 2.0
    scores['oI'] = 2.0
    scores['oF'] = 2.0
    scores['hR'] = 2.5
    scores['hP'] = 2.5
    scores['tP'] = 3.0
    scores['tI'] = 3.0
    scores['cP'] = 4.0
    scores['cI'] = 4.0
    scores['cF'] = 4.0
    return scores[lattice]

# bewarned - you will need a strong cup of coffee before attempting to
# understand the recursion in this function!

def ClimbTree(penalty, lattices, this, penalties, up):
    best = Score(this)
    lattice = this
    for l in lattices:
        # beware fudge factors at this stage! ;o)
        if penalties[l] < max(0.005, 2.0 * penalty):
            l = ClimbTree(penalties[l], up[l], l, penalties, up)
            if Score(l) >= best:
                lattice = l
                best = Score(l)

    return lattice

# image selection functions for determining the best way to
# process a data set

def SelectImagesForAutoindex(header, images):
    '''Takes the header from the first image and a list of
    available images'''

    first = images[0]

    width = float(header['PhiWidth'])
    if width > 0:
        number = first + nint(90.0 / width)
    else:
        number = first + 90

    if number in images:
        Messenger.log_write('Selected images %d, %d for autoindex' % \
                            (first, number))
        return number
    else:
        Messenger.log_write('Selected images %d, %d for autoindex' % \
                            (first, images[-1]))
        return images[-1]

def SelectImagesForCellRefinement(header, mosaic, images):
    '''Takes the header from the first image, the mosaic spread
    and a list of available images - this assumes you want two wedges'''

    first = images[0]

    width = float(header['PhiWidth'])

    if width > 0:
        block_size = nint(mosaic / width)
    else:
        block_size = 3

    if block_size < 1:
        block_size = 1

    if len(images) < (2 * block_size):
        raise RuntimeError, 'Not enough images to refine cell'

    if width > 0:
        number = first + int(90.0 / width)
    else:
        number = first + 90

    if not (number + block_size) in images:
        number = images[-1] - block_size

    starts = []
    ends = []

    starts.append(first)
    ends.append(first + block_size)
    
    starts.append(number)
    ends.append(number + block_size)

    Messenger.log_write('Using images %d to %d and %d to %d for refinement' %
                       (starts[0], ends[0], starts[1], ends[1]))

    return (starts, ends)

def nint(f):
    i = int(f)
    if math.fabs(f - i) > 0.5:
        i += 1

    return i

def CheckHeadersAreConsistent(image1, header1, image2, header2):
    '''Compare the contents of the two headers with the numbers
    image1 and image2 and see if the two are consistent'''

    start = header1['PhiStart']
    width = header1['PhiWidth']

    diff = (image2 - image1) * width

    if math.fabs(start + diff - header2['PhiStart']) > 0.05:
        # there is something wrong with the header information
        Messenger.log_write('Inconsistent rotation information in headers')
        return False

    # all hunky-dory
    return True

def CreateHeader(first, image0, this):
    '''Create an appropriate image header for this based on image0'''

    Messenger.log_write('Synthesising header for image %d' % this)

    header = { }
    header['ExposureTime'] = first['ExposureTime']
    header['BeamX'] = first['BeamX']
    header['BeamY'] = first['BeamY']
    header['Distance'] = first['Distance']
    header['Wavelength'] = first['Wavelength']
    header['PixelX'] = first['PixelX']
    header['PixelY'] = first['PixelY']
    header['Width'] = first['Width']
    header['Height'] = first['Height']
    messages = first['Message']
    header['Message'] = first['Message']
    header['Message'].append('axis not phi')
    start = first['PhiStart']
    end = first['PhiEnd']
    width = first['PhiEnd'] - first['PhiStart']
    diff = this - image0
    header['PhiStart'] = start + diff * width
    header['PhiEnd'] = end = diff * width
    header['PhiWidth'] = width
    
    return header

    
def LookForTwinningAcentric2(listOfValues):
    '''Look for signs of twinning in the second acentric moments
    from TRUNCATE'''

    total = 0.0

    for l in listOfValues:
        total += l

    total /= len(listOfValues)

    sumsq = 0.0

    for l in listOfValues:
        sumsq += (l - total) * (l - total)

    sumsq /= len(listOfValues)

    sumsq = math.sqrt(sumsq)

    # check to see if the mean value is consistent with 2.0

    if math.fabs(total - 2.0) < (sumsq / math.sqrt(len(listOfValues))):
        # we are not twinned
        return 'no'

    # next check to see if this is consistent with 1.5 - a perfect
    # twin

    if math.fabs(total - 1.5) < (sumsq / math.sqrt(len(listOfValues))):
        # we are not twinned
        return 'yes'

    if total > 2.0:
        return 'not sure but probably not'

    return 'not sure'

def AnalyseFALLOFF(falloff_data):
    '''Analyse the output from the program FALLOFF'''

    mn_f_d1 = falloff_data['mn_f_d1'] 
    mn_f_d2 = falloff_data['mn_f_d2'] 
    mn_f_d3 = falloff_data['mn_f_d3'] 
    mn_f_overall = falloff_data['mn_f_overall'] 
    mn_f_sig_d1 = falloff_data['mn_f_sig_d1'] 
    mn_f_sig_d2 = falloff_data['mn_f_sig_d2'] 
    mn_f_sig_d3 = falloff_data['mn_f_sig_d3'] 
    mn_f_sig_overall = falloff_data['mn_f_sig_overall'] 
    resolution = falloff_data['resolution']

    # these arrays should have information about how the intensity
    # falls off as a function of resolution (is the resolution
    # relevent?) so this should be used to see if the strength of
    # diffraction is anisotropic

    total_d1 = 0.0
    total_d2 = 0.0
    total_d3 = 0.0
    resolutions_used = 0
    for i in range(len(resolution)):
        if mn_f_overall[i] > 0:
            resolutions_used += 1
            total_d1 += (mn_f_d1[i] - mn_f_overall[i]) / mn_f_overall[i]
            total_d2 += (mn_f_d2[i] - mn_f_overall[i]) / mn_f_overall[i]
            total_d3 += (mn_f_d3[i] - mn_f_overall[i]) / mn_f_overall[i]

    total_d1 /= resolutions_used
    total_d2 /= resolutions_used
    total_d3 /= resolutions_used

    Messenger.log_write('Anisotropy parameters %f %f %f' % \
                        (total_d1, total_d2, total_d3))

def GuessIOverSig(peaklist):
    '''Guess an appropriate i/sigma value to use for auto indexing'''

    possible_values = [20.0, 10.0, 5.0]

    for p in possible_values:
        top = -1
        for i in range(peaklist.length()):
            if peaklist[i][2] < p:
                top = i
                break
        # we need at least 25 peaks on each image to get a decent index
        # in my humble opinion!
        if top >= 25:
            return p

    # have a lowest value of 5.0
    return 5.0


def PossibleSpacegroups(spacegroup):
    '''Based on a spacegroup, give all possible spacegroups that this could
    be, based on the lattice constraints and the symmetry operations'''

    spacegroups = { }
    spacegroup_symops = { }
    spacegroup_numbers = { }
    spacegroup_altname = { }
    symops = { }

    current_spacegroup = ''

    # initialisation from CCP4 symmetry operation file

    for line in \
        open('%s/lib/data/symop.lib' % os.environ['CCP4'], 'r').readlines():
        if line[0] != ' ':
            # we have a new space group
            number = int(line.split()[0])
            name = line.split()[3].lower()
            longname = line.split("'")[1].lower()

            if number > 230:
                name = ''
            else:
                spacegroup_altname[longname] = name

            spacegroups[number] = name
            spacegroup_numbers[name] = number
            current_spacegroup = name
            spacegroup_symops[name] = []
        else:
            # we have a new symmetry operation for this group[
            operation = line.replace(' ', '').replace('\n', '').lower()
            if not symops.has_key(operation):
                symops[operation] = []
                symops[operation].append(current_spacegroup)
            else:
                symops[operation].append(current_spacegroup)

            spacegroup_symops[current_spacegroup].append(operation)

    if spacegroup in spacegroup_altname.keys():
        # use the short name
        spacegroup = spacegroup_altname[spacegroup]

    # determine the possible spacegroups based on the lattice constraints that
    # this spacegroup satisfies

    lattice = Crystallographer.Lattice(spacegroup)
    sublattices = Crystallographer.Sublattices(lattice)

    possible_spacegroups = []

    for s in sublattices:
        possible_spacegroups.append(Crystallographer.Spacegroups(s)[0])
        # deal with special cases - i.e. when it could be higher symmetry
        # which would make a difference e.g. 432 for cubic or 422 for
        # tetragonal - deal with these later though
        if s == 'hP':
            # Special case - could be p3 or p6
            possible_spacegroups.append('p3')
        if s == 'tP':
            # special case - could be p4 or p422 as the point group
            possible_spacegroups.append('p422')

    spacegroup_operations = spacegroup_symops[spacegroup]

    # now add to this list based on the symmetry operations
    for s in spacegroup_symops.keys():
        subgroup = True
        for operation in spacegroup_symops[s]:
            if not operation in spacegroup_operations:
                subgroup = False
                break

        if subgroup:
            if possible_spacegroups.count(s) == 0:
                possible_spacegroups.append(s)

    # that should be all of the possible subgroups got now - so
    # return a list of (spacegroup, number) tuples

    numbers = []
    for s in possible_spacegroups:
        numbers.append(spacegroup_numbers[s])
    numbers.sort()

    result = []

    for n in numbers:
        result.append((spacegroups[n], n))

    return result

def SpacegroupOrder(spacegroup):
    spacegroups = { }
    spacegroup_symops = { }
    spacegroup_numbers = { }
    spacegroup_altname = { }

    current_spacegroup = ''

    # initialisation from CCP4 symmetry operation file

    for line in \
        open('%s/lib/data/symop.lib' % os.environ['CCP4'], 'r').readlines():
        if line[0] != ' ':
            # we have a new space group
            number = int(line.split()[0])
            name = line.split()[3].lower()
            longname = line.split("'")[1].lower()

            if number > 230:
                name = ''
            else:
                spacegroup_altname[longname] = name

            spacegroups[number] = name
            spacegroup_numbers[name] = number
            current_spacegroup = name
            spacegroup_symops[name] = []
        else:
            # we have a new symmetry operation for this group[
            operation = line.replace(' ', '').replace('\n', '').lower()
            spacegroup_symops[current_spacegroup].append(operation)

    if spacegroup in spacegroup_altname.keys():
        # use the short name
        spacegroup = spacegroup_altname[spacegroup]

    return len(spacegroup_symops[spacegroup])

def CellVolume(a, b, c, alpha, beta, gamma):
    pi = 4.0 * math.atan(1.0)
    dtor = pi / 180.0
    rtod = 180.0 / pi
    ca = math.cos(dtor * alpha)
    cb = math.cos(dtor * beta)
    cc = math.cos(dtor * gamma)
    sa = math.sin(dtor * alpha)
    sb = math.sin(dtor * beta)
    sc = math.sin(dtor * gamma)
    V = a * b * c * math.sqrt(1 - ca * ca - cb * cb - cc * cc + \
                              2 * ca * cb * cc)
    return V

def ShortSpacegroupName(spacegroup):
    spacegroups = { }
    spacegroup_symops = { }
    spacegroup_numbers = { }
    spacegroup_altname = { }
    symops = { }
    
    current_spacegroup = ''

    # initialisation from CCP4 symmetry operation file

    for line in \
        open('%s/lib/data/symop.lib' % os.environ['CCP4'], 'r').readlines():
        if line[0] != ' ':
            # we have a new space group
            number = int(line.split()[0])
            name = line.split()[3].lower()
            longname = line.split("'")[1].lower()

            if number > 230:
                name = ''
            else:
                spacegroup_altname[longname] = name

            spacegroups[number] = name
            spacegroup_numbers[name] = number
            current_spacegroup = name
            spacegroup_symops[name] = []
        else:
            # we have a new symmetry operation for this group[
            operation = line.replace(' ', '').replace('\n', '').lower()
            if not symops.has_key(operation):
                symops[operation] = []
                symops[operation].append(current_spacegroup)
            else:
                symops[operation].append(current_spacegroup)

            spacegroup_symops[current_spacegroup].append(operation)

    spacegroup = spacegroup.lower()

    if spacegroup in spacegroup_altname.keys():
        # use the short name
        spacegroup = spacegroup_altname[spacegroup]

    return spacegroup.upper()

def Molweight(sequence):
    weight = 0.0
    for s in sequence.lower():
        if s == 'a':
            weight += 89
        elif s == 'c':
            weight += 121
        elif s == 'd':
            weight += 133
        elif s == 'e':
            weight += 147
        elif s == 'f':
            weight += 165
        elif s == 'g':
            weight == 75
        elif s == 'h':
            weight += 155
        elif s == 'i':
            weight += 131
        elif s == 'k':
            weight += 146
        elif s == 'l':
            weight += 131
        elif s == 'm':
            weight += 149
        elif s == 'n':
            weight += 132
        elif s == 'p':
            weight += 115
        elif s == 'q':
            weight += 146
        elif s == 'r':
            weight += 174
        elif s == 's':
            weight += 105
        elif s == 't':
            weight += 119
        elif s == 'v':
            weight += 117
        elif s == 'w':
            weight += 204
        elif s == 'y':
            weight += 181

    return weight    

def Sequence(pirfile):
    '''decode a pir format file'''
    file = open(pirfile, 'r')
    line = file.readline()
    if line[0] == '>':
        file.readline()
        seq = ''
    else:
        seq = line
    seq += file.read()
    file.close()
    return seq.lower()

def Selenium(filename):
    sequence = Sequence(filename)
    return sequence[1:].count('m')

def Weight(filename):
    sequence = Sequence(filename)
    return Molweight(sequence)

def BatchesForScaling(duff, start, end):
    '''Compute the batch ranges for scaling which exclude the batches
    listed in "duff"'''

    batch_commands = []

    run = 1

    for d in duff:
        if (d - 1) > start:
            batch_commands.append('run %d batch %d to %d' % \
                                  (run, start, d - 1))
            run += 1
        start = d + 1
    if end > start:
        batch_commands.append('run %d batch %d to %d' % \
                              (run, start, end))

    return batch_commands

if __name__ == '__main__':
    start = 1
    end = 200
    duff = [10, 23]
    print BatchesForScaling(duff, start, end)

if __name__ == '__main__1':
    spacegroup = 'p23'
    if len(sys.argv) > 1:
        spacegroup = sys.argv[1].lower()
    print PossibleSpacegroups(spacegroup)
    print SpacegroupOrder(spacegroup)
