#!/opt/pythonesrf/startup/python
##################################################################
#
# bcm_simulator.py
#
# Authors: The DNA team, http://www.dna.ac.uk  
#
#
# BCM simulator for the DNA project
#
# The purpose of this program is to simulate a DNA Beamline
# Control Module - BCM. The server uses HTTP and will accept a PUT request 
# from a client. The PUT request expects a 'collect_request' 
# command in XML as defined by the XML Schema.
#
# If a directory is given as a command line argument, the bcm_simulator
# copies the images from this directory to the directory given
# in the collect request.
#
##################################################################

import BaseHTTPServer, mimetypes, threading, traceback, sys
import string, pprint, os, time, shutil
from ESConfig import *
import XSD
from XML_utils import *
from ESContext import *
from DetectorDatabase import *
import BcmParametersHelper


class CollectThread(threading.Thread):

  def __init__(self, collect_request):
    threading.Thread.__init__(self)
    self.collect_request = collect_request
  
  def run(self):
    try:
      #
      # Reset the abort flag
      #
      DataCollectRequestHandler.isAbort = False
      #
      # Do the actual dummy data collect
      #
      pprint.pprint( self.collect_request.toDict() )
      run_number = self.collect_request.getFileinfo().getRun_number()
      prefix = self.collect_request.getFileinfo().getPrefix()
      if prefix[-1] == "_":
          prefix = prefix[:-1]
      suffix = self.collect_request.getFileinfo().getSuffix()
      if suffix is None:
          suffix = DetectorDatabase.determine_suffix(config_detector)
      for oscillation_sequence in self.collect_request.getOscillation_sequence():
        number_of_images   = oscillation_sequence.getNumber_of_images()
        start_image_number = oscillation_sequence.getStart_image_number()
        for i in range(number_of_images):
          file_name_from = "%s_%d_%03d.%s"%(prefix, 1, i+start_image_number, suffix)
          file_name_to   = "%s_%d_%03d.%s"%(prefix, run_number, i+start_image_number, suffix)
          if DataCollectRequestHandler.isAbort:
            break
          time.sleep(DataCollectRequestHandler.exposureTime)
          if DataCollectRequestHandler.isAbort:
            break
          #
          # Copy file if from directory is provided
          #
          if not DataCollectRequestHandler.from_directory is None:
            from_file = os.path.join(DataCollectRequestHandler.from_directory,file_name_from)
            to_file   = os.path.join(self.collect_request.getFileinfo().getDirectory(), file_name_to)
            print "Copying file from %s to %s" % (from_file, to_file)
            shutil.copyfile(from_file, to_file)
          print "Image %s collected" % file_name_to
        run_number = run_number+1
        if DataCollectRequestHandler.isAbort:
          break
      #
      # Everything seems to be ok so far...
      #
      status = XSD.Status()
      status.setCode("ok")
      collect_response = XSD.Collect_response()
      collect_response.setStatus(status)
      reply = XML_utils.http_post(ES_host,ES_port,'/collect_response',collect_response.marshal())
    except :
      #raise
      #
      # Something went wrong
      #
      message = "Exception: %s"%(traceback.format_exception_only(sys.exc_type, sys.exc_value)[0])
      print message
      status = XSD.Status()
      status.setCode("error")
      status.setMessage(message)
      collect_response = XSD.Collect_response()
      collect_response.setStatus(status)
      reply = XML_utils.http_post(ES_host,ES_port,'/collect_response',collect_response.marshal())

#
#
#
class DataCollectRequestHandler(BaseHTTPServer.BaseHTTPRequestHandler):

  message = ''
  last_collect_request = None
  config_detector = None
  collect_thread = None

  def do_POST(self):
    self.server_name = os.environ['HOSTNAME']
    #
    # Get incoming XML document
    #
    message_length = string.atoi(self.headers.getheader('Content-Length'))
    DataCollectRequestHandler.message = self.rfile.read(message_length)
    #
    # Distinguish between a data collect command, a request and
    # a status output
    if self.path == '/collect_request':
      #
      # Translate XML to a Python dictionary 
      #
      #print DataCollectRequestHandler.message
      collect_request = XSD.Collect_request()
      collect_request.unmarshal(DataCollectRequestHandler.message)
      DataCollectRequestHandler.last_collect_request = collect_request
      self.collect_thread = CollectThread(collect_request)
      self.collect_thread.start()
      status = XSD.Status()
      status.setCode("ok")
      xml_response = status.marshal()
      self.wfile.write('HTTP/1.1 200 OK\n')
      self.wfile.write('Host: %s\n'%(self.server_name))
      self.wfile.write('Content-Type: text/xml\n')
      self.wfile.write('Content-Length: %d\n\n'%(len(xml_response)))
      self.wfile.write(xml_response)
    elif self.path == '/bcm_parameters_request':
      fileinfo = DataCollectRequestHandler.last_collect_request.getFileinfo()
      if fileinfo.getSuffix() is None:
          fileinfo.setSuffix("img")
      bcm_parameters_response = BcmParametersHelper.getBcmParametersFromImageFile(fileinfo)
      #
      # Add simulated beamline parameters
      #
      maximum_exposure        = 600
      minimum_exposure_time   = 0.1
      minimum_phi_speed       = 0.1
      maximum_phi_speed       = 100
      minimum_phi_oscillation = 0.1
      beamline_parameters = XSD.Beamline_parameters()
      beamline_parameters.setMaximum_exposure(maximum_exposure)
      beamline_parameters.setMinimum_exposure_time(minimum_exposure_time)
      beamline_parameters.setMinimum_phi_speed(minimum_phi_speed)
      beamline_parameters.setMaximum_phi_speed(maximum_phi_speed)
      beamline_parameters.setMinimum_phi_oscillation(minimum_phi_oscillation)
      proposal = XSD.Proposal()
      proposal.setCode( "mx" )
      proposal.setNumber( 415 )
      login_info = XSD.Login_info()
      login_info.setProposal( proposal )
      login_info.setSessionId( 6057 )
      bcm_parameters_response.setBeamline_parameters(beamline_parameters)
      bcm_parameters_response.setLogin_info( login_info )
      bcm_parameters_response.getStatus().setCode("ok")
      pprint.pprint(bcm_parameters_response.toDict())
      xml_response = bcm_parameters_response.marshal()
      self.wfile.write('HTTP/1.1 200 OK\n')
      self.wfile.write('Host: %s\n'%(self.server_name))
      self.wfile.write('Content-Type: text/xml\n')
      self.wfile.write('Content-Length: %d\n\n'%(len(xml_response)))
      self.wfile.write(xml_response)
    elif self.path == '/sample_references_request':
      sample_references_response = XSD.Sample_references_response()
      bar_code_list = [ 'HA00AK0601', \
                        'HA00AK0600', \
                        'HA00AK0596', \
                        'HA00AK0589', \
                        'HA00AK0592', \
                        'HA00AK0594', \
                        'HA00AK0595', \
                        'HA00AK0593', \
                        'HA00AK0591', \
                        ]
      for bar_code in bar_code_list:
        sample_reference = XSD.Sample_reference()
        sample_reference.setCode( bar_code )
        sample_references_response.addSample_reference( sample_reference )
      status = XSD.Status()
      status.setCode("ok")
      sample_references_response.setStatus( status )
      xml_response = sample_references_response.marshal()
      print xml_response
      self.wfile.write('HTTP/1.1 200 OK\n')
      self.wfile.write('Host: %s\n'%(self.server_name))
      self.wfile.write('Content-Type: text/xml\n')
      self.wfile.write('Content-Length: %d\n\n'%(len(xml_response)))
      self.wfile.write(xml_response)      
    elif self.path == '/request':
      self.dummy_request()
    elif self.path == '/message':
      self.dummy_message()
    elif self.path == '/abort_request':
      DataCollectRequestHandler.isAbort = True
      #if not self.collect_thread is None:
      #    self.collect_thread.stop()
      print "Data collection stopped!"
      status = XSD.Status()
      status.setCode("ok")
      abort_response = XSD.Abort_response()
      abort_response.setStatus(status)
      xml_response = status.marshal()
      print xml_response
      self.wfile.write('HTTP/1.1 200 OK\n')
      self.wfile.write('Host: %s\n'%(self.server_name))
      self.wfile.write('Content-Type: text/xml\n')
      self.wfile.write('Content-Length: %d\n\n'%(len(xml_response)))
      self.wfile.write(xml_response)
    else:
      print "Command %s not recognized." % self.path




#
# Start up the HTTP server (if not run by pydoc)
#

if "PYDOC" in os.environ.keys():
  print "Run by pydoc - not starting the bcm simulator."
  sys.exit(0)

#
# Read parameters from the configuration file
#

server_data = ESConfig.getServer_data()
last_collect_request = XSD.Collect_request()
last_collect_request.setFileinfo(ESConfig.getUser_defaults().getDefault_values().getFileinfo())
config_detector = ESConfig.getUser_defaults().getDefault_values().getDetector()
ES_host = server_data.getExpert_system_host_name()
ES_port = server_data.getExpert_system_port_number()

BCM_host = server_data.getBcm_host_name()
if (BCM_host != os.environ["HOSTNAME"]):
    print "Warning! BCM host name in configuration file, %s, " % BCM_host
    print "does not correspond to the actual host name %s." % os.environ["HOSTNAME"]
BCM_port = server_data.getBcm_port_number()


#
#
#
server_class=BaseHTTPServer.HTTPServer
handler_class=DataCollectRequestHandler
DataCollectRequestHandler.last_collect_request = last_collect_request
DataCollectRequestHandler.config_detector = config_detector
DataCollectRequestHandler.isAbort = False
DataCollectRequestHandler.exposureTime = 4
if len(sys.argv) == 2:
  DataCollectRequestHandler.from_directory = sys.argv[1]
else:
  DataCollectRequestHandler.from_directory = None
server_address = (BCM_host, BCM_port)
httpd = server_class(server_address, handler_class)
try:
  httpd.serve_forever()
except:
  print "Shutting down the server."

