#################################################################
#
# ExecutiveSystem.py
#
# Authors: The DNA team, http://www.dna.ac.uk
#
# Version: $Id: ExecutiveSystem.py,v 1.122 2006/06/23 12:55:17 svensson Exp $
#
#################################################################

import os, sys

#
# Configure environment variables for the ES and the scheduler.
#
# This code will probably be moved to a separate file.
#

DNAHOME = None
if "DNAHOME" in os.environ.keys():
    DNAHOME = os.environ[ "DNAHOME" ]
else:
    print ""
    print "ERROR starting the DNA ExecutiveSystem! DNAHOME not defined."
    sys.exit(1)

DNANAME = None
if "DNANAME" in os.environ.keys():
    DNANAME = os.environ[ "DNANAME" ]
else:
    DNANAME = "localhost"
    os.environ[ "DNANAME" ] = DNANAME

#
# Set up PYTHONPATH
#

sys.path.append( os.path.join( DNAHOME, "es", "src" ))
sys.path.append( os.path.join( DNAHOME, "xsd", "python" ))
sys.path.append( os.path.join( DNAHOME, "scheduler", "Scheduler", "Mosflm" ))
sys.path.append( os.path.join( DNAHOME, "scheduler", "DiffractionImage", "lib" ))
sys.path.append( os.path.join( DNAHOME, "ext", "lib" ))

PYTHONPATH = ""
if "PYTHONPATH" in os.environ.keys():
    PYTHONPATH = os.environ[ "PYTHONPATH" ]
AALIB_MODULES = os.path.join( DNAHOME, "libraries", "aalib", "modules" )
PYTHONPATH = PYTHONPATH + ":" + AALIB_MODULES
#os.environ[ "PYTHONPATH" ] = PYTHONPATH

#
# Setup besthome
#

besthome = os.path.join( DNAHOME, "third_party", "best_v3.0" )

#
# Setup PATH
#

MOSFLM_BIN_DIR = os.path.join( DNAHOME, "third_party", "mosflm", "bin")

PATH = os.environ[ "PATH" ]
PATH = besthome + ":" + PATH
PATH = MOSFLM_BIN_DIR + ":" +PATH
os.environ[ "PATH" ] = PATH

#
# Set up DNATMPDIR
#

if not "DNATMPDIR" in os.environ.keys():
    USER = "dna"
    if "USER" in os.environ.keys():
        USER = os.environ[ "USER" ]
    DNATMPDIR = os.path.join( "/tmp", USER, DNANAME )
    os.environ[ "DNATMPDIR" ] = DNATMPDIR
    if not os.path.exists( DNATMPDIR ):
        os.makedirs( DNATMPDIR )
#
# Set up DNALOGDIR
#

if not "DNALOGDIR" in os.environ.keys():
    DNALOGDIR = os.getcwd()
    os.environ[ "DNALOGDIR" ] = DNALOGDIR

    

import xml.sax.saxutils
import dna_httplib, string
import pprint, time, socket, re, traceback, math
import threading, exceptions
import smtplib
import XSD
import Queue
import BcmParametersHelper

from threading import Thread
from ESConfig import *
from ESContext import *
from DNA_Logger import *
from ExecutiveHTTPServer import *
from ExecutiveDBProxy import *
from Debug import *
from XML_utils import *
from DetectorDatabase import *

#
# Do not import the scheduler stuff if run by pydoc
#
if not "PYDOC" in os.environ.keys():
  import MosflmInterface


#################################################################
#
# The "ExecutiveSystem" class
#

class ExecutiveSystem:


  def __init__(self):
    """Initialises the ES and runs the ES main loop."""
    #
    # Initialize some variables
    #
    self.version_short = "DNA version 1.1 pre-release"
    self.version_long  = "DNA version 1.1 pre-release $Id: ExecutiveSystem.py,v 1.122 2006/06/23 12:55:17 svensson Exp $"
    #self.version_short = "DNA version v1.1b-20060404"
    #self.version_long  = "DNA version v1.1b-20060404"
    self.dpm_thread              = None
    self.db_thread               = None
    self.collect_directory       = None
    self.collect_prefix          = None
    #self.bcm_parameters_response = None
    #self.bcm_parameters_flag     = False
    self.saved_index_request     = None
    self.collecting_flag         = False
    self.command_queue           = Queue.Queue()
    #
    # Set server data
    #
    self.system_defaults      = ESConfig.getSystem_defaults()
    self.user_defaults        = ESConfig.getUser_defaults()
    self.server_data          = ESConfig.getServer_data()
    self.default_values       = self.user_defaults.getDefault_values()
    self.local_info           = self.system_defaults.getLocal_info()
    self.index_parameters     = self.user_defaults.getIndex_parameters()
    self.integrate_parameters = self.user_defaults.getIntegrate_parameters()
    self.strategy_parameters  = self.user_defaults.getStrategy_parameters()
    self.GUI_host             = self.server_data.getGui_host_name()
    self.GUI_port             = self.server_data.getGui_port_number()
    #
    # Make sure that rank_parameters are set
    #
    if self.user_defaults.getRank_parameters() is None:
      rank_parameters = XSD.Rank_parameters()
      rank_parameters.setRankEngine( "DREngineDataCollectionTime" )
      rank_parameters.setRankProjectFile( "project-crystal.pdrp" )
      self.user_defaults.setRank_parameters( rank_parameters )
    #
    # Starting up Executive GUI Proxy thread
    #
    Debug.out("Starting up Executive GUI Proxy Thread", 1)
    self._DNA_LoggerThread = None
    try:
      self._DNA_LoggerThread = DNA_Logger(self)
      self._DNA_LoggerThread.start()
    except:
      self.logwrite("Problem starting up Executive GUI Proxy Thread!")
      raise
    #
    # Starting up Executive DB Proxy
    #
    Debug.out("Starting up Executive DB Proxy", 1)
    self._ExecutiveDBProxy = None
    try:
      self._ExecutiveDBProxy = ExecutiveDBProxy(self)
    except:
      self.logwrite("Problem starting up Executive DB Proxy!")
      raise
    #
    #
    #
    self.status = "STARTUP"
    #
    # DNAHOME and DNANAME
    #
    if not "DNAHOME" in os.environ.keys():
      raise "ERROR: DNAHOME not defined"
    self.dnahome = os.environ["DNAHOME"]
    if not "DNANAME" in os.environ.keys():
      raise "ERROR: DNANAME not defined"
    self.dnaname = os.environ["DNANAME"]
    #
    # Path to log file
    #
    ESContext.set_path_to_log_file(None)
    #
    #
    #
    self.ES_host = self.server_data.getExpert_system_host_name()
    if "HOSTNAME" in os.environ.keys():
      if (self.ES_host != os.environ["HOSTNAME"]):
        self.logwrite("Warning! Actual host name for the Executive System is '"+os.environ["HOSTNAME"]+"'")
        self.logwrite("while in the system defaults configuration file it is set to '"+self.ES_host+"'")
    #
    # Get all relevant server data
    #
    self.BCM_host          = self.server_data.getBcm_host_name()
    self.BCM_port          = self.server_data.getBcm_port_number()
    self.DB_host           = self.server_data.getDb_host_name()
    self.DB_port           = self.server_data.getDb_port_number()
    self.ES_port           = self.server_data.getExpert_system_port_number()
    self.DPM_host          = self.server_data.getDpm_host_name()
    
    if self.local_info is None:
      self.dna_contact_email = None
    else:
      self.dna_contact_email = self.local_info.getDna_contact_email()
    #
    # Default run_number
    #
    self.run_number = self.default_values.getFileinfo().getRun_number()
    #
    #
    # Period of time before a time out exception is thrown
    #
    self.timeout_length = self.server_data.getTimeout_length()
    #
    # Create condition object and abort event object 
    #
    self.condition = threading.Condition()
    self.abort_event = threading.Event()
    #
    #
    #
    self.logwrite("")
    self.logwrite("")
    self.logwrite("Startup of the Executive System")
    self.logwrite( self.version_long )
    self.logwrite("DNAHOME = %s" % (self.dnahome))
    self.logwrite("DNANAME = %s" % (self.dnaname))
    Debug.out("GUI host = %s, port = %d" % (self.GUI_host, self.GUI_port) , 1)
    Debug.out("Time out set to %d seconds" % self.timeout_length , 1)
    if not self.dna_contact_email is None:
      Debug.out("DNA contact email: %s" % self.dna_contact_email , 1)
    #
    # Check if DNATMPDIR is present, if it is check that it is writable
    #
    if "DNATMPDIR" in os.environ.keys():
      if not os.access(os.environ["DNATMPDIR"], os.W_OK):
        self.logwrite("ERROR! The environment variable DNATMPDIR is set to %s" % os.environ["DNATMPDIR"])
        self.logwrite("but this directory is not writable. You will be not able to continue to use DNA.")
        self.logwrite("Please make sure this variable is set to a writable directory, e.g. /tmp.")
        raise "DNATMPDIR not writable"
    # Check the detector type in the ES config - this is a fix for bug 911
    #
    detector_type = self.default_values.getDetector().getType()
    if not DetectorDatabase.check_if_known_detector_type(detector_type):
        self.logwrite("Warning! Unknown detector type '%s' in the user_defaults.xml configuration file." % detector_type)
    #
    # Create proxy object for Data Collection Module
    #
    self.logwrite("BCM host = %s, port = %d" % (self.BCM_host, self.BCM_port))
    #
    # Try to connect to the BCM...
    #
    self.check_bcm_communication()
    #
    # Initialize the DPM
    #
    try:
      self.init_dpm()
    except exceptions.Exception, e:
      self.logwrite("Problem starting DPM")
      self.logwrite(str(e))
      raise
    except:
      self.logwrite("Problem connecting to Data Processing Module!")
      raise
    Debug.out("DPM module connected successfully", 1)
    #
    # Info about any extra commands
    #
    Debug.out("DNA configuration user defaults: %s" % pprint.pformat(self.user_defaults.toDict()),1)     
    Debug.out("XSD file: %s" % XSD.__file__,1)
    extra_index_commands = self.index_parameters.getExtra_index_commands()
    if not extra_index_commands is None:
      mosflm_commands = extra_index_commands.getMosflm_commands()
      if not mosflm_commands is None:
        commands = mosflm_commands.getCommand()
        for command in commands:
          self.logwrite("Extra MOSFLM index command: %s" % command )
    #
    if not self.integrate_parameters is None:
      extra_integrate_commands = self.integrate_parameters.getExtra_integrate_commands()
      if not extra_integrate_commands is None:
        mosflm_commands = extra_integrate_commands.getMosflm_commands()
        if not mosflm_commands is None:
          commands = mosflm_commands.getCommand()
          for command in commands:
            self.logwrite("Extra MOSFLM integrate command: %s" % command )     
    #
    #
    #
    if not self.strategy_parameters is None:
      extra_strategy_commands = self.strategy_parameters.getExtra_strategy_commands()
      if not extra_strategy_commands is None:
        mosflm_commands = extra_strategy_commands.getMosflm_commands()
        if not mosflm_commands is None:
          commands = mosflm_commands.getCommand()
          for command in commands:
            self.logwrite("Extra MOSFLM strategy command: %s" % command )     
    #
    # Starting up HTTP Server Thread
    #
    Debug.out("Starting up Executive System HTTP Server Thread", 1)
    Debug.out("HTTP server host = %s, port = %d" % (self.ES_host, self.ES_port), 1)
    self._HTTPServerThread = None
    try:
      self._HTTPServerThread = HTTPServerThread(self, self._DNA_LoggerThread, \
                                                self._ExecutiveDBProxy, \
                                                self.ES_host, self.ES_port, self.GUI_host, self.GUI_port)
      self._HTTPServerThread.start()
    except:
      self.logwrite("Problem starting up HTTP Server Thread!")
      raise
    http_server_status = self._HTTPServerThread.getHttp_server_status()
    if http_server_status.getCode() == "ok":
      Debug.out("HTTP server started up sucessfully", 1)
    else:
      self.logwrite("Problem with HTTP server:")
      self.logwrite("%s"%(http_server_status.getMessage()))
      self.status = "SHUTDOWN"
    #
    # Compile regular expression
    #
    response = re.compile(".*response")
    #
    # Go into main loop
    #
    while (self.status != "SHUTDOWN"):
      #
      #
      #
      self.response_type = None
      self.response = None
      #
      #
      #
      try:
        self.set_gui_bcm_status( "Ready" )
        self.set_gui_dpm_status( "Ready" )
        if self.get_status() != "IDLE":
          self.set_status("IDLE")
        #
        # Wait for new command
        #
        if self.command_queue.empty():
          self.logwrite("Waiting for new command")
          self.condition.acquire()
          self.condition.wait()
          self.condition.release()
        #
        # Here we should have a command on the queue
        #
        if self.command_queue.empty():
          self.logwrite( "ERROR! Severe error in ES - empty command queue." )
          raise "ExecutiveException", "Epmty command queue"
        #
        # 
        #
        command_xml      = self.command_queue.get(0)
        self.command     = command_xml[0]
        self.xml_message = command_xml[1]
        self.logwrite( "New command: %s" % self.command, 2 )
        Debug.out("XML message: %s" % self.xml_message, 2)
        #
        # Check if command is compatible with the ES state...
        # Check if uncaught response. If a command at this point is
        # ending with "response" something is wrong.
        #
        if (not self.check_command(self.command)) or (not response.match(self.command) is None):
          #
          #
          #
          Debug.out("Error processing command: %s" % self.command,1)
          #
          #
          #
        elif self.command == "/characterize_crystal_request":
          #
          #
          #
          self.request = XSD.Characterize_crystal_request()
          self.response = XSD.Characterize_crystal_response()
          self.response_type = "/characterize_crystal_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_characterize_crystal_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("CHARACTERIZE_CRYSTAL_REQUEST")
          else:
            ESContext.set_latest_state("STARTING")            
          #
          #
          #
        elif self.command == "/collect_request":
          #
          #
          #
          self.request = XSD.Collect_request()
          self.response = XSD.Collect_response()
          self.response_type = "/collect_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_collect_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("COLLECT_REQUEST")
          else:
            ESContext.set_latest_state("STARTING")            
          #
          #
          #
        elif self.command == "/collect_reference_request":
          #
          #
          #
          self.request = XSD.Collect_reference_request()
          self.response = XSD.Collect_reference_response()
          self.response_type = "/collect_reference_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_collect_reference_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("COLLECT_REFERENCE_REQUEST")
          else:
            ESContext.set_latest_state("STARTING")            
          #
          #
          #
        elif self.command == "/collect_automatically_request":
          #
          #
          #
          self.request = XSD.Collect_automatically_request()
          self.response = XSD.Collect_automatically_response()
          self.response_type = "/collect_automatically_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_collect_automatically_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("COLLECT_AUTOMATICALLY_REQUEST")
          else:
            ESContext.set_latest_state("STARTING")            
          #
          #
          #
        elif self.command == "/index_request":
          #
          #
          #
          self.request = XSD.Index_request()
          self.response = XSD.Index_response()
          self.response_type = "/index_response"
          self.request.unmarshal(self.xml_message)
          #
          # Go offline
          #
          self.go_offline()
          #
          # Do the index request
          #
          self.response = self.do_index_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("INDEX_REQUEST")
          else:
            ESContext.set_latest_state("STARTING")            
          #
          #
          #
        elif self.command == "/strategy_request":
          #
          #
          #
          self.request = XSD.Strategy_request()
          self.response = XSD.Strategy_response()
          self.response_type = "/strategy_response"
          self.request.unmarshal(self.xml_message)
          #
          # Go offline
          #
          self.go_offline()
          #
          # Do the strategy request
          #
          self.response = self.do_strategy_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("STRATEGY_REQUEST")
          #
          #
          #
        elif self.command == "/kappa_alignment_request":
          #
          #
          #
          self.request = XSD.Kappa_alignment_request()
          self.response = XSD.Kappa_alignment_response()
          self.response_type = "/kappa_alignment_response"
          self.request.unmarshal(self.xml_message)
          #
          # Go offline
          #
          #self.go_offline()
          #
          # Do the strategy request
          #
          self.response = self.do_kappa_alignment_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("STRATEGY_REQUEST")
          #
          #
          #
        elif self.command == "/kappa_strategy_request":
          #
          #
          #
          self.request = XSD.Kappa_strategy_request()
          self.response = XSD.Kappa_strategy_response()
          self.response_type = "/kappa_strategy_response"
          self.request.unmarshal(self.xml_message)
          #
          # Go offline
          #
          #self.go_offline()
          #
          # Do the strategy request
          #
          self.response = self.do_kappa_strategy_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("STRATEGY_REQUEST")
          #
          #
          #
        elif self.command == "/collect_data_request":
          #
          #
          #
          self.request = XSD.Collect_data_request()
          self.response = XSD.Collect_data_response()
          self.response_type = "/collect_data_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_collect_data_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("COLLECT_DATA_REQUEST")
          #
          #
          #
        elif self.command == "/screen_request":
          #
          #
          #
          self.request = XSD.Screen_request()
          self.response = XSD.Screen_response()
          self.response_type = "/screen_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_screen_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("SCREEN_REQUEST")
          #
          #
          #
        elif self.command == "/rank_request":
          #
          #
          #
          self.request = XSD.Rank_request()
          self.response = XSD.Rank_response()
          self.response_type = "/rank_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_rank_request( self.request )
          if self.response.getStatus( ).getCode( ) != "error":
            ESContext.set_latest_state( "RANK_REQUEST" )
          #
          #
          #
        elif self.command == "/proposal_request":
          #
          #
          #
          self.request = XSD.Proposal()
          self.response = XSD.Proposal_response()
          self.response_type = "/proposal_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_proposal_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("SCREEN_REQUEST")
          #
          #
          #
        elif self.command == "/loaded_samples_request":
          #
          #
          #
          self.request = XSD.Loaded_samples_request()
          self.response = XSD.Loaded_samples_response()
          self.response_type = "/loaded_samples_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_loaded_samples_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("SCREEN_REQUEST")
          #
          #
          #
        elif self.command == "/cell_refinement_request":
          #
          #
          #
          self.request = XSD.Cell_refinement_request()
          self.response = XSD.Cell_refinement_response()
          self.response_type = "/cell_refinement_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_cell_refinement_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("CELL_REFINEMENT_REQUEST")
          #
          #
          #
        elif self.command == "/quick_scale_request":
          #
          #
          #
          self.request = XSD.Quick_scale_request()
          self.response = XSD.Quick_scale_response()
          self.response_type = "/quick_scale_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_quick_scale_request(self.request)
          if self.response.getStatus().getCode() != "error":
            ESContext.set_latest_state("QUICK_SCALE_REQUEST")
          #
          #
          #
        elif self.command == "/feedback_request":
          #
          #
          #
          self.request = XSD.Feedback_request()
          self.response = XSD.Feedback_response()
          self.response_type = "/feedback_response"
          self.request.unmarshal(self.xml_message)
          self.response = self.do_feedback_request(self.request)
          #
          #
          #
        elif self.command == "/shutdown":
          self.set_status("SHUTDOWN")
        elif self.command == "/abort":
          self.logwrite("Abort command received")
          self.abort_event.clear()
        else:
          self.logwrite("Unknown command %s received"%(self.command))
          self.logwrite("Known commands are: /collect_request")
          self.logwrite("                    /collect_reference_request")
          self.logwrite("                    /collect_data_request")
          self.logwrite("                    /screen_request")
          self.logwrite("                    /rank_request")
          self.logwrite("                    /index_request")
          self.logwrite("                    /strategy_request")
          self.logwrite("                    /characterize_crystal_request")
          self.logwrite("                    /cell_refinement_request")
          self.logwrite("                    /quick_scale_request")
          self.logwrite("                    /feedback_request")
          self.logwrite("                    /shutdown")
          self.logwrite("                    /abort")
        #
        # Send response to the GUI
        #
        if not self.response is None:
          self.gui_send_message(self.response_type, self.response);
        #
        #
        #
      except "ExecutiveException", e:
        #
        # This is a fix for bug 814, i.e. abort in GUI doesn't abort the
        # BCM.
        #
        if e == "abort":
          #
          # try to abort BCM
          #
          if ESContext.isOnline():
            self.logwrite("Trying to abort BCM.")
            self.bcm_abort()
          #
          # Try to abort DPM - fix for bug 859
          #
          self.logwrite("Trying to abort DPM.")
          self.do_dpm_abort_request(None)
          #
          #
          # Write a message to the user
          #
          self.logwrite("DNA system aborted!")
          #
          #
          #
        else:
          #
          # Not an abort so something more serious!
          #
          message = "ERROR! %s" % e
          self.logwrite(message)
          status = XSD.Status()
          status.setCode("error")      
          status.setMessage(message)
          self.response.setStatus(status)
          #
          # Commented out because of problem with
          # message terminated too early in the GUI
          #
          #self._HTTPServerThread.setBcm_status(status)
          self.gui_send_message(self.response_type, self.response);
          #
          # Do not send email (fix for bug 832)
          #
          #subject = "DNA ERROR on %s: %s" % (self.dnaname, message)
          #self.send_email(subject, message)

      except:
        #
        # This is not an ExecutiveException so something more nasty
        # must have happened.
        #
        self.logwrite("ERROR! Internal error in the DNA system.")
        self.logwrite("The following information is for the DNA developers:")
        # Print stack trace. 
        #
        self.print_traceback()
        #
        message = "Exception: %s" % sys.exc_type
        self.logwrite(message)
        status = XSD.Status()
        status.setCode("error")      
        status.setMessage(message)
        self.response.setStatus(status)
        #
        # Commented out because of problem with
        # message terminated too early in the GUI
        #
        #self._HTTPServerThread.setBcm_status(status)
        self.gui_send_message(self.response_type, self.response);
        #
        # Sleep 2 seconds in order to allow for messages to be
        # recorded in the DNA logger.
        #
        self.logwrite("Please wait, sending error message to DNA local contact")
        time.sleep(2)
        #
        # Send email to contact person (if configured)
        #
        subject = "DNA ERROR on %s: %s" % (self.dnaname, message)
        self.send_email(subject, message)
        if Debug.getLevel() > 0:
          raise

    #
    # Shutdown the ES
    #
    self.logwrite("Shutting down the ExecutiveSystem")
    self._HTTPServerThread.shutdown()
    time.sleep(1)      


  def check_command(self, command):
    """Checks that the command is compatible with the ES status."""
    #
    flag = True
    if (command == "/strategy_request" or command == "/kappa_strategy_request") and  not ESContext.get_latest_state() in ["INDEX_REQUEST", "STRATEGY_REQUEST", "COLLECT_DATA_REQUEST", "CHARACTERIZE_CRYSTAL_REQUEST"]:
        #
        #
        #
        self.logwrite("Error! The DNA system cannot start a strategy calculation")
        self.logwrite("before a successful indexing has been performed.")
        strategy_response = XSD.Strategy_response()
        status = XSD.Status()
        status.setCode("error")
        status.setMessage("Strategy calculation not possible before a successful indexing has been performed.")
        strategy_response.setStatus(status)
        self.gui_send_message("/strategy_response", strategy_response)
        flag = False
        #
        #
        #
    elif command == "/cell_refinement_request":
      #
      #
      #
      if not ESContext.get_latest_state() in ["STRATEGY_REQUEST", "COLLECT_DATA_REQUEST", "CELL_REFINEMENT_REQUEST", "CHARACTERIZE_CRYSTAL_REQUEST", "INTEGRATE_REQUEST"]:
        #
        #
        #
        self.logwrite("Error! The DNA system cannot start a cell refinement calculation")
        self.logwrite("before a successful strategy calculation has been performed.")
        flag = False
        #
        #
        #
    elif command == "/integrate_request":
      #
      #
      #
      if not ESContext.get_latest_state() in ["CELL_REFINEMENT_REQUEST", "INTEGRATE_REQUEST"]:
        #
        #
        #
        self.logwrite("Error! The DNA system cannot start integration")
        self.logwrite("before a successful cell refinement has been performed.")
        flag = False
        #
        #
        #
    elif command == "/collect_data_request":
      #
      #
      #
      if not ESContext.get_latest_state() in ["STRATEGY_REQUEST", "CHARACTERIZE_CRYSTAL_REQUEST", "COLLECT_DATA_REQUEST", "QUICK_SCALE_REQUEST" ]:
        #
        #
        #
        self.logwrite("Error! The DNA system cannot data collection")
        self.logwrite("before a successful strategy calculation has been performed.")
        collect_data_response = XSD.Collect_data_response()
        status = XSD.Status()
        status.setCode("error")
        status.setMessage("Data Collection not possible before a successful strategy calculation.")
        collect_data_response.setStatus(status)
        self.gui_send_message("/collect_data_response", collect_data_response)
        flag = False
        #
        #
        #
    #
    #
    #
    return flag



  def send_message(self, command, xml_message):
    """Called by other threads, mainly the ExecutiveHTTPServer.
    The arguments is the command, e.g. "/index_request", and the corresponding
    XML request."""
    #
    #
    #
    Debug.out("ExecutiveSystem.send_message: command = %s"%(command),1)
    self.command_queue.put( [command, xml_message] )
    Debug.out("ExecutiveSystem.send_message: xml_message = %s"%(pprint.pformat(xml_message)),1)
    self.condition.acquire()
    if command == "/abort":
      self.abort_event.set()
    self.condition.notify()
    self.condition.release()


  def check_abort(self):
    """Called for checking if the ES has received an abort event.
    If this is the case an "ExecutiveException" is raised with the argument "abort"."""
    #
    # 
    #
    if self.abort_event.isSet():
      self.abort_event.clear()
      raise "ExecutiveException", "abort"



  def do_proposal_request(self, proposal):
    """Sends and asynchronous proposal request to the DB proxy"""
    self.db_send_asynchronous_command( "/proposal_request", proposal, "/proposal_response" )
    self.wait_for_response( "/proposal_response" )
    proposal_response = XSD.Proposal_response()
    proposal_response.unmarshal(self.xml_message)
    return proposal_response


  def do_loaded_samples_request(self, loaded_samples_request ):
    """Sends and asynchronous loaded samples request to the DB proxy"""
    self.db_send_asynchronous_command( "/loaded_samples_request", loaded_samples_request, "/loaded_samples_response" )
    self.wait_for_response( "/loaded_samples_response" )
    loaded_samples_response = XSD.Loaded_samples_response()
    loaded_samples_response.unmarshal(self.xml_message)
    return loaded_samples_response



  def do_collect_request(self, _oxsdCollect_request, wait_for_response=True, set_context_flag=True):
    """Calls the BCM to do a collect request. It's called by
    the do_collect_reference_request and the do_collect_data_request."""
    #
    # Make a copy of the incoming request
    #
    collect_request = _oxsdCollect_request.copy()
    #
    # Try to go online
    #
    self.try_to_go_online()
    #
    #
    #
    fileinfo             = collect_request.getFileinfo()
    oscillation_sequence = collect_request.getOscillation_sequence()[0]
    #
    # Set the collect context
    #
    if fileinfo.getSuffix() is None:
      suffix = DetectorDatabase.determine_suffix(ESContext.get_bcm_detector())
      fileinfo.setSuffix(suffix)
    ESContext.set_bcm_fileinfo( fileinfo )
    ESContext.set_bcm_oscillation_sequence( oscillation_sequence )
    ESContext.set_bcm_collect_resolution( collect_request.getResolution() )
    #
    #
    #
    number_of_images = oscillation_sequence.getNumber_of_images()
    start_image_number = oscillation_sequence.getStart_image_number()
    prefix = fileinfo.getPrefix()
    run_number = fileinfo.getRun_number()
    suffix = fileinfo.getSuffix()
    directory = fileinfo.getDirectory()
    first_image = "%s_%d_%03d.%s" % (prefix, run_number, start_image_number, suffix)
    if number_of_images == 1:
      self.logwrite( "Collecting image %s in directory %s" % (first_image, directory) )
    else:
      self.logwrite( "Collecting %d images, first image %s, in directory %s" % \
                    (number_of_images, first_image, directory) )
    self.set_gui_bcm_status( "Collecting: %s_%d" % ( prefix, run_number ) )
    #
    # The ES changes it's status to COLLECT_REQUEST
    #
    old_status = self.get_status()
    self.set_status("COLLECT_REQUEST")
    Debug.out( "Collect request received: %s" % pprint.pformat( collect_request.toDict() ), 2 )
    #
    # Save the collect_request fileinfo in order to know in the
    # index request if the bcm parameters are valable
    #
    self.collect_directory = collect_request.getFileinfo().getDirectory()
    self.collect_prefix = collect_request.getFileinfo().getPrefix()
    #
    # Check that the directory exists!
    #
    if not os.access(self.collect_directory, os.W_OK):
      try:
        self.logwrite("Directory %s does not exist - creating it" % self.collect_directory)
        os.system("mkdir -p %s" % self.collect_directory)
      except:
        raise "ExecutiveException", "Failed to make directory %s" % self.collect_directory
      if not os.access(self.collect_directory, os.W_OK):
        self.logwrite("Tried to create the directory %s but it's not accessible" % self.collect_directory)
        raise "ExecutiveException", "Directory %s not accessible" % self.collect_directory
    #
    # Check if keep loaded sample
    #
    if collect_request.getKeep_sample_loaded():
        self.logwrite( "Keeping the sample loaded." )
    #
    # The collect request
    #
    Debug.out("Sending a collect request to the BCM with the following parameters:", 1)
    Debug.out("Directory: %s"%(collect_request.getFileinfo().getDirectory()), 1)
    Debug.out("Prefix: %s"%(collect_request.getFileinfo().getPrefix()), 1)
    Debug.out("Run number: %s"%(collect_request.getFileinfo().getRun_number()), 1)
    Debug.out("Start image number: %s"%(collect_request.getOscillation_sequence()[0].getStart_image_number()), 1)
    Debug.out("Start oscillation angle: %s"%(collect_request.getOscillation_sequence()[0].getStart()), 1)
    Debug.out("Range: %s"%(collect_request.getOscillation_sequence()[0].getRange()), 1)
    Debug.out("Number of images: %s"%(collect_request.getOscillation_sequence()[0].getNumber_of_images()), 1)
    Debug.out("Overlap: %s"%(collect_request.getOscillation_sequence()[0].getOverlap()), 1)
    Debug.out("Exposure time: %s"%(collect_request.getOscillation_sequence()[0].getExposure_time()), 1)
    Debug.out("Resolution : %s" % collect_request.getResolution().getUpper(), 1)
    if not collect_request.getSample_reference() is None:
      Debug.out("Code : %s" % collect_request.getSample_reference().getCode(), 1)
    #
    # Set the current collect resolution in the ESContext
    #
    ESContext.set_bcm_collect_resolution( collect_request.getResolution() )
    #
    #
    #
    status = self.bcm_send_command("/collect_request", collect_request)
    self.collecting_flag = True
    self.latest_collect_response = None
    #
    # Should we wait?
    #
    if wait_for_response:
      #
      # The ES waits till the images have been collected.
      #
      self.wait_for_response("/collect_response")
      collect_response = XSD.Collect_response()
      collect_response.unmarshal(self.xml_message)
      self.latest_collect_response = collect_response
      Debug.out("Collect response status code: %s" % collect_response.getStatus().getCode(), 1)
      #
      # If no warning, check that the last image is there - wait a while if it's not:
      #
      if collect_response.getStatus().getCode() == "ok":
        start_image_no     = collect_request.getOscillation_sequence()[0].getStart_image_number()
        no_images          = collect_request.getOscillation_sequence()[0].getNumber_of_images()
        last_image_no      = start_image_no + no_images - 1
        path_to_last_image = ESContext.getPath_to_image(collect_request.getFileinfo(), last_image_no)
        time_to_wait       = 20 # seconds
        found_flag         = False
        log_message_flag   = False
        while (time_to_wait > 0):
          #
          # This command is necessary in order to work around a problem
          # on the ESRF ID23-2 and BM14 beamlines who use MAR 225 detectors.
          # If this "ls" command is not issued the image appears only after ca 20 s.
          # This is probably not a NFS but a detector software problem.
          #
          os.system( "ls %s > /dev/null" % collect_request.getFileinfo().getDirectory() )
          if os.access(path_to_last_image, os.F_OK):
            found_flag = True
            time_to_wait = -1
          else:
            if not log_message_flag:
              self.logwrite("Collect response received, waiting for the last image to be written to disk...")
              log_message_flag = True
            time.sleep(1)
            time_to_wait = time_to_wait - 1
        if not found_flag:
          status = XSD.Status()
          status.setCode("error")
          status.setMessage("The last collected image %s is not accessible!" % path_to_last_image)
          collect_response.setStatus(status)
        #
        # Set the ES BCM status panel
        #
        self.set_gui_bcm_status( "Ready" )
      #
      #
      #
      self.collecting_flag = False
      #
      # Restore the old status
      #
      self.set_status(old_status)
      #
      # Return a collect response message.
      #
      return collect_response


  def do_collect_reference_request( self, _oxsdCollect_reference_request, wait_for_response=True, set_context_flag=True ):
    """Does the collection of reference images."""
    collect_reference_response = XSD.Collect_reference_response()
    #
    # Make a copy of the incoming request
    #
    collect_reference_request = _oxsdCollect_reference_request.copy()
    self.logwrite("Collecting reference images")
    Debug.out("Collect reference request received: %s"%(pprint.pformat(collect_reference_request.toDict())),2)
    #
    # Check if archiving of reference images is activated
    #
    if collect_reference_request.getArchive_reference_images():
      self.logwrite("Archiving of reference images enabled")
      ESContext.setArchive_reference_images( True )
    else:
      self.logwrite("Archiving of reference images disabled")
      ESContext.setArchive_reference_images( False )
    #
    #
    #
    collect_request = collect_reference_request.getCollect_request()
    #
    # The ES examines the incoming collect message.
    #
    fileinfo = collect_request.getFileinfo()
    prefix = fileinfo.getPrefix()
    if prefix == None:
        status = XSD.Status()
        status.setCode( "error" )
        message = "ERROR! No prefix in the collect request"
        status.setMessage( message )
        self.logwrite( message )
        collect_reference_response.setStatus( status )
        return collect_reference_response
    if set_context_flag:
      ESContext.set_bcm_fileinfo( fileinfo )
      fileinfo = ESContext.get_bcm_fileinfo()
      ESContext.set_bcm_database_comment( collect_request.getComment() )
    #
    # Check if the run number needs to be determined
    #
    #if fileinfo.getRun_number() is None:
    #  fileinfo.setRun_number( self.determineRunNumber( fileinfo ) )
    #
    # Save the collect_request fileinfo in order to know in the
    # index request if the bcm parameters are valable
    #
    self.collect_directory = collect_request.getFileinfo().getDirectory()
    self.collect_prefix = collect_request.getFileinfo().getPrefix()
    #
    # The ES changes it's status to COLLECT_REQUEST
    #
    self.set_status("COLLECT_REQUEST")
    Debug.out( "Collect request received: %s" % pprint.pformat( collect_request.toDict() ), 2 )
    #
    # Set up the fileinfo
    #
    ref_fileinfo = fileinfo.copy()
    #
    # Check that the prefix hasn't already a "ref-" prefix
    #
    if len(prefix) >= 4:
      if prefix[0:4] == "ref-":
        ref_fileinfo.setPrefix( prefix)
      else:
        ref_fileinfo.setPrefix( "ref-" + prefix )
    ref_fileinfo.setRun_number(  fileinfo.getRun_number()      )
    #
    # Determine unique run number
    #
    ref_fileinfo = self.fileinfo_with_unique_run_number( ref_fileinfo )
    #
    # Set up the oscillation sequence
    #
    ESContext.set_bcm_oscillation_sequence(collect_request.getOscillation_sequence()[0])
    current_oscillation_sequence = ESContext.get_bcm_oscillation_sequence()
    #
    #
    #
    oscillation_sequence = XSD.Oscillation_sequence()
    oscillation_sequence.setStart(0.0)
    oscillation_sequence.setRange(current_oscillation_sequence.getRange())
    oscillation_sequence.setNumber_of_images(2)
    oscillation_sequence.setOverlap(-89.0)
    oscillation_sequence.setExposure_time(current_oscillation_sequence.getExposure_time())
    oscillation_sequence.setStart_image_number(1)
    oscillation_sequence.setNumber_of_passes(current_oscillation_sequence.getNumber_of_passes())
    #
    #    The ES sends a collect request message to the BCM, two images
    #    one at current phi position and one at 90 degrees.
    #
    # Make a new instance of Collect_request in order to not overwrite
    # values in the incoming collect_request.
    #
    collect_request_1 = XSD.Collect_request()
    collect_request_1.setFileinfo(ref_fileinfo)
    collect_request_1.addOscillation_sequence(oscillation_sequence)
    if not collect_request.getResolution() is None:
      collect_request_1.setResolution(collect_request.getResolution())
    collect_request_1.setComment(ESContext.get_bcm_database_comment())
    if not collect_request.getSample_reference() is None:
      collect_request_1.setSample_reference(collect_request.getSample_reference())
    iSessionId = collect_request.getSessionId()
    if not iSessionId is None:
      collect_request_1.setSessionId( iSessionId )
      ESContext.set_bcm_sessionId( iSessionId )
    collect_request_1.setIgnore_maximum_exposure( collect_request.getIgnore_maximum_exposure() )
    #
    #
    #
    self.logwrite("Collecting two reference images for indexing")
    self.logwrite("Directory              : %s " % collect_request_1.getFileinfo().getDirectory() )
    self.logwrite("Prefix                 : %s " % collect_request_1.getFileinfo().getPrefix() )
    self.logwrite("Run number             : %d " % collect_request_1.getFileinfo().getRun_number() )
    self.logwrite("Range                  : %.2f " % collect_request_1.getOscillation_sequence()[0].getRange() )
    self.logwrite("Exposure time          : %.2f " % collect_request_1.getOscillation_sequence()[0].getExposure_time() )
    self.logwrite("Resolution (requested) : %.2f " % collect_request_1.getResolution().getUpper() )
    if not collect_request_1.getComment() is None:
        self.logwrite("Database comment : %s" % collect_request_1.getComment())
    if not iSessionId is None:
        self.logwrite("Session Id             : %d" % iSessionId, 2 )
    if collect_request.getKeep_sample_loaded():
        collect_request_1.setKeep_sample_loaded( True )
    	self.logwrite("Keeping the sample loaded.", 2 )
    #
    #
    #
    collect_response = self.do_collect_request(collect_request_1, wait_for_response, set_context_flag)
    self.latest_collect_response = collect_response
    #
    # Check the status flag
    # 
    if (not wait_for_response):
      return
    elif collect_response.getStatus().getCode() == "error":
      self.logwrite("ERROR! %s" %  collect_response.getStatus().getMessage())
    else:
      #
      # Arcive reference images?
      #
      if ESContext.isArchive_reference_images():
        #
        # Yes!
        #
        if self.local_info is None:
          archive_directory = None
        else:
          archive_directory = self.local_info.getArchive_directory()
        if not archive_directory is None:
          try:
            path_image1 = ESContext.getPath_to_image(ref_fileinfo, 1)
            Debug.out("Path to image 1: %s" % path_image1, 1)
            os.system("cp %s %s &" % (path_image1, archive_directory))
            path_image2 = ESContext.getPath_to_image(ref_fileinfo, 2)
            Debug.out("Path to image 2: %s" % path_image2, 1)
            os.system("cp %s %s &" % (path_image2, archive_directory))
          except:
            logwrite("Error - couldn't copy reference images to archive.")
      #
      # Restore fileinfo and oscillation_sequence context
      #
      ESContext.set_bcm_fileinfo(fileinfo)
      ESContext.set_bcm_oscillation_sequence(oscillation_sequence)
      #
      #
      #
    #
    # register actual Kappa settings
    #
    # Functionality moved to wait_for_response
    #
    #kappa_settings=XSD.Kappa_collect_settings()
    #kappa_settings = MosflmInterface.do_kappa_bcm_settings_request()
    #try:
    #  kappa_fname = os.path.join(ESContext.get_bcm_fileinfo().getDirectory(),ESContext.get_bcm_fileinfo().getTemplate()+"_kappa_settings.xml")
    #except:
    #  kappa_fname=""
    #try:
    #    ostream = open( kappa_fname, "w" )
    #    ostream.write( kappa_settings.marshal() )
    #    ostream.close()
    #except:
    #  self.logwrite("Kappa settings could not been registered")
    #
    #
    #
    collect_reference_response.setStatus( collect_response.getStatus() )
    return collect_reference_response


  def do_collect_automatically_request( self, _oxsdCollect_automatically_request ):
    """
    Complete automatic data collection.
    """
    oxsdCollect_automatically_response = XSD.Collect_automatically_response()
    oxsdStatus = XSD.Status()
    oxsdStatus.setCode( "ok" )
    oxsdCollect_automatically_response.setStatus( oxsdStatus )
    #
    # Make a copy of the incoming request
    #
    oxsdCollect_automatically_request = _oxsdCollect_automatically_request.copy()
    self.logwrite( "-----------------------------------------------------------" )
    self.logwrite( "\n\nStarting automatic data collection\n\n" )
    #
    #
    #
    for oxsdCharacterize_crystal_request in oxsdCollect_automatically_request.getCharacterize_crystal_request():
        self.logwrite( "\n\nAutomatic data collection: characterizing crystal\n" )
        oxsdCharacterize_crystal_response = self.do_characterize_crystal_request( oxsdCharacterize_crystal_request )
        if oxsdCharacterize_crystal_response.getStatus().getCode() == "ok":
            oxsdStrategy_response = oxsdCharacterize_crystal_response.getStrategy_response()
            oxsdStrategy_interpretation = oxsdStrategy_response.getStrategy_interpretation()
            oxsdCollect_data_request = XSD.Collect_data_request()
            oxsdCollect_reference_request = oxsdCharacterize_crystal_request.getCollect_reference_request()
            oxsdCollect_request = oxsdCollect_reference_request.getCollect_request()
            oxsdCollect_request.clearOscillation_sequence()
            oxsdCollect_request.addOscillation_sequence( oxsdStrategy_interpretation.getOscillation_sequence()[0] )
            oxsdCollect_request.setIgnore_maximum_exposure( True )
            oxsdCollect_request.setKeep_sample_loaded( False )
            oxsdCollect_data_request.setCollect_request( oxsdCollect_request )
            oxsdCollect_data_request.setIntegrate( True )
            self.logwrite( oxsdCollect_data_request.marshal_with_indentations() )
            self.logwrite( "\n\nAutomatic data collection: Collecting data\n" )
            oxsdCollect_data_response = self.do_collect_data_request( oxsdCollect_data_request )
            if oxsdCollect_data_response.getStatus().getCode() == "ok":
                oxsdQuick_scale_response = self.do_quick_scale_request()
        else:
            self.logwrite( "Automatic data collection failed!" )
            self.logwrite( "Reason: %s" % oxsdCharacterize_crystal_response.getStatus().getMessage() )
    #
    #
    #
    return oxsdCollect_automatically_response
    
        


  def do_collect_data_request(self, _oxsdCollect_data_request):
    """Does the collection of data. If the integrate flag in
    the collect data request is set the method will also call the DPM
    to integrate the data. This method now includes experimental
    code for concurrent collection and integration.

    If the collect_data_request asks for more than 999 images an
    error message is returned - the DNA system does not yet support
    collections of more than 999 images becasue the template is
    fixed to three digits."""
    #
    # Make a copy of the incoming request
    #
    oxsdCollect_data_request = _oxsdCollect_data_request.copy()
    oxsdCollect_request = oxsdCollect_data_request.getCollect_request()
    oxsdCollect_data_response = XSD.Collect_data_response()
    oxsdStatus = XSD.Status()
    oxsdOscillation_sequence = oxsdCollect_request.getOscillation_sequence()[0]
    ESContext.set_dpm_oscillation_sequence( oxsdOscillation_sequence )
    #
    # Determine unique run number
    #
    oxsdFileinfo_unique = self.fileinfo_with_unique_run_number( oxsdCollect_request.getFileinfo() )
    oxsdCollect_request.setFileinfo( oxsdFileinfo_unique )
    #
    # Check radiation damage...
    #
    dExposureTime      = oxsdOscillation_sequence.getExposure_time()
    iNumberOfImages    = oxsdOscillation_sequence.getNumber_of_images()
    dTotalExposureTime = dExposureTime * iNumberOfImages
    #
    # Retrieve the bcm parameters response
    #
    oxsdBcm_parameters_response = self.get_bcm_parameters()
    self.logwrite( "Total exposure time for this data collection: %.1f s" % dTotalExposureTime )
    ESContext.set_bcm_parameters_response( oxsdBcm_parameters_response )
    self.logwrite( "ignore_maximum_exposure = %s" % oxsdCollect_request.getIgnore_maximum_exposure(), 2 )
    if not oxsdCollect_request.getIgnore_maximum_exposure():
        if oxsdBcm_parameters_response != None:
            oxsdBeamline_parameters = oxsdBcm_parameters_response.getBeamline_parameters()
            if oxsdBeamline_parameters != None:
                dMaximumExposureTime = oxsdBeamline_parameters.getMaximum_exposure()
                self.logwrite( "Maximum exposure time before radiation damage: %.1f" %  dMaximumExposureTime ) 
                if dMaximumExposureTime < dTotalExposureTime:
                    strMessage =   "Total exposure time %.1f s exceeds maximum exposure time of %.1f s\n" % ( dTotalExposureTime, dMaximumExposureTime)
                    dPourcent = ( dTotalExposureTime - dMaximumExposureTime ) / dMaximumExposureTime * 100
                    strMessage +=  "before radiation damage with %.1f %%!\n" % dPourcent
                    strMessage += "\n"
                    strMessage += "To precced with this data collection, click OK\n"
                    strMessage += "To cancel and edit your data collection parameters, click Cancel.\n"
                    self.logwrite( strMessage )
                    oxsdCollect_response = XSD.Collect_response()
                    oxsdStatus.setCode( "warning" )
                    oxsdStatus.setMessage( strMessage )
                    oxsdCollect_data_response.setStatus( oxsdStatus )
                    time.sleep( 1 )
                    return oxsdCollect_data_response
    #
    # This is a fix for bug 922
    #
    if oxsdOscillation_sequence.getNumber_of_images() > 999:
      self.logwrite( "ERROR! This version of DNA does not support data collections with more than 999 images." )
      oxsdStatus.setCode( "error" )
      oxsdStatus.setMessage( "Data collection with more than 999 images not supported." )
      oxsdCollect_data_response.setStatus( oxsdStatus)
      return oxsdCollect_data_response
    elif oxsdOscillation_sequence.getNumber_of_images() <=0:
      self.logwrite( "ERROR! Zero or negative number of images." )
      oxsdStatus.setCode( "error")
      oxsdStatus.setMessage( "Data collection with zero or negaitve number of images." )
      oxsdCollect_response.setStatus( status )
      return oxsdCollect_data_response
    #
    #
    #
    self.set_status( "COLLECT_DATA_REQUEST" )
    #
    # Set the context dpm fileinfo 
    #
    ESContext.set_dpm_fileinfo( oxsdFileinfo_unique )
    #
    # Send an email!
    #
    subject = "DNA collect data reqeust on %s!" % self.dnaname
    old_osc = self.last_strategy_request_oscillation_sequence
    new_osc = oxsdCollect_request.getOscillation_sequence()[0]
    message = "\n\nDNA suggested strategy followed by user strategy:\n"
    #
    # This is a fix for bug 1227:
    #
    if old_osc.getNumber_of_passes() is None:
      message = message + "Start = %4.1f, range = %3.1f, no_images = %3d, exp_time = %3.1f\n" \
                % (old_osc.getStart(), old_osc.getRange(), old_osc.getNumber_of_images(), \
                   old_osc.getExposure_time() )
      message = message + "Start = %4.1f, range = %3.1f, no_images = %3d, exp_time = %3.1f\n" \
                % (new_osc.getStart(), new_osc.getRange(), new_osc.getNumber_of_images(), \
                   new_osc.getExposure_time() )
    else:
      message = message + "Start = %4.1f, range = %3.1f, no_images = %3d, exp_time = %3.1f, no_passes = %1d\n" \
                % (old_osc.getStart(), old_osc.getRange(), old_osc.getNumber_of_images(), \
                   old_osc.getExposure_time(), old_osc.getNumber_of_passes())
      message = message + "Start = %4.1f, range = %3.1f, no_images = %3d, exp_time = %3.1f, no_passes = %1d\n" \
                % (new_osc.getStart(), new_osc.getRange(), new_osc.getNumber_of_images(), \
                   new_osc.getExposure_time(), new_osc.getNumber_of_passes())      
    if oxsdCollect_data_request.getIntegrate():
      message = message + "\nCollecting and integrating!\n"
    else:
      message = message + "\nOnly collecting - no integration.\n"
    self.send_email( subject, message, send_log=False )
    #
    # Store the strategy in the ISPyB data base
    #
    phiStart     = new_osc.getStart()
    phiEnd       = new_osc.getEnd()
    rotation     = new_osc.getRange()
    exposureTime = new_osc.getExposure_time()
    resolution   = oxsdCollect_request.getResolution().getUpper()
    if phiEnd == None:
        #
        # No phiEnd given - must calculate one
        #
        no_images = new_osc.getNumber_of_images()
        phiEnd = phiStart + no_images*rotation
        #
    screeningStrategy = XSD.ScreeningStrategy()
    screeningStrategy.setPhiStart(     phiStart )
    screeningStrategy.setPhiEnd(       phiEnd   )
    screeningStrategy.setRotation(     rotation )
    screeningStrategy.setExposureTime( exposureTime )
    screeningStrategy.setResolution(   resolution  )
    screeningStrategy.setProgram( "User defined strategy used for data collection" )
    status = self._ExecutiveDBProxy.do_store_screening_strategy( screeningStrategy )
    if status.getCode() != "ok":
        print "Error when storing user defined strategy: %s" % status.getMessage()
    #
    # First check if integration is wanted
    #
    Debug.out( "In collect_data_request, collect_data_request = %s" % oxsdCollect_data_request.marshal(),  2)
    if oxsdCollect_data_request.getIntegrate():
      #
      # Yes! First collect post refinement images!
      #
      fileinfo   = oxsdCollect_request.getFileinfo()
      directory  = fileinfo.getDirectory() 
      prefix     = fileinfo.getPrefix() 
      run_number = fileinfo.getRun_number()
      oscillation_sequence  = oxsdCollect_request.getOscillation_sequence()[0]
      exposure_time         = oxsdOscillation_sequence.getExposure_time()
      number_of_passes      = oscillation_sequence.getNumber_of_passes()
      #
      # Add "postref-" to the prefix.
      #
      fileinfo_pr = ESContext.get_dpm_fileinfo()
      fileinfo_pr.setDirectory(directory)
      fileinfo_pr.setPrefix("postref-"+prefix)
      #
      # for multi sweep, the postref should belong to the actual sweep
      # SB: why isn't the ESContext updated after an edm.setRunNumber?
      # after data collection it looks OK - based on the GUI/CollRefImg
      #
      fileinfo_pr.setRun_number(run_number)
      #suffix = index_request.getFileinfo().getSuffix()
      #
      # Ugly! Needs to be fixed:
      #
      #suffix = "img"
      #template = "%s_%d_###.%s"%(prefix, run_number, suffix)
      #fileinfo_pr.setTemplate(template)
      #
      # Three images starting at 90 degrees from phi start:
      #
      phi_start = oxsdCollect_request.getOscillation_sequence()[0].getStart()
      oxsdCollect_request_pr = XSD.Collect_request()
      oxsdCollect_request_pr.setFileinfo( fileinfo_pr )
      os_pr = XSD.Oscillation_sequence()
      os_pr.setExposure_time( exposure_time )
      os_pr.setNumber_of_passes( number_of_passes )
      os_pr.setStart( 90.0+phi_start )
      os_pr.setRange( oscillation_sequence.getRange() )
      os_pr.setStart_image_number( 4 )
      os_pr.setNumber_of_images( 3 )
      os_pr.setOverlap( 0.0 )
      oxsdCollect_request_pr.setFileinfo(fileinfo_pr)
      oxsdCollect_request_pr.addOscillation_sequence(os_pr)
      if not oxsdCollect_request.getSessionId() is None:
        oxsdCollect_request_pr.setSessionId( oxsdCollect_request.getSessionId() )
      if not oxsdCollect_request.getResolution() is None:
        oxsdCollect_request_pr.setResolution( oxsdCollect_request.getResolution() )
      if not oxsdCollect_request.getKappa_collect_settings() is None:
        oxsdCollect_request_pr.setKappa_collect_settings( oxsdCollect_request.getKappa_collect_settings() )
      #
      # Do the collect...
      #
      self.logwrite( "First collecting three postref images at 90 degrees from phi start at %.2d" % phi_start )
      oxsdCollect_response_pr_1 = self.do_collect_request( oxsdCollect_request_pr )
      #
      # Three more images starting at phi start ...
      #
      os_pr.setStart( phi_start )
      os_pr.setStart_image_number(1)
      fileinfo_pr.setPrefix( prefix )
      self.logwrite( "Then collecting three images at phi start at %.2d degrees" % phi_start )
      oxsdCollect_response_pr_2 = self.do_collect_request( oxsdCollect_request_pr )
      #
      # Create the postref links...
      #
      self.create_postref_soft_link( fileinfo_pr, 1 )
      self.create_postref_soft_link( fileinfo_pr, 2 )
      self.create_postref_soft_link( fileinfo_pr, 3 )
      #
      # Reset the fileinfo context...
      #
      ESContext.set_bcm_fileinfo( fileinfo )
      #
      # Collect of data in the background excluding the first three images...
      #
      osc_seq = oxsdCollect_request.getOscillation_sequence()[0]
      number_of_images = osc_seq.getNumber_of_images()
      osc_seq.setNumber_of_images( number_of_images - 3 )
      osc_seq.setStart_image_number( 4 )
      osc_seq.setStart( phi_start + osc_seq.getRange()*3 )
      Debug.out( oxsdCollect_request.toDict() )
      self.do_collect_request( oxsdCollect_request, wait_for_response=False )      
    else:
      #
      # Start collection of data "in the background"...
      #
      self.do_collect_request( oxsdCollect_request, wait_for_response=False )
      #
      # 
      #      
    if oxsdCollect_data_request.getIntegrate():
      #
      # Cell refinement
      #
      self.do_cell_refinement_request( wait_for_response=False )
      #
      # Wait for a response
      #
      self.wait_for_response( "/cell_refinement_response" )
      #
      # Check if ok
      #
      cell_refinement_response = XSD.Cell_refinement_response()
      cell_refinement_response.unmarshal( self.xml_message )
      Debug.out("Cell_refinement response status code: %s" % cell_refinement_response.getStatus().getCode(), 1)
      #
      # Integration.
      #
      integrate_response = self.do_integrate_request( wait_for_response=False )
      #
      # Wait for a response
      #
      self.wait_for_response( "/integrate_response" )
      #
      # Check if ok
      #
      integrate_response = XSD.Integrate_response()
      integrate_response.unmarshal( self.xml_message )
      Debug.out( "Integrate response status code: %s" % integrate_response.getStatus().getCode(), 1 )
      #
      #
      #
    #
    # Check if we are still waiting for the collect response.
    #
    if self.collecting_flag:
      #
      # Yes! We have to wait...
      #
      self.wait_for_response( "/collect_response" )
      oxsdCollect_response = XSD.Collect_response()
      oxsdCollect_response.unmarshal( self.xml_message )
      self.latest_collect_response = oxsdCollect_response
      Debug.out( "Collect response status code: %s" % oxsdCollect_response.getStatus().getCode(), 1 )
    #
    # Return the collect response message.
    #
    collect_response = self.latest_collect_response.copy()
    if collect_response is None:
      oxsdStatus.setCode( "error" )
      oxsdStatus.setMessage( "No collect response obtained from the BCM." )
      oxsdCollect_data_response.setStatus( oxsdStatus )
    else:
      oxsdCollect_data_response.setStatus( collect_response.getStatus() )
    #
    #
    #
    return oxsdCollect_data_response


  def create_postref_soft_link( self, fileinfo, image_number ):
    directory   = fileinfo.getDirectory()
    prefix      = fileinfo.getPrefix()
    run_number  = fileinfo.getRun_number()
    suffix      = fileinfo.getSuffix()
    imageName   = "%s_%d_%03d.%s" % ( prefix, run_number, image_number, suffix )
    pathToImage = os.path.join( directory, "postref-"+imageName ) 
    if os.path.exists( pathToImage ):
      self.logwrite( "Deleting existing link %s" % pathToImage, 2 )
      os.remove( pathToImage )
    os.system( "cd %s; ln -s %s postref-%s" % ( directory, imageName, imageName ) )
    self.logwrite( "Creating soft link from %s to postref-%s" % ( imageName, imageName ) , 2)


  def fileinfo_with_unique_run_number( self, _xsdFileinfo ):
    #
    # Check if images already exists!
    #
    self.logwrite( "Determining the run number", 2 )
    oxsdFileinfo = _xsdFileinfo.copy()
    if oxsdFileinfo.getRun_number() is None:
      oxsdFileinfo.setRun_number( 1 )
    found_flag = False
    while ( not found_flag ):
      strPath_image_001 = ESContext.getPath_to_image( oxsdFileinfo,  1)
      self.logwrite( "Trying path %s" % strPath_image_001, 2)
      if os.path.exists( strPath_image_001 ):
        #
        # Yup, it's there! Increase the run number and try again.
        #
        oxsdFileinfo.setRun_number( oxsdFileinfo.getRun_number() + 1 )
      else:
        #
        # Ok, it's not there - stop searching.
        #
        found_flag = True
        #
    dRun_number = oxsdFileinfo.getRun_number()
    self.logwrite("Using run number %d." % dRun_number )
    return oxsdFileinfo





  def screen_collect(self, characterize_crystal_request, wait_for_response=True, set_context_flag=True):
    #
    #
    #
    target = characterize_crystal_request.getTarget()
    collect_reference_request = characterize_crystal_request.getCollect_reference_request()
    collect_request = collect_reference_request.getCollect_request()
    code = collect_request.getSample_reference().getCode()
    #
    # Collect images
    #
    fileinfo = collect_request.getFileinfo()
    self.logwrite("----------------------------------------------------")
    self.logwrite("Screening sample %s" % fileinfo.getPrefix() )
    if not code is None:
      self.logwrite("Sample code: %s" % code)
    #
    # Determine unique run number
    #
    fileinfo_1 = fileinfo.copy()
    fileinfo_1.setPrefix("ref-%s" % fileinfo.getPrefix())
    fileinfo_1 = self.fileinfo_with_unique_run_number( fileinfo_1 )
    fileinfo.setRun_number( fileinfo_1.getRun_number() )
    #
    #
    #
    collect_reference_request = XSD.Collect_reference_request()
    collect_reference_request.setCollect_request(collect_request)
    collect_reference_request.setArchive_reference_images(True)
    return self.do_collect_reference_request(collect_reference_request, wait_for_response, set_context_flag)




  def do_screen_request(self, _oxsdScreen_request):
    """Screens samples by looping through a list of characterize_crystal_requests."""
    #
    # Make a copy of the incoming request
    #
    screen_request = _oxsdScreen_request.copy()
    status = XSD.Status()
    status.setCode("ok")
    screen_response = XSD.Screen_response()
    screen_response.setStatus(status)
    #
    #
    #
    self.useProjectInputFile = False
    #
    #
    #
    self.logwrite("")
    self.logwrite("")
    self.logwrite("Screening started.")
    characterize_crystal_request_list = screen_request.getCharacterize_crystal_request()
    no_samples_left_to_screen = len(characterize_crystal_request_list)
    #
    # start with a blocking screen_collect
    #
    if no_samples_left_to_screen == 0:
      self.logwrite("No samples to screen!")
      status.setCode("error")
      status.setMessage("No samples to screen!")
      screen_response.setStatus(status)
      return screen_response
    #
    # Go through the samples in the same order as the GUI sent them:
    #
    characterize_crystal_request_list.reverse()
    characterize_crystal_request_1 = characterize_crystal_request_list.pop()
    iSessionId = screen_request.getSessionId()
    ESContext.set_bcm_sessionId( iSessionId )
    characterize_crystal_request_1.getCollect_reference_request().getCollect_request().setSessionId( iSessionId )
    collect_reference_response_1 = self.screen_collect(characterize_crystal_request_1, wait_for_response = True, set_context_flag = True)
    no_samples_left_to_screen = no_samples_left_to_screen-1
    #
    # Then enter the loop!
    #
    more_samples_to_screen_flag = True
    while (more_samples_to_screen_flag):
      #
      # Continue only if no errors...
      #
      if collect_reference_response_1.getStatus().getCode() == "error":
        self.logwrite("ERROR! Reference image collection failed - aborting screening.")
        self.logwrite("Error message: %s" % collect_reference_response_1.getStatus().getMessage() )
        more_samples_to_screen_flag = False
      else:
        #
        # Check that we have more samples in the list...
        #
        characterize_crystal_request_2 = None
        if no_samples_left_to_screen > 0:
          characterize_crystal_request_2 = characterize_crystal_request_list.pop()
          characterize_crystal_request_2.getCollect_reference_request().getCollect_request().setSessionId( screen_request.getSessionId() )
          collect_request = characterize_crystal_request_2.getCollect_reference_request().getCollect_request()
          fileinfo             = collect_request.getFileinfo()
          oscillation_sequence = collect_request.getOscillation_sequence()[0]
          #
          # Set the context 
          #
          ESContext.set_bcm_fileinfo(fileinfo)
          ESContext.set_bcm_oscillation_sequence(oscillation_sequence)
          ESContext.set_bcm_collect_resolution(collect_request.getResolution())
          self.screen_collect(characterize_crystal_request_2, wait_for_response = False, set_context_flag = False)
          no_samples_left_to_screen = no_samples_left_to_screen-1
        else:
          more_samples_to_screen_flag = False
        #
        # Auto index reference images from characterize_crystal_request_1
        # unless the status is set to "warning":
        #
        if collect_reference_response_1.getStatus().getCode() == "warning":
          self.logwrite("Warning message from the BCM - skipping data processing for this sample.")
        else:
          index_request = XSD.Index_request()
          fileinfo_index_request = characterize_crystal_request_1.getCollect_reference_request().getCollect_request().getFileinfo()
          sample_reference = characterize_crystal_request_1.getCollect_reference_request().getCollect_request().getSample_reference()
          index_request.setFileinfo( fileinfo_index_request )
          index_request.addImage(1)
          index_request.addImage(2)
          index_response = self.do_index_request(index_request)
          self.gui_send_message("/index_response", index_response)
          if index_response.getStatus().getCode() == "error":
            self.logwrite("ERROR! Indexing failed - no strategy calculation for this sample.")
            self.logwrite("Error message: %s" % index_response.getStatus().getMessage())
          else:
            #
            # Do the strategy request
            #
            oxsdStrategy_request = XSD.Strategy_request()
            oxsdStrategy_settings = XSD.Strategy_settings()
            oxsdStrategy_settings.setAnomalous( characterize_crystal_request_1.getAnomalous() )
            oxsdStrategy_request.setStrategy_settings( oxsdStrategy_settings )
            oxsdStrategy_response = self.do_strategy_request( oxsdStrategy_request )
            self.gui_send_message( "/strategy_response", oxsdStrategy_response );          
            if oxsdStrategy_response.getStatus().getCode() == "error":
              self.logwrite("ERROR! Strategy calculation failed.")
              self.logwrite("Error messagae: %s" % oxsdStrategy_response.getStatus( ).getMessage( ) )
            else:
              #
              # Rank the result!
              #
              self.rank_screen_result( characterize_crystal_request_1, self.useProjectInputFile )
              if not self.useProjectInputFile:
                self.useProjectInputFile = True
        #
        # Check if we are still waiting for the collect response.
        #
        if more_samples_to_screen_flag and self.collecting_flag:
          #
          # Yes! We have to wait...
          #
          prefix_2 = characterize_crystal_request_2.getCollect_reference_request().getCollect_request().getFileinfo().getPrefix()
          self.logwrite("===============================================================================")
          self.logwrite("Waiting for collection of reference images for sample %s to finnish." % prefix_2)
          self.wait_for_response("/collect_response")
          collect_response = XSD.Collect_response()
          collect_response.unmarshal(self.xml_message)
          self.latest_collect_response = collect_response
          Debug.out("Collect response status code: %s" % collect_response.getStatus().getCode(), 1)
        #
        # 
        #
        characterize_crystal_request_1 = characterize_crystal_request_2
        collect_reference_response_1 = self.latest_collect_response
    #
    #
    #
    return screen_response
        

  def store_in_database_screen_result( self, _oxsdIndex_response, _oxsdStrategy_response ):
    #
    # Here code should be added for writing results back to the database
    #
    pass

  def rank_screen_result( self, _oxsdCharacterize_crystal_request, useProjectInputFile = False ):
    #
    # Rank the result using the drank module 
    #
    oxsdProposal               = XSD.Proposal()
    oxsdScreeningOutput        = self.screeningOutput
    oxsdScreeningOutputLattice = self.screeningOutputLattice
    oxsdScreeningStrategy      = self.screeningStrategy
    oxsdDataCrystal            = XSD.DataCrystal()
    #
    oxsdProposal.setCode( "mx" )
    oxsdProposal.setNumber( 455 )
    oxsdProposal.setPersonId( 1000 )
    oxsdProposal.setProposalId( 1001 )
    oxsdProposal.setTitle( "Test DNA experiment" )
    oxsdDataCrystal.setProposal( oxsdProposal )
    #
    oxsdCollect_reference_request = _oxsdCharacterize_crystal_request.getCollect_reference_request()
    oxsdCollect_request           = oxsdCollect_reference_request.getCollect_request()
    oxsdFileinfo                  = oxsdCollect_request.getFileinfo()
    oxsdSample_reference          = oxsdCollect_request.getSample_reference()
    oxsdDataCrystal.setFileinfo( oxsdFileinfo )
    oxsdDataCrystal.setSample_reference( oxsdSample_reference )
    #
    oxsdDataExtended = XSD.DataExtended()
    oxsdDataExtended.setProposal( oxsdProposal )
    oxsdDataExtended.setCharacterize_crystal_request( _oxsdCharacterize_crystal_request )
    ostrDataExtendedXML = xml.sax.saxutils.escape( oxsdDataExtended.marshal() )
    #self.logwrite(ostrDataExtendedXML, 1)
    #
    oxsdDataCrystal.setDataExtendedXML(        ostrDataExtendedXML        )
    oxsdDataCrystal.setScreeningOutput(        oxsdScreeningOutput        )
    oxsdDataCrystal.setScreeningOutputLattice( oxsdScreeningOutputLattice )
    oxsdDataCrystal.setScreeningStrategy(      oxsdScreeningStrategy      )
    #
    f = open("data-crystal.xml", "w")
    f.write( XML_utils.xml_with_indentations( oxsdDataCrystal.marshal() ) )
    f.close()
    #
    rank_response = self.run_drank_module( oxsdDataCrystal, useProjectInputFile )
    #
    # Send result to GUI
    #
    self.gui_send_message("/rank_response", rank_response);          



  def do_rank_request(self, _oxsdRank_request):
    """Ranking!"""
    #
    # Make a copy of the incoming request
    #
    rank_request = _oxsdRank_request.copy()
    status = XSD.Status()
    status.setCode("ok")
    rank_response = XSD.Rank_response()
    #
    #
    #
    self.logwrite( "Ranking samples..." )
    dataRankProjectDNA = rank_request.getDataRankProjectDNA()
    #self.logwrite( dataRankProjectDNA.marshal() )
    #
    if dataRankProjectDNA is None:
      message = "Empty rank project, no samples to rank!"
      self.logwrite( message )
      status.setCode("error")
      status.setMessage( message )
      rank_response.setStatus(status)
    else:
      magDRProject = dataRankProjectDNA.getMagDRProject()
      if magDRProject is None:
        message = "Empty magDRProject, no samples to rank!"
        self.logwrite( message )
        status.setCode("error")
        status.setMessage( message )
        rank_response.setStatus(status)
      else:
        listDataCrystal = magDRProject.getDataCrystal()
        if listDataCrystal is None:
          message = "No samples to rank!"
          self.logwrite( message )
          status.setCode("error")
          status.setMessage( message )
          rank_response.setStatus(status)
        else:
          noSamplesToRank = len( listDataCrystal )
          self.logwrite( "Ranking %d samples." % noSamplesToRank )
          rankProjectFile = self.user_defaults.getRank_parameters().getRankProjectFile()
          ostreamFile = open( rankProjectFile, "w" )
          ostreamFile.write( dataRankProjectDNA.marshal() )
          ostreamFile.close()
          #
          rank_response = self.run_drank_module( useProjectInputFile = True)
    #
    return rank_response


  def run_drank_module( self, _oxsdDataCrystal = None, useProjectInputFile = False ):
    #
    #
    #
    status = XSD.Status()
    status.setCode("ok")
    rank_response = XSD.Rank_response()
    rank_response.setStatus(status)
    #
    # Run Romeu's drank module
    #
    if not _oxsdDataCrystal is None:
      ostream = open( "data-crystal.xml", "w" )
      ostream.write( _oxsdDataCrystal.marshal() )
      ostream.close()
    #
    cmdEngine = ESConfig.getUser_defaults().getRank_parameters().getRankEngine()
    self.logwrite( "Ranking engine used: %s" % cmdEngine )
    rankProjectFile = self.user_defaults.getRank_parameters().getRankProjectFile()
    self.logwrite( "Ranking project file: %s" % rankProjectFile )
    #    
    cmdLine = "python " + os.path.join( os.environ["DNAHOME"], "libraries", "drank", "modules", "__init__.py " )
    if useProjectInputFile:
      cmdLine += " --ipDNA %s " % rankProjectFile
    cmdLine += " --opDNA %s " % rankProjectFile
    if not _oxsdDataCrystal is None:
      cmdLine += " --rcrystalDNA data-crystal.xml "
    cmdLine += " --rengine %s " % cmdEngine
    cmdLine += " --verbose "
    #self.logwrite(cmdLine)
    #          
    self.set_gui_dpm_status( "Ranking" )
    os.system( cmdLine )
    self.set_gui_dpm_status( "Ready" )
    #
    ostreamFile = open( rankProjectFile, "r" )
    ostrXML = ostreamFile.read()
    ostreamFile.close()
    #    
    oxsdDataRankProjectDNA = XSD.DataRankProjectDNA()
    oxsdDataRankProjectDNA.unmarshal( ostrXML )
    rank_response.setDataRankProjectDNA( oxsdDataRankProjectDNA )
    #
    #
    #
    return rank_response


  def do_cell_refinement_request( self, _oxsdCell_refinement_request = None, wait_for_response = True ):
    """Called by the do_collect_data_request if the integrate
    flag is set to true. It uses the dpm_send_asynchronous_command to
    send the request to the DPM."""
    #
    #
    #
    self.logwrite("Refining the cell")
    #
    # Make a copy if the reqeust exists, if not create an empty one
    #
    cell_refinement_request = None
    if _oxsdCell_refinement_request is None:
      cell_refinement_request = XSD.Cell_refinement_request()
    else:
      cell_refinement_request = _oxsdCell_refinement_request.copy()
    start_images = XSD.Start_images()
    end_images = XSD.End_images()
    start_images.addImage(1)
    start_images.addImage(4)
    end_images.addImage(3)
    end_images.addImage(6)
    cell_refinement_request.setStart_images(start_images)
    cell_refinement_request.setEnd_images(end_images)
    #
    # If necessary, strip the "ref-" prefix
    #
    fileinfo_pr = ESContext.get_dpm_fileinfo()
    prefix_pr   = ESContext.strip_ref_prefix(fileinfo_pr.getPrefix())
    fileinfo_pr.setPrefix("postref-%s" % prefix_pr )
    ESContext.determine_template(fileinfo_pr)
    cell_refinement_request.setFileinfo(fileinfo_pr)
    #
    #
    #
    #self.logwrite(cell_refinement_request.toDict())
    self.dpm_send_asynchronous_command("/cell_refinement_request", cell_refinement_request, "/cell_refinement_response")
    #
    # Check if we should wait for a response...
    #
    if wait_for_response:
      #
      # Wait till the images have been processed.
      #
      self.wait_for_response("/cell_refinement_response")
      #
      # Return the response
      #
      cell_refinement_response = XSD.Cell_refinement_response()
      cell_refinement_response.unmarshal(self.xml_message)
      Debug.out("Cell_refinement response status code: %s" % cell_refinement_response.getStatus().getCode(), 1)
      #
      #
      #
      return cell_refinement_response



  def do_quick_scale_request(self, _oxsdQuick_scale_request = None, wait_for_response=True):
    """Called by the GUI in order to invoke a quick scale.
    It uses the dpm_send_asynchronous_command to send the request to the DPM."""
    #
    # Make a copy of the incoming request 
    #
    if _oxsdQuick_scale_request is None:
      quick_scale_request = XSD.Quick_scale_request()
    else:
      quick_scale_request = _oxsdQuick_scale_request.copy()
    #
    #
    #
    self.logwrite("Perfroming a quick scale, please wait...")
    #
    # Set the fileinfo 
    #
    oxsdFileinfo = ESContext.get_dpm_fileinfo()
    prefix = oxsdFileinfo.getPrefix()
    run_number = oxsdFileinfo.getRun_number()
    template = "%s_%d_###.mtz" % ( prefix, run_number )
    oxsdFileinfo.setTemplate( template )
    oxsdFileinfo.setDirectory( ESContext.get_current_log_dir() )
    quick_scale_request.setFileinfo( oxsdFileinfo )
    #
    # Set the high resolution to the collect resolution
    #
    dCollectResolution = ESContext.get_bcm_collect_resolution().getUpper()
    quick_scale_request.setHigh_resolution_limit( dCollectResolution )
    #self.logwrite( "quick_scale_request sent to the scheduler: %s" % quick_scale_request.marshal_with_indentations() )
    #
    #
    #
    self.dpm_send_asynchronous_command("/quick_scale_request", quick_scale_request, "/quick_scale_response")
    #
    # Check if we should wait for a response...
    #
    if wait_for_response:
      #
      # Wait till the quick scaling is done.
      #
      self.wait_for_response("/quick_scale_response")
      #
      # Return the response
      #
      quick_scale_response = XSD.Quick_scale_response()
      quick_scale_response.unmarshal( self.xml_message )
      Debug.out("Quick scale response status code: %s" % quick_scale_response.getStatus().getCode(), 1)
      #
      #
      #
      return quick_scale_response



  def do_index_request(self, _oxsdIndex_request):
    """First checks that the images are available, fills in
    missing information in the index request (e.g. detector type) and then
    uses the dpm_send_asynchronous_command for sending the index_request to the
    DPM."""
    #
    # Make a copy of the incoming request
    #
    index_request = _oxsdIndex_request.copy()
    #
    # The ES changes its state to INDEX_REQUEST.
    #
    self.set_status("INDEX_REQUEST")
    Debug.out("Index request received: %s"%(pprint.pformat(index_request.toDict())),2)
    #
    #
    #
    self.logwrite("Autoindexing reference images")
    fileinfo = index_request.getFileinfo()
    prefix   = fileinfo.getPrefix()
    if prefix is None:
      raise "ExecutiveException", "Error - prefix is set to None"
    #
    # Check if the images exists. If there's no "ref-" prefix
    # in the index_request fileinfo prefix first check that there is
    # an image with the "ref-" prefix.
    #
    flag_ref_prefix = False
    if len(prefix) > 4:
      if prefix[:4] == "ref-":
        flag_ref_prefix = True
    if not flag_ref_prefix:
      prefix = "ref-%s" % prefix
      fileinfo.setPrefix(prefix)
      index_request.setFileinfo(fileinfo)
      flag_ref_prefix = True
    #
    # Sleep two seconds in order to make sure the images are written to disk
    #
    time.sleep( 2 )
    #
    # Check if image 001 is there
    #
    path_image_001 = ESContext.getPath_to_image(fileinfo,  1)
    if not os.access(path_image_001, os.F_OK):
      #
      # No, it's not there. Try without the ref- prefix.
      #
      path_image_001_save = path_image_001
      prefix = ESContext.strip_ref_prefix(prefix)
      fileinfo.setPrefix(prefix)
      index_request.setFileinfo(fileinfo)
      flag_ref_prefix = False
      path_image_001 = ESContext.getPath_to_image(fileinfo,  1)
      if not os.access(path_image_001, os.F_OK):
        #
        #
        #
        self.logwrite("Tried path: %s" % path_image_001_save)
        self.logwrite("Tried path: %s" % path_image_001)
        raise "ExecutiveException", "Cannot access reference image 001."
        #
        # 
        #
    #
    # If we came so far, reference image 1 exists.
    #
    self.logwrite("Reference image 001: %s" % path_image_001)
    #
    # Check image 002.
    #
    path_image_002 = ESContext.getPath_to_image(fileinfo, 2)
    flag_image_002 = False
    flag_image_091 = False
    if os.access(path_image_002, os.F_OK):
      #
      # 002 is there!
      #
      self.logwrite("Reference image 002: %s" % path_image_002)
      flag_image_002 = True
      #
      #
      #
    else:
      #
      # No, its not there!
      #
      #
      # Check image 91 for legacy reasons...
      #
      path_image_091 = ESContext.getPath_to_image(fileinfo, 91)
      if not os.access(path_image_091, os.F_OK):
        #
        # No, its not there either!
        #
        self.logwrite("Tried path: %s" % path_image_091)
        raise "ExecutiveException", "Cannot access second reference image, tried both 002 and 091"
        #
        #
        #
      #
      # 91 is there!
      #
      self.logwrite("Reference image 091: %s" % path_image_091)
      flag_image_091 = True
      #
      #
      #
    #
    #
    #
    ESContext.set_dpm_fileinfo(index_request.getFileinfo())
    fileinfo = ESContext.get_dpm_fileinfo()
    index_request.setFileinfo(fileinfo)
    #
    # Find out if dector type and beam coordinates are provided
    # in the index request.
    #
    directory = index_request.getFileinfo().getDirectory()
    prefix = index_request.getFileinfo().getPrefix()
    #
    # Check if the fileinfo provided by the index request matches
    # the fileinfo give in the last collect request:
    #
    #self.bcm_parameters_flag = False
    #if (directory == self.collect_directory) and (prefix == self.collect_prefix):
    #  if not self.bcm_parameters_response is None:
    #    self.bcm_parameters_flag = True
    #
    #
    #
    detector = index_request.getDetector()
    if detector is None:
      bcm_parameters_response = ESContext.get_bcm_parameters_response()
      if bcm_parameters_response != None:
        detector = bcm_parameters_response.getDetector()
        self.logwrite("Using the detector type \"%s\" provided by the BCM"%(detector.getType()))
      else:
        #
        # Check that the suffix corresponds to the current detector:
        #
        suffix = fileinfo.getSuffix()
        detector = DetectorDatabase.determine_detector_from_suffix( suffix )
        currentDetector = ESContext.get_dpm_detector()
        if currentDetector.getType() != detector.getType():
          self.logwrite( "Setting detector type to %s determined by the suffix %s" % ( detectorType, suffix ) )
        else:
          self.logwrite( "Using the detector type \"%s\"" % detector.getType() )
      index_request.setDetector(detector)
    else:
      self.logwrite("Detector type provided by the GUI: %s"%(detector.getType()))
    ESContext.set_dpm_detector( detector )
    index_request.setDetector( detector )
    #
    #
    #
    beam = index_request.getBeam()
    if not beam is None:
      self.logwrite("Beam centre provided by the GUI:  %6.3f %6.3f"%(beam.getX(),beam.getY()))
      ESContext.set_dpm_beam(beam)
      index_request.setBeam(beam)
    #
    # Set extra index commands if available
    #
    Debug.out("Check if extra index commands...",1)
    extra_index_commands = self.index_parameters.getExtra_index_commands()
    if not extra_index_commands is None:
      index_request.setExtra_commands(extra_index_commands)
      Debug.out("Extra commands: %s" % pprint.pformat(extra_index_commands.toDict()),1) 
    #
    # Check if target laue group is provided
    #
    if not index_request.getTarget() is None:
      if index_request.getTarget() != "unknown":
        self.logwrite("Target symmetry: %s" % index_request.getTarget().getSymmetry())
    #
    # Send the index request to the DPM for both images
    #
    self.logwrite("Autoindexing first, second and both images - please wait...")
    index_request.clearImage()
    index_request.addImage(1)
    if flag_image_002:
      index_request.addImage(2)
    elif flag_image_091:
      index_request.addImage(91)
    else:
      raise ExecutiveException, "Neither image 002 nor 091 found!"
    self.dpm_send_asynchronous_command("/index_request", index_request, "/index_response")
    #
    # Wait till the images have been collected.
    #
    self.wait_for_response("/index_response")
    #
    # Return the response
    #
    index_response = XSD.Index_response()
    integrate_response = None
    index_response.unmarshal(self.xml_message)
    Debug.out("Index response status code: %s" % index_response.getStatus().getCode(), 1)
    #
    # Send URL to GUI
    #
    web_page_path = os.path.join(ESContext.get_current_log_dir(), "index", "index.html")
    gui_display_url_request = XSD.Gui_display_url_request()
    gui_display_url_request.setUrl("file://%s" % web_page_path)
    self.gui_send_message( "/gui_display_url_request", gui_display_url_request)
    #
    # Let's now check the index response...
    #
    if index_response.getStatus().getCode() == 'error':
        #
        # Something went wrong!
        #
        self.logwrite("INDEXING FAILED!")
        self.logwrite("Error message: %s" % index_response.getStatus().getMessage())
        self.logwrite("Not integrating images.")
        self.logwrite("PLEASE EXAMINE THE IMAGES MANUALLY.")
        #
        #
        #
    elif index_response.getMosaicity_value() < 0.0:
        #
        # Something went wrong with the mosaicity estimation:
        #
        self.logwrite("INDEXING FAILED BECAUSE MOSAIC SPREAD ESTIMATION FAILED!")
        self.logwrite("Not integrating images.")
        #
        #
        #
    else:
        #
        # This was successful - we integrate the images
        #
        self.logwrite("Indexing successful! Now proceeding to integration of images.")
        integrate_request = XSD.Single_integrate_request()
        integrate_request.setFileinfo(self.dpm_request.getFileinfo())
        for image in index_request.getImage():
            integrate_request.addImage(image)
        integrate_response = self.do_single_integrate_request(integrate_request)
        if integrate_response.getStatus().getCode() == "error":
            index_response.setStatus(integrate_response.getStatus())
        else:
            #
            # Do nothing with the response except print the resolution
            #
            if integrate_response.getCalculated_resolution():
                self.logwrite("Calculated resolution: %8.2f" % integrate_response.getCalculated_resolution().getUpper())
                Debug.out("Single integrate response status code: %s" % integrate_response.getStatus().getCode(), 1)
            else:
                self.logwrite("No calculated resolution found in index_response")
    #   #
    #   # Put the resolution into the index_response - if it's there!
    #   #
    #   if not integrate_response is None:
    #     if integrate_response.getStatus().getCode() == "error":
    #       self.logwrite("ERROR after integration of reference images: %s" % integrate_response.getStatus().getMessage() )
    #     else:
    #       if not integrate_response.getCalculated_resolution() is None:
    #         Debug.out("Setting the calculated resolution in the index response to %f" % integrate_response.getCalculated_resolution().getUpper(),2)
    #         index_response.setCalculated_resolution(integrate_response.getCalculated_resolution())
    #
    # Put the collect resolution into the index_response
    #
    if not integrate_response is None:
      if integrate_response.getStatus().getCode() == "error":
        self.logwrite("ERROR after integration of reference images: %s" % integrate_response.getStatus().getMessage() )
      else:
        collect_resolution = ESContext.get_bcm_collect_resolution()
        if collect_resolution is None:
          #
          # We don't have a collect_resolution... try to read it from the image.
          #
          fileinfo = ESContext.get_dpm_fileinfo()
          try:
            bcm_parameters_response = BcmParametersHelper.getBcmParametersFromImageFile( fileinfo )
            collect_resolution = bcm_parameters_response.getExperiment().getResolution()
            self.logwrite( "Obtained resolution used for collecting reference images from image header: %.2f" % collect_resolution.getUpper() )
          except:
            collect_resolution = None
            self.logwrite( "---------------- WARNING! ---------------" )
            self.logwrite( "The DNA system couldn't determine the resolution used for the reference image collection." )
          ESContext.set_bcm_collect_resolution( collect_resolution )
        else:
          self.logwrite( "Obtained resolution used for collecting reference images from the BCM: %.2f" % collect_resolution.getUpper() )          
        if not collect_resolution is None:
          collect_resolution.setUpper( ESContext.round_off_to_n_decimals(collect_resolution.getUpper(), 2) )
          #self.logwrite( "Strategy caluclated to resolution: %.2f" % collect_resolution.getUpper() )
          index_response.setCalculated_resolution( collect_resolution )
    #
    #
    #
    #self.logwrite("Index response:")
    #self.logwrite(pprint.pformat(index_response.toDict()))
    #self.logwrite(pprint.pformat(integrate_response.toDict()))
    #
    # Create screening object
    #
    self.screening = XSD.Screening()
    self.screening.setProgramVersion( self.version_short )
    #
    # Fill in screeningInput values
    #
    self.screeningInput  = XSD.ScreeningInput()
    beam = ESContext.get_bcm_beam()
    if beam != None:
        self.screeningInput.setBeamX( beam.getX() )
        self.screeningInput.setBeamY( beam.getY() )
    else:
        self.screeningInput.setBeamX( -1.0 )
        self.screeningInput.setBeamY( -1.0 )
    self.screeningInput.setRmsErrorLimits( self.index_parameters.getMax_index_spot_rms_error() )
    self.screeningInput.setMaximumFractionRejected( self.index_parameters.getMax_index_spot_frac_rejected() )
    self.screeningInput.setMinimumSignalToNoise( 3.0 )
    #
    #
    #
    self.screeningOutput = XSD.ScreeningOutput()
    if index_response.getStatus().getCode() == "ok":
      self.screeningOutput.setStatusDescription( "Indexing successful" )
      if not integrate_response is None:
        if not integrate_response.getCalculated_resolution() is None:
          self.screeningOutput.setResolutionObtained(  integrate_response.getCalculated_resolution().getUpper()                   )
        else:
          self.screeningOutput.setResolutionObtained( 9999999 )
      else:
        self.screeningOutput.setResolutionObtained( 9999999 )      
      self.screeningOutput.setRejectedReflections( index_response.getSpot_search_response().getRejected() )
      self.screeningOutput.setSpotDeviationR(      index_response.getSolution().getSpot_deviation()                               )
      self.screeningOutput.setBeamShiftX(          index_response.getSolution().getRefinement().getBeam_shift().getShift().getX() )
      self.screeningOutput.setBeamShiftY(          index_response.getSolution().getRefinement().getBeam_shift().getShift().getY() )
      self.screeningOutput.setNumSpotsFound( index_response.getSpot_search_response().getUsed() )
      
      numSpotsUsed = index_response.getSolution().getRefinement().getReflections().getUsed()
      self.screeningOutput.setNumSpotsUsed( numSpotsUsed )

      numSpotsUsedInIndexing = index_response.getSolution().getRefinement().getReflections().getUsed_in_indexing()
      numSpotsRejected = numSpotsUsed - numSpotsUsedInIndexing
      self.screeningOutput.setNumSpotsRejected( numSpotsRejected )

      self.screeningOutput.setMosaicity( index_response.getMosaicity_value() )

      bPowderIceRings = index_response.getPowder_ice_rings()
      if bPowderIceRings == None: 
        self.screeningOutput.setDiffractionRings( False )
      else:
        self.screeningOutput.setDiffractionRings( bPowderIceRings )

      self.screeningOutputLattice = XSD.ScreeningOutputLattice()
      self.screeningOutputLattice.setSpaceGroup( index_response.getSolution().getSymmetry() )
      self.screeningOutputLattice.setPointGroup( "??" )
      self.screeningOutputLattice.setBravaisLattice( "??" )
      self.screeningOutputLattice.setRawOrientationMatrix_a_x( index_response.getSolution().getOrientation().getA_matrix().getE11() )
      self.screeningOutputLattice.setRawOrientationMatrix_a_y( index_response.getSolution().getOrientation().getA_matrix().getE12() )
      self.screeningOutputLattice.setRawOrientationMatrix_a_z( index_response.getSolution().getOrientation().getA_matrix().getE13() )
      self.screeningOutputLattice.setRawOrientationMatrix_b_x( index_response.getSolution().getOrientation().getA_matrix().getE21() )
      self.screeningOutputLattice.setRawOrientationMatrix_b_y( index_response.getSolution().getOrientation().getA_matrix().getE22() )
      self.screeningOutputLattice.setRawOrientationMatrix_b_z( index_response.getSolution().getOrientation().getA_matrix().getE23() )
      self.screeningOutputLattice.setRawOrientationMatrix_c_x( index_response.getSolution().getOrientation().getA_matrix().getE31() )
      self.screeningOutputLattice.setRawOrientationMatrix_c_y( index_response.getSolution().getOrientation().getA_matrix().getE32() )
      self.screeningOutputLattice.setRawOrientationMatrix_c_z( index_response.getSolution().getOrientation().getA_matrix().getE33() )
      self.screeningOutputLattice.setUnitCell_a(     index_response.getSolution().getOrientation().getCell().getA()     )
      self.screeningOutputLattice.setUnitCell_b(     index_response.getSolution().getOrientation().getCell().getB()     )
      self.screeningOutputLattice.setUnitCell_c(     index_response.getSolution().getOrientation().getCell().getC()     )
      self.screeningOutputLattice.setUnitCell_alpha( index_response.getSolution().getOrientation().getCell().getAlpha() )
      self.screeningOutputLattice.setUnitCell_beta(  index_response.getSolution().getOrientation().getCell().getBeta()  )
      self.screeningOutputLattice.setUnitCell_gamma( index_response.getSolution().getOrientation().getCell().getGamma() )
      
      
    else:
      #
      # Deafult values
      #
      self.screeningOutput.setStatusDescription( "Indexing failed" )
      self.screeningOutput.setResolutionObtained( 9999999 )      
      self.screeningOutput.setSpotDeviationR(     -1.0 )
      self.screeningOutput.setBeamShiftX(         -1.0 )
      self.screeningOutput.setBeamShiftY(         -1.0 )
      self.screeningOutput.setNumSpotsFound(      -1 )
      self.screeningOutput.setNumSpotsUsed(       -1 )
      self.screeningOutput.setNumSpotsRejected(   -1 )
      self.screeningOutput.setMosaicity(          -1.0 )
      self.screeningOutput.setDiffractionRings(   False )

      self.screeningOutputLattice = XSD.ScreeningOutputLattice()
      self.screeningOutputLattice.setSpaceGroup(               "Empty" )
      self.screeningOutputLattice.setPointGroup(               "Empty" )
      self.screeningOutputLattice.setBravaisLattice(           "Empty" )
      self.screeningOutputLattice.setRawOrientationMatrix_a_x( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_a_y(  -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_a_z( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_b_x( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_b_y( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_b_z( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_c_x( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_c_y( -1.0 )
      self.screeningOutputLattice.setRawOrientationMatrix_c_z( -1.0 )
      self.screeningOutputLattice.setUnitCell_a(               -1.0 )
      self.screeningOutputLattice.setUnitCell_b(               -1.0 )
      self.screeningOutputLattice.setUnitCell_c(               -1.0 )
      self.screeningOutputLattice.setUnitCell_alpha(           -1.0 )
      self.screeningOutputLattice.setUnitCell_beta(            -1.0 )
      self.screeningOutputLattice.setUnitCell_gamma(           -1.0 )

    #
    # Store screening results in ISPyB data base
    #
    #self.logwrite( self.screeningOutput.marshal_with_indentations(), 1 )
    #
    status = self._ExecutiveDBProxy.do_store_screening_data( self.screening,
                                                             self.screeningInput,
                                                             self.screeningOutput,
                                                             self.screeningOutputLattice )
    if status.getCode() == "error":
      self.logwrite( "Screening results not stored in data base: %s" % status.getMessage(), 1)
    else:
      self.logwrite( "Screening results stored in data base!", 1)        
    return index_response




  def do_integrate_request(self, _oxsdIntegrate_request=None, wait_for_response=True):
    """Adds extra integrate commands if available and then uses
    the dpm_send_asynchronous_command for sending the integrate request to the
    DPM proxy."""
    #
    # Make a copy of the incoming request
    #
    integrate_request = None
    if _oxsdIntegrate_request != None:
      integrate_request = _oxsdIntegrate_request.copy()
    #
    # The ES changes its state to INTEGRATE_REQUEST.
    #
    self.set_status("INTEGRATE_REQUEST")
    if not integrate_request is None:
      Debug.out("Integrate request received: %s"%(pprint.pformat(integrate_request.toDict())),2)
    #
    # Set extra integrate commands if available
    #
    Debug.out("Check if extra integrate commands...",1)
    if not self.integrate_parameters is None:
      extra_integrate_commands = self.integrate_parameters.getExtra_integrate_commands()
      if not extra_integrate_commands is None:
        integrate_request.setExtra_commands(extra_integrate_commands)
        Debug.out("Extra commands: %s" % pprint.pformat(extra_integrate_commands.toDict()),1) 
    #
    #
    #
    oscillation_sequence = ESContext.get_dpm_oscillation_sequence()
    start_image_no = None
    end_image_no = None
    if integrate_request is None:
      start_image_no = oscillation_sequence.getStart_image_number()
      end_image_no = oscillation_sequence.getNumber_of_images()+start_image_no-1
      integrate_request = XSD.Integrate_request()
      integrate_request.setStart(start_image_no)
      integrate_request.setEnd(end_image_no)
    else:
      start_image_no = integrate_request.getStart()
      end_image_no = integrate_request.getEnd()
      if integrate_request.getFileinfo() is not None:
        ESContext.set_dpm_fileinfo(integrate_request.getFileinfo())
    no_images = end_image_no - start_image_no + 1
    # Try to do batches of 10 images...
    no_batches = no_images/10
    integrate_request.setNumber_of_batches(no_batches)
    fileinfo = ESContext.get_dpm_fileinfo()
    #
    # Strip of "ref-" prefix if "multiple" integrate
    #
    if start_image_no != end_image_no:
      prefix = fileinfo.getPrefix()
      prefix = ESContext.strip_ref_prefix(prefix)
      fileinfo.setPrefix(prefix)
    ESContext.determine_template(fileinfo)
    integrate_request.setFileinfo(fileinfo)
    Debug.out("Integrate request: %s" % pprint.pformat(integrate_request.toDict()), 2)
    #
    #
    #
    self.dpm_send_asynchronous_command("/integrate_request", integrate_request, "/integrate_response")
    #
    # Check if we should wait for a response...
    #
    if wait_for_response:
      #
      # Wait till the images have been collected.
      #
      self.wait_for_response("/integrate_response")
      #
      # Return the response
      #
      integrate_response = XSD.Integrate_response()
      integrate_response.unmarshal(self.xml_message)
      Debug.out("Integrate response status code: %s" % integrate_response.getStatus().getCode(), 1)
      return integrate_response



  def do_single_integrate_request(self, _oxsdSingle_integrate_request=None):
    """Adds extra integrate commands if available and then uses
    the dpm_send_asynchronous_command for sending the single integrate request to the
    DPM proxy."""
    #
    # Make a copy of the incoming request
    #
    single_integrate_request = _oxsdSingle_integrate_request.copy()
    #
    # The ES changes its state to SINGLE_INTEGRATE_REQUEST.
    #
    self.set_status("SINGLE_INTEGRATE_REQUEST")
    if not single_integrate_request is None:
      Debug.out("Single integrate request received: %s"%(pprint.pformat(single_integrate_request.toDict())),2)
    #
    # Set extra integrate commands if available
    #
    Debug.out("Check if extra integrate commands...",1)
    if not self.integrate_parameters is None:
      extra_integrate_commands = self.integrate_parameters.getExtra_integrate_commands()
      if not extra_integrate_commands is None:
        single_integrate_request.setExtra_commands(extra_integrate_commands)
        Debug.out("Extra commands: %s" % pprint.pformat(extra_integrate_commands.toDict()),1) 
    #
    #
    #
    ESContext.set_dpm_calculated_resolution(None)
    self.dpm_send_asynchronous_command("/single_integrate_request", single_integrate_request, "/single_integrate_response")
    #
    # Wait till the images have been collected.
    #
    self.wait_for_response("/single_integrate_response")
    #
    # Return the response
    #
    single_integrate_response = XSD.Integrate_response()
    single_integrate_response.unmarshal(self.xml_message)
    Debug.out("Single integrate response status code: %s" % single_integrate_response.getStatus().getCode(), 1)
    #
    #
    #    
    if single_integrate_response.getStatus().getCode() != "error":
      resolution = single_integrate_response.getCalculated_resolution()
      ESContext.set_dpm_calculated_resolution(resolution)
    #
    #
    #
    return single_integrate_response



  def do_strategy_request(self, _oxsdStrategy_request):
    """Adds extra strategy commands if available and then uses the
    dpm_send_asynchronous_command for sending the strategy request to the DPM."""
    #
    # Make a copy of the incoming request
    #
    strategy_request = _oxsdStrategy_request.copy()
    #
    # Set the status to STRATEGY_REQUEST
    #
    self.set_status("STRATEGY_REQUEST")
    #
    # Set extra strategy commands if available
    #
    Debug.out("Check if extra strategy commands...",1)
    if not self.strategy_parameters is None:
      extra_strategy_commands = self.strategy_parameters.getExtra_strategy_commands()
      if not extra_strategy_commands is None:
        strategy_request.setExtra_commands(extra_strategy_commands)
        Debug.out("Extra commands: %s" % pprint.pformat(extra_strategy_commands.toDict()),1) 
      #
      # Set overlap limit if available
      #
      Debug.out("Check if overlap limit...",1)
      overlap_limit = self.strategy_parameters.getOverlap_limit()
      if not overlap_limit is None:
        strategy_settings = strategy_request.getStrategy_settings()
        if strategy_settings is None:
          strategy_settings = XSD.Strategy_settings()
          strategy_settings.setOverlap_limit(overlap_limit)
          strategy_request.setStrategy_settings(strategy_settings)
        else:
          strategy_settings.setOverlap_limit(overlap_limit)          
      self.logwrite("Setting strategy overlap limit to %f" % overlap_limit, 2) 
    #
    # Check for beamline_parameters
    #
    beamline_parameters = None
    bcm_parameters_response = ESContext.get_bcm_parameters_response()
    if not bcm_parameters_response is None:
        beamline_parameters = bcm_parameters_response.getBeamline_parameters()
    if beamline_parameters is None: 
        self.logwrite( "No beamine parameters available, using default values:" )
        maximum_exposure        = 600  # 10 min.
        minimum_exposure_time   = 0.1
        minimum_phi_speed       = 0.1
        maximum_phi_speed       = 100
        minimum_phi_oscillation = 0.1
        beamline_parameters = XSD.Beamline_parameters()
        beamline_parameters.setMaximum_exposure(        maximum_exposure        )
        beamline_parameters.setMinimum_exposure_time(   minimum_exposure_time   )
        beamline_parameters.setMinimum_phi_speed(       minimum_phi_speed       )
        beamline_parameters.setMaximum_phi_speed(       maximum_phi_speed       )
        beamline_parameters.setMinimum_phi_oscillation( minimum_phi_oscillation )
    else:
        self.logwrite( "Using beamline parameters obtained from the BCM:" )        
    #
    #
    #
    maximum_exposure        = beamline_parameters.getMaximum_exposure()
    minimum_exposure_time   = beamline_parameters.getMinimum_exposure_time()
    minimum_phi_speed       = beamline_parameters.getMinimum_phi_speed()
    maximum_phi_speed       = beamline_parameters.getMaximum_phi_speed()
    minimum_phi_oscillation = beamline_parameters.getMinimum_phi_oscillation()
    self.logwrite( "  Maximum exposure        : %d [s]"          % maximum_exposure        )
    self.logwrite( "  Minimum exposure time   : %.1f [s]"        % minimum_exposure_time   )
    self.logwrite( "  Minimum phi speed       : %.1f [degree/s]" % minimum_phi_speed       )
    self.logwrite( "  Maximum phi speed       : %.1f [degree/s]" % maximum_phi_speed       )
    self.logwrite( "  Minimum phi oscillation : %.1f [degree]"   % minimum_phi_oscillation )
    strategy_settings.setBeamline_parameters( beamline_parameters )
    #
    # Do the strategy calculation
    #
    self.logwrite("Calculating the strategy")
    #
    # Check if the resolution is set in the strategy request:
    #
    strategy_resolution = None
    strategy_settings = strategy_request.getStrategy_settings()
    if not strategy_settings is None:
      if not strategy_settings.getResolution() is None:
        strategy_resolution = strategy_settings.getResolution().getUpper()
    else:
      strategy_settings = XSD.Strategy_settings()
    #
    #
    #
    if strategy_resolution is None:
      #
      # This code is implementing the new functionality described in bug 1477
      # The DNA system should use only the collected resolution for calculating
      # the strategy.
      #
      if ESContext.get_bcm_collect_resolution() is None:
        strategy_resolution = 2.75
        self.logwrite( "--------------------------- WARNING! ---------------------------------" )
        self.logwrite( "DNA couldn't determine resolution used for collecting reference images" )
        self.logwrite( "Setting strategy resolution to fixed value %.2f Angstroms" % strategy_resolution)
      else:
        strategy_resolution = ESContext.get_bcm_collect_resolution().getUpper()
    #
    #
    #
    resolution = XSD.Resolution()
    resolution.setUpper(strategy_resolution)
    strategy_settings.setResolution(resolution)
    strategy_request.setStrategy_settings(strategy_settings)
    #
    #
    #
    Debug.out( "Strategy request: %s" % pprint.pformat( strategy_request.toDict() ), 1 )
    self.dpm_send_asynchronous_command("/strategy_request", strategy_request, "/strategy_response")
    #
    # Wait for the strategy response
    #
    self.wait_for_response("/strategy_response")
    #
    # Return the response
    #
    strategy_response = XSD.Strategy_response()
    strategy_response.unmarshal(self.xml_message)
    Debug.out( "Strategy response: %s" % pprint.pformat( strategy_response.toDict() ), 1 )
    Debug.out( "Strategy response status code: %s" % strategy_response.getStatus().getCode(), 1)
    #
    # Fill in the correct number of passes
    #
    for osc_sec in strategy_response.getStrategy_interpretation().getOscillation_sequence():
      osc_sec.setNumber_of_passes( ESContext.get_dpm_oscillation_sequence().getNumber_of_passes() )
    #
    # Force the number of decimals
    #
    oscillation_sequence_list = strategy_response.getStrategy_interpretation().getOscillation_sequence()
    for oscillation_sequence in oscillation_sequence_list:
      exposure_time = oscillation_sequence.getExposure_time()
      exposure_time_rounded = ESContext.round_off_to_n_decimals(exposure_time, 1)
      oscillation_sequence.setExposure_time(exposure_time_rounded)
    self.last_strategy_request_oscillation_sequence = oscillation_sequence_list[0]
    #
    # Return the strategy response
    #
    osc_sec = strategy_response.getStrategy_interpretation().getOscillation_sequence()[0]
    exposure_time = osc_sec.getExposure_time()
    phiStart = osc_sec.getStart()
    phiEnd   = osc_sec.getEnd()
    rotation = osc_sec.getRange()
    if phiEnd == None:
        #
        # No phiEnd given - must calculate one
        #
        no_images = osc_sec.getNumber_of_images()
        phiEnd = phiStart + no_images*rotation
        #
    self.screeningStrategy = XSD.ScreeningStrategy()
    self.screeningStrategy.setPhiStart( phiStart )
    self.screeningStrategy.setPhiEnd(   phiEnd   )
    self.screeningStrategy.setRotation( rotation )
    self.screeningStrategy.setExposureTime( exposure_time )
    self.screeningStrategy.setResolution(   strategy_resolution        )
    self.screeningStrategy.setProgram( "BEST 3.0" )
    status = self._ExecutiveDBProxy.do_store_screening_strategy( self.screeningStrategy )    
    if status.getCode() != "ok":
        print "Error when storing user defined strategy: %s" % status.getMessage()
    #self.logwrite(pprint.pformat( XML_utils.xml_with_indentations(self.screeningStrategy.marshal() ) ) )
    #
    # Print information about the different resolutions...
    #
    calculated_resolution = ESContext.get_dpm_calculated_resolution().getUpper()
    collect_resolution    = ESContext.get_bcm_collect_resolution().getUpper()
    if abs( calculated_resolution - strategy_resolution ) > 0.1:
        self.logwrite( "" )
        self.logwrite( "=================================================================================" )
        self.logwrite( "Strategy caluclated to resolution: %.2f (edge of the detector)" % strategy_resolution )
        if abs( collect_resolution - strategy_resolution ) < 0.1:
            self.logwrite( "This is the current detector resolution." )
        self.logwrite( "" )
        self.logwrite( "The proposed exposure time is: %.2f s" % exposure_time )
        self.logwrite( "This exposure time is necessary in order to obtain this resolution."  )
        self.logwrite( ""  )
        self.logwrite( "The observed resolution calculated from reference images is %.2f (I/sigma(I) > 2)." % calculated_resolution )
        self.logwrite( ""  )
        self.logwrite( "Please check your images and recalculate strategy if you want to"  )
        self.logwrite( "collect to the observed resolution."  )
        self.logwrite( "=================================================================================" )
        self.logwrite( ""  )
    #
    # Return the strategy response
    #
    return strategy_response


  def do_kappa_alignment_request(self, _oxsdKappa_alignment_request):
    """ processing Kappa_alignment_requests
        It is the first step in kappa strategy """
    #
    # Make a copy of the incoming request
    #
    kappa_alignment_request = _oxsdKappa_alignment_request.copy()
    #
    # Set the status to STRATEGY_REQUEST
    #
    self.set_status("STRATEGY_REQUEST")
    #
    # Do the calculation
    #
    self.logwrite("Calculating the kappa alignment")
    self.dpm_send_asynchronous_command("/kappa_alignment_request", kappa_alignment_request, "/kappa_alignment_response")
    #
    # Wait for the calculation
    #
    self.wait_for_response("/kappa_alignment_response")
    #
    # Return the response
    #
    kappa_alignment_response = XSD.Kappa_alignment_response()
    kappa_alignment_response.unmarshal(self.xml_message)
    Debug.out("Strategy response status code: %s" % kappa_alignment_response.getStatus().getCode(), 1)
    #
    # Return the strategy response
    #
    return kappa_alignment_response


  def do_kappa_strategy_request(self, _oxsdKappa_strategy_request):
    """ processing Kappa_strategy_requests
        It is the second step in kappa strategy """
    #
    # Make a copy of the incoming request
    #
    kappa_strategy_request = _oxsdKappa_strategy_request.copy()
    #
    # standard strategy request (threadable!)
    #
    standard_asnwer = XSD.Strategy_response()
    standard_answer = self.do_strategy_request(kappa_strategy_request.getStandard_request())
    #
    # Set the status to STRATEGY_REQUEST
    #
    self.set_status("STRATEGY_REQUEST")
    #
    # Do the calculation
    #
    self.logwrite("Calculating the kappa strategies")
    self.dpm_send_asynchronous_command("/kappa_strategy_request", kappa_strategy_request, "/kappa_strategy_response")
    #
    # Wait for the calculation
    #
    self.wait_for_response("/kappa_strategy_response")
    #
    # Return the response
    #
    kappa_strategy_response = XSD.Kappa_strategy_response()
    kappa_strategy_response.unmarshal(self.xml_message)
    Debug.out("Strategy response status code: %s" % kappa_strategy_response.getStatus().getCode(), 1)
    #
    # Return the strategy response
    #
    kappa_strategy_response.setStandard_response(standard_answer)
    return kappa_strategy_response


  def do_get_proposal_request(self, db_proposal):
    """An experimental method for the database access. It is not used for DNA 1.0."""
    #
    # set the status to DATABASE_REQUEST
    #
    self.set_status('DATABASE_REQUEST')
    proposal_response_xml = self.db_send_synchronous_command('/db_get_proposal_request', db_proposal)
    self.logwrite('Response %s' % proposal_response_xml)
    get_proposal_response = XSD.Db_proposal_response()
    get_proposal_response.unmarshal(proposal_response_xml)
    self.response = get_proposal_response
    return get_proposal_response


  def do_characterize_crystal_request(self, _oxsdCharacterize_crystal_request):
    """Does the whole chain of tasks involved in a characterize
    crystal request, i.e. collecting reference images, indexing, integration
    and calculation of the strategy."""
    #
    # Make a copy of the incoming request
    #
    oxsdCharacterize_crystal_request = _oxsdCharacterize_crystal_request.copy()
    #
    # The characterize_crystal_response
    #
    oxsdStatus = XSD.Status()
    oxsdStatus.setCode("ok")      
    oxsdCharacterize_crystal_response = XSD.Characterize_crystal_response()
    oxsdCharacterize_crystal_response.setStatus( oxsdStatus )
    #
    # Determine unique run number
    #
    oxsdCollect_reference_request = oxsdCharacterize_crystal_request.getCollect_reference_request()
    oxsdCollect_request = oxsdCollect_reference_request.getCollect_request()
    oxsdFileinfo = oxsdCollect_request.getFileinfo()
    oxsdFileinfoRef = oxsdFileinfo.copy()
    strPrefix = oxsdFileinfoRef.getPrefix()
    #
    # Check that the prefix hasn't already a "ref-" prefix
    #
    if len( strPrefix ) >= 4:
      if strPrefix[0:4] == "ref-":
        oxsdFileinfoRef.setPrefix( strPrefix)
      else:
        oxsdFileinfoRef.setPrefix( "ref-" + strPrefix )
    oxsdFileinfoRef_unique = self.fileinfo_with_unique_run_number( oxsdFileinfoRef )
    oxsdFileinfo.setRun_number( oxsdFileinfoRef_unique.getRun_number() )
    oxsdCollect_request.setFileinfo( oxsdFileinfo )
    #
    #  Collect reference images.
    #
    oxsdCollect_reference_response = self.do_collect_reference_request( oxsdCollect_reference_request )
    #self.gui_send_message( "/collect_reference_response", oxsdCollect_reference_response );          
    if oxsdCollect_reference_response.getStatus().getCode() != "ok":
      oxsdCharacterize_crystal_response.setStatus( oxsdCollect_reference_response.getStatus() )
      return oxsdCharacterize_crystal_response
    #
    # Auto index reference images
    #
    oxsdIndex_request = XSD.Index_request()
    oxsdIndex_request.setFileinfo( oxsdCollect_reference_request.getCollect_request().getFileinfo() )
    oxsdIndex_request.addImage( 1 )
    oxsdIndex_request.addImage( 2 )
    if not oxsdIndex_request.getBeam() is None:
      oxsdIndex_request.setBeam( oxsdCharacterize_crystal_request.getBeam() )
    if not oxsdIndex_request.getDetector() is None:
      oxsdIndex_request.setDetector( oxsdCharacterize_crystal_request.getDetector() )
    if not oxsdCharacterize_crystal_request.getTarget() is None:
      oxsdIndex_request.setTarget( oxsdCharacterize_crystal_request.getTarget() )
    oxsdIndex_response = self.do_index_request( oxsdIndex_request )
    self.gui_send_message( "/index_response", oxsdIndex_response )
    if oxsdIndex_response.getStatus().getCode() == "error":
      oxsdCharacterize_crystal_response.setStatus( oxsdIndex_response.getStatus() )
      return oxsdCharacterize_crystal_response
    #
    # Do the strategy request
    #
    oxsdStrategy_request  = XSD.Strategy_request()
    oxsdStrategy_settings = XSD.Strategy_settings()
    oxsdStrategy_settings.setAnomalous( oxsdCharacterize_crystal_request.getAnomalous() )
    oxsdStrategy_settings.setMultiplicity( oxsdCharacterize_crystal_request.getMultiplicity() )
    oxsdStrategy_settings.setUser_desired_minimum_phi_oscillation( 
      oxsdCharacterize_crystal_request.getUser_desired_minimum_phi_oscillation() )
    oxsdStrategy_request.setStrategy_settings( oxsdStrategy_settings )
    oxsdStrategy_response = self.do_strategy_request( oxsdStrategy_request )
    self.gui_send_message("/strategy_response", oxsdStrategy_response);          
    if oxsdStrategy_response.getStatus().getCode() == "error":
      oxsdCharacterize_crystal_response.setStatus(strategy_response.getStatus())
      return oxsdCharacterize_crystal_response
    oxsdCharacterize_crystal_response.setStrategy_response( oxsdStrategy_response )
    #
    # Send success email
    #
    subject = "DNA characterize crystal succeeded on %s!" % (self.dnaname)
    self.send_email(subject, subject)
    #
    #
    #
    return oxsdCharacterize_crystal_response


  def get_status(self):
    """Returns the current status of the ES, e.g. "IDLE"."""
    return self.status


  def set_status(self, newStatus):
    """Sets the status of the ES."""
    #    self.logwrite("Status set to %s" % newStatus )
    Debug.out("Status set to %s" % newStatus ,1)
    self.status = newStatus


  def set_gui_bcm_status(self, newBCMStatusMessage, error = None):
    """Sets the status of the BCM in the GUI ES status panel."""
    #    self.logwrite("Status set to %s" % newStatus )
    Debug.out("BCM status message sent to GUI: %s" % newBCMStatusMessage ,1)
    bcm_status = XSD.Status()
    bcm_status.setCode( "ok" )
    bcm_status.setMessage( newBCMStatusMessage)
    gui_display_es_status_request = XSD.Gui_display_es_status_request()
    gui_display_es_status_request.setBcm_status( bcm_status )
    #self.logwrite( gui_display_es_status_request.toDict() )
    self.gui_send_message( "/gui_display_es_status_request", gui_display_es_status_request )


  def set_gui_dpm_status(self, newDPMStatusMessage, error = None):
    """Sets the status of the DPM in the GUI ES status panel."""
    #    self.logwrite("Status set to %s" % newStatus )
    Debug.out("DPM status message sent to GUI: %s" % newDPMStatusMessage ,1)
    dpm_status = XSD.Status()
    dpm_status.setCode( "ok" )
    dpm_status.setMessage( newDPMStatusMessage)
    gui_display_es_status_request = XSD.Gui_display_es_status_request()
    gui_display_es_status_request.setDpm_status( dpm_status )
    #self.logwrite( gui_display_es_status_request.toDict() )
    self.gui_send_message( "/gui_display_es_status_request", gui_display_es_status_request )


  def logwrite(self, logText, level=1):
    """Forwads log messages to the DNA logger thread."""
    #
    # Send message to the GUI
    #
    if not self._DNA_LoggerThread is None:
      dna_message = XSD.Dna_message()
      dna_message.setType(    "es-log" )
      dna_message.setMessage( logText  )
      dna_message.setLevel(   level    )
      self._DNA_LoggerThread.forward_dna_message(dna_message)



  def wait_for_response(self, response_type):
    """This new version waits for a response and passes through collect_responses."""
    #
    #
    #
    continue_flag = True
    while(continue_flag):
      #
      # Wait for an event
      #
      self.condition.acquire()
      Debug.out("Waiting for response of type %s..." % response_type, 1)
      start_time = time.time()
      self.condition.wait(self.timeout_length)
      self.condition.release()
      self.check_abort()
      if (time.time() - start_time) > (self.timeout_length-0.1):
          raise "ExecutiveException", "Client time out"
      #
      # Check if it was what we wanted...
      #
      command_xml      = self.command_queue.get(0)
      self.command     = command_xml[0]
      self.xml_message = command_xml[1]
      #
      #
      #
      if self.command == "/collect_response":
        #
        # Ok, the collect has finished.
        #
        self.collecting_flag = False
        collect_response = XSD.Collect_response()
        collect_response.unmarshal(self.xml_message)
        self.latest_collect_response = collect_response
        Debug.out("Collect response status code: %s" % collect_response.getStatus().getCode(), 1)
        self.logwrite( "------------------------------------------------------------------------------")
        fileinfo = ESContext.get_bcm_fileinfo()
        self.logwrite( "Data collection finished for sample %s_%d" % ( fileinfo.getPrefix(), fileinfo.getRun_number() ) )
        #
        # The ES sends a bcm parameters request to the BCM.
        #
        Debug.out("Sending a parameters request to the BCM", 1)
        bcm_parameters_response = self.get_bcm_parameters()
        status = bcm_parameters_response.getStatus()
        if status.getCode() == "error":
          self.logwrite("Problem sending bcm parameters request!")
          raise "ExecutiveException", "BCM parameters request error!"
        #
        # Check if dbstatus is present...
        #
        dbstatus = collect_response.getDbstatus()
        if not dbstatus is None:
          self.logwrite( "Dbstatus object present: %s" % dbstatus.toDict(), 2 )
          dataCollectionId = dbstatus.getDataCollectionId()
          if not dataCollectionId is None:
            self.logwrite( "ISPyB data collection id: %d" % dataCollectionId, 2 )
            ESContext.set_bcm_dataCollectionId( dataCollectionId )
        #
        # register actual Kappa settings
        #
        #self.logwrite( self.system_defaults.getServer_data().getBcm_kappa_in_use() )
        if self.system_defaults.getServer_data().getBcm_kappa_in_use():
          kappa_settings=XSD.Kappa_collect_settings()
          kappa_settings = MosflmInterface.do_kappa_bcm_settings_request()
          kappa_fname = os.path.join(ESContext.get_bcm_fileinfo().getDirectory(),ESContext.get_bcm_fileinfo().getTemplate()+"_kappa_settings.xml")
          if not kappa_settings is None:
            ostream = open( kappa_fname, "w" )
            ostream.write( kappa_settings.marshal() )
            ostream.close()
            self.logwrite("Kappa settings registered")
          else:
            self.logwrite("ERROR! Kappa settings could not been registered")
        #
        #      
        self.logwrite( "------------------------------------------------------------------------------")
        #
        # Tell the user that the data collection has finished but that the
        # ES is still waiting for either the cell refinement or the integration
        # to stop.
        #
        if response_type == "/collect_response":
          continue_flag = False
        elif response_type == "/index_response":
          self.logwrite("Waiting for indexing to finish")
        elif response_type == "/cell_refinement_response":
          self.logwrite("Waiting for cell refinement to finish")
        elif response_type == "/integration_response":
          self.logwrite("Waiting for integration to finish")
        else:
          self.logwrite("Waiting for %s." % response_type)
        self.set_gui_bcm_status( "Ready" )
        #
        #
        #
      elif self.command == response_type:
        #
        # Yes! We stop here.
        #
        continue_flag = False
        Debug.out("Received response %s" % self.command, 1)
        #
        #
        #
      #
      # This is a fix for bug 932:
      #
      elif self.command == "/collect_reference_request" or self.command == "/collect_request":
        self.logwrite("Please wait for the current task to finish before collecting new reference images!")
      elif self.command == "/characterize_crystal_request":
        self.logwrite("Please wait for the current task to finish before starting a new crystal characterization!")
      elif self.command == "/index_request":
        self.logwrite("Please wait for the current task to finish before making an index request!")
      elif self.command == "/strategy_request":
        self.logwrite("Please wait for the current task to finish before making a strategy calculation request!")
      elif self.command == "/kappa_alignment_request":
        self.logwrite("Please wait for the current task to finish before making a strategy calculation request!")
      elif self.command == "/kappa_strategy_request":
        self.logwrite("Please wait for the current task to finish before making a strategy calculation request!")
      elif self.command == "/collect_data_request":
        self.logwrite("Please wait for the current task to finish before making a new data collection!")
      else:
        #
        # Anything else is an error - but we don't raise an
        # exception if we are collecting.
        #
        self.logwrite("Received unexpected response %s while waiting for %s!" % (self.command, response_type))
        if not self.collecting_flag:
          raise "ExecutiveException", "Unexpected response %s while waiting for %s!" % (self.command, response_type)
      


  def print_traceback(self):
    """Prints the python traceback as log messages - very useful for debugging."""
    #
    #
    #
    (exc_type, exc_value, exc_traceback) = sys.exc_info()
    self.logwrite(traceback.format_exception_only(exc_type, exc_value)[0])
    list = traceback.extract_tb(exc_traceback)
    for line in list:
      self.logwrite("File %s, method name %s, line no %d: %s"%(line[0],line[2],line[1],line[3]))
    exc_traceback = None


  def do_feedback_request( self, _oxsdFeedback_request ): 
    """
    Sends an email to the DNA contact person with
    the information in the feedback request.
    """
    message = "\n\n"
    if _oxsdFeedback_request.getCause() != None:
        for cause in _oxsdFeedback_request.getCause():
            message += "Cause: %s\n" % cause
    message += "\nComments: %s" % _oxsdFeedback_request.getComments()
    message += "\n\nContact email: mailto:%s\n\n" % _oxsdFeedback_request.getContact_email()
    self.send_email( "DNA %s feedback request" % self.dnaname,
                     message,
                     send_log=True
                     ) 
    self.logwrite( "Feedback request sent to DNA local contact." )
    
  def send_email(self, subject, message, send_log=True):
    """Sends an email to the DNA contact person (if configured)."""
    #
    # Send error email to DNA contact person (if configured)
    #
    try:
      if not self.dna_contact_email is None:
        self.logwrite("Sending message to %s." % self.dna_contact_email, 2 )
        self.logwrite( "Message: %s" % message, 2)
        sender = "dna-support@esrf.fr"
        new_message = """
DNAHOME = %s
DNANAME = %s
log_dir = %s
%s

""" % (self.dnahome, self.dnaname, ESContext.get_current_log_dir(), message)
        #
        # Add ES messages...
        #
        if send_log:
          new_message = new_message + "\n\nES log:\n\n"
          for line in self._DNA_LoggerThread.get_last_log_lines():
            new_message = new_message + line + "\n"
          #
          # Add DPM messages...
          #
          dpmloglines = string.split(self.get_dpm_log(), "\n")
          if len(dpmloglines) > 50:
            new_message = new_message + "\n\nDPM log:\n\n"
            for line in dpmloglines[-50:]:
              new_message = new_message + line + "\n"
        #
        #
        #
        email_msg = ("From: %s\r\nTo: %s\r\nSubject: %s\r\n\r\n%s"%(sender,self.dna_contact_email,subject,new_message))
        server = smtplib.SMTP("localhost")
        server.sendmail(sender,self.dna_contact_email,email_msg)
        server.quit()
    except:
      self.logwrite("Error when sending email message!", 2)
      # self.logwrite("Error when sending email message!")
      message = "Exception: %s"%(traceback.format_exception_only(sys.exc_type, sys.exc_value)[0])



  def bcm_send_command(self, command, request):
    """Sends a command to the BCM."""
    #
    #
    #
    Debug.out("In bcm_send_command: command=%s" % command,1)
    status = XSD.Status()
    try:
      #
      #handle special kappa settings
      #
      if command == "/collect_request":
        kappa_collect_settings=request.getKappa_collect_settings()
        if kappa_collect_settings != None:
          #old_gui_bcm_status = "Collecting: %s" % prefix_run_number
          #self.set_gui_bcm_status( "Kappa datum change: %s" % prefix_run_number )
          #KappaSettings=XSD.Kappa_collect_settings
          #KappaSettings=request.getKappa_collect_settings()
          kappa_collect_status = MosflmInterface.do_kappa_bcm_request(request.getKappa_collect_settings())
          #self.set_gui_bcm_status( old_gui_bcm_status )
      #end of kappa settings
      xml_message = XML_utils.http_post(self.BCM_host, self.BCM_port, command, request.marshal())
      Debug.out("BCM reply: %s"%(pprint.pformat(xml_message)),2)
      if xml_message is None:
        raise "NULL reply from the BCM"
      status.unmarshal(xml_message)
    except:
      status.setCode("error")
      status.setMessage("Exception: %s"%(traceback.format_exception_only(sys.exc_type, sys.exc_value)[0]))
    if status.getCode() == "error":
      self.logwrite("Error when sending command to BCM: %s" % status.getMessage())
    return status



  def get_bcm_parameters(self):
    """Returns the response after sending a BCM parameters request to the BCM."""
    #
    #
    #
    status = XSD.Status()
    bcm_parameters_request = XSD.Bcm_parameters_request()
    bcm_parameters_request.setSynchronous('true')
    bcm_parameters_response = XSD.Bcm_parameters_response()
    try:
      xml_message = XML_utils.http_post(self.BCM_host, self.BCM_port, \
                                   '/bcm_parameters_request', bcm_parameters_request.marshal())
      Debug.out("XML response: %s" % xml_message,1)
      bcm_parameters_response.unmarshal(xml_message)
      #
      #
      #
      if not bcm_parameters_response.getBeam() is None:
        beam       = bcm_parameters_response.getBeam()
        ESContext.set_bcm_beam( beam )
        self.logwrite("Beam centre            : %.2f %.2f" % (beam.getX(),beam.getY()), 2)
      #
      #
      #
      if not bcm_parameters_response.getExperiment() is None:
        experiment = bcm_parameters_response.getExperiment()
        detector   = bcm_parameters_response.getDetector()
        ESContext.set_bcm_experiment( experiment )
        self.logwrite("Wavelength             : %.4f" % experiment.getWavelength(), 2)
        self.logwrite("Distance               : %.1f" % experiment.getDistance(), 2)
        ESContext.set_bcm_detector( detector )
        self.logwrite("Detector type          : %s"   % detector.getType(), 2)
        #
        # Check if the resolution is provided
        #
        resolution = experiment.getResolution()
        if not resolution is None:
          self.logwrite("Resolution (edge of detector): %.2f" % resolution.getUpper(), 1)
          #
          # Set the current collect resolution in the ESContext
          #
          ESContext.set_bcm_collect_resolution( resolution )
      ESContext.set_bcm_parameters_response( bcm_parameters_response )
      #
      # Check if log_info is available
      #
      if not bcm_parameters_response.getLogin_info() is None:
        login_info = bcm_parameters_response.getLogin_info()
        proposal = login_info.getProposal()
        self.logwrite( "BCM logged in as user %s%d" % ( proposal.getCode(), proposal.getNumber() ), 2 )
        self.gui_send_message( "/login_info_request" , login_info )
    except:
      Debug.out_traceback()
      status.setCode("error")
      status.setMessage("Exception: %s"%(traceback.format_exception_only(sys.exc_type, sys.exc_value)[0]))
      bcm_parameters_response.setStatus(status)
      ESContext.set_bcm_parameters_response( bcm_parameters_response )
    return bcm_parameters_response


  def get_sample_references(self):
    """Returns a list of sample references if the system is configured to use this functionality."""
    #
    #
    #
    sample_references_response = XSD.Sample_references_response()
    status = XSD.Status()
    status.setCode("ok")
    if "DNA_USE_BCM_SAMPLE_REFERENCES_REQUEST" in os.environ.keys():
      self.logwrite("Sending a sample_references_request to the BCM...")
      sample_references_request = XSD.Sample_references_request()
      try:
        Debug.out("sample_references_request_xml: %s" % sample_references_request.marshal(), 1)
        xml_message = XML_utils.http_post(self.BCM_host, self.BCM_port, \
                                          '/sample_references_request', sample_references_request.marshal())
        Debug.out("sample_references_response_xml: %s" % xml_message, 1)
        Debug.out("len(sample_references_response_xml): %d" % len(xml_message),1)
        Debug.out("XML message: %s" % XML_utils.xml_with_indentations(xml_message),1)
        sample_references_response.unmarshal(xml_message)
        Debug.out("Sample references request succeeded!",1)
        Debug.out("Got the following response: %s" % pprint.pformat(sample_references_response.toDict()),1)
      except:
        status.setCode("error")
        status.setMessage("Exception: %s"%(traceback.format_exception_only(sys.exc_type, sys.exc_value)[0]))
        Debug.out(status.getMessage(),1)
        sample_references_response.setStatus(status)
        self.logwrite("Didn't succeed! Got the following error message: %s" % status.getMessage())
    #else:
    #  status.setCode("error")
    #  status.setMessage("DNA system not configured to use the sample_references_request.")
    #  Debug.out(status.getMessage(),1)
    sample_references_response.setStatus(status)      
    return sample_references_response


  def bcm_abort(self):
    """Tries to abort the BCM system."""
    #
    # 
    #
    try:
      abort_request = XSD.Abort_request()
      abort_request.setLevel("stop_after_finishing_current_task")
      abort_response = self.bcm_send_command("/abort_request",abort_request)
    except:
      self.logwrite("ERROR - couldn't send abort command to the BCM")

    
  def gui_send_message(self, path, message):
    """Sends a message to the GUI."""
    xml_message = message.marshal()
    reply = None
    try:
      reply = XML_utils.http_post(self.GUI_host, self.GUI_port, path, xml_message)
    except:
      self.logwrite("ERROR - couldn't send %s command to the GUI" % path)


  def dpm_send_asynchronous_command(self, request_type, the_request, response_type):
    """Handles the asynchronous communication with the DPM proxy. It starts the dpm_do_request
    method in a new thread."""
    #
    #
    #
    self.dpm_check_thread()
    self.dpm_request_type = request_type
    self.dpm_request = the_request
    self.dpm_response_type = response_type
    self.dpm_thread=Thread(target=self.dpm_do_request)
    self.dpm_thread.start()


  def db_send_asynchronous_command(self, request_type, the_request, response_type):
    """Handles the asynchronous communication with the DB proxy. It starts the db_do_request
    method in a new thread."""
    #
    #
    #
    self.db_check_thread()
    self.db_request_type = request_type
    self.db_request = the_request.copy()
    self.db_response_type = response_type
    self.db_thread=Thread(target=self.db_do_request)
    self.db_thread.start()


  def dpm_do_request(self):
    """
    Is started in a new thread by the dpm_send_asynchronous_command method.
    It forward the request (index, integrate etc.) to the DPM proxy.
    """
    #
    #
    #
    Debug.out("In dpm_do_request, request_type = %s"%(self.dpm_request_type),1)
    fileinfo = ESContext.get_dpm_fileinfo()
    prefix_run_number = "%s_%d" % ( fileinfo.getPrefix(), fileinfo.getRun_number() )
    if not self.dpm_request is None:
      Debug.out("Request received: %s" % pprint.pformat(self.dpm_request.toDict()),2)
    response      = None
    response_type = None
    try:
      if self.dpm_request_type == "/index_request":
        #
        #
        #
        self.set_gui_dpm_status( "Indexing: %s" % prefix_run_number )
        response = XSD.Index_response()
        response_type = "/index_response"
        response = MosflmInterface.do_index_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/strategy_request":
        #
        #
        #
        self.set_gui_dpm_status( "Calculating strategy: %s" % prefix_run_number )
        response = XSD.Strategy_response()
        response_type = "/strategy_response"
        response = MosflmInterface.do_strategy_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/kappa_alignment_request":
        #
        #
        #
        self.set_gui_dpm_status( "Kappa alignment: %s" % prefix_run_number )
        response = XSD.Kappa_alignment_response()
        response_type = "/kappa_alignment_response"
        response = MosflmInterface.do_kappa_alignment_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/kappa_strategy_request":
        #
        #
        #
        self.set_gui_dpm_status( "Calculating kappa strategy: %s" % prefix_run_number )
        response = XSD.Kappa_strategy_response()
        response_type = "/kappa_strategy_response"
        response = MosflmInterface.do_kappa_strategy_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/integrate_request":
        #
        #
        #
        self.set_gui_dpm_status( "Integrating: %s" % prefix_run_number )
        response = XSD.Integrate_response()
        response_type = "/integrate_response"
        response = MosflmInterface.do_integrate_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/single_integrate_request":
        #
        #
        #
        self.set_gui_dpm_status( "Integrating: %s" % prefix_run_number )
        response = XSD.Integrate_response()
        response_type = "/single_integrate_response"
        response = MosflmInterface.do_single_integrate_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/cell_refinement_request":
        #
        #
        #
        self.set_gui_dpm_status( "Cell refinement: %s" % prefix_run_number )
        response = XSD.Cell_refinement_response()
        response_type = "/cell_refinement_response"
        response = MosflmInterface.do_refine_cell_request(self.dpm_request)
        #
        #
        #
      elif self.dpm_request_type == "/quick_scale_request":
        #
        #
        #
        self.set_gui_dpm_status( "Quick scaling: %s" % prefix_run_number )
        response = XSD.Quick_scale_response()
        response_type = "/quick_scale_response"
        response = MosflmInterface.do_quick_scale_request(self.dpm_request)
        #
        #
        #
      else:
        raise "ERROR: Unknown request type: %s"%(self.dpm_request_type)
      #
      #
      #
      Debug.out(pprint.pformat(response.toDict()),3)
      self.send_message(response_type, response.marshal())
      self.set_gui_dpm_status( "Ready" )
      #
      #
      #
    except:
      #
      # If we have an exception here we have a serious problem.
      #
      self.logwrite("ERROR! Internal error in the DNA system.")
      self.logwrite("The following information is for the DNA developers:")
      self.print_traceback()
      #
      #
      #
      message = "DPMException"
      status = XSD.Status()
      status.setCode("error")      
      status.setMessage(message)
      #
      # Sleep 2 seconds in order to allow for messages to be
      # recorded in the DNA logger.
      #
      self.logwrite("Please wait, sending error message to DNA local contact")
      time.sleep(2)
      #
      # Send email to contact person (if configured)
      #
      subject = "DNA ERROR on %s: %s" % (self.dnaname, message)
      self.send_email(subject, message)
      #
      #
      #
      if response_type is not None:
        response.setStatus(status)
        self.send_message(response_type, response.marshal())
      #
      #
      #
      self.set_gui_dpm_status( "Ready" )
      

  def dpm_check_thread(self):
    """Checks if the dpm_thread is alive and sleeps until
    the thread is terminated."""
    if not self.dpm_thread is None:
      while self.dpm_thread.isAlive():
        time.sleep(0.1)

  def db_check_thread(self):
    """Checks if the db_thread is alive and sleeps until
    the thread is terminated."""
    if not self.db_thread is None:
      while self.db_thread.isAlive():
        time.sleep(0.1)

  def db_do_request(self):
    """Is started in a new thread by the db_send_asynchronous_command method.
    It forward the request (proposal, loaded sample etc.) to the DB proxy."""
    #
    #
    #
    Debug.out("In db_do_request, request_type = %s"%(self.db_request_type),1)
    if not self.db_request is None:
      Debug.out("Request received: %s" % pprint.pformat(self.db_request.toDict()),2)
    response      = None
    response_type = None
    try:
      if self.db_request_type == "/proposal_request":
        #
        #
        #
        response = XSD.Proposal_response()
        response_type = "/proposal_response"
        response = self._ExecutiveDBProxy.do_proposal_request(self.db_request)
        #
        #
        #
      elif self.db_request_type == "/loaded_samples_request":
        #
        #
        #
        response = XSD.Loaded_samples_response()
        response_type = "/loaded_samples_response"
        response = self._ExecutiveDBProxy.do_loaded_samples_request(self.db_request)
        #
        #
        #
      else:
        raise "ERROR: Unknown request type: %s"%(self.db_request_type)
      #
      #
      #
      Debug.out(pprint.pformat(response.toDict()),3)
      self.send_message(response_type, response.marshal())
      #
      #
      #
    except:
      #
      # If we have an exception here we have a serious problem.
      #
      self.logwrite("ERROR! Internal error in the DNA system.")
      self.logwrite("The following information is for the DNA developers:")
      self.print_traceback()
      #
      #
      #
      message = "DBException"
      status = XSD.Status()
      status.setCode("error")      
      status.setMessage(message)
      #
      # Sleep 2 seconds in order to allow for messages to be
      # recorded in the DNA logger.
      #
      self.logwrite("Please wait, sending error message to DNA local contact")
      time.sleep(2)
      #
      # Send email to contact person (if configured)
      #
      subject = "DNA ERROR on %s: %s" % (self.dnaname, message)
      self.send_email(subject, message)
      #
      #
      #
      if response_type is not None:
        response.setStatus(status)
        self.send_message(response_type, response.marshal())



  def go_online(self):
    """Switches the ESContext online attribute to true if it's not already true."""
    if not ESContext.isOnline():
      ESContext.setOnline(True)
      self.logwrite("Switching to online mode.")


  def go_offline(self):
    """Switches the ESContext online attribute to false if it's not already false."""
    if ESContext.isOnline():
      ESContext.setOnline(False)
      self.logwrite("Switching to offline mode.")


  def check_bcm_communication(self):
    """Checks the comminucation to the BCM by sending a bcm_parameter_request.
    If the request succeeds the system is set to online, otherwise to offline."""
    try:
      Debug.out("Sending a parameters request to the BCM to test communication", 1)
      bcm_parameters_response = self.get_bcm_parameters()
      status = bcm_parameters_response.getStatus()
      if status.getCode() == "error":
        self.logwrite("Error when trying to contact the BCM!")
        self.logwrite("Error message: %s" % status.getMessage())
        self.go_offline()
      else:
        self.logwrite("BCM communication OK.", 2)
        self.go_online()
    except:
      self.logwrite("Error when trying to contact the BCM!")
      self.go_offline()


  def try_to_go_online(self):
    """
    Checks the communication with the BCM.
    """
    #
    # Check the BCM commuincation
    #
    if not ESContext.isOnline():
      self.logwrite("Currently offline, trying to go online")
    self.check_bcm_communication()
    if not ESContext.isOnline():
      raise "ExecutiveException", "Couldn't go online!"


  def init_dpm(self):
    #
    # Get the path to the MOSFLM log file
    #
    log_file = self.server_data.getDpm_log_file()
    MosflmInterface.SetMosflmInterfaceFile(log_file)
    MosflmInterface.InitMessenger(self.ES_host, self.ES_port)
    #
    # Get the path to the MOSFLM executable
    #

    # surely we should be checking this stuff we get from the DNA
    # configuration?

    mosflm_executable = self.server_data.getMosflm_executable()
    MosflmInterface.SetMosflmExecutable(mosflm_executable)
    #
    # If DNATMPDIR present, redirect MOSFLM output files to that directory
    #
    if "DNATMPDIR" in os.environ.keys():
      MosflmInterface.SetWorkingDirectory(os.environ["DNATMPDIR"])
    else:
      MosflmInterface.SetWorkingDirectory(os.getcwd())


  def moveLogFile(self, new_filename):
    #
    # Moves the logfile to a new location
    #
    MosflmInterface.MoveMosflmInterfaceFile(new_filename)


  def get_dpm_log(self):
    #
    # Returns the log file as a string.
    #
    return MosflmInterface.GetMosflmInterfaceLog()

  def do_dpm_abort_request(self, abort_request):
    #
    #  This is a fix for bug 859 - abort in GUI does not abort DPM
    #
    MosflmInterface.AbortAllNow()
    self.logwrite("Waiting for processes to be terminated...")
    time.sleep(1)
    MosflmInterface.ResetAbortAllNow()


#################################################################
#
# The main function.
#

if not "PYDOC" in os.environ.keys():
  executive_system = ExecutiveSystem()

