#################################################################
#
# ExecutiveDBProxy.py
#
# Authors: The DNA team, http://www.dna.ac.uk
#
# Version: $Id: ExecutiveDBProxy.py,v 1.22 2006/06/23 08:27:44 svensson Exp $
#
#################################################################

import XSD
import time, os

from ESConfig import *
from ESContext import *
from XML_utils import *
from Debug import *


#################################################################
#
# The "ExecutiveDBProxy" class
#


class ExecutiveDBProxy:


  def __init__(self, the_es):
    self.es                 = the_es
    self.user_defaults      = ESConfig.getUser_defaults()
    self.server_data        = self.user_defaults.getServer_data()
    self.DB_host            = self.server_data.getDb_host_name()
    self.DB_port            = self.server_data.getDb_port_number()
    self.m_iScreeningId     = None
    self.m_iScreeningOutputId = None


  def do_proposal_request(self, proposal):
    if "DNA_DB_SIMULATOR" in os.environ.keys():
        if os.environ[ "DNA_DB_SIMULATOR" ] == "1":
            return self.do_proposal_request_simulated( proposal )
    proposal_name = "%s%d" % ( proposal.getCode(), proposal.getNumber() )
    self.es.logwrite("Getting proposal information for %s, please wait..." % proposal_name )
    proposal_response = XSD.Proposal_response()
    try:
      Debug.out("Sending proposal_request to DB server %s:%d" % (self.DB_host, self.DB_port), 1)
      proposal_response_xml = XML_utils.http_post(self.DB_host, self.DB_port, \
                                                  '/proposal_request', proposal.marshal())
      Debug.out("proposal_response_xml: %s" % XML_utils.xml_with_indentations(proposal_response_xml), 1)
      proposal_response.unmarshal(proposal_response_xml)
      Debug.out("proposal_response: %s" % proposal_response.toDict(), 1)
      status = proposal_response.getStatus()
      if status.getCode() == "error":
        self.es.logwrite("Error when trying to process %s%d:" % (proposal.getCode(), proposal.getNumber()) )
        self.es.logwrite(status.getMessage() )
      else:
        no_sessions = len( proposal_response.getSession() )
        if no_sessions == 0:
          self.es.logwrite( "No sessions recorded for proposal %s!" % proposal_name )
        else:
          self.es.logwrite( "Proposal %s information received: %d sessions" % ( proposal_name, no_sessions ) )
    except:
      message = "ERROR when trying to send proposal_request to the database on host %s port %d" % \
                (self.DB_host, self.DB_port)
      self.es.logwrite(message)
      Debug.out_traceback(2)
      status = XSD.Status()
      status.setCode("error")
      status.setMessage(message)
      proposal_response.setStatus(status)
    #self.es.gui_send_message( "/proposal_response", proposal_response )
    #f = open( "proposal_response.xml", "w")
    #f.write( XML_utils.xml_with_indentations( proposal_response.marshal() ) )
    #f.close()
    #
    # Sort the session table with respect to the start date
    #
    oxsdListSession = proposal_response.getSession()
    if oxsdListSession != None:
        oxsdListSession.sort( lambda a,b: cmp( b.getStartDate(),
                                               a.getStartDate() ) )
        #for oxsdSession in oxsdListSession:  
        #    pprint.pprint( oxsdSession.toDict() )
    self.m_iScreeningId = None
    self.m_iScreeningOutputId = None
    return proposal_response

  def do_proposal_request_simulated(self, proposal):
    proposal_name = "mx415"
    self.es.logwrite("Getting proposal information for %s, please wait..." % proposal_name )
    proposal_response_xml = """<?xml version="1.0"?>
<proposal_response>
  <status>
    <code>ok</code>
    <message>testmessage</message>
  </status>
  <Proposal>
    <proposalId>993</proposalId>
    <personId>1007937</personId>
    <title>TEST</title>
    <code>MX</code>
    <number>415</number>
  </Proposal>
  <Session>
    <sessionId>6957</sessionId>
    <beamLineSetupId>6957</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2004-12-09 00:00:00</startDate>
    <endDate>2004-12-10 00:00:00</endDate>
    <beamLineName>ID14 3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>3</nbShifts>
    <comments>None</comments>
  </Session>
  <Session>
    <sessionId>6958</sessionId>
    <beamLineSetupId>6958</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-09 00:00:00</startDate>
    <endDate>2005-02-09 00:00:00</endDate>
    <beamLineName>ID14-3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>pipeline</comments>
  </Session>
  <Session>
    <sessionId>6959</sessionId>
    <beamLineSetupId>6959</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-10 00:00:00</startDate>
    <endDate>2005-02-10 00:00:00</endDate>
    <beamLineName>ID14-3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>pipeline test</comments>
  </Session>
  <Session>
    <sessionId>6960</sessionId>
    <beamLineSetupId>6960</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-03-02 00:00:00</startDate>
    <endDate>2005-03-04 00:00:00</endDate>
    <beamLineName>ID14 3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>6</nbShifts>
    <comments>Scheduled experiment</comments>
  </Session>
  <Session>
    <sessionId>6961</sessionId>
    <beamLineSetupId>6961</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-22 00:00:00</startDate>
    <endDate>2005-02-22 00:00:00</endDate>
    <beamLineName>ID14 3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>MDT test without beam</comments>
  </Session>
  <Session>
    <sessionId>6962</sessionId>
    <beamLineSetupId>6962</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-23 00:00:00</startDate>
    <endDate>2005-02-23 00:00:00</endDate>
    <beamLineName>ID14_3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Test with beam</comments>
  </Session>
  <Session>
    <sessionId>6963</sessionId>
    <beamLineSetupId>6963</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-28 00:00:00</startDate>
    <endDate>2005-02-28 00:00:00</endDate>
    <beamLineName>ID14_3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>BAG demonstration</comments>
  </Session>
  <Session>
    <sessionId>6964</sessionId>
    <beamLineSetupId>6964</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-02-28 00:00:00</startDate>
    <endDate>2005-02-28 00:00:00</endDate>
    <beamLineName>ID</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Bag</comments>
  </Session>
  <Session>
    <sessionId>6965</sessionId>
    <beamLineSetupId>6965</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-03-08 00:00:00</startDate>
    <endDate>2005-03-08 00:00:00</endDate>
    <beamLineName>ID14</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>test</comments>
  </Session>
  <Session>
    <sessionId>6966</sessionId>
    <beamLineSetupId>6966</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-03-08 00:00:00</startDate>
    <endDate>2005-03-08 00:00:00</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>2nd test</comments>
  </Session>
  <Session>
    <sessionId>6967</sessionId>
    <beamLineSetupId>6967</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2004-10-09 00:00:00</startDate>
    <endDate>2004-10-09 00:00:00</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>1</nbShifts>
    <comments>3rd test</comments>
  </Session>
  <Session>
    <sessionId>6968</sessionId>
    <beamLineSetupId>6968</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-03-11 00:00:00</startDate>
    <endDate>2005-03-12 00:00:00</endDate>
    <beamLineName>ID29</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>3</nbShifts>
    <comments>None</comments>
  </Session>
  <Session>
    <sessionId>6969</sessionId>
    <beamLineSetupId>6969</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-05-09 00:00:00</startDate>
    <endDate>2005-05-09 00:00:00</endDate>
    <beamLineName>ID14 3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>2</nbShifts>
    <comments>None</comments>
  </Session>
  <Session>
    <sessionId>7261</sessionId>
    <beamLineSetupId>7261</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-06-23 00:00:00</startDate>
    <endDate>2005-06-24 00:00:00</endDate>
    <beamLineName>ID14 3</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>3</nbShifts>
  </Session>
  <Session>
    <sessionId>7262</sessionId>
    <beamLineSetupId>7262</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-06-29 00:00:00</startDate>
    <endDate>2005-06-30 00:00:00</endDate>
    <beamLineName>ID14 2</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>3</nbShifts>
  </Session>
  <Session>
    <sessionId>7263</sessionId>
    <beamLineSetupId>7263</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-07-06 00:00:00</startDate>
    <endDate>2005-07-09 00:00:00</endDate>
    <beamLineName>ID14 2</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>9</nbShifts>
    <comments>blo</comments>
  </Session>
  <Session>
    <sessionId>7266</sessionId>
    <beamLineSetupId>7266</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-09-02 00:00:00</startDate>
    <endDate>2005-09-03 00:00:00</endDate>
    <beamLineName>ID23 1</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>3</nbShifts>
  </Session>
  <Session>
    <sessionId>7302</sessionId>
    <beamLineSetupId>7302</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-09 00:00:00</startDate>
    <endDate>2005-10-09 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7303</sessionId>
    <beamLineSetupId>7303</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-10 00:00:00</startDate>
    <endDate>2005-10-10 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7304</sessionId>
    <beamLineSetupId>7304</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-10 00:00:00</startDate>
    <endDate>2005-10-10 23:59:59</endDate>
    <beamLineName>ID29</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7353</sessionId>
    <beamLineSetupId>7353</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-12 00:00:00</startDate>
    <endDate>2005-10-14 00:00:00</endDate>
    <beamLineName>ID23 1</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>6</nbShifts>
  </Session>
  <Session>
    <sessionId>7354</sessionId>
    <beamLineSetupId>7354</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-12 00:00:00</startDate>
    <endDate>2005-10-12 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7355</sessionId>
    <beamLineSetupId>7355</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-12 00:00:00</startDate>
    <endDate>2005-10-14 00:00:00</endDate>
    <beamLineName>BM14U</beamLineName>
    <scheduled>1</scheduled>
    <nbShifts>6</nbShifts>
  </Session>
  <Session>
    <sessionId>7356</sessionId>
    <beamLineSetupId>7356</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-12 00:00:00</startDate>
    <endDate>2005-10-12 23:59:59</endDate>
    <beamLineName>ID29</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7357</sessionId>
    <beamLineSetupId>7357</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-15 00:00:00</startDate>
    <endDate>2005-10-15 23:59:59</endDate>
    <beamLineName>ID29</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7358</sessionId>
    <beamLineSetupId>7358</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-17 00:00:00</startDate>
    <endDate>2005-10-17 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7359</sessionId>
    <beamLineSetupId>7359</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-18 00:00:00</startDate>
    <endDate>2005-10-18 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7360</sessionId>
    <beamLineSetupId>7360</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-19 00:00:00</startDate>
    <endDate>2005-10-19 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7361</sessionId>
    <beamLineSetupId>7361</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-19 00:00:00</startDate>
    <endDate>2005-10-19 23:59:59</endDate>
    <beamLineName>ID14 2</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7362</sessionId>
    <beamLineSetupId>7362</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-20 00:00:00</startDate>
    <endDate>2005-10-20 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7363</sessionId>
    <beamLineSetupId>7363</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-21 00:00:00</startDate>
    <endDate>2005-10-21 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7364</sessionId>
    <beamLineSetupId>7364</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-23 00:00:00</startDate>
    <endDate>2005-10-23 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7365</sessionId>
    <beamLineSetupId>7365</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-24 00:00:00</startDate>
    <endDate>2005-10-24 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7366</sessionId>
    <beamLineSetupId>7366</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-25 00:00:00</startDate>
    <endDate>2005-10-25 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7368</sessionId>
    <beamLineSetupId>7368</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-26 00:00:00</startDate>
    <endDate>2005-10-26 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
  <Session>
    <sessionId>7369</sessionId>
    <beamLineSetupId>7369</beamLineSetupId>
    <proposalId>993</proposalId>
    <startDate>2005-10-27 00:00:00</startDate>
    <endDate>2005-10-27 23:59:59</endDate>
    <beamLineName>ID14 1</beamLineName>
    <scheduled>0</scheduled>
    <nbShifts>1</nbShifts>
    <comments>Session created by the BCM</comments>
  </Session>
</proposal_response>
"""
    proposal_response = XSD.Proposal_response()
    proposal_response.unmarshal( proposal_response_xml )
    return proposal_response
    

  def do_loaded_samples_request(self, loaded_samples_request):
    if "DNA_DB_SIMULATOR" in os.environ.keys():
        if os.environ[ "DNA_DB_SIMULATOR" ] == "1":
            return self.do_loaded_samples_request_simulated( loaded_samples_request )
    status = XSD.Status()
    #
    # Check if proposal id is provided...
    #
    proposal = loaded_samples_request.getProposal()
    proposalId = proposal.getProposalId()
    if proposalId is None:
      proposal_response = self.do_proposal_request( proposal )
      proposal = proposal_response.getProposal()
      loaded_samples_request.setProposal( proposal )
    self.es.logwrite("Getting a list of loaded samples - please wait...")
    loaded_samples_response = XSD.Loaded_samples_response()
    #
    # Try to get a list of sample references from the BCM
    #
    sample_references_response = self.es.get_sample_references()
    Debug.out("%s" % sample_references_response.marshal(),2)
    if sample_references_response.getStatus().getCode() == "error":
      self.es.logwrite("ERROR when trying to send sample_references_request:")
      self.es.logwrite("%s" % sample_references_response.getStatus().getMessage())
    else:
      sample_references = sample_references_response.getSample_reference()
      if sample_references is not None:
        self.es.logwrite( "%d sample references provided by the BCM." % len( sample_references ) )
        for sample_reference in sample_references:
          code = sample_reference.getCode()
          if code is None:
            code = "None"
          container_reference = sample_reference.getContainer_reference()
          if container_reference is None:
            container_reference = "None"
          loaded_samples_request.addSample_reference(sample_reference)
      else:
        Debug.out( "No sample references provided by the BCM.", 1 )
      try:
        Debug.out("Sending loaded_samples_request to DB server %s:%d" % (self.DB_host, self.DB_port), 1)
        loaded_samples_response_xml = XML_utils.http_post(self.DB_host, self.DB_port, \
                                                          '/loaded_samples_request', \
                                                          loaded_samples_request.marshal())
        Debug.out("loaded_samples_response_xml: %s" % loaded_samples_response_xml, 1)
        loaded_samples_response.unmarshal(loaded_samples_response_xml)
        Debug.out("loaded_samples_response: %s" % loaded_samples_response.toDict(), 1)
      except:
        message = "ERROR when trying to send loaded_samples_request to the database on host %s port %d" % \
                  (self.DB_host, self.DB_port)
        self.es.logwrite(message)
        Debug.out_traceback(2)
        status.setCode("error")
        status.setMessage(message)
        loaded_samples_response.setStatus(status)
    #
    # Merge the sample references with the loaded_samples_response
    #
    loaded_samples = loaded_samples_response.getLoaded_sample() 
    if loaded_samples is None:
      message = "No information about loaded samples provided by the database server!"
      self.es.logwrite( message )
      status.setCode("error")
      status.setMessage(message)
      loaded_samples_response.setStatus(status)
    else:
      self.es.logwrite( "Information about %d crystals received" % len( loaded_samples ) )
      #
      # 
      #
      for loaded_sample in loaded_samples:
        #
        # Fix for bug 1489:
        #
        if ( sample_references != None ) and ( loaded_sample.getBLSample() != None ):
          bl_sample_code = loaded_sample.getBLSample().getCode()
          for sample_reference in sample_references:
            sample_reference_code = sample_reference.getCode()
            #self.es.logwrite("%s %s" % (bl_sample_code, sample_reference_code) )
            if bl_sample_code == sample_reference_code:
              loaded_sample.setSample_reference(sample_reference)
        # 
        # If DiffractionPlan_BLSample is available, use it
        #
        if loaded_sample.getDiffractionPlan_BLSample() != None:
          loaded_sample.setDiffractionPlan( loaded_sample.getDiffractionPlan_BLSample() )
        # 
        # else if DiffractionPlan_CrystalType is available, use it
        #
        elif loaded_sample.getDiffractionPlan_CrystalType() != None:
          loaded_sample.setDiffractionPlan( loaded_sample.getDiffractionPlan_CrystalType() )          
    #
    # Sort the results with respect to the container ID
    #
    #loaded_samples = loaded_samples_response.getLoaded_sample()
    #loaded_samples.sort( lambda a,b: cmp( a.getSample_reference().getContainer_reference(),
    #                                      b.getSample_reference().getContainer_reference() ) )
    #f = open( "loaded_samples_response.xml", "w")
    #f.write( XML_utils.xml_with_indentations( loaded_samples_response.marshal() ) )
    #f.close()
    return loaded_samples_response




  def do_loaded_samples_request_simulated(self, loaded_samples_request):
    status = XSD.Status()
    self.es.logwrite("Getting a list of loaded samples - please wait...")
    loaded_samples_response = XSD.Loaded_samples_response()
    loaded_samples_response_xml = """<?xml version="1.0"?>
<loaded_samples_response>
  <status>
    <code>ok</code>
  </status>
  <loaded_sample>
    <sample_reference>
      <code>HA00AA7090</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100326</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps1</name>
      <code>HA00AA7090</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AB9604</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100327</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps2</name>
      <code>HA00AB9604</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AC1683</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100328</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps3</name>
      <code>HA00AC1683</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AB0026</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100329</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps4</name>
      <code>HA00AB0026</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AC6369</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100330</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps5</name>
      <code>HA00AC6369</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AA6357</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100332</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps7</name>
      <code>HA00AA6357</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AC6565</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100333</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps6</name>
      <code>HA00AC6565</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AC6543</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100334</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps8</name>
      <code>HA00AC6543</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
  <loaded_sample>
    <sample_reference>
      <code>HA00AC0518</code>
    </sample_reference>
    <Protein>
      <proteinId>100107</proteinId>
      <name>Dps_P23_</name>
      <acronym>Dps_P23_</acronym>
      <proposalId>993</proposalId>
    </Protein>
    <Crystal>
      <crystalId>100107</crystalId>
      <proteinId>100107</proteinId>
      <size_X>0.000000</size_X>
      <size_Y>0.000000</size_Y>
      <size_Z>0.000000</size_Z>
      <cell_a>0.000000</cell_a>
      <cell_b>0.000000</cell_b>
      <cell_c>0.000000</cell_c>
      <cell_alpha>0.000000</cell_alpha>
      <cell_beta>0.000000</cell_beta>
      <cell_gamma>0.000000</cell_gamma>
      <comments>From Celia</comments>
    </Crystal>
    <BLSample>
      <blSampleId>100335</blSampleId>
      <diffractionPlanId>0</diffractionPlanId>
      <crystalId>100107</crystalId>
      <name>Dps9</name>
      <code>HA00AC0518</code>
      <holderLength>22.000000</holderLength>
      <loopType>Nylon</loopType>
    </BLSample>
  </loaded_sample>
</loaded_samples_response>
"""
    loaded_samples_response.unmarshal( loaded_samples_response_xml )
    return loaded_samples_response

  def do_store_screening_data( self,
                               _oxsdScreening, 
                               _oxsdScreeningInput,
                               _oxsdScreeningOutput, 
                               _oxsdScreeningOutputLattice ):
    """
    Stores ScreeningInput object in the databse.
    """
    #
    # First create Screening entry.
    #
    status = XSD.Status()
    status.setCode( "ok" )
    if "DNA_DB_SIMULATOR" in os.environ.keys():
      status.setCode( "error" )
      status.setMessage( "Running DB simulator" )
      return status
    iDataCollectionId = ESContext.get_bcm_dataCollectionId()
    iSessionId = ESContext.get_bcm_sessionId()
    if iDataCollectionId == None :
      status.setCode( "error" )
      status.setMessage( "No valid Data Collection Id" )
    else:
      _oxsdScreening.setDataCollectionId( iDataCollectionId )
      xml_response = XML_utils.http_post( 
                       self.DB_host, self.DB_port, \
                       '/store_object_request', 
                       _oxsdScreening.marshal() )
      store_response = XSD.Dbstatus()
      store_response.unmarshal( xml_response )
      #pprint.pprint( store_response.toDict() )
      self.m_iScreeningId = store_response.getScreeningId()
      #
      # Then create ScreeningInput entry.
      #
      _oxsdScreeningInput.setScreeningId( self.m_iScreeningId )
      xml_response = XML_utils.http_post( 
                       self.DB_host, self.DB_port, \
                       '/store_object_request', 
                       _oxsdScreeningInput.marshal() )
      #
      # Then the ScreeningOutput entry.
      #
      _oxsdScreeningOutput.setScreeningId( self.m_iScreeningId )
      xml_response = XML_utils.http_post( 
                       self.DB_host, self.DB_port, \
                       '/store_object_request', 
                       _oxsdScreeningOutput.marshal() )
      #
      # Then the ScreeningOutputLattice entry.
      #
      store_response.unmarshal( xml_response )
      #pprint.pprint( store_response.toDict() )
      self.m_iScreeningOutputId = store_response.getScreeningOutputId()
      _oxsdScreeningOutputLattice.setScreeningOutputId( self.m_iScreeningOutputId )
      xml_response = XML_utils.http_post( 
                       self.DB_host, self.DB_port, \
                       '/store_object_request', 
                       _oxsdScreeningOutputLattice.marshal() )
    return status

    
  def do_store_screening_strategy( self, _oxsdScreeningStrategy ):
    """
    Stores ScreeningStrategy object in the databse.
    """
    status = XSD.Status()
    status.setCode( "ok" )
    if "DNA_DB_SIMULATOR" in os.environ.keys():
      status.setCode( "error" )
      status.setMessage( "Running DB simulator" )
      return status
    if self.m_iScreeningOutputId == None :
      status.setCode( "error" )
      status.setMessage( "No valid Screening Output Id" )
    else:
      _oxsdScreeningStrategy.setScreeningOutputId( self.m_iScreeningOutputId )
      xml_response = XML_utils.http_post( 
                       self.DB_host, self.DB_port, \
                       '/store_object_request', 
                       _oxsdScreeningStrategy.marshal() )
    return status
