#################################################################
#
# ESContext.py
#
# Authors: The DNA team, http://www.dna.ac.uk  
#
# Version: $Id: ESContext.py,v 1.40 2006/04/18 12:21:38 svensson Exp $
#
# This class holds all the context variables, i.e. all the
# variables defined at a particular moment that needs to be
# transmitted to other functions.
#
#################################################################

import os, XSD, pprint, math
from ESConfig import *
from DetectorDatabase import *
from Debug import *


class ESContext:

  def __init__( self ):
    #
    # Get values from configuration file
    #
    self.latest_state                   = "STARTING"
    self.__online                       = False
    #
    # BCM parameters
    #
    self.__bcm_fileinfo                 = None
    self.__bcm_oscillation_sequence     = None
    self.__bcm_collect_resolution       = None
    self.__bcm_dataCollectionId         = None
    self.__bcm_sessionId                = None
    self.__bcm_database_comment         = None
    self.__bcm_parameters_response      = None
    #
    # DPM parameters
    #
    self.__dpm_fileinfo                 = None
    self.__dpm_calculated_resolution    = None
    self.__dpm_detector                 = None
    self.__dpm_experiment               = None
    self.__dpm_beam                     = None
    #
    # ES parameters
    #
    self.__current_log_dir              = "startup"
    DNALOGDIR = None
    if "DNALOGDIR" in os.environ.keys():
      DNALOGDIR = os.environ["DNALOGDIR"]
      Debug.out("DNALOGDIR set to: %s" % DNALOGDIR,2)
      if DNALOGDIR[0] != "/":
        if DNALOGDIR[0:2] == "./":
          DNALOGDIR = os.path.join(os.getcwd(), DNALOGDIR[2:])
        elif DNALOGDIR[0] == ".":
          DNALOGDIR = os.getcwd()
        else:
          DNALOGDIR = os.path.join(os.getcwd(), DNALOGDIR)
        Debug.out("New absolute path DNALOGDIR: %s" % DNALOGDIR,2)
    else:
      DNALOGDIR = os.getcwd()
      os.environ[ "DNALOGDIR" ] = DNALOGDIR
    self.__default_log_dir              = DNALOGDIR
    self.__current_archive_reference_images = "false"
    #
    #
    #
    self.user_defaults = ESConfig.getUser_defaults()
    self.default_values = self.user_defaults.getDefault_values()
    #
    detector = self.default_values.getDetector()
    self.set_bcm_detector(detector)
    self.set_dpm_detector(detector)
    #
    fileinfo = self.default_values.getFileinfo()
    self.set_bcm_fileinfo(fileinfo)
    self.set_dpm_fileinfo(fileinfo)
    #
    oscillation_sequence = self.default_values.getOscillation_sequence()
    self.set_bcm_oscillation_sequence( oscillation_sequence )
    self.set_dpm_oscillation_sequence( oscillation_sequence )
    #
    beam = self.default_values.getBeam()
    self.set_bcm_beam(beam)


  def __call__( self ):
    #
    # Singleton!
    #
    return self    

  def set_latest_state( self, the_state ):
    """Set's the latets state of the ES."""
    self.latest_state = the_state
    
      
  def get_latest_state( self ):
    """Returns the latest state."""
    #
    return self.latest_state


  def set_dpm_fileinfo( self, the_fileinfo ):
    """Sets the dpm fileinfo in the ES context. If both the
    prefix and the run number are different to the previous fileinfo
    the BCM parameters are invalidated. It then sets the path
    to the log file. If there's no run number provided the default
    one in the configuration is used. if the ES is in online mode
    or if the suffix is not profided the suffix is determined,
    if not the detector type is determined from the suffix.
    Finally the template is controlled with a call to the
    determine_template method."""
    #
    #
    #
    if not the_fileinfo is None:
      Debug.out("In ESContext.set_dpm_fileinfo: %s" % \
                pprint.pformat(the_fileinfo.toDict()), 2)
    #
    # Check if the fileinfo is different - if it is,
    # then change the logfile and invalidate the
    # bcm_parameters.
    #
    if self.__dpm_fileinfo is not None:
      old_prefix     = self.__dpm_fileinfo.getPrefix()
      old_prefix     = self.strip_ref_prefix(old_prefix)
      new_prefix     = the_fileinfo.getPrefix()
      new_prefix     = self.strip_ref_prefix(new_prefix)
      old_run_number = self.__dpm_fileinfo.getRun_number()
      new_run_number = the_fileinfo.getRun_number()
      if (old_prefix != new_prefix) or (old_run_number != new_run_number ):
        #
        # Inavildate the bcm_parameters only if offline!
        #
        if not self.isOnline():
          Debug.out("Invalidating DPM parameters!!!---------------------------------------------", 1)
          self.__dpm_experiment = None
          self.__bcm_collect_resolution = None
          self.__dpm_beam       = None
          self.__dpm_detector   = None
    #
    # Change the log file
    #
    self.set_path_to_log_file(the_fileinfo)
    #
    # Copies to current fileinfo
    #
    self.__dpm_fileinfo = XSD.Fileinfo(the_fileinfo.toDict())
    #
    # Add a run number if missing
    #
    if self.__dpm_fileinfo.getRun_number() is None:
      #
      #
      #
      fileinfo.setRun_number( self.default_values.getFileinfo().getRun_number())
      #
      #
      #
    #
    # If we are in online mode or the suffix is not provided:
    #
    if self.__online or self.__dpm_fileinfo.getSuffix() is None:
      #
      # Determine the suffix
      #
      if self.__dpm_detector is None:
        #
        # Use the config detector type
        #
        self.__dpm_detector = ESConfig.getUser_defaults().getDefault_values().getDetector()
        #
        #
        #
      suffix = DetectorDatabase.determine_suffix( self.__dpm_detector)
      self.__dpm_fileinfo.setSuffix(suffix)  
      #
      #
      #
    else:
      #
      # Determine the detector type
      #
      self.__dpm_detector = DetectorDatabase.determine_detector_from_suffix( self.__dpm_fileinfo.getSuffix())
      #
      #
      #
    #
    # Make sure the template is ok
    #
    self.determine_template( self.__dpm_fileinfo)
    #
    #
    #
    Debug.out("Current DPM fileinfo: %s" % pprint.pformat( self.__dpm_fileinfo.toDict()),2)


  def set_bcm_fileinfo( self, the_fileinfo ):
    """
    Sets the BCM fileinfo in the ES context.
    """
    #
    #
    #
    if not the_fileinfo is None:
      Debug.out("In ESContext.set_current_fileinfo: %s" % \
                pprint.pformat(the_fileinfo.toDict()), 2)
    #
    # Invalidate the bcm_parameters.
    #
    Debug.out("Invalidating BCM parameters!!!---------------------------------------------", 1)
    self.__bcm_experiment       = None
    self.__bcm_resolution       = None
    self.__bcm_beam             = None
    self.__bcm_detector         = None
    #self.__bcm_dataCollectionId = None
    #
    # Change the log file
    #
    self.set_path_to_log_file(the_fileinfo)
    #
    # Copies to bcm fileinfo
    #
    self.__bcm_fileinfo = XSD.Fileinfo(the_fileinfo.toDict())
    #
    # Add a run number if missing
    #
    if self.__bcm_fileinfo.getRun_number() is None:
      #
      #
      #
      fileinfo.setRun_number( self.default_values.getFileinfo().getRun_number())
      #
      #
      #
    #
    # If the suffix is not provided:
    #
    if self.__bcm_fileinfo.getSuffix() is None:
      #
      # Determine the suffix
      #
      if self.__bcm_detector is None:
        #
        # Use the config detector type
        #
        self.__bcm_detector = ESConfig.getUser_defaults().getDefault_values().getDetector()
        #
        #
        #
      suffix = DetectorDatabase.determine_suffix( self.__bcm_detector)
      self.__bcm_fileinfo.setSuffix(suffix)  
      #
      #
      #
    else:
      #
      # Determine the detector type
      #
      self.__bcm_detector = DetectorDatabase.determine_detector_from_suffix( self.__bcm_fileinfo.getSuffix())
      #
      #
      #
    #
    # Make sure the template is ok
    #
    self.determine_template( self.__bcm_fileinfo)
    #
    #
    #
    Debug.out("Bcm fileinfo: %s" % pprint.pformat( self.__bcm_fileinfo.toDict()),2)


  def determine_template( self, the_fileinfo ):
    """Determines the template given a fileinfo object."""
    #
    prefix     = the_fileinfo.getPrefix()
    run_number = the_fileinfo.getRun_number()
    suffix     = the_fileinfo.getSuffix()
    template = "%s_%d_###.%s" % (prefix, run_number, suffix)
    the_fileinfo.setTemplate(template)



  def get_bcm_parameters_response( self ):
    """Returns a copy of the current bcm parameters."""
    #
    value = None
    if self.__bcm_parameters_response != None:
        value = self.__bcm_parameters_response.copy()
    return value


  def set_bcm_parameters_response( self, _oxsdBcm_parameters_response ):
    """Sets the current bcm parameters."""
    #
    #
    #
    self.__bcm_parameters_response = _oxsdBcm_parameters_response.copy()

    
  def get_bcm_fileinfo( self ):
    """Returns a copy of the current bcm fileinfo."""
    #
    return XSD.Fileinfo( self.__bcm_fileinfo.toDict())


  def get_dpm_fileinfo( self ):
    """Returns a copy of the current dpm fileinfo."""
    #
    return XSD.Fileinfo( self.__dpm_fileinfo.toDict())


  def set_bcm_oscillation_sequence( self, the_oscillation_sequence ):
    """Sets the current oscillation sequence in the ES context. If any parameter
    is missing it is read from the configuration."""
    #
    self.__bcm_oscillation_sequence = XSD.Oscillation_sequence(the_oscillation_sequence.toDict())
    if self.__bcm_oscillation_sequence.getStart() is None:
      self.__bcm_oscillation_sequence.setStart( self.default_values.getOscillation_sequence().getStart())
    if self.__bcm_oscillation_sequence.getEnd() is None:
      self.__bcm_oscillation_sequence.setEnd( self.default_values.getOscillation_sequence().getEnd())
    if self.__bcm_oscillation_sequence.getRange() is None:
      self.__bcm_oscillation_sequence.setRange( self.default_values.getOscillation_sequence().getRange())
    if self.__bcm_oscillation_sequence.getNumber_of_images() is None:
      self.__bcm_oscillation_sequence.setNumber_of_images( self.default_values.getOscillation_sequence().getNumber_of_images())
    if self.__bcm_oscillation_sequence.getOverlap() is None:
      self.__bcm_oscillation_sequence.setOverlap( self.default_values.getOscillation_sequence().getOverlap())
    if self.__bcm_oscillation_sequence.getStart_image_number() is None:
      self.__bcm_oscillation_sequence.setStart_image_number( self.default_values.getOscillation_sequence().getStart_image_number())
    if self.__bcm_oscillation_sequence.getNumber_of_passes() is None:
      self.__bcm_oscillation_sequence.setNumber_of_passes( self.default_values.getOscillation_sequence().getNumber_of_passes())
    if self.__bcm_oscillation_sequence.getExposure_time() is None:
      self.__bcm_oscillation_sequence.setExposure_time( self.default_values.getOscillation_sequence().getExposure_time())
    #
    Debug.out("Current BCM oscillation_sequence: %s" % pprint.pformat( self.__bcm_oscillation_sequence.toDict()),2)


  def set_dpm_oscillation_sequence( self, the_oscillation_sequence ):
    """Sets the current oscillation sequence in the ES context. If any parameter
    is missing it is read from the configuration."""
    #
    self.__dpm_oscillation_sequence = XSD.Oscillation_sequence(the_oscillation_sequence.toDict())
    if self.__dpm_oscillation_sequence.getStart() is None:
      self.__dpm_oscillation_sequence.setStart( self.default_values.getOscillation_sequence().getStart())
    if self.__dpm_oscillation_sequence.getEnd() is None:
      self.__dpm_oscillation_sequence.setEnd( self.default_values.getOscillation_sequence().getEnd())
    if self.__dpm_oscillation_sequence.getRange() is None:
      self.__dpm_oscillation_sequence.setRange( self.default_values.getOscillation_sequence().getRange())
    if self.__dpm_oscillation_sequence.getNumber_of_images() is None:
      self.__dpm_oscillation_sequence.setNumber_of_images( self.default_values.getOscillation_sequence().getNumber_of_images())
    if self.__dpm_oscillation_sequence.getOverlap() is None:
      self.__dpm_oscillation_sequence.setOverlap( self.default_values.getOscillation_sequence().getOverlap())
    if self.__dpm_oscillation_sequence.getStart_image_number() is None:
      self.__dpm_oscillation_sequence.setStart_image_number( self.default_values.getOscillation_sequence().getStart_image_number())
    if self.__dpm_oscillation_sequence.getNumber_of_passes() is None:
      self.__dpm_oscillation_sequence.setNumber_of_passes( self.default_values.getOscillation_sequence().getNumber_of_passes())
    if self.__dpm_oscillation_sequence.getExposure_time() is None:
      self.__dpm_oscillation_sequence.setExposure_time( self.default_values.getOscillation_sequence().getExposure_time())
    #
    Debug.out("Current dpm oscillation_sequence: %s" % pprint.pformat( self.__dpm_oscillation_sequence.toDict()),2)


  def get_bcm_oscillation_sequence( self ):
    """
    Returns a copy of the current bcm oscillation_sequence.
    """
    #
    return XSD.Oscillation_sequence( self.__bcm_oscillation_sequence.toDict())


  def get_dpm_oscillation_sequence( self ):
    """
    Returns a copy of the current dpm oscillation_sequence.
    """
    #
    return XSD.Oscillation_sequence( self.__dpm_oscillation_sequence.toDict())


  def set_bcm_detector( self, the_detector ):
    """
    Sets the current bcm detector in the ES context.
    """
    #
    self.__bcm_detector = XSD.Detector(the_detector.toDict())
    #
    # Set the suffix for fileinfo
    #
    if not self.__bcm_fileinfo is None:
      suffix = DetectorDatabase.determine_suffix( self.__bcm_detector )
      self.__bcm_fileinfo.setSuffix( suffix )
      self.determine_template( self.__bcm_fileinfo )

  def get_bcm_detector( self ):
    """
    Returns a copy of the current bcm detector object.
    """
    #
    if self.__bcm_detector is None:
      the_detector = None
    else:
      the_detector = XSD.Detector( self.__bcm_detector.toDict() )
    #
    return the_detector

  
  def set_dpm_detector( self, the_detector ):
    """Sets the current dpm detector in the ES context."""
    #
    self.__dpm_detector = XSD.Detector(the_detector.toDict())
    #
    # Set the suffix for fileinfo
    #
    if not self.__dpm_fileinfo is None:
      suffix = DetectorDatabase.determine_suffix( self.__dpm_detector )
      self.__dpm_fileinfo.setSuffix( suffix )
      self.determine_template( self.__dpm_fileinfo )

  def get_dpm_detector( self ):
    """Returns a copy of the current dpm detector object."""
    #
    if self.__dpm_detector is None:
      the_detector = None
    else:
      the_detector = XSD.Detector( self.__dpm_detector.toDict() )
    #
    return the_detector

  
  def set_bcm_beam( self, the_beam ):
    """Sets the current beam object."""
    #
    self.__bcm_beam = XSD.Beam(the_beam.toDict())


  def get_bcm_beam( self ):
    """Returns a copy of the current beam object."""
    #
    if self.__bcm_beam is None:
      the_beam = None
    else:
      the_beam = XSD.Beam( self.__bcm_beam.toDict())
    #
    return the_beam
    

  def set_dpm_beam( self, the_beam ):
    """Sets the current beam object."""
    #
    self.__dpm_beam = XSD.Beam(the_beam.toDict())


  def get_dpm_beam( self ):
    """Returns a copy of the current beam object."""
    #
    if self.__dpm_beam is None:
      the_beam = None
    else:
      the_beam = XSD.Beam( self.__dpm_beam.toDict())
    #
    return the_beam
    

  def set_dpm_calculated_resolution( self, the_resolution ):
    """
    Sets the resolution calculated by the DPM.
    """
    #
    self.__dpm_calculated_resolution = the_resolution


  def get_dpm_calculated_resolution( self ):
    """
    Returns a copy of the current calculated resolution.
    """
    #
    return self.__dpm_calculated_resolution
    

  def set_bcm_collect_resolution( self, the_resolution ):
    """
    Sets the bcm collect resolution.
    """
    #
    self.__bcm_collect_resolution = the_resolution


  def get_bcm_collect_resolution( self ):
    """
    Returns a copy of the current resolution.
    """
    #
    return self.__bcm_collect_resolution
    

  def set_bcm_database_comment( self, the_comment ):
    """
    Sets the current data base comment.
    """
    #
    self.__bcm_database_comment = the_comment


  def get_bcm_database_comment( self ):
    """
    Returns a copy of the current database comment.
    """
    #
    return self.__bcm_database_comment
    

  def set_bcm_experiment( self, the_experiment ):
    """
    Sets the current bcm experiment.
    """
    #
    self.__bcm_experiment = XSD.Experiment(the_experiment.toDict())


  def get_bcm_experiment( self ):
    """
    Returns a copy of the current experiment object.
    """
    #
    if self.__bcm_experiment is None:
      the_experiment = None
    else:
      the_experiment = XSD.Experiment( self.__bcm_experiment.toDict())
    #
    return the_experiment


  def set_bcm_dataCollectionId( self, the_dataCollectionId ):
    """
    Sets the current ISPyB dataCollectionId reference.
    """
    self.__bcm_dataCollectionId = the_dataCollectionId


  def get_bcm_dataCollectionId( self ):
    """
    Returns the current ISPyB dataCollectionId reference.
    """
    return self.__bcm_dataCollectionId


  def set_bcm_sessionId( self, the_sessionId ):
    """
    Sets the current ISPyB sessionId reference.
    """
    self.__bcm_sessionId = the_sessionId


  def get_bcm_sessionId( self ):
    """
    Returns the current ISPyB sessionId reference.
    """
    return self.__bcm_sessionId

  def isOnline( self ):
    """Returns the online status."""
    #
    return self.__online


  def setOnline( self, online ):
    """Sets the online status."""
    #
    self.__online = online


  def set_path_to_log_file( self, fileinfo ):
    """
    Sets the path to the log file.
    "log_dir" is the location of the log file determined
    by the content of the fileinfo, log_dir is later compared
    to self.__current_log_dir in order to see if it has changed.

    The log_prefix is <prefix>_<run_number>_dnafiles if the
    fileinfo is provided.
    """
    #
    log_dir = None
    self.log_prefix = "dnafiles"
    #
    # Make sure that DNALOGDIR is an absolute path
    #
    DNALOGDIR = os.environ["DNALOGDIR"]
    if not DNALOGDIR is None:
      Debug.out("DNALOGDIR set to: %s" % DNALOGDIR,2)
      if DNALOGDIR[0] != "/":
        if DNALOGDIR[0:2] == "./":
          DNALOGDIR = os.path.join(os.getcwd(), DNALOGDIR[2:])
        elif DNALOGDIR[0] == ".":
          DNALOGDIR = os.getcwd()
        else:
          DNALOGDIR = os.path.join(os.getcwd(), DNALOGDIR)
        Debug.out("New absolute path DNALOGDIR: %s" % DNALOGDIR,2)
    else:
      DNALOGDIR = os.getcwd()
    #
    # Check if called without a fileinfo
    #
    top_directory = None
    if fileinfo is None:
      #
      # No fileinfo...
      #
      top_directory = DNALOGDIR
    else:
      #
      # Ok, we got a fileinfo...
      #
      top_directory = fileinfo.getDirectory()
      prefix = fileinfo.getPrefix()
      run_number = fileinfo.getRun_number()
      #
      # Strip off the prefix...
      #
      prefix = self.strip_ref_prefix(prefix)
      #
      #
      #
      self.log_prefix = "%s_%d_dnafiles" % (prefix, run_number)
    #
    # Create log_dir...
    #
    log_dir = "%s/%s" % (top_directory, self.log_prefix)
    #
    # Check if the directory has changed
    #
    if log_dir != self.__current_log_dir:
      #
      # Check that we can write in this directory
      #
      if not os.access(top_directory, os.W_OK ):
        top_directory = DNALOGDIR
        #
        # Create new log_dir...
        #
        log_dir = "%s/%s" % (top_directory, self.log_prefix)
      #
      # Keep the new directory
      #
      self.__current_log_dir = log_dir 
      #
      #
      #
      if not os.access( self.__current_log_dir, os.W_OK ):
        #
        # Create a new directory if it's not already present
        #
        os.system("mkdir -p %s" % self.__current_log_dir)
      elif top_directory == DNALOGDIR:
        #
        # Do nothing if we are in DNALOGDIR
        #
        pass
      else:
        #
        # List all directories
        #
        dir_list = os.listdir(top_directory)
        Debug.out("Content of %s: %s" % (top_directory, dir_list),3)
        #
        # Check if directories with log_prefix present...
        #
        log_prefix_list = []
        log_prefix_length = len( self.log_prefix)
        for name in dir_list:
          if len(name) >= log_prefix_length:
            if name[0:log_prefix_length] == self.log_prefix:
              log_prefix_list.append(name)
        Debug.out("Directories starting with %s: %s" % ( self.log_prefix,
                                                        log_prefix_list),3)
        max_number = 0
        if len(log_prefix_list) > 1:
          #
          # We have more than one directory, find out which
          # has the highest number
          #
          max_number = 0
          for name in log_prefix_list:
            if len(name) > log_prefix_length:
              try:
                number = int(name[log_prefix_length+1:])
                if number > max_number:
                  max_number = number
              except:
                pass
          #
          # Increase the max_number with one
          #
          max_number = max_number+1
        elif len(log_prefix_list) == 1:
          #
          # Just one directory
          #
          max_number = 1
        if max_number > 0:
          Debug.out("New number of directory: %d" % max_number,3)
          #
          # Move the old directory
          #
          os.system("mv %s %s~%d" % ( self.__current_log_dir, self.__current_log_dir, max_number))
        #
        # Make the new directory
        #
        os.system("mkdir -p %s" % self.__current_log_dir)          
        #      self.logwrite("Writing log file in %s" % self.__current_log_dir)
    Debug.out("Current log dir set to: %s" % self.__current_log_dir, 2)


  def get_current_log_dir( self ):
    """
    Returns the current log directory.
    """
    #
    Debug.out("Get current log dir: %s" % self.__current_log_dir , 3)
    return self.__current_log_dir


  def get_default_log_dir( self ):
    """
    Returns the default (startup) log directory.
    """
    #
    Debug.out("Get default log dir: %s" %  self.__default_log_dir, 3)
    return self.__default_log_dir


  def calculate_max_resolution( self ):
    """
    Calculates the max resolution at the edge of the detecor
    given parameters from the BCM.
    """
    #
    resolution_max = None
    if self.__bcm_experiment is not None:
      #
      # Calculates the max resolution at the edge of the detecor
      # given parameters from the BCM.
      #        
      wavelength    = self.__bcm_experiment.getWavelength()
      distance      = self.__bcm_experiment.getDistance()
      detector_type = self.__bcm_detector.getType()
      detector_width = None
      if detector_type == "adsc":
        detector_width = 184.32 # mm, 2304 (no pixels) * 0.080 (pixel size)
      else:
        #
        # Assuming detector width to be two times the average beam centre,
        # i.e. just the sum of the beam centre X and Y
        #
        detector_width = self.__bcm_beam.getX()+self.__bcm_beam.getY()
      #
      # Resolution calculated with the minimum distance to
      # the detector border
      #
      d1 = self.__bcm_beam.getX()
      d2 = detector_width - d1
      d3 = self.__bcm_beam.getY()
      d4 = detector_width - d3
      min_d = min([d1,d2,d3,d4])
      #
      # Angle corresponding to min_d
      #
      two_theta = math.atan(min_d/distance)
      theta = two_theta / 2.0
      #
      # And, finally, the maximum resolution
      #
      resolution_max = wavelength / (2.0 * math.sin(theta))
    #
    return resolution_max


  def strip_ref_prefix( self, the_prefix ):
    """
    Strips off "ref-" and "postref-" prefix-prefix.
    """
    #
    if the_prefix != None:
      if len(the_prefix) > 4:
        if the_prefix[0:4] == "ref-":
          the_prefix = the_prefix[4:]
        elif len(the_prefix) > 8:
          if the_prefix[0:8] == "postref-":
            the_prefix = the_prefix[8:]
    #
    return the_prefix


  def setArchive_reference_images( self, value ):
    """
    Sets the flag determining if reference images should be archived or not.
    """
    #
    self.__current_archive_reference_images = value


  def isArchive_reference_images( self ):
    """
    Returns the flag determining if reference images should be archived or not.
    """
    #
    value = False
    if self.__current_archive_reference_images:
      value = True
    return value


  def getPath_to_image( self, fileinfo, image_number ):
    """
    Determines the full path to an image given a fileinfo object and an image number.
    """
    #
    prefix         = fileinfo.getPrefix()
    run_number     = fileinfo.getRun_number()
    suffix         = fileinfo.getSuffix()
    if suffix is None:
      suffix = DetectorDatabase.determine_suffix( self.__bcm_detector)
    directory      = fileinfo.getDirectory()
    path_to_image  = os.path.join(directory, "%s_%d_%03d.%s" % (prefix, run_number, image_number, suffix))
    #
    return path_to_image


  def get_dna_context( self ):
    """
    Returns the current DNA context as a "Dna_context" object.
    """
    #
    dna_context = XSD.Dna_context()
    dna_context.setOnline( self.__online)
    dna_context.setCurrent_fileinfo(                 self.__bcm_fileinfo                     )
    dna_context.setCurrent_oscillation_sequence(     self.__bcm_oscillation_sequence         )
    dna_context.setCurrent_detector(                 self.__bcm_detector                     )
    dna_context.setCurrent_experiment(               self.__bcm_experiment                   )
    dna_context.setCurrent_beam(                     self.__bcm_beam                         )
    dna_context.setCurrent_calculated_resolution(    self.__dpm_calculated_resolution        )
    dna_context.setCurrent_collect_resolution(       self.__bcm_collect_resolution           )
    dna_context.setCurrent_database_comment(         self.__bcm_database_comment             )
    dna_context.setCurrent_log_dir(                  self.__current_log_dir                  )
    dna_context.setCurrent_archive_reference_images( self.__current_archive_reference_images )
    #
    return dna_context
    
    
  def round_off_to_n_decimals( self, value, no_decimals ):
    """
    Funtion that rounds of a decimal number to a given number of decimals.
    """
    #
    tmp1 = value*(10**no_decimals)
    tmp2 = int(tmp1+0.5)
    tmp3 = tmp2*1.0/(10**no_decimals)
    #
    return tmp3


#
# Not very elegant singleton definition but it works.
#
ESContext = ESContext()

