/*=======================================================================
 * All files in the distribution of the DPS system are Copyright
 * 1996 by the Computational Biology group in the Department of Biological
 * Sciences at Purdue University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this entire copyright notice is duplicated in all such
 * copies, and that any documentation, announcements, and other materials
 * related to such distribution and use acknowledge that the software was
 * developed by the Computational Biology group in the Department of
 * Biological Sciences at Purdue University, W. Lafayette, IN by Ingo
 * Steller and Michael G. Rossmann. No charge may be made for copies,
 * derivations, or distributions of this material without the express
 * written consent of the copyright holder.  Neither the name of the
 * University nor the names of the authors may be used to endorse or
 * promote products derived from this material without specific prior
 * written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *======================================================================*/

/*=====================================================================*
 *                                                                     *
 *                         Data Processing Suit                        *
 *                                                                     *
 *                              dps_index                              *
 *                                                                     *
 *                        Written by Ingo Steller                      *
 *                                                                     *
 *                              File: index.c                          *
 *                                                                     *
 *=====================================================================*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "util.h"
#include "index.h"
#include "ind_gen_direct.h"
/* do_index(): This routine is the main routine for indexing. On a given*
 *	    list of xyz-coordinates and the maximum expected length	*
 *	    of the cell edges,  it tries to index the reflections. The	*
 *	    output is a 3 by 3 matrix with the orientation matrix.	*
 *	    At the moment is it only possible to run index in automatic *
 *	    mode. In this mode index will select the orientation matrix *
 *	    by itself.	Later other modes with certain influence of the *
 *	    will be included.						*/

int do_index(int number_xyz, 
	     struct vector xyz[], 
	     float d_max, 
	     Index_Mode index_mode, 
	     struct matrix_3x3 *om_out)

{
  int i,dpserrflg;				/* Counter */
  int number_dir;			/* Number of direct lattice vectors (t_vectors)*/
  int number_cut;
  struct tvector t_vector[N_TVECTOR];	/* Array for the t-vectors. */
  int num_om = 0;			/* Number of generated orientation matrices (num_om < NUM_OM_MAX) */
	int selected_om;			/* Finally selected orientation matrix */
	struct cell_struct cell, cell_rec;	/* Direct and reciprocal cell generated from om */
	struct matrix_3x3 om[MAX_OM_COUNT];	/* Orientation matrix array*/
	float cut_crit = CUT_CRIT;		/* Criterium for cutting t-vector list */
	/* First generate a list of possible direct space vectors. This is
       the main part of the indexing program				*/
	/* for debugging...*/
	number_dir = gen_direct_lattice_vectors(N_TVECTOR, d_max, xyz, &number_xyz, &t_vector[0]);

	/* Merge t_vectors with same length, same or opposite direction and
       similar number of fitting reflections. For more detail look at the
       comments in ind_sort_merge.c					*/
#ifdef INDEX_DEBUG
	printf("t_vectors after gen_direct_lattice_vectors\n");
	print_tvector_list(number_dir,&t_vector[0]);
#endif

	number_dir = merge_tvectors_same_n(number_dir, &t_vector[0]);

#ifdef INDEX_DEBUG
	printf("t_vectors after merge\n");
	print_tvector_list(number_dir,&t_vector[0]);
#endif

	/* Sort the t_vectors after their number of fitting reflections 
       (1st crtiterium) and after their length (2nd criterion). This
       is necessary for the following routine (cut_tvector_list)
     */

	sort_tvectors_on_n_and_l(number_dir, &t_vector[0]);

#ifdef INDEX_DEBUG
	printf("t_vectors after sort\n");
	print_tvector_list(number_dir,&t_vector[0]);
#endif

	/* Normally only a few direct space vectors have many fitting
       reflections and after sorting in regard to this number you will 
       find a sharp break at a certain point. To avoid wrong t_vectors
       we like to cut the t_vector list at this point. cut_tvector_list
       does this job. 
     */

	number_cut = cut_tvector_list(number_dir, cut_crit, &t_vector[0]);

#ifdef INDEX_DEBUG
	printf("t_vectors after cut\n");
	print_tvector_list(number_cut,&t_vector[0]);
#endif

	/* Now finally sort the t_vector list in respect to the length of 
       the vectors (Shortest first!). Thsi will generate the smalles unit
       cell as the first generated orientation matrix.			*/

	/* Now we loop until we have at least one OM. If we have no OM, we
       increase number_cut and resort. That way we include more and more 
       direct space vectors. */

	while(num_om < 1) {

		/* If we run out of t_vectors we have to exit at the moment.... */
		if(number_cut > number_dir) {
			printf("DIRECT SPACE VECTORS DID NOT RESULT INTO A ORIENTATION MATRIX\n\nTry indexing with a smaller or larger maximum cell edge or try REFIX!\nA useful idea is to use the MEASURE CELL option before indexing\n");
			/*exit(1);*/
			return (1);
       	}
		dpserrflg = 0;
#ifdef INDEX_DEBUG
		printf("NEW CYCLE with %d t_vectors\n", number_cut+1);
#endif

		sort_tvectors_on_l(number_cut, &t_vector[0]);

#ifdef INDEX_DEBUG
		printf("t_vectors after sort\n");
		print_tvector_list(number_cut,&t_vector[0]);
#endif

		/* Choose non linear combinations of direct space vectors to generate unit
	   cells, make the reciprocal unit cell (orientation matrix) out of this
	   and store them in an array. Generate MAX_OM_COUNT or as much as possible
	   orientation matrices.
	*/
		num_om = choose_non_lin(number_xyz, xyz, number_cut, &t_vector[0], MAX_OM_COUNT,  &om[0]);

		/* Increase number_cut for a possible next cycle */
		number_cut++;

	}
	/* Finally, select the 'best' of these as the result. More on how the
       best is chosen in ind_sort_merge.c				*/

	selected_om = choose_best_om(num_om,  om,  number_xyz,  xyz);
	printf("selected %d\n",selected_om);

	/* Sort, copy and return the final orientation matrix to the calling routine  */

	*om_out = om[selected_om];
	/* hrp 01.09.2004
	   I don't think this is necessary as the reduce_cell stuff will do
	   all the permutations anyway, and test against type I and type II 
	   cells
	   sort_cell_vectors(om_out);
	*/
	reduce_cell(om_out);
	return (0);
}


