/*=======================================================================
 * All files in the distribution of the DPS system are Copyright
 * 1996 by the Computational Biology group in the Department of Biological
 * Sciences at Purdue University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this entire copyright notice is duplicated in all such
 * copies, and that any documentation, announcements, and other materials
 * related to such distribution and use acknowledge that the software was
 * developed by the Computational Biology group in the Department of
 * Biological Sciences at Purdue University, W. Lafayette, IN by Ingo
 * Steller and Michael G. Rossmann. No charge may be made for copies,
 * derivations, or distributions of this material without the express
 * written consent of the copyright holder.  Neither the name of the
 * University nor the names of the authors may be used to endorse or
 * promote products derived from this material without specific prior
 * written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *======================================================================*/

/*=====================================================================*
 *                                                                     *
 *                         Data Processing Suit                        *
 *                                                                     *
 *                              dps_index                              *
 *                                                                     *
 *                        Written by Ingo Steller                      *
 *                                                                     *
 *                                                                     *
 *                                                                     *
 *=====================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
/* #include <malloc.h> */
#include "util.h"
#include "ind_gen_direct.h"

/* get_projections_unsorted() Generates an array with the projections	*
 *		of all reflections in the xyz array onto the vector	*
 *		n_vector. Store the smallest projection in the proj[0]	*
 *		and the largest in proj[number_xyz-1].			*/

void get_projections_unsorted(struct vector xyz[], int number_xyz, 
struct vector n_vector, float *proj)
{
	int i;		/* Counter */
	int min_i, max_i;	/* Indices of the array elements with the maximum
				   and minimum values	*/
	float max, min;	/* Minimum and maximum values of the projections. */


	/* For initializing the variables for finding the minimum
       and maximum we have to handle the first coordinate outside
       the loop. */

	/* The projection is the scalar product of the two vectors	*/
	proj[0]=scalar_prodf(xyz[0],n_vector);

	/* Initialize variables for maximum, minimum and the corresponding
       number of the element in the proj array.	*/
	max=proj[0];
	min=proj[0];
	min_i=0;
	max_i=0;

	/* That's the main loop, looping through the reflections, generating
       the projections and getting the new maxima or minima if necessary    */
	for( i = 1; i < number_xyz; i++) {
		/* generate the projection and store it in the array */
		proj[i] = scalar_prodf(xyz[i],n_vector);

		/* Is the value bigger than the old max? */
		if(proj[i]>max) {
			max=proj[i];
			max_i=i;
			continue;
		}
		/* IS the value smaller than the old min? */
		if(proj[i]<min) {
			min=proj[i];
			min_i=i;
		}
	}
	/* Now we have to exchange the array element with the minimum
       value against the array element in proj[number_xyz-1] and
       the maximum value in proj[0]. */

	/* First save the values of proj[number_xyz-1] and proj[0] in the
       array elements of the maximum and minimum value. */
	proj[min_i]=proj[number_xyz-1];
	proj[max_i]=proj[0];
	/* Now store the minimum and maximum value in proj[number_xyz-1]
       and proj[0]. */
	proj[number_xyz-1]=min;
	proj[0]=max;

}


/* get_fitting(): This routine takes a list of projections, bins them and   *
 *		  does a FFT on the bins. After calculating the absolute    *
 *		  value of the complex result, the highest value between    *
 *		  the first minimum after 0 and a value corresponding to    *
 *		  maximum expected cell edge length is searched. The heigths*
 *		  of this peak and the corresponding d_spacing value is	    *
 *		  returned.						    */

void get_fitting(float grid, float stretch, struct vector xyz[], int number_xyz, float d_max, 
struct vector *n_vector, float proj[], float *d_experiment, float *n_fit, int debug)

{

	int i;		    /* Counter */
	float extent;	    /* The range coverd by the projections (max-min) */
	float shift;	    /* The shift necessary to bring the min value to zero */
	int max;		    /* Number of bins needed */
	float exp;		    /* Exponent calcualted from max (log2(max))*/
	int max_int;	    /* New max value calculated from the rounded exponent */
	static int last_exp = 0;/* Exponet in the last call of get_fitting (static!!) */
	static float fft_help[30000]; /* Help array for storing arithm. tables etc for the FFT.
			       Also static, becasue this array needs only to be changed 
				       if max_int changes. */
	static float data[30000];	    /* Array for holding the binned data. Also static because
				       the size only changes if max_int changes. */

	float step;		    /* Stepsize from bin to bin */
	float fit;		    /* Heights of the highest peak in the FFT and temporary
				       variable during calculation. */
	float d_exp;	    /* Corresponding d-spacing for fit. */
	FILE     *fftout,
	    *projout,
	    *binout;     	    /* Files for debugging output */
	int itemp;
	/*    debug=1;*/
	if(debug != 0) {
		fftout = fopen("fft.out","w");
		projout = fopen("proj.out","w");
		binout = fopen("bin.out","w");
	}

	/* First we have to bin the data. Therefore we have to now the maximum
       and minimum values of the projections. Also the projections list
       can be unsorted,  the first and last value have to correspond to the
       highest and lowest values. */
	extent=proj[0]-proj[number_xyz-1];

	/* We have to start at zero and therefore move the origin. This makes
       no difference for the FFT. */
	shift=-proj[number_xyz-1];

	/* max is the number of bins we need. This is calculated in a way to
       assure that the introduced error by binning is not to large. The 
       value of grid gives the minimum number of bins per maximum expected
       cell edge length. The value of stretch can be used to get a higher 
       resolution for the returned d-spacing. With stretch != 1,  only
       part of the array will be used for the binned data. The rest is 
       padded with zeros. In the FFT this will result in a higher resolution
       in the returned values. Empirically we found,  that values of 5 for
       grid in 1 for stretch are sufficient. */
	max=extent*d_max*grid*stretch;

	/* Most of the FFT routines work best with arrays of the length of the 
       power of two. Because we are binning anyway, we can modify stretch to
       match this requirement. */

	/* Calculating the logarithm on the basis 2... */
	/*  exp=truncf(logf(max)/logf(2)+1); */
	exp = logf(max)/logf(2)+1.0;
	itemp = exp;
	exp = itemp;

	/* ...recalculating max after rounding exp_fract to the next higher integer... */
	max_int=powf(2, (int)exp);

	/* ... modifying stretch to fit the array. */
	stretch=stretch*(max_int/max);

	if(debug != 0) {
		printf("max %d exp %f \n", max, exp);
	}

	/* Using the new value for max. */
	max=max_int;

	if(debug != 0) {
		printf("max_int %d stretch %f\n", max_int, stretch);
	}

	/* Calculating the stepsize between two consecutive bins. This value is
       needed to calculate the bin number of a projection. */
	step=extent/((float)(max-1)/stretch);

	if(debug != 0) {
		printf("max %d step %f \n", max, step);
	}

	/* In the first call or when the length of the bin array changes, we
       have to reallocate memory for the array and initialize the FFT routine. */
	if(exp != last_exp) {
		last_exp=exp;

		/* Free the old arrays */
		/* free(fft_help);
        free(data);*/

		/* Allocate the new memory. Due to the documentation of the FFT
	   routine the size of fft_help should be 2*max+15. Unfortunately, 
	   this seems not to be right,  because with this value following
	   malloc or free calls lead to segmentation faults. 15000 is a value 
	   that works at the moment. This must be further investigated. */
		/*        fft_help=dps_malloc("get_fitting", "fft_help", (size_t)2*max+15000);
        data=dps_malloc("get_fitting", "data", (size_t)max*sizeof(float));
*/
		/* Call the initialization for the FFT routine */
		rffti_(&max, &fft_help[0]);
	}

	/* Zero the data array */
	for(i=0;i<max;i++) {
		data[i]=0.0;
	}

	/* Go through the list of reflections and calculate the bin number; than
       increment the number in the bin by one. */
	for(i=0;i<number_xyz;i++) {
		++data[(int)((proj[i]+shift)/step)];
	}
	if(debug != 0) {
		for(i=1;i<max;i++) {
			fprintf(binout, "%d %f\n",i, data[i]);
		}
	}

	/* Apply the FFT */
	rfftf_(&max, &data[0], &fft_help[0]);

	/* Calculate the absolut value of the complex numbers. The complex numbers
       are stored as:
       c_real[1], c_imag[1], c_real[2], c_imag[2].....
       
       The absolut values are finally stored the following way:
       abs[1], abs[2], ....
       
       !! We do not take the square root,  because we only want to compare the
       size of the values. This saves a lot of time!! */
	/* Harry chaNGED next line from for(i=1;i<max;i=i+2) */
	for(i=1;i<max-1;i=i+2) {
		data[(i+1)/2]=data[i]*data[i]+data[i+1]*data[i+1];
	}

	if(debug != 0) {
		for(i=1;i<max/2;i++) {
			fprintf(fftout, "%d %f %f\n",i, 1.0/i*(max-1)*step, sqrt(data[i]));
		}
		for(i=1;i<number_xyz;i++) {
			fprintf(projout, "%d %f\n",i, proj[i]);
		}
	}
	/* We need now to find the highest maximum correspoding to cell edges 
       between 0 and the maximum expected cell edge length. We also have to 
       acoid to sit on the slope of the maximum at zero. Therefore,  we
       start at 0 move along until we find a local minimum and looking the 
       in the are between the minimum and the maximum expected cell edge
       length for the highest peak.
    */

	/* First we need the value at 0 */
	fit=data[0]*data[0];

	/* The following loop finds the minimum, assign the value at the minimum to
       fit and sets i to  start with the next value */
	i=d_max*(max-1)*step*0.1;
	if (debug != 0) {
		printf("start at %d ",i);
	}
	while(data[i]<fit) {
		fit=data[i];
		i++;
	}
	if(debug != 0) {
		printf("minimum %d with %f\n", i-1, data[i-1]);
	}
	i++;

	/* This loop, running from the minimum+1 to the bin correspoding to the
       maximum cell edge length,  finds the maximum inthis area. It assigns
       the maximum value to fit and stores the correcponding bin number
       in d_exp. */
	d_exp=i; /* If the bin after the minimum is already the global maximum. */
	fit=data[i];

	while(i<d_max*(max-1)*step) {
		if(data[i]>fit) {
			fit=data[i];
			d_exp=i;
			if(debug != 0) {
				printf("%d ", i);
			}
		}
		i++;
	}

	if(debug != 0) {
		printf("d_exp before %f ", d_exp);
	}

	/* Calculating the d_spacing from the bin number */
	d_exp=1.0/(d_exp)*(max-1)*step;

	if(debug != 0) {
		printf("max_search %f fit %f d_exp %f\n", fabsf(d_max*(max-1)*step), fit, d_exp);
	}

	/* Preparing the return values */
	*d_experiment=d_exp;
	*n_fit=fit;
	if(debug != 0) {
		fclose(fftout);
		fclose(binout);
		fclose(projout);
	}
}


/* ref_n_vector() Refines a direct space vector recursively and returns *
 *		  the refined vector.				        */

void ref_n_vector(struct vector xyz[],  int number_xyz, float d_max, float proj[], 
float theta1_start, float theta2_start, float theta1_step, 
float theta2_step, struct vector *n_vector,  float *n_fit, float *d_exp)

{
	float max_fit, work_fit;
	int i, j;
	int i_max, j_max;
	float work_exp, max_exp;
	float r;
	float theta1, theta2;
	float theta1_max, theta2_max;
	struct vector work_vector, max_vector;
	float theta2_step_r;
	float pi=acos(-1.0);

	max_fit=0;
	theta1_step=theta1_step/2;
	theta2_step=theta2_step/2;
#ifdef REF_N_VECTOR_DEBUG
	printf("step1 %f step2 %f theta1_start %f theta2_start %f\n", theta1_step, theta2_step, theta1_start,  theta2_start);
#endif
	for(i=-2;i<=2;i++) {
		theta1=theta1_start-i*theta1_step;
		work_vector.z=cosf(theta1);
		if(theta1<0) theta1=-theta1;
		r=sinf(theta1);
#ifdef REF_N_VECTOR_DEBUG
		printf("radius %f ", r);
#endif
		if(r<0.000001) {
			theta2_step_r=0;
		}
		else {
			if(2.0*pi/5.0>theta2_step) {
				theta2_step_r=theta2_step/r;
			}
			else {
				theta2_step_r=(2.0*pi)/5.0;

			}
		}
		for(j=-2;j<=2;j++) {
			theta2=theta2_start+j*theta2_step_r;
			work_vector.x=sinf(theta2)*r;
			work_vector.y=cosf(theta2)*r;
			get_projections_unsorted(xyz, number_xyz, work_vector, proj);
			get_fitting(5.0, 1.0,  xyz, number_xyz, d_max, &work_vector, proj, &work_exp, &work_fit, GET_FITTING_DEBUG);
#ifdef REF_N_VECTOR_DEBUG
			printf("work_vector: %f %f %f fit: %f d_exp %f thetas: %f %f \n", work_vector.x, work_vector.y, work_vector.z,
			    work_fit, work_exp, theta1, theta2);
#endif
			if(work_fit>max_fit){
				cp_vectorf(work_vector, &max_vector);
				max_fit=work_fit;
				max_exp=work_exp;
				theta1_max=theta1;
				theta2_max=theta2;
				i_max=i;
				j_max=j;
			}

		}
	}
#ifdef REF_N_VECTOR_DEBUG
	printf("new_vector: %f %f %f fit: %f d_exp %f thetas: %f %f count: %d %d\n", max_vector.x, max_vector.y, max_vector.z,
	    max_fit, max_exp, theta1_max, theta2_max, i_max, j_max);
#endif
	if(((i_max<=-2 || i_max>=2) || (j_max<=-2 || j_max>=2)) && ( theta1_max >= 0.000001)) {
		theta1_step=theta1_step*1.5;
		theta2_step=theta2_step*1.5;
	}
	if((theta1_step>0.0004) || (theta2_step>0.0004)){
		ref_n_vector(xyz,  number_xyz, d_max, proj, theta1_max, theta2_max, theta1_step, theta2_step, n_vector, n_fit,
		    d_exp);
	}
	else {
		cp_vectorf(max_vector, n_vector);
		*d_exp=max_exp;
		*n_fit=max_fit;
		return;
	}
}

/* gen_direct_lattice_vectors(): Generates direct lattice vectors of unity  *
 *	length,  calculates the projections of the reciprocal lattice points*
 *	onto this direction and uses FFT to find a periodicity in the	    *
 *	projections. Than find the most propable directions,  refine	    *
 *	the directions and return a list of direct space vectors with the   *
 *	heigths of the highest peak in the FFT. The return value is the	    *
 *	number of found direct space vectors.				    */

int  gen_direct_lattice_vectors(int max_tvectors, float d_max,
struct vector xyz[], int *number_xyz,
struct tvector t_vector[])

{

	int i, j;				/* Counters	*/
	float pi=acos(-1);			/* pi = 3.14..... */

	float *proj;	/* Array for the projection on one direct space direction */

	float theta1_area;			/* Area to be covered by theta1 */

	float theta1, theta2;		/* Actually sphere coordinates */
	float theta1_step, theta2_step;	/* Stepsize for the sperical coordinates */
	float r;				/* Radius for the calculation of theta2 */
	float theta2_step_r;		/* Modified theta2_step value for the circle
						   with radius r. */

	int i_no, j_no;			/* Number of values in the array for storing
					   the heights of the highest peaks in the
						   FFT of a direct lattice directions. */
	float *circle;			/* Pointer to this array */
	float mean;				/* Mean value of the values of the array */
	float max;				/* Maximum value of the values of the array */
	int points;				/* Filled points in the array */

	struct vector work_vector;		/* direct space vector direction */
	float d_exp;			/* d-spacing found for the projections
						   on work vector in the FFT. */
	float n_fit;			/* Heights of the highest peak in the
						   FFT (at d_exp). */

	float low;				/* Actual threshold for identifying maxima
						   in the array */
	int mean_factor;			/* Factor for modifying threshold */
	float old_low;			/* Low value of the round before. */
	int k, l;				/* Counters */
	int yes;				/* Boolean if there is a higher maximum in the
						   neighborhood... */

	int t_vectors;			/* Number of direct space vectors, index for the array */

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
	FILE *out;
	printf(" GEN_DIRECT_LATTICE_VECTORS entered...\n");
	out=fopen("2d_data.out","w");
#endif

	/* Do some output */
	printf("Generating direct lattice vectors (This may take some time):\n");

	/* Allocate some memory for the projections list... */
	proj = dps_malloc("gen_direct_lattice_vectors", "proj", sizeof(float)*(*number_xyz+1));

	/* The direct space vectors are generated by applying sphere	    *
     * coordinates. The two angles are theta1 and theta2. For a more	    *
     * detailed description please have a look in the publication	    */

	/* Area to cover for theta1 pi/2 == semi-sphere */
	theta1_area=pi/2;

	/* Step size for theta1. 0.03 was found empirically to work. */
	theta1_step=0.03;

	/* Choose the same stepsize for theta2 */
	theta2_step=theta1_step;

	/* The heights of the highest peak in the FFT for one direction is	    *
     * stored in an array with th indices i and j. i_max and j_max are the  *
     * maximum values of i and j to cover theta1_area with the stepsize	    */
	i_no=(int)(theta1_area/theta1_step+2);
	j_no=(int)(2*pi/theta2_step+2);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
	printf("max_i max_j %d %d\n", i_no, j_no);
#endif

	/* Allocate memeory for the array */
	/*	 free(circle); harry removes this as it's implicated in crashes after indexing failure with a too-small cell specified*/
	circle = dps_malloc("gen_direct_lattice_vectors", "circle", i_no*j_no*sizeof(float));
	/* Harry initializes "circle" here as it causes a crash on a second call*/
	for(i=0;i<=i_no*j_no-1;i++)
	{
		circle[i] = 0.0;
	}
	/* Start values for some statistics. These will be used in finding the  *
     * maxima in the array afterwards.					    */
	mean=0.0;
	max=0.0;
	points=0;

	/* Start value for thata1. theta 1 will run from 0 to theta1_area */
	theta1=-theta1_step;

	/* Some more output */
	printf ("|");
	for(i=0; i<i_no; i++) printf("-");
	printf("|\n");
	/* theta1 loop */
	for(i=0;theta1<theta1_area;i++) {
		theta1=theta1+theta1_step;

		printf(".");
		fflush(stdout);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
		printf("theta1 %f i %d\n",theta1, i);
#endif

		/* The z-coordinate of the direct lattice vector is only dependend
	   on theta1 */
		work_vector.z=cosf(theta1);

		/* Calculating the radius of the circle, theta2 will be generated
	   in. */
		r=sinf(theta1);

		/* If theta1 == 0, we have only one point for theta2. */
		if(r==0) {
			/* These values will result in exact one step in theta2 */
			theta2_step_r=2*pi;
			theta2=-3*pi;
		}
		else {
			/* calculating the stepsize for theta2. */
			theta2_step_r=theta2_step/r;
			theta2=-pi-theta2_step_r;
		}
		for(j=0;theta2<pi;j++){
			theta2=theta2+theta2_step_r;

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
			printf("theta2 %f j %d\n", theta2, j);
#endif

			/* Calculating the two other components of the direct lattice
	       vector */
			work_vector.x=sinf(theta2)*r;
			work_vector.y=cosf(theta2)*r;
#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
			printf("work_vector.x %f work_vector.y %f\n",work_vector.x,work_vector.y);
#endif

			/* Now we can calculate the projections of the reciprocal
	       lattice points on the direct lattice vector. */
			get_projections_unsorted(xyz, *number_xyz, work_vector, proj);
			/* Finding a periodicity in the projections using FFT */
			get_fitting(5.0, 1.0, xyz, *number_xyz, d_max, &work_vector, proj, &d_exp, &n_fit, GET_FITTING_DEBUG);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
			fprintf(out, "%f %f %f %d %d\n", sinf(theta2)*theta1, cosf(theta2)*theta1, sqrt(n_fit),i,j);
#endif

			/* Store n_fit, the heights of the highest peaks in the FFT in
	       the array. */
			circle[i*j_no+j]=n_fit;

			/* For the statistics of mean and maximum value */
			mean=mean+n_fit;
			if(n_fit>max) max=n_fit;
			++points;
		}

		/* Padding of unused fields (Some platforms (OSF)) will have problems otherwise) */
		for(j=j+1;j<j_no;j++) circle[i*j_no+j]=0.0;
	}

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
	fclose(out);
#endif
	printf(".\n");

	/* Calculating the mean value */
	mean=mean/points;

	/* At this point we have an array (circle) with the heights of the highest
       peak of the FFT for the calculated direct space vectors. Now we have
       to find the largest of them and refine their directions. To minimise
       the number of direct space directions to be refined, we are looking
       for local maxima. This is done by looking at the array points around
       a point that is above a certain threshold. If one of the surrounding 
       points is higher,  we do not refine the direction of the actual
       direction. Also the array is distorted and the above described 
       algorithm will not work for small theta1's, it seems to be working.   */

	/* Initializing some variables */
	t_vectors=0;
	mean_factor=1;
	low=max;

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
	printf("mean %f max %f\n", mean, max);
#endif

	/* We decrease our threshold for finding maxima until we have at least
       max_tvectors vectors refined. */
	printf(" Refining  %d direct space vectors:\n", max_tvectors);

	printf("|");
	for(i=0;i<max_tvectors-2;i++) printf("-");
	printf("|\n");

	while(t_vectors < max_tvectors) {
		/* We need the old low value (in the first round = max) to not refine
	   the same directions in every round. */

		old_low=low;

		/* The threshold low will be varied from max to mean until we find
	   enough maxima. */
		low=((float)mean_factor*mean+max)/(1.0+(float)mean_factor);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
		printf("NEW MEAN_FACTOR!!! %d low %f \n", mean_factor, low);
#endif

		/* Now looping through the array */
		for(i=0;i<i_no;i++) {
			/* if we have enough t_vectors, exit the loop */
			if(t_vectors == max_tvectors) break;
			for(j=0;j<j_no;j++) {

				/* Is the value between our threshold (low) and the threshold 
		   of the last round (old_low). If yes, the check the 
		   neighborhood... */
				if(circle[i*j_no+j]>low && circle[i*j_no+j]<old_low) {

					/* Sorry for reusing max. Here we store the value of the
		       central point in it. */
					max=circle[i*j_no+j];

					/* If yes stays 0, we do not have a higher value in the
		       neighboorhood. */
					yes=0;

					/* Looking one step in the array in each direction */
					for(k=i-1;k<=i+1;k++) {
						for(l=j-1;l<=j+1;l++) {
							if(k >= 0 && k<i_no && l >= 0 && l < j_no) {

								/* Did we find a higerh value? */
								if(circle[k*j_no+l]>max) {
/* code folded from here */
	yes=1;
/* unfolding */
								}
							}
						}
					}

					/* If have not found a higher value, let's refine this
		       directions. */
					if(! yes) {

						/* theta1 can be calculated from i and theta1_step */
						theta1=i*theta1_step;

						/* Theta2 will be calcualted from j, theta2_step and theta1 */
						if(theta1==0) {
							theta2=0;
						}
						else {
							/* printf("theta1 = %f and sinf(theta1) = %f\n",theta1,
				 sinf(theta1));*/
							theta2=j*theta2_step/sinf(theta1)-pi;
						}

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
						printf("theta1 %f theta2 %f\n", theta1, theta2);
#endif

						/* Refine the direction recursively... */
						ref_n_vector(xyz,  *number_xyz, d_max, proj, theta1, theta2, theta1_step,
						    theta2_step, &work_vector, &n_fit, &d_exp);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
						printf("final work_vector: %f %f %f fit: %f d_exp %f \n", work_vector.x,
						    work_vector.y, work_vector.z, n_fit, d_exp);
#endif

						/* We have to recalculate the d-spacing on the final
			   vector (d_exp), because it got lost
			   in the refinement recursion. For this purpose we
			   have to recalculate the directions and do the FFT. */
						get_projections_unsorted(xyz, *number_xyz, work_vector, proj);
						get_fitting(5.0, 1.0, xyz, *number_xyz, d_max, &work_vector, proj, &d_exp,
						    &n_fit, GET_FITTING_DEBUG);

						/* Now we can store the vector, scaled by the inverse 
			   d-spacing found in the FFT. */
						t_vector[t_vectors].vec.x=work_vector.x/d_exp;
						t_vector[t_vectors].vec.y=work_vector.y/d_exp;
						t_vector[t_vectors].vec.z=work_vector.z/d_exp;

						/* Store also the heigths of the highest peak in the 
			   FFT. We do the square-root now, which bring n_fit
			   onto a scale that makes it comparabel to the
			   number of reflections we put in. */
						t_vector[t_vectors].fit=sqrtf(n_fit);

						/* Calculate also the length of the vector and store it */
						t_vector[t_vectors].len=vec_lenf(t_vector[t_vectors].vec);

#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
						printf("final vector %f %f %f fit: %f len: %f d_exp: %f\n", t_vector[t_vectors].vec.x,
						    t_vector[t_vectors].vec.y, t_vector[t_vectors].vec.z, t_vector[t_vectors].fit,
						    t_vector[t_vectors].len, d_exp);
						printf("continue\n");
						fgetc(stdin);
#endif

						/* Finally, increment the t-vector counter. */
						t_vectors++;
						printf(".");
						fflush(stdout);

						/* if we have enough t_vectors, exist the loop */
						if(t_vectors == max_tvectors) break;
					}
				}
			}
		}

		/* We now looped through the array. If we have to do a next round we
	   nead the mean_factor increased for a lower threshold. */
		mean_factor++;
	}

	printf("\n");

	/* Free the memory from the array */
	free(circle);
#ifdef GEN_DIRECT_LATTICE_VECTORS_DEBUG
	printf(" GEN_DIRECT_LATTICE_VECTORS left with %d t_vectors...\n", t_vectors);
	fclose(out);
#endif
	return(t_vectors-1);
}

