/*=======================================================================
 * All files in the distribution of the DPS system are Copyright
 * 1996 by the Computational Biology group in the Department of Biological
 * Sciences at Purdue University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this entire copyright notice is duplicated in all such
 * copies, and that any documentation, announcements, and other materials
 * related to such distribution and use acknowledge that the software was
 * developed by the Computational Biology group in the Department of
 * Biological Sciences at Purdue University, W. Lafayette, IN by Ingo
 * Steller and Michael G. Rossmann. No charge may be made for copies,
 * derivations, or distributions of this material without the express
 * written consent of the copyright holder.  Neither the name of the
 * University nor the names of the authors may be used to endorse or
 * promote products derived from this material without specific prior
 * written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *======================================================================*/

/*=====================================================================*
 *                                                                     *
 *                         Data Processing Suit                        *
 *                                                                     *
 *                              dps_index                              *
 *                                                                     *
 *                        Written by Ingo Steller                      *
 *                                                                     *
 *                          File: ind_choose.c                         *
 *                                                                     *
 *=====================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include "index.h"
#include "vector_math.h"
#include "ind_choose.h"
#include "util.h"

/* choose_non_lin(): Choose the first three non linear depended 	*
 *		direct lattice vectors.					*/

int choose_non_lin(int t_vectors, struct tvector t[], int max_om, 
struct matrix_3x3 *om )
{
	int i,j,k;		/* counter */
	float	volume;		/* volume of the cell */
	float	tmp;		/* temporaery variable */
	int 	om_count = 0;   /* Found orientation matrices */
	int a,b,c;		/* Number of vectors for generating om after
					   sorting */

#ifdef CHOOSE_FIRST_NON_LINEAR_DEBUG
	printf("CHOOSE_NON_LINEAR entered...\n");
#endif

	for(i=2;i<=t_vectors;i++) {
		for(j=0;j<=i-2;j++) {
			for(k=j+1;k<=i-1;k++) {
				volume = vec_volumef(t[i].vec,t[j].vec,t[k].vec);
				tmp=t[i].len * t[j].len * t[k].len;
				if ( (volume > tmp*LIN_DEP_FACTOR) ) {
					if (t[i].len > t[j].len) {
						if (t[i].len > t[k].len) {
							a=i;
							if (t[j].len > t[k].len) {
								b=j;
								c=k;
							}
							else {
								b=k;
								c=j;
							}
						}
						else {
							a=k;
							b=i;
							c=j;
						}
					}
					else {
						if (t[j].len > t[k].len) {
							a=j;
							if (t[i].len > t[k].len) {
								b=i;
								c=k;
							}
							else {
								b=k;
								c=i;
							}
						}
						else {
							a=k;
							b=j;
							c=i;
						}
					}
#ifdef CHOOSE_FIRST_NON_LINEAR_DEBUG
					printf("No. %d choosen %d %d %d volume %f tmp %f rel %f\n",om_count,c,b,a,volume,
					    tmp,tmp/volume);
#endif

					om_from_vectors(t[c].vec,t[b].vec,t[a].vec,&om[om_count]);

					++om_count;
					if (om_count > (max_om-1)) return(om_count-1);
				}
#ifdef CHOOSE_FIRST_NON_LINEAR_DEBUG
				printf("%d %d %d volume %f tmp %f\n",i,j,k,volume,tmp);
#endif
			}
		}
	}

#ifdef CHOOSE_FIRST_NON_LINEAR_DEBUG
	printf("CHOOSE_NON_LINEAR left with %d matrices ...\n",om_count);
#endif
	return(om_count);

}

/* choose_best_om(): Given a list of possible orientation matrices, this routine returns the	*
 *		     number of the 'best' OM. This is decided on the number of reflections	*
 *		     that can be indexed within a certain margin around inter (MAX_DIFF_HKL)	*
 *		     and on the size of the unit cell. In principle the smallest unit cell	*
 *		     with the highest number of reflections is choosen.				*
 * Prerequirements: The list of orientation matrices should start with a OM generated by the	*
 *		    three shortes vectors (See routine index() ).				*/

int choose_best_om(int num_om, struct matrix_3x3 om[], int number_xyz, struct vector xyz[])

{
	int i;			/* Counter */
	int best_indexed;	/* Number of indexed reflections for the actually choosen OM */
	int tmp_indexed;	/* Temporarely variable, same usage as best_indexed. */
	float best_volume;	/* Volume of the actually choosen OM */
	float tmp_volume;	/* Temporarely variable, same usage as best_volume. */
	struct hkl_int	*hkl;		/* Integer hkl, same index as xyz */
	struct hkl_fract *hkl_calc;	/* Fractional hkl, same index as xyz */
	int selected;		/* Finally selected best OM. */

#ifdef CHOOSE_BEST_OM_DEBUG
	printf("CHOOSE_BEST_OM entered...\n");
#endif
	/* Allocate memory for the hkl arrays. */
	hkl = dps_malloc("choose_best_om", "hkl", sizeof(struct hkl_int)*number_xyz);
	hkl_calc = dps_malloc("choose_best_om", "hkl_calc", sizeof(struct hkl_fract)*number_xyz);

	/* Set starting values, so that we definetely reach the statements of the 'if' statement */
	best_indexed = 0;
	best_volume = 1.0/vec_volumef(om[0].r[0], om[0].r[1], om[0].r[2]);

	/* Go through all OM's and find the 'best' */
	for(i=0;i<num_om;i++) {
		/* Generate fractional and intetger (rounded) HKL */
		xyz_2_hkl(om[i], xyz, number_xyz, hkl, hkl_calc);
		/* Calculate the number of reflections indexable with this OM. */
		tmp_indexed = num_ok_hkl(hkl_calc, number_xyz, hkl, MAX_DIFF_HKL);
		/* Calcualte the volume of this OM. */
		tmp_volume = 1.0/vec_volumef(om[i].r[0], om[i].r[1], om[i].r[2]);
		/* Decide whether this OM is 'better' than the best found until now */
		/* First: Is the number of indexable reflections higher? */
#ifdef CHOOSE_BEST_OM_DEBUG
		printf("OM %d: indexed: %d volume %f\n",i,tmp_indexed,tmp_volume);
		print_3x3(om[i]);
#endif
		if (best_indexed < tmp_indexed) {
			/* If yes, is the volume not to much higher. This should only avoid	    *
			 * to find a higher order cell ( face/body centered etc.) which would have  *
			 * twice the volume							    */
			if (best_volume*VOLUME_FACTOR > tmp_volume) {
				/* Ok, OM is better... */
				best_indexed = tmp_indexed;
				selected = i;
				best_volume=tmp_volume;
			}
		}
	}
#ifdef CHOOSE_BEST_OM_DEBUG
	printf("CHOOSE_BEST_OM left with %d selected...\n",selected);
#endif
	return(selected);
}
