/* jpeg writing for mosflm - v0.1
 * originally by g.winter
 * 15th october 2001
 * 
 * changelog
 * 8th ov 01 i realised that the image may be being rotated by the 
 * jpeg creation process - or even worse having x and y swapped over - 
 * so check this. transposing is bad news!
 * 
 * 6th August 2002
 * Allowing the size to be specified is bringing touble. If should 
 * be the case that the region to be zoomed and the zoom factor
 * only should be supplied, to allow the subroutine to determine everything 
 * and prevent segmentation violations.
 * 
 * Therefore changing the calling structure to write_jpeg etc. Since this
 * actually involves the create_image subroutine more, this message is in
 * the wrong place!
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 

Specification for the image requests/responses
15th April 2002
G. Winter
 
1. All coordinates will be given in terms of the display, in which the origin
   is in the top lefthand corner of the image, and `x' is perpendicular to the
   lower edge of the image:

   +----->x
   |
   |
   |
   V
   y

2. Creation of the jpeg images should handle the possibility that the limits 
   may be a little back-to-front. For instance, the possibility that the x
   coordinate of the tlc might be greater than tat of the brc.

3. The section of the image should be specified in terms of the tlc and brc,
   and nothing else. This will make for simple tracing of the variables. Thus,
   in every subroutine/function, the input must be of the form of two arrays
   each of two integers. These will be in the pixel coordinates, rotated to
   match the screen orientation.

4. When referring to the x and y ordinates in an ordered pair, the order will
   be (x, y).

5. The jpeg writing routine should be:

   subroutine write_jpeg(destination,
			 nrec, 
			 iylen, 
			 image, 
			 factor, 
			 tlc, 
			 brc,
			 newline) 

   Where destination is the file descriptor of the socket, nrec and iylen are
   the dimensions of the image in memory, image is the pointer to the image
   array, factor is the desired `zoom' factor, tlc and brc are as discussed
   and newline is an integer to say whether to append a newline to the image
   or no (relevent for embedding the image in an XML document).

   In this subroutine, the relative orientation of the tlc and brc are 
   important, since this should still be in the display coordinates.	       
 */

#include <stdio.h>
#include <stdlib.h>
#include "jinclude.h"
#include "jpeglib.h"
#include "jerror.h"
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

typedef struct triple{
  unsigned char r;
  unsigned char g;
  unsigned char b;
} triple;

/* TGGB added typdef greylevel to be used instead of triple for greyscale image */

typedef unsigned char greylevel ;

static int jpeg_quality=85;
static char imagename[80] = "";
static unsigned char base64convert[64];

/* structures used internally */

typedef struct mosflm_jpeg_mgr {
  struct jpeg_destination_mgr pub;
  unsigned char cmpbuffer[54]; /* 54 bytes at a time will be written here */
  unsigned char encbuffer[72]; /* then base64 encoded to 72 bytes to write */
  int socket;
} mosflm_jpeg_mgr;

/* utility functions, for base64 encoding */
/* this should be called before any encoding is done */

int base64init() {
  int i;

  for(i = 0; i < 26; i++) {
    base64convert[i] = 65 + i;
    base64convert[i + 26] = 97 + i;
  }

  for(i = 0; i < 10; i++) {
    base64convert[i + 52] = 48 + i;
  }

  base64convert[62] = 43;
  base64convert[63] = 47;

  return 0;
}

int base64get(
  int i
)
{
  return base64convert[i];
}

int base64enc(
  unsigned char *in,
  unsigned char *out,
  int endbyte
)
{
  int i;
  unsigned char sixbits[4];
  int endflag;

  endflag = (4 * endbyte) / 3;

  sixbits[0] = (in[0] & 0xfc) >> 2;
  sixbits[1] = ((in[0] & 0x03) << 4) + ((in[1] & 0xf0) >> 4);
  sixbits[2] = ((in[1] & 0x0f) << 2) + ((in[2] & 0xc0) >> 6);
  sixbits[3] = (in[2] & 0x3f);

  for(i = 0; i < 4; i++) {
    if(endflag == 0) {
      out[i] = base64get(sixbits[i]);
    } else {
      if(i > endflag) {
        if(sixbits[i] == 0) {
          out[i] = 61;
        } else {
          out[i] = base64get(sixbits[i]);
        }
      } else {
        out[i] = base64get(sixbits[i]);
      }
    }
  }

  return 0;
}

int base64encline(
  unsigned char *in,
  unsigned char *out,
  int nbytes
)
{
  int i, n, remainder;

  n = nbytes / 3;
  remainder = nbytes % 3;

  for(i = 0; i < n; i++) {
    base64enc(&(in[3 * i]), &(out[4 * i]), 0);
  }

  if(remainder != 0) {
    base64enc(&(in[3 * n]), &(out[4 * n]), remainder);
  }

  return 0;
}

/* this routine is called to initiate the jpeg writing process */

void mosflm_init_destination(
  j_compress_ptr cinfo
)
{
  struct mosflm_jpeg_mgr *dest = (struct mosflm_jpeg_mgr *) cinfo->dest;
  /* the storage here is staticly allocated */
  dest->pub.next_output_byte = dest->cmpbuffer;
  memset(dest->cmpbuffer, 0, 54);
  dest->pub.free_in_buffer = 54;
}

/* this routine is called each time the buffer gets full */

boolean mosflm_empty_output_buffer(
  j_compress_ptr cinfo
)
{
  struct mosflm_jpeg_mgr *dest = (struct mosflm_jpeg_mgr *) cinfo->dest;
  base64encline(dest->cmpbuffer, dest->encbuffer, 54);
  write(dest->socket, dest->encbuffer, 72);
  memset(dest->cmpbuffer, 0, 54);
  memset(dest->encbuffer, 0, 72);
  dest->pub.next_output_byte = dest->cmpbuffer;
  dest->pub.free_in_buffer = 54;
  return TRUE;
}

boolean mosflm_empty_output_bin(
  j_compress_ptr cinfo
)
{
  struct mosflm_jpeg_mgr *dest = (struct mosflm_jpeg_mgr *) cinfo->dest;
  write(dest->socket, dest->cmpbuffer, 54);
  dest->pub.next_output_byte = dest->cmpbuffer;
  dest->pub.free_in_buffer = 54;
  return TRUE;
}

/* and this routine gets called when the writing process is finished */

void mosflm_term_destination(
  j_compress_ptr cinfo
)
{
  int i, n;
  struct mosflm_jpeg_mgr *dest = (struct mosflm_jpeg_mgr *) cinfo->dest;
  i = 54 - dest->pub.free_in_buffer;
  n = 4 * (i / 3);
  if((4 * i) > (3 * n)) {
    n += 4;
  }

  base64encline(dest->cmpbuffer, dest->encbuffer, i);
  write(dest->socket, dest->encbuffer, n);
}

void mosflm_term_destination_bin(
  j_compress_ptr cinfo
)
{
  int i, n;
  struct mosflm_jpeg_mgr *dest = (struct mosflm_jpeg_mgr *) cinfo->dest;
  i = 54 - dest->pub.free_in_buffer;
  write(dest->socket, dest->cmpbuffer, i);
}

int mosflm_write_jpeg(
  int sock,
  int nx,
  int ny,
  triple *image,
  int newline,
  int binary
)
{
  struct mosflm_jpeg_mgr dest;
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  JSAMPROW row_pointer[1];


  base64init();
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);
  cinfo.dest = (struct jpeg_destination_mgr *) &dest;

  dest.socket = sock;
  dest.pub.init_destination = mosflm_init_destination;
  if(binary == 0) {
      printf("--> mosflm_write_jpeg: doing non-binary write!\n") ;
    dest.pub.empty_output_buffer = mosflm_empty_output_buffer;
    dest.pub.term_destination = mosflm_term_destination;
  } else {
      printf("--> mosflm_write_jpeg: doing binary write!\n") ;
    dest.pub.empty_output_buffer = mosflm_empty_output_bin;
    dest.pub.term_destination = mosflm_term_destination_bin;
  }
 

  cinfo.image_width = nx;
  cinfo.image_height = ny;
  cinfo.input_components = 3; 
  cinfo.in_color_space = JCS_RGB;
  jpeg_set_defaults(&cinfo);
  jpeg_set_quality(&cinfo, jpeg_quality, TRUE);
  jpeg_start_compress(&cinfo, TRUE);

  while(cinfo.next_scanline < cinfo.image_height) {
    row_pointer[0] = (unsigned char *) &(image[cinfo.next_scanline * ny]);
    jpeg_write_scanlines(&cinfo, row_pointer, 1);
  }
  
  jpeg_finish_compress(&cinfo);
  jpeg_destroy_compress(&cinfo);
  if(newline != 0) {
    write(sock, "\n", 1);
  }
  return 0;
}

/* mosflm_write_grey_jpeg:
 *   called by write_grey_jeg_ (this file)
 *   creates a jpeg from the 8-bit greyscale image data it is passed
 */

int mosflm_write_grey_jpeg(int sock, int x, int y, greylevel *image)
{
    struct mosflm_jpeg_mgr dest;
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;
    JSAMPROW row_pointer[1];

    printf("Writing greyscale jpeg to socket: %d\n",sock) ;

    base64init();
    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_compress(&cinfo);
    cinfo.dest = (struct jpeg_destination_mgr *) &dest;

    dest.socket = sock;
    dest.pub.init_destination = mosflm_init_destination;
    printf("--> mosflm_write_jpeg: doing non-binary write!\n") ;
    dest.pub.empty_output_buffer = mosflm_empty_output_buffer;
    dest.pub.term_destination = mosflm_term_destination;
 
    cinfo.image_width = x;
    cinfo.image_height = y;
    cinfo.input_components = 1; 
    cinfo.in_color_space = JCS_GRAYSCALE;
    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, jpeg_quality, TRUE);
    jpeg_start_compress(&cinfo, TRUE);
    
    while(cinfo.next_scanline < cinfo.image_height) {
	row_pointer[0] = (unsigned char *) &(image[cinfo.next_scanline * y]);
	jpeg_write_scanlines(&cinfo, row_pointer, 1);
    }
    
    jpeg_finish_compress(&cinfo);
    jpeg_destroy_compress(&cinfo);
    write(sock, "\n", 1);
    return 0;
}

int drawstuff(
  int x,
  int y,
  triple *image
)
{
  int i, j;
  triple red, green, blue, white;

  red.r = 255; red.g = 0; red.b = 0;
  green.r = 0; green.g = 255; green.b = 0;
  blue.r = 0; blue.g = 0; blue.b = 255;
  white.r = 255; white.g = 255; white.b = 255;

  for(i = 0; i < 10; i++) {
    for(j = 0; j < 10; j++) {
      image[j + i * x] = red;
      image[(x - j - 1) + i * x] = green;
      image[j + (y - i - 1) * x] = blue;
      image[(x - j - 1)  + (y - i - 1) * x] = white;
    }
  }

  return 0;
}

void jpeg_set_quality_(
  int *newvalue
)
{
  jpeg_quality = *newvalue;
}

int jpeg_set_filename_(
  char *filename,
  int *length
)
{
  if(*length < 80) {
    memset(imagename, 0, 80);
    strncpy(imagename, filename, *length);
    printf("Output filename set to %s\n", filename);
  }
  return 0;
}


/* this is supposed to be akin to a fortran subroutine */

int write_jpeg_(
  int *_fd,
  int *_imwidth,
  int *_imheight,
  short int *_image,
  int *_pixmin,
  int *_pixmax,
  int *_zoom,
  int *_tlc,
  int *_brc,
  int *_newline,
  int *_theta,
  int *_invert,
  int *_binary
)
{
  int zoom, xmin, xmax, ymin, ymax, x, y, dx, dy;
  int invert_x, invert_y, min, max, nx, ny, i, j, k, l, temp;
  int theta, invert, pix, file, binary;
  double total;
  double mostimea,mostimeb;
  triple *image, pixel;

  if(*_fd == -1) {
    /* create an output file */
    if(strlen(imagename) > 0) {
      *_fd = open(imagename, O_TRUNC | O_WRONLY | O_CREAT, S_IRWXU);
      file = 1;
    }
  }


  /* this may be called if the file we tried to open belonged to someone 
     else, so open would return -1 */

  if(*_fd == -1) {
    char tempnam[20];
    sprintf(tempnam, "tempfile_XXXXXX");
    *_fd = mkstemp(tempnam);
    file = 1;
  }

  /* first, see if the limits are the right way around, and if they're not,
     swap them */

  xmin = _tlc[0];
  ymin = _tlc[1];
  xmax = _brc[0];
  ymax = _brc[1];
  binary = *_binary;

  if(xmax < xmin) {
    temp = xmin;
    xmin = xmax;
    xmax = temp;
    invert_x = 1;
  } else {
    invert_x = 0;
  }

  if(ymax < ymin) {
    temp = ymin;
    ymin = ymax;
    ymax = temp;
    invert_y = 1;
  } else {
    invert_y = 0;
  }

  theta = *_theta;
  invert = *_invert;
  zoom = *_zoom;

  if(zoom == 0) zoom = 1;

  if(zoom > 0) {
    /* we're zooming in on the image */
    nx = (xmax - xmin) * zoom;
    ny = (ymax - ymin) * zoom;
  } else {
    /* we must be zooming out */
    nx = - (xmax - xmin) / zoom;
    ny = - (ymax - ymin) / zoom;
  }

  image = (triple *) malloc (sizeof(triple) * nx * ny);

  x = *_imwidth;
  y = *_imheight;

  /*  fprintf(stderr, "Writing a jpeg - zoom %d\n", zoom);
      fprintf(stderr, "Width %d Height %d\n", nx, ny);
  */

  if((*_pixmin == 0) &&
     (*_pixmax == 0)) {
    /* we need to manually determine the limits on the pixel values -
       this should be done over the image as a whole */
    total = 0;
    for(i = 0; i < y; i++) {
      for(j = 0; j < x; j++) {
	if(_image[i * x + j] >= 0) {
	  total += _image[i * x + j];
	} else {
	  total += 32767;
	}
      }
    }

    min = 4.0 * total / (x * y);
    max = 0;
  } else {
    min = *_pixmin;
    max = *_pixmax;
  }

  /* next make the actual image from the shortmap */

  dx = xmax - xmin;
  dy = ymax - ymin;

  if(zoom > 0) {
    for(i = ymin; i < ymax; i++) {
      for(j = xmin; j < xmax; j++) {
	temp = _image[i * x + j];
	if(temp < 0) temp = 32767;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	for(k = 0; k < zoom; k++) {
	  for(l = 0; l < zoom; l++) {
	    image[((i - ymin) * zoom + k) * zoom * dx + 
		  ((j - xmin) * zoom + l)] = pixel;
	  }
	}
      }
    }
  } else {
    int factor = zoom * -1;
    int here;
    
    for(i = 0; i < ny; i++) {
      for(j = 0; j < nx; j++) {
	temp = 0;
	for(k = 0; k < factor; k++) {
	  for(l = 0; l < factor; l++) {
	    here = ymin * x + xmin + 
	      j * factor + i * x * factor +
	      k * x + l;
	    if(_image[here] >= 0) {
	      temp += _image[here];
	    } else {
	      temp += 32767;
	    }
	  }
	}
	temp /= factor * factor;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	image[i * nx + j] = pixel;
      }
    }
  }

  /* the next stage is to make the appropriate transformations */

  if(invert_y == 1) image_swap_y(image, nx, ny);
  if(invert == 0) {
    if(invert_x == 1) image_swap_x(image, nx, ny);
  } else {
    if(invert_x == 0) image_swap_x(image, nx, ny);
  }
 
  if(theta != 0) {
       fprintf(stderr, "1 Rotating image by %d degrees\n", theta);
    image_rotate_acw(image, nx, ny, theta);
     fprintf(stderr, "Done rotating\n");
  }

  if((theta == 90) || 
     (theta == 270)) {
    if(nx != ny) {
      int nt;
      nt = nx;
      nx = ny;
      ny = nt;
    }
  }

  /* finally, all we need to do is send this image out */
  ctimer_(&mostimea);
  mosflm_write_jpeg(*_fd, nx, ny, image, *_newline, binary);
  ctimer_(&mostimeb);
  printf("write_jpeg took: %6.3f\n",mostimeb-mostimea);
  fflush(stdout);

  free(image);

  /* if we wrote to a file, close it */
  if(file == 1) {
    close(*_fd);
    *_fd = -1;
  }

  return 0;
}

/* write_grey_jpeg_:
 *   Called from create_grey_image (in create_image.f)
 *   Prepares image data to make a full-size, greyscale jpeg from an image
 *   Calculates (pseudo-)contrast limits, and creates array with 8-bit greyscale pixel values
 *   then calls mosflm_write_grey_jpeg to turn that array into a jpeg
 *   (supposed to be akin to a fortran subroutine)
 */

int write_grey_jpeg_(
  int *_fd,
  int *_imwidth,
  int *_imheight,
  short int *_image
)
{
    int file, x, y, i, min, max, temp, pixel ;
    greylevel *image ;
    double total, mostimea,mostimeb;

    /* if no socket destination is given... */
    if(*_fd == -1) {
	/* create an output file */
	if(strlen(imagename) > 0) {
	    *_fd = open(imagename, O_TRUNC | O_WRONLY | O_CREAT, S_IRWXU);
	    file = 1;
	}
    }
    /* this may be called if the file we tried to open belonged to someone 
       else, so open would return -1 */
    if(*_fd == -1) {
	char tempnam[20];
	sprintf(tempnam, "tempfile_XXXXXX");
	*_fd = mkstemp(tempnam);
	file = 1;
    }

    x= *_imwidth ;
    y= *_imheight ;

    image= (greylevel *) malloc(sizeof(greylevel) * x * y) ;


    /* we need to manually determine the limits on the pixel values -
       this should be done over the image as a whole */
    total = 0;
    for(i = 0; i < (x * y) ; i++) {
	if(_image[i] >= 0) {
	    total += _image[i];
	} else {
	    total += 32767;
	}
    }

    min = 4.0 * total / (x * y);
    max = 0;
    
    /* next make the actual image from the shortmap */

    for (i= 0 ; i < (x * y) ; i++) {
	temp= _image[i] ;
	if (temp < 0) temp= 32767 ;
	pixel= (255 * (temp - min)) / (max - min) ;
	if (pixel < 0) {
	    pixel= 0 ;
	}
	else if (pixel > 255) {
	    pixel= 255 ;
	}
	image[i]= pixel ;
    }
        
    /* finally, all we need to do is send this image out */
    ctimer_(&mostimea);

    mosflm_write_grey_jpeg(*_fd, x, y, image);

    ctimer_(&mostimeb);
    printf("write_jpeg took: %6.3f\n",mostimeb-mostimea);
    fflush(stdout);

    free(image);

    /* if we wrote to a file, close it */
    if(file == 1) {
	close(*_fd);
	*_fd = -1;
    }

    return 0;
}



int write_jpeg_spots_(
  int *_fd,
  int *_imwidth,
  int *_imheight,
  short int *_image,
  int *_pixmin,
  int *_pixmax,
  int *_zoom,
  int *_tlc,
  int *_brc,
  int *_newline,
  int *_theta,
  int *_invert,
  short int *_spots,
  int *_nspot
)
{
  int zoom, xmin, xmax, ymin, ymax, x, y, dx, dy, _i, _j;
  int invert_x, invert_y, min, max, nx, ny, i, j, k, l, temp;
  int theta, invert, pix, file, nspot;
  double total, p_x, p_y;
  triple *image, pixel;

  if(*_fd == -1) {
    /* create an output file */
    if(strlen(imagename) > 0) {
      *_fd = open(imagename, O_TRUNC | O_WRONLY | O_CREAT, S_IRWXU);
      file = 1;
    }
  }


  /* this may be called if the file we tried to open belonged to someone 
     else, so open would return -1 */

  if(*_fd == -1) {
    char tempnam[20];
    sprintf(tempnam, "tempfile_XXXXXX");
    *_fd = mkstemp(tempnam);
    file = 1;
  }

  /* first, see if the limits are the right way around, and if they're not,
     swap them */

  nspot = *_nspot;

  xmin = _tlc[0];
  ymin = _tlc[1];
  xmax = _brc[0];
  ymax = _brc[1];

  if(xmax < xmin) {
    temp = xmin;
    xmin = xmax;
    xmax = temp;
    invert_x = 1;
  } else {
    invert_x = 0;
  }

  if(ymax < ymin) {
    temp = ymin;
    ymin = ymax;
    ymax = temp;
    invert_y = 1;
  } else {
    invert_y = 0;
  }

  theta = *_theta;
  invert = *_invert;
  zoom = *_zoom;

  if(zoom == 0) zoom = 1;

  if(zoom > 0) {
    /* we're zooming in on the image */
    nx = (xmax - xmin) * zoom;
    ny = (ymax - ymin) * zoom;
  } else {
    /* we must be zooming out */
    nx = - (xmax - xmin) / zoom;
    ny = - (ymax - ymin) / zoom;
  }

  image = (triple *) malloc (sizeof(triple) * nx * ny);

  x = *_imwidth;
  y = *_imheight;

  /*  fprintf(stderr, "Writing a jpeg - zoom %d\n", zoom);
      fprintf(stderr, "Width %d Height %d\n", nx, ny);
  */

  if((*_pixmin == 0) &&
     (*_pixmax == 0)) {
    /* we need to manually determine the limits on the pixel values -
       this should be done over the image as a whole */
    total = 0;
    for(i = 0; i < y; i++) {
      for(j = 0; j < x; j++) {
	if(_image[i * x + j] >= 0) {
	  total += _image[i * x + j];
	} else {
	  total += 32767;
	}
      }
    }

    min = 4.0 * total / (x * y);
    max = 0;
  } else {
    min = *_pixmin;
    max = *_pixmax;
  }

  /* next make the actual image from the shortmap */

  dx = xmax - xmin;
  dy = ymax - ymin;

  if(zoom > 0) {
    for(i = ymin; i < ymax; i++) {
      for(j = xmin; j < xmax; j++) {
	temp = _image[i * x + j];
	if(temp < 0) temp = 32767;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	for(k = 0; k < zoom; k++) {
	  for(l = 0; l < zoom; l++) {
	    image[((i - ymin) * zoom + k) * zoom * dx + 
		  ((j - xmin) * zoom + l)] = pixel;
	  }
	}
      }
    }
  } else {
    int factor = zoom * -1;
    int here;
    
    for(i = 0; i < ny; i++) {
      for(j = 0; j < nx; j++) {
	temp = 0;
	for(k = 0; k < factor; k++) {
	  for(l = 0; l < factor; l++) {
	    here = ymin * x + xmin + 
	      j * factor + i * x * factor +
	      k * x + l;
	    if(_image[here] >= 0) {
	      temp += _image[here];
	    } else {
	      temp += 32767;
	    }
	  }
	}
	temp /= factor * factor;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	image[i * nx + j] = pixel;
      }
    }
  }

  /* the next stage is to make the appropriate transformations */

  /* now draw the spots onto the image */

  pixel.r = 255;
  pixel.g = 0;
  pixel.b = 0;

  for(i = 0; i < nspot; i++) {
    l = _spots[2 * i];
    k = _spots[2 * i + 1];
    /* 0.5 here to make the spot drawn on in the middle of the image */
    p_x = ((float) (nx * (k - xmin - 0.5)) / ((float) xmax - xmin));
    p_y = ((float) (ny * (l - ymin - 0.5)) / ((float) ymax - ymin));
    for(_i = -1; _i < 2; _i++) {
      for(_j = -1; _j < 2; _j++) {
	k = p_x + _i;
	l = p_y + _j;
	if((l >= 0) && (l < ny) && (k >= 0) && (k < nx)) {
	  image[l * nx + k] = pixel;
	}
      }
    }
  }

  if(invert_y == 1) image_swap_y(image, nx, ny);
  if(invert == 0) {
    if(invert_x == 1) image_swap_x(image, nx, ny);
  } else {
    if(invert_x == 0) image_swap_x(image, nx, ny);
  }
 
  if(theta != 0) {
       fprintf(stderr, "2 Rotating image by %d degrees\n", theta);
    image_rotate_acw(image, nx, ny, theta);
        fprintf(stderr, "Done rotating\n");
  }

  if((theta == 90) || 
     (theta == 270)) {
    if(nx != ny) {
      int nt;
      nt = nx;
      nx = ny;
      ny = nt;
    }
  }

  /* finally, all we need to do is send this image out */

  mosflm_write_jpeg(*_fd, nx, ny, image, *_newline, 0);

  free(image);

  /* if we wrote to a file, close it */
  if(file == 1) {
    close(*_fd);
    *_fd = -1;
  }

  return 0;
}

int write_jpeg_predict_(
  int *_fd,
  int *_imwidth,
  int *_imheight,
  short int *_image,
  int *_pixmin,
  int *_pixmax,
  int *_zoom,
  int *_tlc,
  int *_brc,
  int *_newline,
  int *_theta,
  int *_invert,
  short int *_predictions,
  short int *_types,
  int *_nspot,
  int *_binary
)
{
  int zoom, xmin, xmax, ymin, ymax, x, y, dx, dy, _i, _j;
  int invert_x, invert_y, min, max, nx, ny, i, j, k, l, temp;
  int theta, invert, pix, file, nspot, binary;
  double total, p_x, p_y;
  triple *image, pixel;

  triple _red, _green, _blue, _yellow;

  if(*_binary) {
    printf("writing a nice binary JPEG\n");
  }


  binary = *_binary;

  if(*_fd == -1) {
    /* create an output file */
    if(strlen(imagename) > 0) {
      *_fd = open(imagename, O_TRUNC | O_WRONLY | O_CREAT, S_IRWXU);
      file = 1;
    }
  }


  /* this may be called if the file we tried to open belonged to someone 
     else, so open would return -1 */

  if(*_fd == -1) {
    char tempnam[20];
    sprintf(tempnam, "tempfile_XXXXXX");
    *_fd = mkstemp(tempnam);
    file = 1;
  }

  /* first, see if the limits are the right way around, and if they're not,
     swap them */

  nspot = *_nspot;

  xmin = _tlc[0];
  ymin = _tlc[1];
  xmax = _brc[0];
  ymax = _brc[1];

  if(xmax < xmin) {
    temp = xmin;
    xmin = xmax;
    xmax = temp;
    invert_x = 1;
  } else {
    invert_x = 0;
  }

  if(ymax < ymin) {
    temp = ymin;
    ymin = ymax;
    ymax = temp;
    invert_y = 1;
  } else {
    invert_y = 0;
  }

  theta = *_theta;
  invert = *_invert;
  zoom = *_zoom;

  if(zoom == 0) zoom = 1;

  if(zoom > 0) {
    /* we're zooming in on the image */
    nx = (xmax - xmin) * zoom;
    ny = (ymax - ymin) * zoom;
  } else {
    /* we must be zooming out */
    nx = - (xmax - xmin) / zoom;
    ny = - (ymax - ymin) / zoom;
  }

  image = (triple *) malloc (sizeof(triple) * nx * ny);

  x = *_imwidth;
  y = *_imheight;

  /*  fprintf(stderr, "Writing a jpeg - zoom %d\n", zoom);
      fprintf(stderr, "Width %d Height %d\n", nx, ny);
  */

  if((*_pixmin == 0) &&
     (*_pixmax == 0)) {
    /* we need to manually determine the limits on the pixel values -
       this should be done over the image as a whole */
    total = 0;
    for(i = 0; i < y; i++) {
      for(j = 0; j < x; j++) {
	if(_image[i * x + j] >= 0) {
	  total += _image[i * x + j];
	} else {
	  total += 32767;
	}
      }
    }

    min = 4.0 * total / (x * y);
    max = 0;
  } else {
    min = *_pixmin;
    max = *_pixmax;
  }

  /* next make the actual image from the shortmap */

  dx = xmax - xmin;
  dy = ymax - ymin;

  if(zoom > 0) {
    for(i = ymin; i < ymax; i++) {
      for(j = xmin; j < xmax; j++) {
	temp = _image[i * x + j];
	if(temp < 0) temp = 32767;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	for(k = 0; k < zoom; k++) {
	  for(l = 0; l < zoom; l++) {
	    image[((i - ymin) * zoom + k) * zoom * dx + 
		  ((j - xmin) * zoom + l)] = pixel;
	  }
	}
      }
    }
  } else {
    int factor = zoom * -1;
    int here;
    
    for(i = 0; i < ny; i++) {
      for(j = 0; j < nx; j++) {
	temp = 0;
	for(k = 0; k < factor; k++) {
	  for(l = 0; l < factor; l++) {
	    here = ymin * x + xmin + 
	      j * factor + i * x * factor +
	      k * x + l;
	    if(_image[here] >= 0) {
	      temp += _image[here];
	    } else {
	      temp += 32767;
	    }
	  }
	}
	temp /= factor * factor;
	pix = (255 * (temp - min)) / (max - min);
	if(pix > 255) pix = 255;
	if(pix < 0) pix = 0;
	pixel.r = pix;
	pixel.g = pix;
	pixel.b = pix;
	image[i * nx + j] = pixel;
      }
    }
  }

  /* the next stage is to make the appropriate transformations */

  /* now draw the predictions onto the image */

  pixel.r = 0;
  pixel.g = 0;
  pixel.b = 255;

  _red.r = 255;
  _red.g = 0;
  _red.b = 0;

  _green.r = 0;
  _green.g = 255;
  _green.b = 0;

  _blue.r = 0;
  _blue.g = 0;
  _blue.b = 255;

  _yellow.r = 255;
  _yellow.g = 255;
  _yellow.b = 0;

  for(i = 0; i < nspot; i++) {
    l = _predictions[2 * i];
    k = _predictions[2 * i + 1];
    /* 0.5 here to make the spot drawn on in the middle of the image */
    p_x = ((float) (nx * (k - xmin - 0.5)) / ((float) xmax - xmin));
    p_y = ((float) (ny * (l - ymin - 0.5)) / ((float) ymax - ymin));
    
    if(_types[i] == 0) {
      pixel = _blue;
    } else if(_types[i] == 2) {
      pixel = _red;
    } else if(_types[i] == 3) {
      pixel = _green;
    } else {
      pixel = _yellow;
    }

    for(_i = -1; _i < 2; _i++) {
      for(_j = -1; _j < 2; _j++) {
	k = p_x + _i;
	l = p_y + _j;
	if((l >= 0) && (l < ny) && (k >= 0) && (k < nx)) {
	  image[l * nx + k] = pixel;
	}
      }
    }
  }

  if(invert_y == 1) image_swap_y(image, nx, ny);
  if(invert == 0) {
    if(invert_x == 1) image_swap_x(image, nx, ny);
  } else {
    if(invert_x == 0) image_swap_x(image, nx, ny);
  }
 
  if(theta != 0) {
        fprintf(stderr, "3 Rotating image by %d degrees\n", theta);
    image_rotate_acw(image, nx, ny, theta);
        fprintf(stderr, "Done rotating\n");
  }

  if((theta == 90) || 
     (theta == 270)) {
    if(nx != ny) {
      int nt;
      nt = nx;
      nx = ny;
      ny = nt;
    }
  }

  /* finally, all we need to do is send this image out */

  mosflm_write_jpeg(*_fd, nx, ny, image, *_newline, binary);

  free(image);

  /* if we wrote to a file, close it */
  if(file == 1) {
    close(*_fd);
    *_fd = -1;
  }

  return 0;
}
