/**********************************************************************
 *
 * work.c
 *
 * Version:     4.3
 * Date:        26/01/2001
 *
 * Version	Date		Description
 * 4.6		15/02/2001	Fix Q4 output to mar2300
 * 4.3		26/01/2001	Allow for STER3072 fast axis 2560 
 * 4.2		30/11/2000	Added funtions flip + rotate + addvalue + scale
 * 4.1		23/11/2000	Added Sterling 3072 support
 * 4.0		18/10/2000	Added CBF/imgCIF support
 * 3.3		21/06/2000	Core dump with -o: fixed
 *				Allow same o/p-format with -f
 * 3.2		23/05/2000	Sterling 1536*1280 image implemented (STER1536)
 *
 ***********************************************************************/

/*
 * System includes
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <limits.h>
#include <utime.h>
#include <math.h>

/*
 * Local includes
 */

#include "marcvt.h"
#include "version.h"
#include <mar345_header.h>
#include <mar300_header.h>
#include <marccd_header.h>

/*
 * Defines 
 */
#define SIZE		3450

#define ltell(a)	lseek( a, 0, SEEK_CUR )
#define FTW_F   	1
#define FTW_D   	2
#define FTW_DNR 	3
#define FTW_NS  	4

/*
 * Typdefs
 */

typedef int DirList( const char *, const struct stat *, int );

/*
 * Local declarations
 */

MAR345_HEADER 		h345;
MAR300_HEADER 		h300;
MARCCD_HEADER		ccd;
typedef union {
	unsigned int	i4_img[SIZE*SIZE];
	unsigned short	i2_img[2*SIZE*SIZE];
} UIMG;
UIMG			uimg;
unsigned short		i2_rec[SIZE];
static int           	i4_buf[SIZE];
static unsigned char    i1_buf[SIZE];
static int		x,size,nov,norip,norop;
static int		val2[2];
static	FILE 		*fp;
static int		nfile		= 0;
static int		ndir		= 0;
static int		nreg		= 0;
static int		fd		= 0;
static int		raw_size	= 0;
static float		raw_psize	= 0.10;
static char		swap		= 0;
static char		pck		= 0;
static char		truncate_16_bit = 1;
static DirList		dirlist;

/*
 * External declarations
 */
float			pixelsize;

/*
 * External functions
 */

extern int		Putmar345Header	(int,	MAR345_HEADER);
extern int		Putmar300Header	(int,int,MAR300_HEADER);
extern void		swaplong	();
extern void		swapshort	();
extern float		GetResol	( float, float, float ); 

extern void		usage		();
extern int		GetCIFHeader	(char *, char *, FILE *);
extern int              PutCIFHeader    (char *, char *);
extern int              GetCIFData      (char *, char *, FILE *, unsigned int *);
extern int              PutCIFData      (char *, char *, char  , unsigned int *);
extern MAR345_HEADER	CIF2mar345Header(char *);
extern void             mar3452CIFHeader(char *, char *, MAR345_HEADER );

/*
 * Local functions
 */
void			Work		(void);
static void		AsciiHeader	(void);
static void		Truncate_i4_img	(int, int, char *);
static void		TiffHeader	(void);
static void		MakeHeader	(int );
static void		rotate_anticlock(unsigned short *, int);
static void		flip_image	(unsigned short *, int);
static void		scale_image	(unsigned short *, int);
static void		add_image	(unsigned short *, int);
static void		rotate_clock	(unsigned short *, int);
int		        is_mar    	(char *f);
static int		mar_convert    	(char *f);
static int		open_file	(char *f);
static int		dodir		(char *, DirList * );
static int		dopath		(DirList * );

/******************************************************************
 * Function: Work
 ******************************************************************/
void
Work()
{
register int	i,j,k;
struct stat     sbuf;
int             ret;

	/* 
	 * Only one file was given. 
	 * If this is a directory, check all files in all directories
	 */
	for ( i=nfile=0; i<nfiles; i++, nfile++ ) {

		strcpy( path, files[i] );

#if ( DEBUG )
		printf("================================================\n");
		printf("ARGUMENT %2d: %s\n",i+1,path);
		printf("================================================\n");
#endif

        	if ( lstat( path, &sbuf ) < 0 ) {
			printf("%s: ERROR: (%s) Cannot get status!\n",prg,path);
			return;
		}

		/* Argument files[i] is a directory */
		if ( S_ISDIR( sbuf.st_mode ) ) { 
			if ( verbose && !list ) 
				printf("%s: searching  directory '%s' ...\n\n", prg,path);
			dodir( path, dirlist );
		}

		/* Argument files[i] is a regular file or link to a file */
		else if ( S_ISREG( sbuf.st_mode ) || S_ISLNK( sbuf.st_mode ) ) { 

			if ( (ipf = is_mar( path )) == NONE )
				return;

			if ( verbose ) {
				printf("%s: %s: format = %d\n",prg,path,ipf);
			}

			/* Check for necessity of output */
			if (  ipf == opf && force == 0 && strlen(outdir)==0 ) { 
				printf("%s: skipping %s: same op-format. Use -f to force o/p!\n",prg,path);
				continue;
			}

			/* Does input format match format of file */

			/**********************/
			j = mar_convert( path );
			/**********************/
		}

		/* Argument files[i] is neither a regular file nor a dir */
		else {
			printf("%s: argument '%s' must be a directory or a filename...\n",prg,path);
		}
	}

	printf("\n");
}

/******************************************************************
 * Function: dodir
 ******************************************************************/
static int dodir( char *d, DirList *func)
{
        strcpy( path, d);
        return( dopath( func ));
}

/******************************************************************
 * Function: dopath
 ******************************************************************/
static int dopath(DirList *func)
{
struct stat     sbuf;
struct dirent   *dirp;
DIR             *dp;
int             i,ret;
char            *p;
char		f1[128],f2[256];

        if ( lstat( path, &sbuf ) < 0 )
                return( func( path, &sbuf, FTW_NS ));

        if ( S_ISDIR( sbuf.st_mode ) == 0 )
                return( func( path, &sbuf, FTW_F ));

        if ( ( ret =  func( path, &sbuf, FTW_D ) ) != 0 )
                return( ret );

        p = path + strlen( path );

        *p++ = '/';
        *p = 0;

#if (  DEBUG )
	if ( recursive && verbose )
		printf("directory: %s\n",path);
#endif
/*
*/

        if ( ( dp = opendir( path )) == NULL )
                return( func( path, &sbuf, FTW_DNR ));

        while( ( dirp = readdir( dp )) != NULL ) {

		/* Skip . and .. */
                if ( strcmp( dirp->d_name , "."  ) ==  0 ||
                     strcmp( dirp->d_name , ".." ) ==  0 ) continue;

		/* 
		 * Skip all files that do not have extensions .pck,
		 * .mar or .image
		 */
		strcpy( p, dirp->d_name );
		strcpy(f1, p );
#if (  DEBUG )
		printf("file: %s\n",f1);
#endif
		/* 
		 * Is this really a mar image ?
		 */

		if ( (ipf = is_mar( path )) != NONE ) {

			/* Check for necessity of output */
			if (  ipf == opf && force == 0 && strlen(outdir)==0 ) { 
				printf("%s: skipping %s: same op-format. Use -f to force o/p!\n",prg,path);
				continue;
			}

			/**********************/
			i = mar_convert( f2 );
			/**********************/
		}

		if ( recursive )
			if ( ( ret = dopath( func)) != 0 ) break;


        }

        p[-1]=0;

        if ( closedir( dp ) < 0 )
                printf( "%s: Cannot close directory %s\n",prg,path);

        return ret;
}

/******************************************************************
 * Function: dirlist
 ******************************************************************/
static int dirlist( const char *path, const struct stat *sptr, int type )
{
        if ( type == FTW_F ) {
                if ( ( sptr->st_mode & S_IFMT ) == S_IFREG ) {
                        nreg++;
                }
        }
        else if ( type == FTW_D ) {
                        ndir++;

        }

        return 0;
}

/******************************************************************
 * Function: open_file
 ******************************************************************/
static int open_file ( char *f )
{

	if ( ( fp = fopen( f, "rb" ) ) == NULL ) {
		printf("%s: ERROR: Cannot open file %s\n", prg, f );
		return -1;
	}

	return 0;
}

/******************************************************************
 * Function: is_mar
 ******************************************************************/
int is_mar ( char *f )
{
int		i,j,k,i1,i2;
int		bo;
double		q1,q2;
static short	isize[13] = { 1024, 1152, 2048, 2304, 1200, 1600, 2000, 2300, 1800, 2400, 3000, 3450, 0 };

	swap = 0;

	/* Try to open file */
	if ( open_file( f ) < 0 )
			return 0;

	if ( force_ipf) return force_ipf;

        /*
         * Is this a CIF image: check bytes 1-15
         */
        fseek( fp, 0, SEEK_SET );
        if ( fread( str, sizeof(char), 15, fp) < 15 )
                return NONE;

        if ( strstr( str, "###CBF: VERSION" ) )
                return IMG_CIF;

	/* 
	 * Is this a CCD image: check bytes 11024-1028: string MMX
	 */ 
	fseek( fp, 1028, SEEK_SET );
	str[4] = '\0';
	str[6] = '\0';
	if ( fread( str, sizeof(char), 6, fp) < 4 )
		return NONE;

	if ( strstr( str, "MMX" ) || strstr( str, "MARCCD" ) ) return CCD;

	/* No: this is not a CCD: try MAR345 */

	fseek( fp, 0, SEEK_SET );
	if ( fread( &bo, sizeof(int), 1, fp) < 1) return NONE;

	if ( bo == 1234 ) 
		return MAR345;
	else if ( bo >= 100 && bo <= 5000 ) {
		if ( strstr( f, ".pck") ) return PCK; 
		return IMAGE;
	}

	swaplong( &bo, 1*sizeof(int) );
	swap = 1;

	if ( bo == 1234 ) 
		return MAR345;
	else if ( bo >= 100 && bo <= 5000 ){
		if ( strstr( f, ".pck") ) return PCK; 
		return IMAGE;
	}

	/* This could be a raw image file, but its size must
	 * be quadratic
	 */
	fseek( fp, 0, SEEK_END);
	i = ftell( fp );

	/* Sterling detector May 2000: x=1536 y=1280 pixels: */
	j = i/(1536*1280);
	if ( i % (1536*1280) == 0 ){
		raw_psize = 0.139;
		if ( j == 2 ) {
			raw_size  = 1600;
			return STER1536;
		}
		raw_size  = 3000;
		return STER3072;
	}
	else if ( i == ( 1536*1280*2*4  )){
		raw_psize = 0.139;
		raw_size  = 1600;
		return STER3072;
	}
	else if ( i == ( 2304*2304*2 +512 ) ) {
		raw_psize = 0.081;
		raw_size  = 2304;
		return  Q4_2304;
	}

	q1 = sqrt( (i-jump)/2. );
	q2 = q1/2;
	i1 = (int)q1;
	i2 = (int)q2;
	j = 0;	
	if ( verbose ) {
		printf("%s: %s: size = %d (i2) or %d (i4)\n",prg,f,i1,i2);
	}
	while ( (k=isize[j]) > 0 ) {
		raw_size = k;
		if ( j == 0 ) raw_psize = 0.161;
		else if ( j == 2 ) raw_psize = 0.0805;
		else if ( j <  8 ) raw_psize = 0.150;
		else raw_psize = 0.10;
		if ( ( k == i2 && ipf == RAW32 ) || i2 == in_nx ) return RAW32;
		if ( k == i1 || i1 == in_nx ) return RAW16;
		if ( k == i2 ) return RAW32;
		j++;
	}

	return NONE;
}

/******************************************************************
 * Function: mar_convert
 ******************************************************************/
static int mar_convert( char *f )
{
static short		isize[9] = { 3450, 3000, 2400, 2300, 2000, 1800, 1600, 1200, 0};
int			i, j, k, n, m, x, pos, hleno,hleni,xoff;
int			row,col,pair[2];
char			*f1, *p1, *p2=NULL, op[256],buf[128];
unsigned short		*i2;
unsigned char 		*i1;
int  			*i4,i4_val[10];
int			cutdiff=0,original_size;
FILE			*fo=NULL;
struct stat     	sbuf;
struct utimbuf		tbuf;
extern MAR300_HEADER    Getmar300Header(FILE *);
extern MAR345_HEADER    Getmar345Header(FILE *);
extern int		put_pck();
extern int 		get_pck();

	/* List: just print the file name */
	if ( list ) { 
		printf("%s: #%4d: %s \n",prg,list,f);
		if ( fp != NULL )
			fclose( fp );
		list++;
		return 1;
	}

	/* Try to open file, unless already open */
	if ( fp == NULL )
		if ( open_file( f ) < 0 )
			return 0;

	/* Read mar345 header */
	if ( ipf == MAR345 ) {
		h345 = Getmar345Header( fp );
		size = x = h345.size;
		nov = h345.high;
		if ( h345.byteorder != 1234 ) swap = 1;
		pixelsize	= h345.pixel_height/1000.;
	}
	/* Read mar300 header */
	else if ( ipf == PCK || ipf == IMAGE ) {
		h300 = Getmar300Header( fp );
		size = x = h300.pixels_x;
		nov = h300.high_pixels;
		pixelsize	= h300.p_x;
	}
	/* Raw data */
	else if ( ipf == RAW16 || ipf == RAW32 || ipf == STER1536 || ipf == STER3072 || ipf == Q4_2304 ) {
		if ( ipf == Q4_2304 ) {
			if ( jump == 0 ) jump = 512;
			size = x 	= 2304;
			cutccd 		= 2300;
			cutdiff	  	= x-cutccd;
			original_size 	= size;
		}
		if ( in_nx ) {
			size = x = in_nx;
		}
		else
			size = x = raw_size;
		nov 		= 0;
		swap		= 0;
		pixelsize	= raw_psize;
		h345.byteorder  = 1234;                       
		h345.high       = nov;                    
		h345.scanner    = 1;
		h345.format     = 1;     
		h345.size	= size;
		h345.pixels	= size*size;
		h345.time	= 60.;
		h345.wave	= 1.541789;
		h345.dist    	= 100.0;
		h345.phibeg	= 0.0;
		h345.phiend	= 1.0;
		h345.omebeg	= 0.0;
		h345.omeend	= 0.0;
		h345.theta 	= 0.0;
		h345.chi   	= 0.0;
		h345.xcen	= raw_size/2.;
		h345.ycen	= raw_size/2.;
		h345.pixel_length = raw_psize*1000.;
		h345.pixel_height = raw_psize*1000.;
		strcpy( h345.version, VERSION );                      
		strcpy( h345.program, prg);                     
		if ( use345 ) {
			i=0;
			while ( x < (k=isize[i++]) ) continue;
			if ( i>1 ) 
				i = i-2;
			else
				i = 0;
			printf("%s: %s: using size %d instead of %d for output\n",prg,f,isize[i],x);
			x = isize[i];
		}
		if ( x > 3450 ) {
			printf("%s: %s: size of image is %d. Truncating to 3450\n",prg,f,x);
			size = x = in_nx = in_ny = 3450;
		}
		if ( ipf == Q4_2304 ) {
			x		= cutccd;
			h345.size 	= x;
			h345.pixels	= x*x;
			h345.wave       = 0.933;
			h345.dist       = 150.;
			h345.pixel_length = 82;
			h345.pixel_height = 82;
		}
	}

	/* CCD data */
	else if ( ipf == CCD ) {
		fseek( fp, 1024, SEEK_SET );
		if ( fread( &ccd, sizeof(MARCCD_HEADER), 1, fp) < 1 ) {
			printf("%s: %s: Cannot read CCD header\n",prg,f);
			fclose( fp );
			return 0;
		}
		
		swap = 0;

		if ( ccd.header_byte_order <  1234 || ccd.header_byte_order > 4321 ) {
			swaplong( &ccd, 1*sizeof(MARCCD_HEADER) );
		}

		if ( ccd.header_byte_order <  1234 || ccd.header_byte_order > 4321 ) {
			printf("%s: %s: Cannot swap bytes in CCD header\n",prg,f);
			fclose( fp );
			return 0;
		}

#if ( __linux__ || VMS || __ALPHA || __osf__ )
		if ( ccd.data_byte_order == 4321 ) swap=1;
#elif ( __sgi || SUN || __hpux )
		if ( ccd.data_byte_order == 1234 ) swap=1;
#endif
		if ( cutccd ) {
			cutdiff	  = ccd.nfast - cutccd;
			if ( cutdiff < 0 ) {
				printf("%s: %s: Output size (%d) MUST be smaller than input (%d)!\n", prg,f,cutccd,ccd.nfast);
				exit(-1);
			}
			original_size = ccd.nfast;
			ccd.nfast = cutccd;
		}
		size 		= x = ccd.nfast;
		nov = 0;
		h345.size	= size;
		h345.time	= ccd.exposure_time     / 1000.;
		h345.wave	= ccd.source_wavelength / 100000.;
		h345.dist    	= ccd.xtal_to_detector  / 1000.;
		h345.phibeg	= ccd.start_phi         / 1000.;
		h345.phiend	= ccd.end_phi           / 1000.;
		h345.omebeg	= ccd.start_omega       / 1000.;
		h345.omeend	= ccd.end_omega         / 1000.;
		h345.theta 	= ccd.start_twotheta    / 1000.;
		h345.chi   	= ccd.start_chi         / 1000.;
		h345.xcen	= ccd.beam_x            / 1000.;
		h345.ycen	= ccd.beam_y            / 1000.;
		h345.pixel_length = ccd.pixelsize_x     / 1000.;
		h345.pixel_height = ccd.pixelsize_y     / 1000.;
		h345.multiplier	= 1.0;	

		h345.dosemax	= ccd.max;
		h345.dosemin	= ccd.min;
		h345.doseavg	= ccd.mean;
		h345.dosesig	= ccd.rms;
		h345.dosemin	= ccd.min;
		h345.dosebeg	= ccd.total_counts[0];
		h345.doseend	= ccd.total_counts[1];

		pos 		= ccd.header_size + 1024;
		pixelsize	= h345.pixel_height/1000.;
	}
	
	/* CBF/imgCIF data */
	else if ( ipf == IMG_CIF || ipf == IMG_CBF ) {
		/* Read CIF header */
		i = GetCIFHeader( prg, f, fp );

		/* Copy contents of CIF header into mar345 header */
		h345 = CIF2mar345Header(prg);

		nov     	= h345.high;
		pixelsize	= h345.pixel_height;
		size = x   	= h345.size;
	}

	if ( pixelsize < 0.005 ) pixelsize = 0.15;

	if ( in_nx != in_ny ) 
		size *= in_ny;
	else
		size *= size;

	i=j=0;

	/* Create output file name */
	f1 = f;

	/* Put directory component into 'op' */
	/* Put file name - directory into p2 */
	p1 = strrchr( f, '/' );
	if ( p1 != NULL && strlen( outdir ) == 0 ) {
		strcpy( op, f );
		op [ strlen(op) - strlen(p1) + 1 ] = '\0';
		p2 = p1 + 1;
	}
	else if ( strlen( outdir ) ) {
		strcpy( op, outdir );
		if ( op[ strlen(op) - 1 ] != '/' ) strcat( op, "/");
		if ( p1 != NULL )
			p2 =  p1 + 1;
		else
			p2 = f;
	}
	else {
		strcpy( op, "./");
		p2 = f;
	}

	/* Append filename to directory */
	if ( p2 != NULL )
		strcat( op, p2);

	/* p1 = File name extension ... */
	p1 = strrchr( op, '.' );
	
	if ( ipf != CCD && p1 != NULL )
		op[ strlen(op) - strlen(p1) ] = '\0';

	if      ( opf == MAR345 )
		strcat( op, ".mar" );
	else if ( opf == IMG_CIF )
		strcat( op, ".cif" );
	else if ( opf == IMG_CBF )
		strcat( op, ".cbf" );
	else if ( opf == PCK ) 
		strcat( op, ".pck" );
	else if ( opf == IMAGE )
		strcat( op, ".image" );
	else if ( opf == RAW16 )
		strcat( op, ".raw" );
	else if ( opf == RAW32 )
		strcat( op, ".raw32_" );
	else if ( opf == RAW8 )
		strcat( op, ".raw8_" );
	else if ( opf == ASCII )
		strcat( op, ".ascii" );
	else if ( opf == TIFF )
		strcat( op, ".tiff" );

	/* Append image size to extension for mar345 formats */
	if ( opf == MAR345 || opf == ASCII || opf == IMG_CIF || opf == IMG_CBF || ( opf>=RAW8 && opf<=RAW32) ) {
		if ( ipf == Q4_2304 )
			sprintf( op, "%s%d",op,cutccd);
		else
			sprintf( op, "%s%d",op,x);
		hleno = 64;
		norop = (int)(nov/8.0 + 0.875);	/* No. of overflow records */
	}
	else {
		hleno = 2*x;
		norop = (nov+x/4-1) / (x/4);
	}

	if ( ipf == MAR345 ) {
		hleni = 64;
		norip = (int)(nov/8.0 + 0.875);	/* No. of overflow records */
	}
	else {
		hleni = 2*x;
		norip = (nov+x/4-1) / (x/4);
	}

	if ( verbose < 2  ) {
		printf("%s: %s -> %s\n",prg,f,op);
	}

	if ( strlen( h345.program ) < 2 ) {
		strcpy( h345.program, prg);
		strcpy( h345.version, VERSION  );
	}

	/* Does output file already exist ? */
	if ( ( fd = open( op, O_RDONLY ) ) != -1 ) {
		if ( force == 0 ) {
			printf("%s: NOTE: '%s' exists. Skip ...\n",prg,op);
			fclose( fp );
			close( fd  );
			return 0;
		}
		close( fd );
	}

	/* Open output file */
	if ( ( fd = open( op, O_CREAT | O_TRUNC | O_RDWR )) < 0 ) {
		printf("%s: ERROR: (%s) Cannot create file!\n",prg,op);
		goto ERROR;
	}

	/* Change access mode of output file to ... */ 
	i=chmod( op, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH );

	/* Obtain creation time from old file and transfer to new file */
	if ( stat( f, &sbuf ) >= 0 ) {
		tbuf.actime  = sbuf.st_atime;
		tbuf.modtime = sbuf.st_mtime;
		if ( utime( op, &tbuf) < 0 )
			printf("%s: NOTE: (%s) Cannot give old time!\n",prg,op);
	}

	if ( ipf == IMG_CIF || ipf == IMG_CBF )
		MakeHeader(0);
	else if ( ipf != PCK && ipf != IMAGE && ( opf == PCK || opf == IMAGE ) )
		MakeHeader(0);
	else if ( (ipf == PCK || ipf == IMAGE ) && ( opf == MAR345 || opf == ASCII ) )
		MakeHeader(1);

	/* Write header */
	i = 1;
	if ( opf == MAR345 ) 
		i = Putmar345Header( fd, h345 );
	else if ( opf == PCK || opf == IMAGE )
		i = Putmar300Header( fd, 1, h300 );
	else if ( opf == TIFF )
		TiffHeader();
	else if ( opf == ASCII )
		AsciiHeader();
	else if ( opf == IMG_CIF || opf == IMG_CBF ) {
		sprintf( str, "%s (V %s)",prg,VERSION);
		mar3452CIFHeader( str, op, h345);
		close( fd );
		fd = (-1);
		i = PutCIFHeader(prg, op);
	}
	
	if ( i < 1 ) {
		printf("%s: ERROR: (%s) Cannot write header!\n",prg,op);
		fclose( fp );
		if ( fd >= 0 ) close( fd );
		return 0;
	}

	if ( opf == ASCII && nov ) goto HIGH;

CORE:
	/* Read IMAGE format */
	if ( ipf == IMAGE || ( ipf == CCD && cutccd == 0) ) {

		if ( ipf == IMAGE )
			fseek( fp, 2*x, SEEK_SET );
		else
			fseek( fp, pos, SEEK_SET );

		/* Read core */
		if ( ( i=fread( uimg.i2_img, sizeof(short), size, fp ) ) != size ) {
			printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i, size);
			goto ERROR;
		}

		/* Swap i2 array if necessary */
		if ( swap || force_swap ) 
			swapshort( uimg.i2_img, size*sizeof(short) );

	}

	/* Read Q4_2304 format */
	else if ( ipf == Q4_2304 ) {
		memset( (char *)uimg.i2_img, 0, 2*x*x*sizeof( short) );
		fseek( fp, jump + (2304-2300)*2304, SEEK_SET );

		/* Read core */
		for ( j = 0; j<cutccd; j++ ) {
			fseek( fp, cutdiff, SEEK_CUR );
			if ( ( i=fread( uimg.i2_img+j*cutccd, sizeof(short), cutccd, fp ) ) != cutccd) {
				printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, j*cutccd+i, cutccd*cutccd);
				goto ERROR;
			}
			fseek( fp, cutdiff, SEEK_CUR );
		}

		/* Swap i2 array if necessary */
		if ( swap || force_swap ) 
			swapshort( uimg.i2_img, size*sizeof(short) );

	}

	/* Read RAW16 format */
	else if ( ipf == RAW16 ) {
		fseek( fp, jump, SEEK_SET );

		if ( use345 )
			memset( (char *)uimg.i2_img, 0, 2*x*x*sizeof( short) );

		if ( in_nx+in_ny ) {
			col = in_nx;
			row = in_ny;
	  		xoff = 0;
	  		xoff = ((x-in_ny)/2)*x+(x-in_nx)/2;
		}
		else {
			row = col = x;
	  		xoff = 0;
		}
		if ( verbose ) {
			printf("%s: %s -> x=%d  in_nx=%d  size=%d  xoff=%d\n",prg,f,x,in_nx,size,xoff);
		}

		if ( !use345 ) {
			if ( ( i=fread( uimg.i2_img, sizeof(short), size, fp ) ) != size ) {
				printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i, size);
				goto ERROR;
			}
		}
		else {
			for ( j=0; j<row; j++ ) {
				if ( ( i=fread( i2_rec, sizeof(short), col, fp ) ) != col) {
					printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i,col);
					goto ERROR;
				}
				memcpy( (char *)&uimg.i2_img[xoff], (char *)i2_rec, sizeof(short)*col);
				xoff += x;
			}
		}

		/* Swap i2 array if necessary */
		if ( swap || force_swap ) 
			swapshort( uimg.i2_img, size*sizeof(short) );

	}

	/* Read STER1536 format */
	else if ( ipf == STER1536 ) {
		fseek( fp, 0, SEEK_SET );
		k = 155*1600+32;
		memset( (char *)uimg.i2_img, 0, 1600*1600*sizeof( short) );
		for ( j=0; j<1280; j++ ) {
			if ( ( i=fread( i2_rec, sizeof(short), ipf, fp ) ) != ipf) {
				printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i+j*1280,ipf*1280);
				goto ERROR;
			}
			/* Swap i2 array if necessary */
			if ( force_swap ) 
				swapshort( i2_rec, ipf*sizeof(short) );
			for ( n=0; n<ipf; n++ ) {
				uimg.i2_img[ j*1600 + k + n] = i2_rec[n];
			}
		}

	}

	/* Read STER3072 format */
	else if ( ipf == STER3072 ) {
		memset( (char *)uimg.i2_img, 0, 3000*3000*sizeof( short) );
		if ( in_nx == 3072 ) {
			/* Put into 3072*3072 array */
			fseek( fp, 0, SEEK_SET );
			k = 220*3000;
			for ( j=0; j<2560; j++ ) {
				if ( ( i=fread( i2_rec, sizeof(short), ipf, fp ) ) != ipf) {
					printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i+j*2560, ipf*2560);
					goto ERROR;
				}
				/* Swap i2 array if necessary */
				if ( force_swap ) 
					swapshort( i2_rec, ipf*sizeof(short) );
				for ( m=n=0; n<3036; n++, m++ ) {
					uimg.i2_img[ j*3000 + k + m] = i2_rec[n];
				}
			}
		}	
		else {
			/* Truncate to 3000x3000 pixels */
			if ( in_nx == 0 ) {
				fseek( fp, 2560*32*2, SEEK_SET );
				k = 220;
				for ( j=0; j<3000; j++ ) {
					if ( ( i=fread( i2_rec, sizeof(short), 2560, fp ) ) != 2560) {
						printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i+j*2560, ipf*2560);
						goto ERROR;
					}
					/* Swap i2 array if necessary */
					if ( force_swap ) 
						swapshort( i2_rec, 2560*sizeof(short) );
					for ( m=n=0; n<2560; n++, m++ ) {
						if ( i2_rec[n] >= 16383 ) i2_rec[n] = 0;
						uimg.i2_img[ j*3000 + k + m] = i2_rec[n];
					}
				}
			}
			/* Put into 2560*2560 array */
			else {
				fseek( fp, (220+36)*2560, SEEK_SET );
				k = 0;
				for ( j=0; j<2560; j++ ) {
					if ( ( i=fread( i2_rec, sizeof(short), 2560, fp ) ) != 2560) {
						printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i+j*2560, ipf*2560);
						goto ERROR;
					}
					/* Swap i2 array if necessary */
					if ( force_swap ) 
						swapshort( i2_rec, 2560*sizeof(short) );
					for ( m=n=0; n<2560; n++, m++ ) {
						uimg.i2_img[ j*2560 + k + m] = i2_rec[n];
					}
				}
			}
		}

	}

	/* Read RAW32 format */
	else if ( ipf == RAW32 ) {
		fseek( fp, jump, SEEK_SET );
		if ( use345 ) {
			memset( (char *)uimg.i4_img, 0, x*x*sizeof( int ) );
		}
		if ( in_nx+in_ny ) {
			col = in_nx;
			row = in_ny;
	  		xoff = ((x-in_ny)/2)*x+(x-in_nx)/2;
		}
		else {
			row = col = x;
	  		xoff = 0;
		}

		/* Read 32-bit array */
		if ( ( i=fread( uimg.i4_img, sizeof(int), size, fp ) ) != size) {
			printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, i, size);
			goto ERROR;
		}

		/* Truncate 32-bits to 16-bits, unless for CIF */
		if ( opf == IMG_CIF || opf == IMG_CBF ) {
			/* Swap i4 array if necessary */
			if ( swap || force_swap ) 
				swaplong( uimg.i4_img, size*sizeof(int) );
		}
		else {
			for ( i=0; i<size; i++ ) {
				j = uimg.i4_img[i];
				if ( j > 65535 )
					uimg.i2_img[i] = 65535;
				else 
					uimg.i2_img[i] = (unsigned short)(j);
			}
			/* Swap i2 array if necessary */
			if ( swap || force_swap ) 
				swapshort( uimg.i2_img, size*sizeof(short) );
		}


	}

	/* Read CCD format: cut to cutccd */
	else if ( ipf == CCD && cutccd) {

		fseek( fp, pos + (original_size - x)*original_size, SEEK_SET );

		/* Read core */
		for ( j = 0; j<cutccd; j++ ) {
			fseek( fp, cutdiff, SEEK_CUR );
			if ( ( i=fread( uimg.i2_img+j*cutccd, sizeof(short), cutccd, fp ) ) != cutccd) {
				printf("%s: ERROR: (%s) Only %d out of %d pixels read!\n", prg, f, j*cutccd+i, cutccd*cutccd);
				goto ERROR;
			}
			fseek( fp, cutdiff, SEEK_CUR );
		}

		/* Swap i2 array if necessary */
		if ( swap || force_swap ) 
			swapshort( uimg.i2_img, size*sizeof(short) );
	}

	/* Read MAR345 & PCK format */
	else if ( ipf == PCK || ipf == MAR345 ) {
		/* Read core */
		if ( get_pck( fp, uimg.i2_img ) == 0 ) {
			printf("%s: ERROR: (%s) Cannot read pck core!\n",prg,f);
			goto ERROR;
		}
	}

	/* Read CBF/imgCIF format */
	else if ( ipf == IMG_CIF || ipf == IMG_CBF ) {
		if ( GetCIFData( prg, f, fp, uimg.i4_img) < size ) {
			printf("%s: ERROR: (%s) Cannot read CIF data!\n",prg,f);
			goto ERROR;
		}

		/* CBF/imgCIF data are a 32 bit array. Truncate it to 16-bit */
		if ( opf != RAW32 ) {
			Truncate_i4_img(x,size,op);
			/* Swap i2 array if necessary */
			if ( swap || force_swap ) 
				swapshort( uimg.i2_img, size*sizeof(short) );
		}

		norip = 0;
	}

	/* Position file */
	if ( opf == MAR345 ) 
		lseek( fd, 4096 + norop*hleno, SEEK_SET );
	else if ( opf == PCK )
		lseek( fd, 2*x  + norop*hleno, SEEK_SET );
	else if ( opf == IMAGE )
		lseek( fd, 2*x               , SEEK_SET );
	else if ( opf == RAW8 || opf == RAW16 || opf == RAW32 )
		lseek( fd, jump, SEEK_SET );

	/* Rotate images by 90 deg. when converting between mar345 and mar300 */
	if ( (ipf == MAR345 || ipf == CCD || ipf == IMG_CIF || ipf == IMG_CBF ) && ( opf == PCK || opf == IMAGE ) ) 
		rotate_clock( uimg.i2_img, x);
	else if ( (ipf == IMAGE || ipf == PCK ) && opf != IMAGE && opf != PCK ) 
		rotate_anticlock( uimg.i2_img, x);

	/* Rotate image if requested */
	if ( rotate == 90 ) {
		rotate_anticlock( uimg.i2_img, x);
	}
	else if ( rotate == (-90) || rotate == 270 ) {
		rotate_clock( uimg.i2_img, x);
	}

	/* Flip image if requested */
	if ( flip ) flip_image( uimg.i2_img, x );

	/* Add a value to all data */
	if ( addvalue ) {
		add_image( uimg.i2_img, x );
	}

	/* Apply scale factor to all data */
	if ( scale_factor > 0.f ) {
		scale_image( uimg.i2_img, x );
	}
	
	/* Expand 16 bit to 32-bits for RAW32 and CBF/imgCIF output */
	if ( ( ipf != IMG_CIF && ipf != IMG_CBF && ipf != RAW32 ) && 
	     ( opf == IMG_CIF || opf == IMG_CBF || opf == RAW32 ) ) {
			for ( i=size-1; i>=0; i-- ) {
				j = (int)uimg.i2_img[i];
				uimg.i4_img[i] = j;
			}
	}

	/* Write IMAGE & RAW16 format: 16 bit array  */
	if ( opf == IMAGE || opf == RAW16 ) {
		if ( ( i=write( fd, uimg.i2_img, 2*size ) ) != 2*size ) {
			printf("%s: ERROR: (%s) Only %d out of %d pixels written!\n", prg,op, i/2, size);
			goto ERROR;
		}
	}

	/* Write RAW8 format: 8 bit array  */
	else if ( opf == RAW8 ) {
		i2 = uimg.i2_img;
		for ( i=0; i<x; i++ ) {
			i1 = i1_buf;
			for ( j=0; j<x; j++, i2++, i1++ ) {
				/* Divide 16-bit array by 256 */
				*i1 = (unsigned char)( *i2/256);
			}

			if ( ( j=write( fd, i1_buf, x ) ) != x ) {
				printf("%s: ERROR: (%s) Only %d out of %d pixels written!\n", prg,op, i*x+j, size);
				goto ERROR;
			}
		}
	}

	/* Write MAR345 & PCK300 format: 16 bit array */
	else if ( opf == MAR345 || opf == PCK ) {
		/* Write core */
		if ( put_pck(uimg.i2_img,x,x,fd ) < 1 ) {
			printf("%s: ERROR: (%s) Cannot write pck core!\n", prg, op);
			goto ERROR;
		}
	}

	/* Write ASCII format: 16 bit array */
	else if ( opf == ASCII ) {
		i = 0;
		i2 = uimg.i2_img;
		while ( i<size ) {
			str[0] = '\0';
			for ( j=0; j<12; j++, i++, i2++ ) {
				if ( i == size ) break;
				sprintf( buf, "%6d", *i2);
				strcat( str, buf);
			}
			strcat( str, "\n");
			write( fd, str, strlen(str) );
		}
		goto CLOSE;
	}

	/* Write TIFF format: 16 bit array */
	else if ( opf == TIFF ) {
		if ( WriteTiff( fd, x, uimg.i2_img) == 0 )
			goto ERROR;
		goto CLOSE;
	}

	/* In case of CBF/imgCIF and mar output we need to update header */
	if ( (ipf == IMG_CIF || ipf == IMG_CBF) && nov ) { 
		lseek( fd, 0, SEEK_SET );
		if ( opf == MAR345 ) 
			i = Putmar345Header( fd, h345 );
		else if ( opf == PCK || opf == IMAGE )
			i = Putmar300Header( fd, 1, h300 );
	}

	/* In case of scaling and adding we need to update header */
#ifdef WORK_IN_PROGRESS
	if ( addvalue || scale_factor ) {
		ptr1 = data+size;
		for ( nov=i=0;i<size;i++,ptr1++) 
			if ( *ptr1 ) nov++;
		h345.high = h300.high_pixels	= nov;
		lseek( fd, 0, SEEK_SET );
		if ( opf == MAR345 )  {
			norop = (int)(nov/8.0 + 0.875);	/* No. of overflow records */
			i = Putmar345Header( fd, h345 );
		}
		else if ( opf == PCK || opf == IMAGE ) {
			h300.high_records = norop = (nov+x/4-1) / (x/4);
			i = Putmar300Header( fd, 1, h300 );
		}
	}
#endif

	if ( norip <= 0 ) goto CLOSE;

HIGH:
	/* Position input file for overflow records */
	if ( ipf == MAR345 )
		fseek( fp, 4096, SEEK_SET );
	else if ( ipf == PCK )
		fseek( fp, 2*x         , SEEK_SET );
	else if ( ipf == IMAGE )
		fseek( fp, 2*x + 2*size, SEEK_SET );

	/* Position output file */
	if ( opf == MAR345 )
		lseek( fd, 4096, SEEK_SET );
	else if ( opf == PCK )
		lseek( fd, 2*x         , SEEK_SET );
	else if ( opf == IMAGE )
		lseek( fd, 2*x + 2*size, SEEK_SET );

	/* Read and write high intensities, pairwise */
	j = 0;
	while ( j<nov ) {
		if ( ( i=fread( pair, sizeof(int), 2, fp ) ) != 2 ) {
			printf("%s: ERROR: (%s) Cannot read high intensity pixel %d!\n",prg, f,j+1);
			goto ERROR;
		}

		/* Swap i2 array if necessary */
		if ( swap || force_swap ) 
			swaplong( pair, 2*sizeof(int) );

		/* Modify address */
		if ( ( ipf == PCK || ipf == IMAGE ) && opf != IMAGE && opf != PCK  ) {
			row = pair[0] / x;
			col = pair[0] % x ;
			i = (x-col)*x + row;
		}
		else if ( ipf == MAR345 && ( opf == PCK || opf == IMAGE ) ) {
			row = pair[0] / x;
			col = pair[0] % x - 1;
			i   = (col+1)*x + x - row;
		}
		else {
			row = pair[0] / x;
			col = pair[0] % x;
			i = pair[0];
		}
		
/*
printf("xxxx fp=%8d fd=%8d pair= %10d %10d (%d,%d)\n",ftell(fp),ltell(fd),pair[0],i,row,col);
*/
		pair[0] = i;
		
		if ( opf == ASCII ) {
			sprintf( buf, "high_intensity\taddress %10d\t value %10d\n", pair[0],pair[1]);
			write( fd, buf, strlen(buf) );
		}
		else if ( opf == PCK || opf == IMAGE || opf == MAR345 ) {
		    if ( ( i=write( fd, pair, 2*sizeof(int)) ) != 2*sizeof(int) ) {
			printf("%s: ERROR: (%s) Only %d out of %d high intensity bytes written!\n", prg, op, j, nov);
			goto ERROR;
		    }
		}
		/* 32-bit output: update array value */
		else if ( opf == RAW32 || opf == IMG_CIF || opf == IMG_CBF ) {
			uimg.i4_img[i] = pair[1];
			
#ifdef MARCVT_V_3
		    lseek( fd, 4*pair[0], SEEK_SET);
		    if ( ( i=write( fd, &pair[1], sizeof(int)) ) != sizeof(int) ) {
			printf("%s: ERROR: (%s) Only %d out of %d high intensity bytes written!\n", prg, op, j, nov);
			goto ERROR;
		    }
#endif
		}

		/* Increase counter */
		j++;
	}

	/* In MAR300 output mode, fill rest of overflow record with 0 */
	if ( opf == IMAGE ) {
		memset( i4_buf, 0, hleno*sizeof( char ) );
		j = (nov*8) % hleno;
		if ( j > 0 ) {
			j = hleno - j;
			i=write( fd, i4_buf, j);
		}
	}

	if ( opf == ASCII ) goto CORE;

	/* Close files */
CLOSE:
	/* Write RAW32 format: 32 bit array */
	if ( opf == RAW32 ) {
		if ( ( j=write( fd, uimg.i4_img, 4*size ) ) != (4*size) ) {
			printf("%s: ERROR: (%s) Only %d out of %d pixels written!\n", prg, op, j/4, size);
			goto ERROR;
		}
	}

	/* Write CBF/imgCIF format: 32 bit array */
	else if ( opf == IMG_CIF || opf == IMG_CBF ) {
		if ( PutCIFData( prg, op, (char)(opf-IMG_CIF), uimg.i4_img) == 0 ) {
			printf("%s: ERROR: (%s) Cannot write CIF data!\n",prg,op);
			goto ERROR;
		}
	}

	if ( fp != NULL )
		fclose( fp );
	if ( fd >= 0 ) close( fd );

	/* Remove input file , unless ... */
	if ( !keep_input )
		remove( f );

	return 1;

	/*
	 * Go here in case of I/O errors
	 */
ERROR:	
	if ( fp != NULL )
		fclose( fp );
	if ( fd >= 0 ) {
		close(  fd );
		remove( op );
	}
	return 0;
}

/******************************************************************
 * Function: MakeHeader
 ******************************************************************/
static void MakeHeader( int mode )
{
    if ( mode == 1 ) {
        h345.byteorder          = 1234;                       
        h345.high               = nov;                    
        h345.scanner            = 1;
        h345.size               = h300.pixels_x;             
        h345.format             = 1;     
        h345.mode               = 1;            
        h345.high               = nov;                    
        h345.pixels             = size;                  
	if ( h300.p_x < 0.01 || h300.p_x > 200 )
		h300.p_x = h300.p_y = 0.15;
        h345.pixel_length       = h300.p_x;
        h345.pixel_height       = h300.p_y;
        h345.xcen               = h300.centre_x;
        h345.ycen               = h300.centre_y;
        h345.gain               = 1.0;
        h345.time               = h300.exptime_sec;           

        h345.wave               = h300.lambda;             
                                                              
        h345.dist               = h300.distance;                  
        h345.resol              = GetResol( (float)(h300.pixels_x*h300.p_x/2.0), (float)h345.dist, (float)h345.wave );
        h345.phibeg             = h300.phi_start;
        h345.phiend             = h300.phi_end;                
        h345.omebeg             = 0.0;
        h345.omeend             = 0.0;
        h345.theta              = 0.0;
        h345.chi                = 0.0;                   
        h345.phiosc             = 1;                
        h345.omeosc             = 0;                

        h345.valavg             = 10;                       
        h345.valmin             = 0;                     
        h345.valmax             = 1;                       
        h345.valsig             = 0.0;                       
                                                              
        h345.histbeg            = 0;                 
        h345.histend            = 200;                   
        h345.histmax            = 1000;

        strcpy( h345.version, VERSION );                      
        strcpy( h345.program, prg);                     
   }

   else if ( mode == 0 ) {
	h300.pixels_x 		= h345.size;
	h300.pixels_y 		= h345.size;
	h300.lrecl    		= 2*h345.size;
	h300.max_rec  		= h345.size;
	h300.high_pixels	= nov;
	h300.high_records	= norop;
	h300.counts_start	= (int)h345.dosebeg;
	h300.counts_end		= (int)h345.doseend;
	h300.exptime_sec	= (int)h345.time;
	h300.exptime_units	= (int)h345.doseavg;
	h300.prog_time		= h345.time;
	h300.r_max		= (float)h345.size/2.;
	h300.r_min		= (float)h345.size/2.;
	h300.p_r		= h345.pixel_height/1000.;
	h300.p_l		= h345.pixel_length/1000.;
	h300.p_x		= h345.pixel_height/1000.;
	h300.p_y		= h345.pixel_length/1000.;
	h300.centre_x		= h345.xcen;
	h300.centre_y		= h345.ycen;
	h300.lambda		= h345.wave;
	h300.distance		= h345.dist;
	h300.phi_start		= h345.phibeg;
	h300.phi_end		= h345.phiend;
	h300.omega		= h345.omebeg;
	h300.multiplier		= h345.multiplier;
	
	strcpy( h300.date, h345.date );
   }

}

/******************************************************************
 * Function: rotate_anticlock = rotates image anticlockwise
 *  			0 1 2       2 5 8
 *  			3 4 5  -->  1 4 7
 *  			6 7 8       0 3 6
 ******************************************************************/
static void
rotate_anticlock(register unsigned short *data, int nx)
{
register unsigned short *ptr1, *ptr2, *ptr3, *ptr4, temp;
register int 		i, j;
int			nx2 = (nx+1)/2;

	for ( i=nx/2; i--; ) {

		/* Set pointer addresses */

		j    = nx2 - 1;
		ptr1 = data + nx*i        + j;		/* 1. Quadrant */
		ptr2 = data + nx*j        + nx-1-i;	/* 2. Quadrant */
		ptr3 = data + nx*(nx-1-i) + nx-1-j;	/* 4. Quadrant */
		ptr4 = data + nx*(nx-1-j) + i;		/* 3. Quadrant */

		for ( j = nx2; j--; ) {

			/* Restack: anticlockwise rotation by 90.0 */
			temp  = *ptr1;
			*ptr1 = *ptr2;
			*ptr2 = *ptr3;
			*ptr3 = *ptr4;
			*ptr4 = temp;

			/* Increase pointer */
			 ptr1 --;
			 ptr2 -= nx;
			 ptr3 ++;
			 ptr4 += nx;
		}
	}
}

/******************************************************************
 * Function: rotate)clock = rotates image clockwise
 *  			0 1 2       2 5 8
 *  			3 4 5  -->  1 4 7
 *  			6 7 8       0 3 6
 ******************************************************************/
static void
rotate_clock(register unsigned short *data, int nx)
{
register unsigned short *ptr1, *ptr2, *ptr3, *ptr4, temp;
register int 		i, j;
int			nx2 = (nx+1)/2;

	for ( i=nx/2; i--; ) {

		/* Set pointer addresses */
		j    = nx2 - 1;
		ptr1 = data + nx*i        + j;		/* 1. Quadrant */
		ptr2 = data + nx*j        + nx-1-i;	/* 2. Quadrant */
		ptr3 = data + nx*(nx-1-i) + nx-1-j;	/* 4. Quadrant */
		ptr4 = data + nx*(nx-1-j) + i;		/* 3. Quadrant */

		for ( j = nx2; j--; ) {

			/* Restack: clockwise rotation  by 90.0 */
			 temp = *ptr4;
			*ptr4 = *ptr3;
			*ptr3 = *ptr2;
			*ptr2 = *ptr1;
			*ptr1 =  temp;
			 
			/* Increase pointer */
			 ptr1 --;
			 ptr2 -= nx;
			 ptr3 ++;
			 ptr4 += nx;
		}
	}
}

/******************************************************************
 * Function: AsciiHeader
 ******************************************************************/
static void AsciiHeader( void )
{
	sprintf( str,"header\t%s version         :\t%s\n",prg,VERSION);
	write( fd, str, strlen(str) );
	sprintf( str,"header\t# pixels               :\t%d x %d\n",h345.size,h345.size);
	write( fd, str, strlen(str) );
	sprintf( str,"header\t# high intensity pixels:\t%d\n",h345.high);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tpixelsize              :\t%1.3f\n",h345.pixel_height);
	write( fd, str, strlen(str) );
	sprintf( str,"header\texposure time          :\t%1.1f\n",h345.time);
	write( fd, str, strlen(str) );
	sprintf( str,"header\twavelength             :\t%1.5f\n",h345.wave);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tdistance               :\t%1.3f\n",h345.dist);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tresolution at edge     :\t%1.5f\n",h345.resol);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tphi start              :\t%1.3f\n",h345.phibeg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tphi end                :\t%1.3f\n",h345.phiend);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tphi oscillations       :\t%d\n",h345.phiosc);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts start           :\t%1.2f\n",h345.dosebeg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts end             :\t%1.2f\n",h345.doseend);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts minimum         :\t%1.2f\n",h345.dosemin);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts maximum         :\t%1.2f\n",h345.dosemax);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts average         :\t%1.2f\n",h345.doseavg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts sigma           :\t%1.2f\n",h345.dosesig);
	write( fd, str, strlen(str) );
	sprintf( str,"header\tcounts readings        :\t%d\n",h345.dosen);
	write( fd, str, strlen(str) );
	sprintf( str,"header\timage minimum          :\t%d\n",h345.valmin);
	write( fd, str, strlen(str) );
	sprintf( str,"header\timage maximum          :\t%d\n",h345.valmax);
	write( fd, str, strlen(str) );
	sprintf( str,"header\timage average          :\t%1.1f\n",h345.valavg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\timage average          :\t%1.1f\n",h345.valavg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\timage sigma            :\t%1.1f\n",h345.valsig);
	write( fd, str, strlen(str) );
	sprintf( str,"header\thistogram start        :\t%d\n",h345.histbeg);
	write( fd, str, strlen(str) );
	sprintf( str,"header\thistogram end          :\t%d\n",h345.histend);
	write( fd, str, strlen(str) );
	sprintf( str,"header\thistogram maximum      :\t%d\n",h345.histmax);
	write( fd, str, strlen(str) );
}

/******************************************************************
 * Function: TiffHeader
 ******************************************************************/
static void  TiffHeader( void )
{
}

/******************************************************************
 * Function: Truncate_i4_img
 ******************************************************************/
static void  Truncate_i4_img(int x, int size, char *op)
{
int	i,j,k;
int	pair[2];

	/* Position output file */
	if ( opf == MAR345 )
		lseek( fd, 4096, SEEK_SET );
	else if ( opf == PCK ) 
		lseek( fd, 2*x         , SEEK_SET );
	else if ( opf == IMAGE )
		lseek( fd, 2*x + 2*size, SEEK_SET );

	nov = 0;

	/* Go through 32-bit array, count values > 16 bit
	 * and write overflows */
	for ( i=0; i<size; i++ ) {
	    j = uimg.i4_img[i];
	    if ( j <= 65535 )  {
		uimg.i2_img[i] = (unsigned short)(j);
	    }
	    else  {
		uimg.i2_img[i] = 65535;
			nov++;
		if ( opf != MAR345 && opf != PCK && opf != IMAGE ) continue;

		/* Write high intensities, pairwise */

		pair[0] = i;	/* Address */
		pair[1] = j;	/* Intensity value */

		if ( opf == PCK || opf == IMAGE )
			pair[0]   = (i%x)*x + x - i/x;

		if ( verbose > 1 ) {
			printf("%s: Val=%8d @ %-10d at position %d\n",prg,j,i,ltell( fd ));
		}

		if ( ( j=write( fd, pair, 2*sizeof(int) ) ) != 2*sizeof(int) ) {
			printf("%s: ERROR: (%s) Cannot write high intensity pixel %d!\n",prg, op,nov);
		}
	    }
	}

	if ( nov < 1 ) return;

	h345.high 		= nov;
	h300.high_pixels	= nov;
	h300.high_pixels	= nov;
	h300.high_records	= (nov+x/4-1) / (x/4);
	norop 			= (int)(nov/8.0 + 0.875);
	if ( opf != MAR345 )	norop = h300.high_records;
}

/******************************************************************
 * Function: flip_image
 ******************************************************************/
static void flip_image(unsigned short *data, int nx)
{
unsigned short *ptr1, *ptr2, i1;
int 		i, j;

	for ( j=0;j<nx;j++) {
		ptr1 = data + j*nx;
		ptr2 = data + (j+1)*nx - 1;
		for ( i=0;i<nx/2;i++,ptr1++,ptr2--) {
			i1 	= *ptr1;
			*ptr1	= *ptr2;
			*ptr2	= i1;
		}
	}	
}

/******************************************************************
 * Function: add_image
 ******************************************************************/
static void add_image(unsigned short *data, int nx)
{
unsigned short *ptr1, *ptr2, i1;
register int	v, i, j;

	i = nx*nx;
	ptr1 = data;
	ptr2 = data + i;
	memset( (char *)ptr2, 0, sizeof(short)*nx );

	/* Store values > 16 bit in second part of i2_img */
	for ( j=0;j<i;j++, ptr1++, *ptr2++) {
		v = (*ptr1) + addvalue;
		if ( v < 0 ) v = 0;
		*ptr1 = (unsigned short)v;
		while ( v > 65535 ) {
			v -= 65535;
			(*ptr2) = (*ptr2)+1;
			*ptr1 = 65535;
		}
	}	
}

/******************************************************************
 * Function: scale_image
 ******************************************************************/
static void scale_image(unsigned short *data, int nx)
{
unsigned short *ptr1, *ptr2, i1;
register int	v, i, j;

	i = nx*nx;
	ptr1 = data;
	ptr2 = data + i;
	memset( (char *)ptr2, 0, sizeof(short)*nx );

	/* Store values > 16 bit in second part of i2_img */
	for ( j=0;j<i;j++, ptr1++, *ptr2++) {
		v = (*ptr1)*scale_factor;
		*ptr1 = (unsigned short)v;
		while ( v > 65535 ) {
			v -= 65535;
			(*ptr2) = (*ptr2)+1;
			*ptr1 = 65535;
		}
			
	}	
}
