/***********************************************************************
 * 
 * marcvt: savetiff
 *
 * Copyright by:	Dr. Claudio Klein
 *			Dr. Michael Blum
 * 			X-ray Research GmbH, Hamburg
 *
 * Version:     3.0
 * Date:        30/04/1999
 *
 ***********************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "tiff.h"
#include "marcvt.h"
#define  BIT8

/*
 * Local variables
 */

typedef struct _tiff_frame_header {
        char tiff_part[1024];
} tiff_frame_header;


/*
 * Local functions
 */

int				WriteTiff		(int, int, unsigned short *);
static void 			fill_tiff_dir_entry	();
static tiff_frame_header 	*Create_Tiff_Header	(int);
static int 			HistoMinMax		(int, unsigned short *);

/*
 * External functions
 */

/******************************************************************
 * Function: SaveTiff = prepares everything for Tiff output
 ******************************************************************/
int WriteTiff(int fd,int nx, unsigned short *i2_buf )
{
register int 		i,k,m,n;
register unsigned short	j, *i2;
int			min,max;
int			nxy;
unsigned char		*c,cbuf[1024];
double			logmin,factor;
unsigned short		cred  [256];
unsigned short		cblue [256];
unsigned short		cgreen[256];
tiff_frame_header	*th;

	if ( ncolors > 256 ) ncolors = 256;
	max = histomax;
	min = histomin;
	nxy = nx*nx;

	max = HistoMinMax( nx, i2_buf);

	if ( min < 0 ) min = 0;
	if ( max < 1 ) max = 250;

	if ( (max-min ) < ncolors ) ncolors = max-min;

	factor = ncolors/(float)(max-min);

	th = Create_Tiff_Header(nx);

   	/* Output header */
   	i = write( fd, th, sizeof(tiff_frame_header));

	if ( i == 0 ) {
		printf("MARCVT: Cannot write Tiff header\n");
		return( 0 );
	}

	/* Min. color: always white */
        cred[0] = cgreen[0] = cblue[0] = 65535;

        /* All colors black, white (BLUE) or red (RAINBOW)  ...*/
        for(i=ncolors-1;i<256;i++) {
        	cred[i] = cgreen[i] = cblue[i] = 0;
		if ( color_scheme == BLUE )
			cred[i] = cgreen[i] = cblue[i] = 65535;
		else if ( color_scheme == RAINBOW )
			cred[i]	  = 65535;
	}

	if ( color_scheme == GREY ) {
		for(i=1;i<ncolors;i++)
			cred[i] = cgreen[i] = cblue[i] = 65535 - 65535*i/ncolors;
	}
	else if ( color_scheme == BLUE ) {
		for(i=1;i<ncolors;i++) {
			/* Black ==> Blue */
			if (i<ncolors*1/3) {
				cred[i]   = 0; 
				cgreen[i] = 0;
				cblue[i]  = i*65535/(ncolors/3);
			}
			/* Blue ==> Cyan */
			else if (i>=ncolors*1/3 && i<ncolors*2/3 ) {
				cred[i]   = 0; 
				cgreen[i] = (i-ncolors/3)*65535/(ncolors/3);
				cblue[i]  = 65535;
			}
			else if (i>=ncolors*2/3 ) {
				cred[i]   = (i-ncolors*2/3)*65535/(ncolors/3); 
				cgreen[i] = 65535; 
				cblue[i]  = 65535;
			}
		}
	}
	else if ( color_scheme == RAINBOW ) {
		for(i=1;i<ncolors;i++) {
			/* White ==> Yellow */
			if (i<ncolors*1/5) {
				cred[i]   = 65535; 
				cgreen[i] = 65535;
				cblue[i]  = 65535 - i*65535/(ncolors*1/5);
			}
			/* Yellow ==> Green */
			else if (i>=ncolors/5 && i<ncolors*3/10) {
				cred[i]   = 65535- (i-ncolors/5)*65535/(ncolors/10); 
				cgreen[i] = 65535;
				cblue[i]  = 0;
			}
			/* Green  ==> Cyan  */
			else if (i>=ncolors*3/10 && i<ncolors*2/5) {
				cred[i]   = 0; 
				cgreen[i] = 65535;
				cblue[i]  = (i-ncolors*3/10)*65535/(ncolors/10);
			}
			/* Cyan   ==> Blue  */
			else if (i>=ncolors*2/5 && i<ncolors*3/5) {
				cred[i]   = 0; 
				cgreen[i] = 65535- (i-ncolors*2/5)*65535/(ncolors/5);
				cblue[i]  = 65535;
			}
			/* Blue   ==> Navy */
			else if (i>=ncolors*6/10 && i<ncolors*7/10) {
				cred[i]   = 0; 
				cgreen[i] = 0;
				cblue[i]  = 65535 - (i-ncolors*3/5)*(32767)/(ncolors/9 );
			}
			/* Navy ==> Purple */
			else if (i>=ncolors*7/10 && i<ncolors*9/10) {
				cred[i]   = (i-ncolors*7/10)*32767/(ncolors/5);
				cgreen[i] = 0;
				cblue[i]  = 32767;
			}
			/* Purple ==> Red   */
			else if (i>=ncolors*9/10) {
				cred[i]   = 65535/2-(i-ncolors*9/10)*32767/(ncolors/10); 
				cgreen[i] = 0;
				cblue[i]  = 65535/2+(i-ncolors*9/10)*32767/(ncolors/10);
			}
		}
		cred	[ncolors-1]   	= 65535;
		cgreen	[ncolors-1] 	= 0;
		cblue   [ncolors-1] 	= 0;
	}

	/* Position file at byte no. 1024 */
	i = lseek( fd, 1024, SEEK_SET );

	/* Write rgb */
   	i = write( fd, cred,   sizeof(short) * 256);
   	i = write( fd, cgreen, sizeof(short) * 256);
   	i = write( fd, cblue,  sizeof(short) * 256);

	/* Position file at byte no. 3072 */
	i = lseek( fd, 3*1024, SEEK_SET );

	i2 = i2_buf;
#ifdef BIT16
	i=write(fd, i2_buf, 2*nxy);
	if ( i!= 2*nxy ) {
		printf("MARCVT: Only %d out of %d pixels written in TIFF core\n",i/2,nxy);
		return( 0 );
	}
#else
	c  = cbuf;
	if ( log_scale ) {
		logmin = log( (double) min );
		factor = max / ( log( (double) max ) - logmin );
printf("xxx factor=%f %f\n",factor,logmin);

		for ( k=0,i=0; i<nxy; i++ ) {
			j = *i2++;
			if ( j > max ) 
				*c = (unsigned char)(ncolors-1);
			else if ( j < min ) 
				*c = 0;
			else 
				*c = (unsigned char)( (log(j) - logmin)*factor);

			k++;
			c++;
			if ( k == 1000 ) {
				write(fd, cbuf, k);
				k = 0;
				c = cbuf;
			}
		}
	}

	else {
		for ( k=0,i=0; i<nxy; i++ ) {
			j = *i2++;
			if ( j > max ) 
				*c = 255;
			else if ( j < min ) 
				*c = 0;
			else 
				*c = (unsigned char)( (j-min)*factor );

			k++;
			c++;
			if ( k == 1000 ) {
				write(fd, cbuf, k);
				k = 0;
				c = cbuf;
			}
		}
	}

	if ( k )
		write(fd, cbuf, k);
#endif

	return 1;
}

/******************************************************************
 * Function: Create_Tiff_Header
 ******************************************************************/
static tiff_frame_header
*Create_Tiff_Header(int nx)
{
TIFFHeader 		tiff_header;
TIFFDirEntry 		tiff_dir_entry;
unsigned short 		n_dir_entries;
unsigned int  		next_diroff;
int 			entries;
int 			i;
int 			orientation;
int 			origin;
int 			tiff_ifd_length;
extern float 		pixelsize;
unsigned char 		*cp;
unsigned char 		*next_long_value;
unsigned int  		numerator, denominator;
tiff_frame_header 	*tiff_frame_headerp;


   /* No. of header entries: this is fixed */
   entries =  13;

   tiff_ifd_length = sizeof(short) + entries*sizeof(TIFFDirEntry) + sizeof(int);

   if ((tiff_frame_headerp = (tiff_frame_header *)malloc(sizeof(tiff_frame_header))) == NULL) {
      printf("MARCVT: Failed to allocate  memory for TIFF header.\n");
      return(0);
   }
   memset(tiff_frame_headerp, 0, sizeof(tiff_frame_header));

   /* Start filling the header */
   cp = (unsigned char *)tiff_frame_headerp; 
   next_long_value =  (unsigned char *)tiff_frame_headerp;
   next_long_value += sizeof(TIFFHeader) + tiff_ifd_length;

#if ( __sgi || _hpux__ )
      tiff_header.tiff_magic =  TIFF_BIGENDIAN;
#else
      tiff_header.tiff_magic =  TIFF_LITTLEENDIAN;
#endif

   tiff_header.tiff_version 	= TIFF_VERSION;
   tiff_header.tiff_diroff 	= sizeof(TIFFHeader);

   memcpy(cp, &tiff_header, sizeof(TIFFHeader));
   cp += sizeof(TIFFHeader);

   n_dir_entries = entries;
   memcpy(cp, &n_dir_entries, sizeof(n_dir_entries));
   cp += sizeof(n_dir_entries);

   /* Entry no. 1: width of image (x)  */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_IMAGEWIDTH;
   tiff_dir_entry.tdir_type 	= TIFF_LONG;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= nx;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 2: height of image (y)  */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_IMAGELENGTH;
   tiff_dir_entry.tdir_type 	= TIFF_LONG;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= nx;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 3: bits per sample */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_BITSPERSAMPLE;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= 8 ;		/*significant bits */
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 4: compression  */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_COMPRESSION;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= COMPRESSION_NONE;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 5: color scheme  */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_PHOTOMETRIC;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= PHOTOMETRIC_PALETTE;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 6: where do image data start? */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_STRIPOFFSETS;
   tiff_dir_entry.tdir_type 	= TIFF_LONG;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= 3*sizeof(tiff_frame_header);
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 7: orientation */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_ORIENTATION;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= ORIENTATION_TOPLEFT;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 8: no. of pixels per row */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_ROWSPERSTRIP;
   tiff_dir_entry.tdir_type 	= TIFF_LONG;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= nx;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 9: no. of pixels */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_STRIPBYTECOUNTS;
   tiff_dir_entry.tdir_type 	= TIFF_LONG;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= nx * nx;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 10: pixelsize x */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_XRESOLUTION;
   tiff_dir_entry.tdir_type 	= TIFF_RATIONAL;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= (unsigned int)next_long_value - (unsigned int)tiff_frame_headerp;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   /* Pixelsize: default - 150 microns */
   numerator 	= 10000;
   denominator 	= pixelsize*1000;
   memcpy(next_long_value, &numerator, sizeof(int));
   next_long_value += sizeof(int);
   memcpy(next_long_value, &denominator, sizeof(int));
   next_long_value += sizeof(int);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 11: pixelsize y */
   tiff_dir_entry.tdir_tag = TIFFTAG_YRESOLUTION;
   tiff_dir_entry.tdir_type = TIFF_RATIONAL;
   tiff_dir_entry.tdir_count = 1;
   tiff_dir_entry.tdir_offset = (unsigned int)next_long_value - (unsigned int)tiff_frame_headerp;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   /* in pixels/cm */
   numerator = 10000;
   denominator = pixelsize*1000;
  
   memcpy(next_long_value, &numerator, sizeof(int));
   next_long_value += sizeof(int);
   memcpy(next_long_value, &denominator, sizeof(int));
   next_long_value += sizeof(int);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 12: units of resolution */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_RESOLUTIONUNIT;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 1;
   tiff_dir_entry.tdir_offset 	= RESUNIT_CENTIMETER;
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);

   /* Entry no. 13: color map  */
   tiff_dir_entry.tdir_tag 	= TIFFTAG_COLORMAP;
   tiff_dir_entry.tdir_type 	= TIFF_SHORT;
   tiff_dir_entry.tdir_count 	= 3*256;
   tiff_dir_entry.tdir_offset 	= sizeof(tiff_frame_header); /* Start of cmap */
   fill_tiff_dir_entry(cp, &tiff_dir_entry);

   cp += sizeof(tiff_dir_entry);


   /* This signals the end of the TIFF directories */
   next_diroff = 0;
   memcpy(cp, &next_diroff, sizeof(next_diroff));

   cp += sizeof(next_diroff);


   return (tiff_frame_headerp);
}

/******************************************************************
 * Function: fill_tiff_dir_entry     
 ******************************************************************/
static void fill_tiff_dir_entry(void *destination, TIFFDirEntry *source)
{

/* when offset (value) contains a value of less then 4 byte length, it must
   be stored in the bytes at the beginning of offset, not at the end
*/
   if(source->tdir_count == 1) {
#if ( __sgi || _hpux__ )
	      switch (source->tdir_type) {
		 case TIFF_SHORT:
		 case TIFF_SSHORT:
		    source->tdir_offset = (source->tdir_offset << 16);
		    break;
		 case TIFF_BYTE:
		 case TIFF_SBYTE:
		 case TIFF_UNDEFINED:
		    source->tdir_offset = (source->tdir_offset << 24);
		    break;
		 default:
		    break;
	      }
#endif
   }
   memcpy(destination, source, sizeof(TIFFDirEntry));
}
	
/******************************************************************
 * Function: HistoMinMax = generates an intensity histogram of the image
 *		    Range is from 0 ... max
 *		    n is the number of data points
 ******************************************************************/
static int HistoMinMax(int nx, unsigned short *i2_buf)
{
register unsigned short *i2i;
register int 		i, j,nI, index, d1,d2;
float 			dh, hsumr, hsuml;
int			imax;
int			nr,nxy,nx2;
int			Imax;
int			hist_begin,hist_end;
int			isum;
float			fsum;
float			ftmp;
unsigned int  		histogram[65600];

	memset( histogram, 0, 65600*sizeof(int) );

        /*
         * Make histogram and find maximum value in image...
         */
	nI   = 0;
	nxy  = nx*nx;
	nx2  = nx/2;
	nr   = (nx2-10)*(nx2-10);
	Imax = 0;

	i2i = i2_buf;

	for(i=0; i<nx; i++) {
	    d1 = i-nx2;
	    for(j=0; j<nx; j++, i2i++) {
		index = (int)*i2i;
		if ( index <= 0 || index > 65530 ) continue;
		d2 = j-nx2;
		if ( (d1*d1 + d2*d2) > nr ) continue;
		nI++;
		if ( index > Imax ) 
			Imax = index;
		histogram[index]++;
	    }
	}

	/* 
	 * Find maximum of histogram 
	 */
	hist_begin = 0;

	index = 0;
	imax  = 0;
	for (i=65530; i>0; i-- ) {
		if ( histogram[i] > index ) {
			index = histogram[i]; 
			imax = i;
		}
	}

	/* 
	 * Integrate whole histogram 
	 */
	ftmp = 0.001*Imax;

	for (i=imax; i>=0; i-- ) {
		if ( histogram[i] <  ftmp ) { 
			hist_begin = i;
			break;
		}
	}

	hsuml=0;
	for (i=0; i<65530; i++ ) {
		hsuml = hsuml + histogram[i];
	}

	/* From high intensity end go down stepwise, until right integral
	 * is 0.002 % from left  integral */

	hsumr 	   = 0;
	for (i=65530; i>0; i-- ) {
		hsumr = hsumr + histogram[i];
		hsuml = hsuml - histogram[i];
		if (hsuml>(float)0.0) dh = hsumr/hsuml;
		if (dh>0.002) break;
	}

	hist_end = i;

	if (hist_end < 200) hist_end = 200;

	if ( histomin > 0 )
		hist_begin = histomin;
	else
		histomin   = hist_begin;
	
	if ( histomax > 0 ) 
		hist_end   = histomax;
	else
		histomax   = hist_end;

	i = hist_end - hist_begin;
	if ( ncolors > i ) i = ncolors;
	if ( i > 256 ) i= 256;
	if ( ncolors < i ) i = ncolors;

	printf("        %d colors ranging from %d to %d\n",i,hist_begin,hist_end);

	return( hist_end );
}
