
/*********************************************************************
 *
 * MARCVT  - main program
 * 
 *********************************************************************
 *
 * Author:  	Claudio Klein, X-Ray Research GmbH.
 * Version:     4.2
 * Date:        30/11/2000
 *
 *********************************************************************/
 
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
#include <stdlib.h>

/*
 * mar software include files
 */
#define  MAIN
#include "version.h"
#include "marcvt.h"

/*
 * Definitions
 */

/*
 * External functions
 */
extern void		Work( void );

/*
 * Local functions
 */

void			usage(void);

/*
 *************************************************************************
 *
 * Main Program.
 *
 *************************************************************************
 */
int main(int argc, char **argv)
{
int 			i,j,k,iin=0;
int			cpu1, cpu2;
time_t			tick;
char			file[132];

	/* Set defaults */
	keep_input      = 1;
	recursive	= 0;
	pack		= 0;
	force		= 0;
	list		= 0;
	verbose		= 0;
	nfiles		= 0;
	cutccd		= 0;
	jump		= 0;
	in_nx		= 0;
	in_ny		= 0;
	opf		= -1;
	ipf		= -1;
	addvalue	= 0;
	scale_factor	= 0.0;
	histomax	= 0;
	histomin	= 0;
	log_scale	= 0;
	force_swap	= 0;
	color_scheme	= GREY;
	ncolors		= 256;
	use345		= 0;
	rotate		= 0;
	flip		= 0;
	
	strcpy( suffix, ".QWER_12398=qxayzwkdlpoqlllbv" );
	strcpy( files[0], "*" );
	strcpy( buf     , "\0" );
	strcpy( str     , "\0" );
	strcpy( path    , "\0" );
	strcpy( outdir  , "\0" );
	strcpy( opform  , "\0" );
	strcpy( ipform  , "\0" );
	strcpy( prg     , "marcvt" );

	/*
	 * Parse command line
	 */
	for (argv++; argc-- > 1; argv++ ) {

		/* Output format */
		if (!strcmp( *argv, "-out") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				strcpy( opform, *argv );
			}
		}

		else if (strstr( *argv, "-mar345") )
			strcpy ( opform, "mar345" );
		else if (strstr( *argv, "-pck") )
			strcpy ( opform, "pck" );
		else if (strstr( *argv, "-image") )
			strcpy ( opform, "image" );
		else if (strstr( *argv, "-raw32") )
			strcpy ( opform, "raw32" );
		else if (strstr( *argv, "-raw8") )
			strcpy ( opform, "raw8" );
		else if (strstr( *argv, "-raw") )
			strcpy ( opform, "raw16" );
		else if (strstr( *argv, "-tif") )
			strcpy ( opform, "tiff" );
		else if (strstr( *argv, "-ascii") )
			strcpy ( opform, "ascii" );
		else if (strstr( *argv, "-cif") )
			strcpy ( opform, "cif" );
		else if (strstr( *argv, "-cbf") )
			strcpy ( opform, "cbf" );

		/* Add */
		else if (!strcmp( *argv, "--add") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				addvalue = atoi( *argv );
			}
		}

		/* Scale factor */
		else if (!strcmp( *argv, "--scale") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				scale_factor = atof( *argv );
			}
		}

		/* Cutoff CCD */
		else if (!strcmp( *argv, "-cut") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				cutccd = atoi( *argv );
			}
		}

		/* Number of colors */
		else if (strstr( *argv, "-col") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				ncolors = atoi( *argv );
			}
		}

		/* Histogram maximum */
		else if (strstr( *argv, "-max") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				histomax = atoi( *argv );
			}
		}

		/* Jump N bytes (header) */
		else if ( !strcmp( *argv, "-j") || !strcmp( *argv, "--jump" ) ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				jump = atoi( *argv );
			}
		}

		/* xaxis: # of elements in x */
		else if ( !strcmp( *argv, "-x") || !strcmp( *argv, "--xaxis" ) ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				in_nx = atoi( *argv );
			}
		}

		/* rotate */
		else if ( !strcmp( *argv, "--rotate" ) ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				if ( strstr( *argv, "+" ) )
					rotate = 270;
				else if ( strstr(*argv, "-" ))
					rotate = 90;
				else
					rotate = atoi( *argv );
			}
		}

		/* Use mar345 equivalent format on output */
		else if ( !strcmp( *argv, "-u") || !strcmp( *argv, "--use345" ) ) {
			use345 = 1;
		}

		/* Flip image */
		else if ( !strcmp( *argv, "--flip" ) ) {
			flip= 1;
		}

		/* yaxis: # of elements in y */
		else if ( !strcmp( *argv, "-y") || !strcmp( *argv, "--yaxis" ) ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				in_ny = atoi( *argv );
			}
		}

		/* Force input type */
		else if ( !strcmp( *argv, "-type" ) ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				strcpy( ipform, *argv );
			}
		}

		/* Histogram minimum */
		else if (strstr( *argv, "-min") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				histomin = atoi( *argv );
			}
		}

		/* Output directory */
		else if (!strcmp( *argv, "-o") ) {
			argc--;
			if (argc<=0)
				usage();
			else {  
				argv++;	
				strcpy( outdir, *argv );
			}
		}

		/* Recursive directory search */
		else if (!strcmp( *argv, "-r") || !strcmp( *argv, "--recursive" ) ) {

			recursive = 1;
		}

		/* Force byteswap */
		else if (!strcmp( *argv, "-s") || !strcmp( *argv, "--swap" ) ) {
			force_swap = 1;
		}

		/* Logarithmic scaling */
		else if (!strcmp( *argv, "-log") ) {
			log_scale = 1;
		}

		/* Rainbow colorscheme */
		else if (strstr( *argv, "-rain") ) {
			color_scheme = RAINBOW;
		}

		/* Blue colorscheme */
		else if (!strcmp( *argv, "-blue") ) {
			color_scheme = BLUE;
		}

		/* List */
		else if (!strcmp( *argv, "-l") || !strcmp( *argv, "--list")) {
			list = 1;
		}

		/* Force */
		else if (!strcmp( *argv, "-f") || !strcmp( *argv, "--force")) {
			force = 1;
		}

		/* Delete input file */
		else if (!strcmp( *argv, "-d") || !strcmp( *argv, "--delete")) {
			keep_input = 0;
		}
		/* Verbose */
		else if (!strcmp( *argv, "-v") || !strcmp( *argv, "--verbose")) {
			verbose++;
		}

		/* Quit */
		else if (!strcmp( *argv, "-q") || !strcmp( *argv, "--quiet")) {
			verbose = 0;
		}

		/* Help */
		else if (!strcmp( *argv, "-h") || !strcmp( *argv, "--help")) {
			usage();
		}

		else {
			strcpy( file, *argv );
			for ( i=0; i<=strlen(file); i++ ) 
				files[nfiles][i] = file[i];
			nfiles++;
			if ( nfiles == MAX_FILES ) break;
		}
	}

	if ( list ) verbose++;

	/* Input file RAW16 with defined size ? */
	if ( in_nx != in_ny ) {
		/* Is nx=ny ? */
		if ( in_nx > 0 && in_ny == 0 ) in_ny = in_nx;
		if ( in_ny > 0 && in_nx == 0 ) in_nx = in_ny;
	}

	if ( strlen( opform ) < 3 ) usage();

	/* Parse output format */
	for ( i=0; i<strlen(opform); i++ ) opform[i] = tolower( opform[i] );
	for ( i=0; i<strlen(ipform); i++ ) ipform[i] = tolower( ipform[i] );
	
	if      ( strstr(opform, "pck345" ) || strstr(opform, "mar345" ) ) 
		opf = MAR345;
	else if ( strstr(opform, "pck300" ) || ( strstr(opform, "pck") && strlen( opform ) < 4 ) )
		opf = PCK;
	else if ( strstr(opform, "mar300" ) || strstr( opform, "image" ) )
		opf = IMAGE;
	else if ( strstr(opform, "raw8" ) )
		opf = RAW8;
	else if ( strstr(opform, "raw16" ) )
		opf = RAW16;
	else if ( strstr(opform, "raw32" ) )
		opf = RAW32;
	else if ( strstr(opform, "tif" ) )
		opf = TIFF;
	else if ( strstr(opform, "ascii" ) )
		opf = ASCII;
	else if ( strstr(opform, "cif" ) )
		opf = IMG_CIF;
	else if ( strstr(opform, "cbf" ) )
		opf = IMG_CBF;
	else if (opf < 0 ) {
		printf("%s: invalid output format '%s'!\n",prg,opform );
		usage();
	} 

	if ( strlen ( ipform ) > 2  ) {
		if      ( strstr(ipform, "pck345" ) || strstr(ipform, "mar345" ) ) 
			ipf = MAR345;
		else if ( strstr(ipform, "pck300" ) || ( strstr(ipform, "pck") && strlen( ipform ) < 4 ) )
			ipf = PCK;
		else if ( strstr(ipform, "mar300" ) || strstr( ipform, "image" ) )
			ipf = IMAGE;
		else if ( strstr(ipform, "ccd" ) )
			ipf = CCD;
		else if ( strstr(ipform, "raw32" ) )
			ipf = RAW32;
		else if ( strstr(ipform, "raw8" )  )
			ipf = RAW8;
		else if ( strstr(ipform, "raw" )  ) 
			ipf = RAW16;
		else if ( strstr(ipform, "img" )  || strstr( ipform, "IMG") )
			ipf = STER1536;
		else if (  strstr(ipform, "ascii" ) || strstr(ipform, "tiff" ) ) {
			printf("%s: invalid input format '%s'!\n",prg,opform );
			usage();
		}
		else 
			ipf = 0;
		if ( ipf ) force_ipf = ipf;
	}

	if ( force_ipf == RAW16 || force_ipf == RAW32 || force_ipf == RAW8 ) {
		if ( in_nx == 0 ) {
			printf("%s: When forcing output type also give -x!\n",prg);
			usage();
		}
	}

	if ( (opf == RAW32 || opf == IMG_CIF || opf == IMG_CBF) && ( addvalue>0 || scale_factor > 0.f) ) {
		printf("%s: 32-bit output not allowed for scaling and/or adding!\n",prg);
		exit(0);
	}	
 
    	/* 
	 * Log the time to the errorlog for startup 
	 */

	cpu1 = clock();
	cpu2 = time(NULL);

    	time( &tick );

	if ( verbose ) {
        fprintf( stdout, " ********************************************************\n\n");
        fprintf( stdout, "      Program   : MARCVT = mar image format conversion\n");
        fprintf( stdout, "      Version   : %s  (%s)\n",VERSION,__DATE__);
        fprintf( stdout, "      Author    : Claudio Klein, mar research\n");
        fprintf( stdout, "      Started on: %s" ,(char *)ctime( &tick ) );
	if ( strlen( outdir ) > 0 )
        fprintf( stdout, "      Output in : %s\n" ,outdir );
        fprintf( stdout, "\n");
        fprintf( stdout, " ********************************************************\n\n");
	}

	if ( list ) {
		printf("%s: List of mar image files\n",prg);
		printf("===============================\n\n");
	}

	/************************************************
	 * Do the work
	 ***********************************************/

	Work();

	/************************************************
	 * Print ...
	 ***********************************************/
	if ( list > 0 )
	fprintf( stdout, "%s: No. of mar images found: %d\n", prg,list-1);

	cpu1 = clock();
	cpu2 = time(NULL) - cpu2;
	
	/* All done */
	exit( 0 );

}


/******************************************************************
 * Function: usage
 ******************************************************************/
void
usage()
{
static char *help[] = {
	 " -d   --delete        delete input file after succesful conversion",
	 " -f   --force         force overwrite of output file and links",
	 "      --flip          flip image on output",
	 "      --rotate N      rotate image counter-/clockwise by 90 deg.",
	 "                      N = '+', '270', '-' or '90'",
	 "      --add    N      add N to all pixels in image",
	 "      --scale  S      apply factor F to all pixels in image",
	 " -h   --help          give this help",
	 " -l   --list          list files only, no output",
	 " -r   --recursive     operate recursively on directories",
	 " -s   --swap          force swapping of bytes in input file",
	 " -v   --verbose       verbose mode (default)",
	 " -j   --jump   N      skip first N bytes of i/p-file",
	 " -x   --xaxis  NX     # of elements of fast varying axis (i/p-file)",
	 " -y   --yaxis  NY     # of elements of fast varying axis (i/p-file)",
	 " -type TYPE           force input of type TYPE: see -out",
	 " -o   DIR 		use directory 'DIR' for image output",
	 " -cut XXXX		cutoff ccd images to XXXX pixels (e.g. 2000)",
	 " -out FORMAT          output format (compulsory)",
	 " -mar345              ==> -out mar345",
	 " -pck                 ==> -out pck",
	 " -image               ==> -out image",
	 " -raw8                ==> -out raw8",
	 " -raw16               ==> -out raw16",
	 " -raw32               ==> -out raw32",
	 " -ascii               ==> -out ascii",
	 " -tiff                ==> -out tiff",
	 " -cif                 ==> -out cif",
	 " -cbf                 ==> -out cbf",
	 " -use345              when dealing with non-mar345 images, inflate to",
	 "                      next available mar345 image size",
	 " file...      	files to convert",
	 " ",
	 " Additional TIFF output options:",
	 " -max MAX             pixel > MAX = black ",
	 " -min MIN     	pixel < MIN = white",
	 " -colors N    	use N colors (default: 256)",
/*
	 " -log                 apply logarithmic scaling",
*/
	 " -rainbow             apply rainbow color scheme",
	 " -blue                apply blue color scheme",
	 NULL };
char **p=help;

	printf("\n");
	printf(" %s: Version %s (%s) by Claudio Klein\n",prg,VERSION,__DATE__);
	printf(" Usage:  %s [-dfhlrsv] [-o DIR] [-out FORMAT] file ...\n",prg);
	printf("         Valid output formats: 'mar345', 'pck', 'image',\n");
	printf("                               'ascii',  'tiff',\n");
	printf("                               'cif'    (ASCII encoded data)\n");
	printf("                               'cbf'    (BINARY endoced data)\n");
	printf("                               'raw8'   ( 8-bit array)\n");
	printf("                               'raw16'  (16-bit array)\n");
	printf("                               'raw32'  (32-bit array)\n");
	printf("         Valid  input formats: 'mar345', 'pck', 'image', 'ccd'\n");
	printf("                               'raw16', ' raw32'\n");
	printf(" For 'raw16' input, also -x, -y and -j may be given.\n");
	printf(" For 'cif' and 'cbf' output, CBF_PACKED compression is used.\n");
	printf(" For 'cif' and 'cbf' input, all types of compression are handled.\n");

	while (*p) 
		printf("          %s\n", *p++);

	printf("\n");

	exit(0);
}
