/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

/*
$Id: uMatrix.cc,v 1.10 1996/06/17 15:36:24 weber Exp $
$Author: weber $
$Source: /home/weber/work/uMatrix/RCS/uMatrix.cc,v $
$Revision: 1.10 $
$Log: uMatrix.cc,v $
// Revision 1.10  1996/06/17  15:36:24  weber
// Added Gnu library public license header
//
// Revision 1.9  1996/04/19  16:55:15  weber
// Removed operator/(scalar,uMatrix) because this is stupid and
// illogical.
//
// Revision 1.8  1996/03/13  15:25:12  weber
// Added a max and min fucntion for matrices
//
// Revision 1.7  1996/01/29  20:43:21  weber
// Added a uRandom function to fill a matrix with random numbers
//
// Revision 1.6  1995/10/03  19:03:12  weber
// Changed the norm( uMatrix<type> ) function to use pointer arithmetic.
//
// Revision 1.5  1995/09/29  02:34:30  weber
// Optimized the real(), imag() and conj() functions for Complex and
// fComplex data types. No use is made of the libg++ routines and pointer
// arithmetic is used to index the matrices.
//
// Revision 1.4  1995/09/28  04:18:42  weber
// Added BLAS based matrix multiplies for the float, double, fComplex and
// Complex data types. The use of BLAS matrix multiply stuff is enabled
// by defining uMatrix_use_BLAS_matrix_multiply at compile time. I have
// also made sure that a templated version of the pointer based matrix
// multiply exists for all the other data types and NOT for the 4 types
// mentioned here. This has resulted in a speed improvement by a factor
// of 25 or so over the old index based matrix multiply.
//
// Revision 1.3  1995/09/26  19:20:44  weber
// Modified the conj(uMatrix<Complex>) function to use pointers as a
// speed experiment for pqnn. The mod is functionally identical to the
// previous version
//
// Revision 1.2  1995/09/21  17:09:18  weber
// Added RCS headers
//

   Implementation of all sorts of stuff that cannot be templated in
   uMatrix.h. This includes casts, conjugates, transposes etc. Some
   of these routines may seem a bit thick (like conjugate a float
   type) but this allows easy templating of functions etc. that
   would otherwise require explicit overloaded versions to be implemented.
   This stuff replaces most of the stuff in uMatUtils which will now
   become obsolete.

*/

#include "nMatrix.h"

// #ifdef uMatrix_use_BLAS_matrix_multiply
#include "lapack.h"
// #endif

#ifdef explicit_instantiations

// We instantiate all the types of matrix template we anticipate here.

#define Instantiate(Type) \
template class uMatrix<Type>;\
template ostream& operator<<(ostream& outStream, const uMatrix<Type>& data);\
template istream& operator>>(istream& inStream, uMatrix<Type>& data);\
template uMatrix<Type> operator+( const uMatrix<Type>& data1,const uMatrix<Type>& data2 );\
template uMatrix<Type> operator-( const uMatrix<Type>& data1,const uMatrix<Type>& data2 );\
template uMatrix<Type> operator-( const uMatrix<Type>& data1);\
template uMatrix<Type> operator*( const uMatrix<Type>& data2, const Type data1 );\
template uMatrix<Type> operator*( const Type data1, const uMatrix<Type>& data2 );\
template uMatrix<Type> operator+( const uMatrix<Type>& data2, const Type data1 );\
template uMatrix<Type> operator+( const Type data1, const uMatrix<Type>& data2 );\
template uMatrix<Type> operator-( const uMatrix<Type>& data2, const Type data1 );\
template uMatrix<Type> operator-( const Type data1, const uMatrix<Type>& data2 );\
template uMatrix<Type> operator/( const uMatrix<Type>& data2, const Type data1 );\
template uMatrix<Type> transpose( const uMatrix<Type>& data );\
template uMatrix<Type>  vect( const uMatrix<Type>& data );\
template void zeroPad(  uMatrix<Type>& Zp, const uMatrix<Type>& data);\
template class uVector< uMatrix<Type> >;  

// Here we invoke the above macro to instantiate the templates we need

Instantiate(char)
Instantiate(unsigned char)
Instantiate(int)
Instantiate(unsigned int)
Instantiate(short int)
Instantiate(unsigned short int)
Instantiate(long int)
Instantiate(unsigned long int)
Instantiate(float)
Instantiate(double)
Instantiate(Complex)
Instantiate(fComplex)


// We instantiate the matrix multiply operators separately because some
// of these are overloaded and implemented using the BLAS



#define declareScalar_X_Matrix(Ts,Tm,Tr)                            \
 uMatrix<Tr>  operator*( const uMatrix<Tm>& data2, const Ts data1 ) \
  {                                                                 \
   uMatrix<Tr> temp( data2.rows(), data2.columns() );               \
   Tr *d = temp.address();					    \
   Tm *e = data2.address();					    \
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )	    \
   {								    \
      *d++ = data1* (*e++) ;				            \
   }								    \
   temp.markAsTemporary();					    \
   return temp;                                                     \
}                                                                   \
 uMatrix<Tr>  operator*( const Ts data1, const uMatrix<Tm>& data2 ) \
 {                                                                  \
   uMatrix<Tr> temp( data2.rows(), data2.columns() );               \
   Tr *d = temp.address();					    \
   Tm *e = data2.address();					    \
   for ( int i = 0; i < data2.rows()*data2.columns(); i++ )	    \
   {								    \
      *d++ = data1* (*e++) ;				            \
   }   						                    \
   temp.markAsTemporary();					    \
   return temp;                                                     \
}					                            
 
declareScalar_X_Matrix(double , Complex, Complex) 
declareScalar_X_Matrix(float  , Complex, Complex)
declareScalar_X_Matrix(float  , fComplex, fComplex)
declareScalar_X_Matrix(double  , fComplex, fComplex)
declareScalar_X_Matrix(Complex  , double , Complex)
declareScalar_X_Matrix(Complex  , float ,  Complex)
declareScalar_X_Matrix(fComplex  , double , Complex)
declareScalar_X_Matrix(fComplex  , float , fComplex);




#define InstantiateMM(Type)\
template uMatrix<Type> operator*( const uMatrix<Type>& data1, const uMatrix<Type>& data2 );

// template uMatrix<Type> operator*( const uMatrix<Type>& data2 , const uMatrix<Type>& data1);

InstantiateMM(char)
InstantiateMM(unsigned char)
InstantiateMM(int)
InstantiateMM(unsigned int)
InstantiateMM(short int)
InstantiateMM(unsigned short int)
InstantiateMM(long int)
InstantiateMM(unsigned long int)
/*#ifndef uMatrix_use_BLAS_matrix_multiply
InstantiateMM(float)
InstantiateMM(double)
InstantiateMM(Complex)
InstantiateMM(fComplex)
#endif */
#endif

// Overloaded matrix multiply operators
// #ifdef uMatrix_use_BLAS_matrix_multiply
uMatrix<float> operator*( const uMatrix<float>& A, 
			  const uMatrix<float>& B )
{
   if ( A.columns() != B.rows() )
   {
      cerr << "uMatrix<float> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<float> C(A.rows(),B.columns());

   int lda = A.rows();
   int ldb = B.rows(); 
   int ldc = C.rows();
   int m = A.rows();
   int n = B.columns();
   int k = B.rows();
   
   char transa='N', transb='N';
   float alpha = 1.0;
   float beta = 0.0;

   sgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
   
   C.markAsTemporary();
   return C;
}

uMatrix<double> operator*( const uMatrix<double>& A, 
			   const uMatrix<double>& B )
{
   if ( A.columns() != B.rows() )
   {
      cerr << "uMatrix<double> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<double> C(A.rows(),B.columns());

   int lda = A.rows();
   int ldb = B.rows(); 
   int ldc = C.rows();
   int m = A.rows();
   int n = B.columns();
   int k = B.rows();
   
   char transa='N', transb='N';
   double alpha = double(1.0);
   double beta = double(0.0);

   dgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
   
   C.markAsTemporary();
   return C;
}

uMatrix<fComplex> operator*( const uMatrix<fComplex>& A, 
			     const uMatrix<fComplex>& B )
{
   if ( A.columns() != B.rows() )
   {
      cerr << "uMatrix<fComplex> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<fComplex> C(A.rows(),B.columns());

   int lda = A.rows();
   int ldb = B.rows(); 
   int ldc = C.rows();
   int m = A.rows();
   int n = B.columns();
   int k = B.rows();
   
   char transa='N', transb='N';
   fComplex alpha(1.0);
   fComplex beta(0.0);

   cgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
   
   C.markAsTemporary();
   return C;
}

uMatrix<Complex> operator*( const uMatrix<Complex>& A, 
			    const uMatrix<Complex>& B )
{
   if ( A.columns() != B.rows() )
   {
      cerr << "uMatrix<Complex> operator*(): Arguments have incompatible sizes\n";
      abort();
   }
   uMatrix<Complex> C(A.rows(),B.columns());

   int lda = A.rows();
   int ldb = B.rows(); 
   int ldc = C.rows();
   int m = A.rows();
   int n = B.columns();
   int k = B.rows();
   
   char transa='N', transb='N';
   Complex alpha(1.0);
   Complex beta(0.0);

   zgemm_( &transa, &transb, &m, &n, &k, &alpha, A.address(), &lda, 
           B.address(), &ldb, &beta, C.address(), &ldc );
   
   C.markAsTemporary();
   return C;
}

// #endif













/*
#ifdef explicit_instantiations
// Random number generator function instantiation
#define intantiateRandom(Type)\
   template void uRandom(uMatrix<Type>& A, Random & rng );

intantiateRandom(char)
intantiateRandom(unsigned char)
intantiateRandom(int)
intantiateRandom(unsigned int)
intantiateRandom(short int)
intantiateRandom(unsigned short int)
intantiateRandom(long int)
intantiateRandom(unsigned long int)
intantiateRandom(float)
intantiateRandom(double)

#endif
  */
// Override the templating for complex types
	/*	
void uRandom(uMatrix<Complex>& A,  Random & rng )
{
   for ( int i = 0; i < A.rows(); i++ )
   {
      for ( int j = 0; j < A.columns(); j++ )
      {
	 A(i,j) = Complex(rng(),rng());
      }
   }
}
		     
void uRandom(uMatrix<fComplex>& A, Random & rng )
{
   for ( int i = 0; i < A.rows(); i++ )
   {
      for ( int j = 0; j < A.columns(); j++ )
      {
	 A(i,j) = fComplex(rng(),rng());
      }
   }
}
*/
// Max and min functions for matrices. We do not have a version for complex
// types for obvious reasons. I still hate macros but...
	
#define MaxMinMacro(Type1)                                   \
Type1 max( const uMatrix<Type1>& data1)                      \
{                                                            \
   Type1 maxV = data1(0,0);                                  \
   Type1 *s = data1.address();                               \
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )  \
   {                                                         \
      maxV = maxV > *s ? maxV : *s;                          \
      s++;                                                   \
   }                                                         \
   return maxV;                                              \
}                                                            \
Type1 min( const uMatrix<Type1>& data1)                      \
{                                                            \
   Type1 minV = data1(0,0);                                  \
   Type1 *s = data1.address();                               \
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )  \
   {                                                         \
      minV = minV < *s ? minV : *s;                          \
      s++;                                                   \
   }                                                         \
   return minV;                                              \
}                                                            \

MaxMinMacro(char);
MaxMinMacro(unsigned char);
MaxMinMacro(int);
MaxMinMacro(unsigned int);
MaxMinMacro(short int);
MaxMinMacro(unsigned short int);
MaxMinMacro(long int);
MaxMinMacro(unsigned long int);
MaxMinMacro(float);
MaxMinMacro(double);

#undef MinMaxMacro

// Casting functions. Note that I really do dispise macros but this appears
// to be the only way I can prevent attempts to cast Complex types to
// real types. There is a similar ugly in uMatrix.h for the headers

#define castMacro(fName,Type1,Type2)                         \
uMatrix<Type1> fName( const uMatrix<Type2>& data1)           \
{                                                            \
   uMatrix<Type1> temp(data1.rows(),data1.columns());        \
   Type1 *d = temp.address();                                \
   Type2 *s = data1.address();                               \
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )  \
   {                                                         \
      *d++ = (Type1)(*s++);                                  \
   }                                                         \
   temp.markAsTemporary();                                   \
   return temp;                                              \
}                                                            \

castMacro(floatCast,float,char)
castMacro(floatCast,float,unsigned char)
castMacro(floatCast,float,int)
castMacro(floatCast,float,unsigned int)
castMacro(floatCast,float,short int)
castMacro(floatCast,float,unsigned short int)
castMacro(floatCast,float,long int)
castMacro(floatCast,float,unsigned long int)
castMacro(floatCast,float,float)   // Odd but it helps one or two boneheads
castMacro(floatCast,float,double)

castMacro(doubleCast,double,char)
castMacro(doubleCast,double,unsigned char)
castMacro(doubleCast,double,int)
castMacro(doubleCast,double,unsigned int)
castMacro(doubleCast,double,short int)
castMacro(doubleCast,double,unsigned short int)
castMacro(doubleCast,double,long int)
castMacro(doubleCast,double,unsigned long int)
castMacro(doubleCast,double,float) 
castMacro(doubleCast,double,double)

castMacro(intCast,int,char)
castMacro(intCast,int,unsigned char)
castMacro(intCast,int,int)
castMacro(intCast,int,unsigned int)
castMacro(intCast,int,short int)
castMacro(intCast,int,unsigned short int)
castMacro(intCast,int,long int)
castMacro(intCast,int,unsigned long int)
castMacro(intCast,int,float) 
castMacro(intCast,int,double)

castMacro(unsignedIntCast,unsigned int,char)
castMacro(unsignedIntCast,unsigned int,unsigned char)
castMacro(unsignedIntCast,unsigned int,int)
castMacro(unsignedIntCast,unsigned int,unsigned int)
castMacro(unsignedIntCast,unsigned int,short int)
castMacro(unsignedIntCast,unsigned int,unsigned short int)
castMacro(unsignedIntCast,unsigned int,long int)
castMacro(unsignedIntCast,unsigned int,unsigned long int)
castMacro(unsignedIntCast,unsigned int,float) 
castMacro(unsignedIntCast,unsigned int,double)

// long and short int types
castMacro(shortIntCast,short int,char)
castMacro(shortIntCast,short int,unsigned char)
castMacro(shortIntCast,short int,int)
castMacro(shortIntCast,short int,unsigned int)
castMacro(shortIntCast,short int,short int)
castMacro(shortIntCast,short int,unsigned short int)
castMacro(shortIntCast,short int,long int)
castMacro(shortIntCast,short int,unsigned long int)
castMacro(shortIntCast,short int,float) 
castMacro(shortIntCast,short int,double)

castMacro(unsignedShortIntCast,unsigned short int,char)
castMacro(unsignedShortIntCast,unsigned short int,unsigned char)
castMacro(unsignedShortIntCast,unsigned short int,int)
castMacro(unsignedShortIntCast,unsigned short int,unsigned int)
castMacro(unsignedShortIntCast,unsigned short int,short int)
castMacro(unsignedShortIntCast,unsigned short int,unsigned short int)
castMacro(unsignedShortIntCast,unsigned short int,long int)
castMacro(unsignedShortIntCast,unsigned short int,unsigned long int)
castMacro(unsignedShortIntCast,unsigned short int,float) 
castMacro(unsignedShortIntCast,unsigned short int,double)

castMacro(longIntCast,long int,char)
castMacro(longIntCast,long int,unsigned char)
castMacro(longIntCast,long int,int)
castMacro(longIntCast,long int,unsigned int)
castMacro(longIntCast,long int,short int)
castMacro(longIntCast,long int,unsigned short int)
castMacro(longIntCast,long int,long int)
castMacro(longIntCast,long int,unsigned long int)
castMacro(longIntCast,long int,float) 
castMacro(longIntCast,long int,double)

castMacro(unsignedLongIntCast,unsigned long int,char)
castMacro(unsignedLongIntCast,unsigned long int,unsigned char)
castMacro(unsignedLongIntCast,unsigned long int,int)
castMacro(unsignedLongIntCast,unsigned long int,unsigned int)
castMacro(unsignedLongIntCast,unsigned long int,short int)
castMacro(unsignedLongIntCast,unsigned long int,unsigned short int)
castMacro(unsignedLongIntCast,unsigned long int,long int)
castMacro(unsignedLongIntCast,unsigned long int,unsigned long int)
castMacro(unsignedLongIntCast,unsigned long int,float) 
castMacro(unsignedLongIntCast,unsigned long int,double)

// Character types
castMacro(charCast,char,char)
castMacro(charCast,char,unsigned char)
castMacro(charCast,char,int)
castMacro(charCast,char,unsigned int)
castMacro(charCast,char,short int)
castMacro(charCast,char,unsigned short int)
castMacro(charCast,char,long int)
castMacro(charCast,char,unsigned long int)
castMacro(charCast,char,float) 
castMacro(charCast,char,double)

castMacro(unsignedCharCast,unsigned char,char)
castMacro(unsignedCharCast,unsigned char,unsigned char)
castMacro(unsignedCharCast,unsigned char,int)
castMacro(unsignedCharCast,unsigned char,unsigned int)
castMacro(unsignedCharCast,unsigned char,short int)
castMacro(unsignedCharCast,unsigned char,unsigned short int)
castMacro(unsignedCharCast,unsigned char,long int)
castMacro(unsignedCharCast,unsigned char,unsigned long int)
castMacro(unsignedCharCast,unsigned char,float) 
castMacro(unsignedCharCast,unsigned char,double)

castMacro(fComplexCast,fComplex,char)
castMacro(fComplexCast,fComplex,unsigned char)
castMacro(fComplexCast,fComplex,int)
castMacro(fComplexCast,fComplex,unsigned int)
castMacro(fComplexCast,fComplex,short int)
castMacro(fComplexCast,fComplex,unsigned short int)
castMacro(fComplexCast,fComplex,long int)
castMacro(fComplexCast,fComplex,unsigned long int)
castMacro(fComplexCast,fComplex,float) 
castMacro(fComplexCast,fComplex,double)
castMacro(fComplexCast,fComplex,fComplex)

castMacro(ComplexCast,Complex,char)
castMacro(ComplexCast,Complex,unsigned char)
castMacro(ComplexCast,Complex,int)
castMacro(ComplexCast,Complex,unsigned int)
castMacro(ComplexCast,Complex,short int)
castMacro(ComplexCast,Complex,unsigned short int)
castMacro(ComplexCast,Complex,long int)
castMacro(ComplexCast,Complex,unsigned long int)
castMacro(ComplexCast,Complex,float) 
castMacro(ComplexCast,Complex,double)
castMacro(ComplexCast,Complex,Complex)


// Cast between complex types

#define castComplexMacro(fName,Type1,Type2)                  \
uMatrix<Type1> fName( const uMatrix<Type2>& data1)           \
{                                                            \
   uMatrix<Type1> temp(data1.rows(),data1.columns());        \
   Type1 *d = temp.address();                                \
   Type2 *s = data1.address();                               \
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )  \
   {                                                         \
      *d++ = Type1(real(*s),imag(*s));                       \
      s++;                                                   \
   }                                                         \
   temp.markAsTemporary();                                   \
   return temp;                                              \
}                                                            \


castComplexMacro(fComplexCast,fComplex,Complex)
castComplexMacro(ComplexCast,Complex,fComplex)


// Construct complex types from ordinal types

#define castToComplexMacro(fName,Type1,Type2)                \
uMatrix<Type1> fName( const uMatrix<Type2>& data1,           \
		      const uMatrix<Type2>& data2)           \
{                                                            \
   uMatrix<Type1> temp(data1.rows(),data1.columns());        \
   Type1 *d = temp.address();                                \
   Type2 *s1 = data1.address();                              \
   Type2 *s2 = data2.address();                              \
   for ( int i = 0; i < data1.rows()*data1.columns(); i++ )  \
   {                                                         \
      *d++ = Type1(*s1++,*s2++);                             \
   }                                                         \
   temp.markAsTemporary();                                        \
   return temp;                                              \
}                                                            \

castToComplexMacro(fComplexCast,fComplex,float)
castToComplexMacro(fComplexCast,fComplex,double)

castToComplexMacro(ComplexCast,Complex,float)
castToComplexMacro(ComplexCast,Complex,double)


// Stuff to handle complex data

/*
uMatrix<Complex> conj( const uMatrix<Complex>& data )
{
   uMatrix<Complex> A(data.rows(),data.columns());
   int i,j;
   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
	 A(i,j) = conj(data(i,j)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}
*/

// Conjugate using pointer arithmetic
uMatrix<Complex> conj( const uMatrix<Complex>& data )
{
   uMatrix<Complex> A(data.rows(),data.columns());
   double *a = (double *) A.address();
   double *d = (double *) data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a++ = *d++;
      *a++ = -(*d++);
   }
   A.markAsTemporary();
   return A;
}

uMatrix<fComplex> conj( const uMatrix<fComplex>& data )
{
   uMatrix<fComplex> A(data.rows(),data.columns());
   float *a = (float *) A.address();
   float *d = (float *) data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a++ = *d++;
      *a++ = -(*d++);
   }
   A.markAsTemporary();
   return A;
}

// This may seem insane but trust me. It will make templating fuctions
// designed for float or Complex types MUCH easier
uMatrix<float> conj( const uMatrix<float>& data )
{
   return data;
}

uMatrix<double> conj( const uMatrix<double>& data )
{
   return data;
}

uMatrix<double> real( const uMatrix<Complex>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   double *a = A.address();
   double *d = (double *) data.address();
   for ( int i = 0; i < data.rows() * data.columns(); i++ )
   {
      *a++ = *d;
      d += 2;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<double> imag( const uMatrix<Complex>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   double *a = A.address();
   double *d = (double *) data.address();
   d += 1;
   for ( int i = 0; i < data.rows() * data.columns(); i++ )
   {
      *a++ = *d;
      d += 2;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> real( const uMatrix<fComplex>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   float *a = A.address();
   float *d = (float *) data.address();
   for ( int i = 0; i < data.rows() * data.columns(); i++ )
   {
      *a++ = *d;
      d += 2;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> imag( const uMatrix<fComplex>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   float *a = A.address();
   float *d = (float *) data.address();
   d += 1;
   for ( int i = 0; i < data.rows() * data.columns(); i++ )
   {
      *a++ = *d;
      d += 2;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> real( const uMatrix<float>& data )
{
   return data;
}

uMatrix<float> imag( const uMatrix<float>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   A = 0.0;
   A.markAsTemporary();
   return A;
}


uMatrix<double> real( const uMatrix<double>& data )
{
   return data;
}

uMatrix<double> imag( const uMatrix<double>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   A = 0.0;
   A.markAsTemporary();
   return A;
}


uMatrix<double> norm( const uMatrix<Complex>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   double *a = A.address();
   double *d = (double *) data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a = *d * *d;
      d++;
      *a++ += *d * *d;
      d++;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> norm( const uMatrix<fComplex>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   float *a = A.address();
   float *d = (float *) data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a = *d * *d;
      d++;
      *a++ += *d * *d;
      d++;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> norm( const uMatrix<float>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   float *a = A.address();
   float *d = data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a++ = *d * *d;
      d++;
   }
   A.markAsTemporary();
   return A;
}

uMatrix<double> norm( const uMatrix<double>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   double *a = A.address();
   double *d = data.address();
   for ( int i = 0; i < data.rows()*data.columns(); i++ )
   {
      *a++ = *d * *d;
      d++;
   }
   A.markAsTemporary();
   return A;
}


uMatrix<double> abs( const uMatrix<Complex>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   int i,j;
   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
	 A(i,j) = abs(data(i,j)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> abs( const uMatrix<fComplex>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   int i,j;
   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
	 A(i,j) = abs(data(i,j)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}


uMatrix<float> abs( const uMatrix<float>& data )
{
   uMatrix<float> A(data.rows(),data.columns());
   int i,j;
   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
	 A(i,j) = fabs(data(i,j)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<double> abs( const uMatrix<double>& data )
{
   uMatrix<double> A(data.rows(),data.columns());
   int i,j;
   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
	 A(i,j) = fabs(data(i,j)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}



// Hermitian (conjugate) transposes

uMatrix<Complex> hTranspose( const uMatrix<Complex>& data )
{
   uMatrix<Complex> A(data.columns(),data.rows());
   int i,j;
   for ( i = 0; i < data.columns(); i++ )
   {
      for ( j = 0; j < data.rows(); j++ )
      {
	 A(i,j) = conj(data(j,i)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<fComplex> hTranspose( const uMatrix<fComplex>& data )
{
   uMatrix<fComplex> A(data.columns(),data.rows());
   int i,j;
   for ( i = 0; i < data.columns(); i++ )
   {
      for ( j = 0; j < data.rows(); j++ )
      {
	 A(i,j) = conj(data(j,i)); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<float> hTranspose( const uMatrix<float>& data )
{
   uMatrix<float> A(data.columns(),data.rows());
   int i,j;
   for ( i = 0; i < data.columns(); i++ )
   {
      for ( j = 0; j < data.rows(); j++ )
      {
	 A(i,j) = data(j,i); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<double> hTranspose( const uMatrix<double>& data )
{
   uMatrix<double> A(data.columns(),data.rows());
   int i,j;
   for ( i = 0; i < data.columns(); i++ )
   {
      for ( j = 0; j < data.rows(); j++ )
      {
	 A(i,j) = data(j,i); // simple but slow.
      }
   }
   A.markAsTemporary();
   return A;
}



/* 
   Normalize a matrix
*/

uMatrix<float> normalize( const uMatrix<float>& data, float minv, float maxv )
{
   float Max = data(0,0);
   float Min = data(0,0);
   int i,j;

   uMatrix<float> A(data.rows(),data.columns());
   if ( minv >= maxv )
   {
      cout << "normalize<float> error: parameter min > max\n";
      abort();
   }

   Max = max( data );
   Min = min( data );

   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
         A(i,j) =  (maxv-minv)*(data(i,j) - Min)/fabs(Max - Min) + minv;
      }
   }
   A.markAsTemporary();
   return A;
}

uMatrix<double> normalize( const uMatrix<double>& data, double minv, 
			   double maxv)
{
   double Max = data(0,0);
   double Min = data(0,0);
   int i,j;

   uMatrix<double> A(data.rows(),data.columns());
   if ( minv >= maxv )
   {
      cout << "normalize<double> error: parameter min > max\n";
      abort();
   }

   Max = max( data );
   Min = min( data );

   for ( i = 0; i < data.rows(); i++ )
   {
      for ( j = 0; j < data.columns(); j++ )
      {
         A(i,j) =  (maxv-minv)*(data(i,j) - Min)/fabs(Max - Min) + minv;
      }
   }
   A.markAsTemporary();
   return A;
}



template uMatrix<Complex> JoinV( uMatrix<Complex> const  &A,uMatrix<Complex> const &B);
template uMatrix<Complex> JoinH( uMatrix<Complex> const &A,uMatrix<Complex> const &B);



template uMatrix<double> JoinV( uMatrix<double> const  &A,uMatrix<double> const &B);
template uMatrix<double> JoinH( uMatrix<double> const &A,uMatrix<double> const &B);


